#pragma once


template<typename T>
class UniquePtr
{
	T*   data;
public:
	UniquePtr()
		: data(nullptr)
	{}
	// Explicit constructor
	explicit UniquePtr(T* data)
		: data(data)
	{}
	~UniquePtr()
	{
		delete data;
	}

	// Constructor/Assignment that binds to nullptr
	// This makes usage with nullptr cleaner
	UniquePtr(std::nullptr_t)
		: data(nullptr)
	{}
	UniquePtr& operator=(std::nullptr_t)
	{
		reset();
		return *this;
	}

	// Constructor/Assignment that allows move semantics
	UniquePtr(UniquePtr&& moving) noexcept
	{
		moving.swap(*this);
	}
	UniquePtr& operator=(UniquePtr&& moving) noexcept
	{
		moving.swap(*this);
		return *this;
	}

	// Constructor/Assignment for use with types derived from T
	template<typename U>
	UniquePtr(UniquePtr<U>&& moving)
	{
		UniquePtr<T>   tmp(moving.release());
		tmp.swap(*this);
	}
	template<typename U>
	UniquePtr& operator=(UniquePtr<U>&& moving)
	{
		UniquePtr<T> tmp(moving.release());
		tmp.swap(*this);
		return *this;
	}

	// Remove compiler generated copy semantics.
	UniquePtr(UniquePtr const&) = delete;
	UniquePtr& operator=(UniquePtr const&) = delete;

	// Const correct access owned object
	T* operator->() const { return data; }
	T& operator*()  const { return *data; }

	// Access to smart pointer state
	T* get()                 const { return data; }
	explicit operator bool() const { return data != nullptr; }

	// Modify object state
	T* release() noexcept
	{
		T* result = nullptr;
		std::swap(result, data);
		return result;
	}
	void swap(UniquePtr& src) noexcept
	{
		std::swap(data, src.data);
	}
	void reset()
	{
		T* tmp = release();
		delete tmp;
	}
};

