// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Tokenizer.cpp
//
////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Tokenizer.h"


bool Tokenizer::ReadNext() 
{
    if(!m_NextToken.IsEmpty()) 
    {
        m_Token = m_NextToken;
        m_NextToken.Empty();
        return true;
    }

    const TCHAR doubleQuote = _T('\"');
    const TCHAR singleQuote = _T('\'');

    m_Token.Empty();
    m_IsDoubleQuoted = false;
    m_IsSingleQuoted = false;
    
    while(m_Pos < m_strLength)
    {
        TCHAR ch = m_pszStr[m_Pos++];

        switch(m_State) 
        {
        case stPlain:

            if(ch == m_Comment && m_Comment != 0) {
                m_State = stComment;
                if(!m_Token.IsEmpty()) return true;
            }
            else if(ch == doubleQuote) {
                m_State = stDoubleQuoted;
                if(!m_Token.IsEmpty()) return true;         
            }
            else if(ch == singleQuote) {
                m_State = stSingleQuoted;
                if(!m_Token.IsEmpty()) return true;      
            }
            else if(m_Delimiters.Find(ch) > -1) {
                if(!m_Token.IsEmpty()) return true;               
            }
            else if(m_Delimiters_preserve.Find(ch) > -1) {

                if(!m_Token.IsEmpty()) {

                    m_NextToken = CString(ch);  // delimiter will be the next token
                    return true;
                }
                else 
                {
                    m_Token = CString(ch);
                    return true;
                }
            }
            else {
                m_Token += ch;
            }

            break;
    
        case stDoubleQuoted:

            if(ch == doubleQuote && !m_IsEscaped) {
                m_State = stPlain;
                if(!m_Token.IsEmpty()) {
                    m_IsDoubleQuoted = true;
                    return true; 
                }
            }
            else if(ch == m_Esc && !m_IsEscaped) {
                m_IsEscaped = true;
            }
            else {
                m_Token += ch;
                m_IsEscaped = false;
            }

            break;

        case stSingleQuoted:

            if(ch == singleQuote && !m_IsEscaped) {
                m_State = stPlain;
                if(!m_Token.IsEmpty()) {
                    m_IsSingleQuoted = true;
                    return true; 
                }        
            }
            else if(ch == m_Esc && !m_IsEscaped) {
                m_IsEscaped = true;
            }
            else {
                m_Token += ch;
                m_IsEscaped = false;
            }

            break;

        case stComment:

            if(ch == '\r' || ch == '\n') 
            {
                m_State = stPlain;
                --m_Pos;  // we need to reconsider the line separator char in case it us used as a preserving delimiter
            }

            break;
        }
    }
    
    if(m_State == stPlain && !m_Token.IsEmpty())
    {
        return true;
    }

    return false;
}

/*

enum STATE {
    stPlain = 0,
    stDoubleQuoted,
    stSingleQuoted,
    stComment
};

enum QUOTED {
    NotQuoted = 0,
    DoubleQuoted,
    SingleQuoted
};

void addToken(CTokens& result, CString& token, QUOTED quoted) 
{
    if(!token.IsEmpty()) 
    {
        ANNOTATED_TOKEN theToken;
        theToken.token = token;
        theToken.isSingleQuoted = (quoted == SingleQuoted);
        theToken.isDoubleQuoted = (quoted == DoubleQuoted);

        result.Add(theToken);
        token.Empty();
    }
}

void tokenize ( const CString& str, 
                      CTokens& result,
			    const CString& delimiters, 
                const CString& delimiters_preserve,
			    const TCHAR    esc, 
                const TCHAR    comment )
{
	if (result.GetSize() > 0) result.RemoveAll();

    const TCHAR doubleQuote = _T('\"');
    const TCHAR singleQuote = _T('\'');

	CString token;		// string buffer for the token

    STATE state = stPlain;
    bool isEscaped = false;

    for(int i = 0; i < str.GetLength(); ++i) 
    {
        TCHAR ch = str[i];

        switch(state) 
        {
        case stPlain:

            if(ch == comment && comment != 0) {
                state = stComment;
                addToken(result, token, NotQuoted);
            }
            else if(ch == doubleQuote) {
                state = stDoubleQuoted;
                addToken(result, token, NotQuoted);         
            }
            else if(ch == singleQuote) {
                state = stSingleQuoted;
                addToken(result, token, NotQuoted);       
            }
            else if(delimiters.Find(ch) > -1) {
                addToken(result, token, NotQuoted);                
            }
            else if(delimiters_preserve.Find(ch) > -1) {
                addToken(result, token, NotQuoted);
                addToken(result, CString(ch), NotQuoted); // add the delimiter as new token
            }
            else {
                token += ch;
            }

            break;
    
        case stDoubleQuoted:

            if(ch == doubleQuote && !isEscaped) {
                state = stPlain;
                addToken(result, token, DoubleQuoted);         
            }
            else if(ch == esc && !isEscaped) {
                isEscaped = true;
            }
            else {
                token += ch;
                isEscaped = false;
            }

            break;

        case stSingleQuoted:

            if(ch == singleQuote && !isEscaped) {
                state = stPlain;
                addToken(result, token, SingleQuoted);         
            }
            else if(ch == esc && !isEscaped) {
                isEscaped = true;
            }
            else {
                token += ch;
                isEscaped = false;
            }

            break;

        case stComment:

            if(ch == '\r' || ch == '\n') 
            {
                state = stPlain;
                --i;  // we need to reconsider the line separator char in case it us used as a preserving delimiter
            }

            break;
        }
    }
    
    if(state == stPlain && !token.IsEmpty())
    {
        addToken(result, token, NotQuoted);
    }
}
*/































/*

void tokenize ( const CString& str, CStringArray& result, CByteArray& TokenQuoted,
			    const CString& delimiters, const CString& delimiters_preserve,
			    TCHAR quote, TCHAR esc, TCHAR comment )
{
	if (result.GetSize() > 0) result.RemoveAll();
	if (TokenQuoted.GetSize() > 0) TokenQuoted.RemoveAll();

	int   pos = 0; // the current position (char) in the string
	TCHAR ch  = 0; // buffer for the current character
	TCHAR delimiter = 0;	// the buffer for the delimiter char which will be added to the tokens 
							// if the delimiter is preserved
	bool quoted = false;	// indicator if there is an open quote
	bool token_complete = false; // indicates if the current token is
								 // ready to be added to the result vector
	CString token;		// string buffer for the token
	int len = str.GetLength();  // length of the input-string

	// for every char in the input-string
	while ( len > pos )
	{
		ch = str[pos];
		delimiter = 0;

		bool add_char = true;	// assume ch isn't a delimiter

	
		// ... if the delimiter is an escaped character
		bool escaped = false; // indicates if the next char is protected
		if ( esc != 0 ) // check if esc-char is provided
		{
			if ( ch == esc )
			{
				// get the escaped char
				++pos;
				if ( pos < len ) // if there are more chars left
				{
					ch = str[pos];   // get the next one
					add_char = true;  // add the escaped character to the token
				}
				else // cannot get any more characters
				{
					add_char = false; // don't add the esc-char
				}

				escaped = true;  // ignore the remaining delimiter checks
			}
		}


		if(comment != 0 && ch == comment && !escaped && !quoted) 
		{
			++pos;
			while ( len > pos ) 
			{
				ch = str[pos];
				if(ch == '\r' || ch == '\n')
					break;
				++pos;
			}
			continue;
		}

		
		// ... if the delimiter is a quote
		if ( quote != 0 && ch == quote && !escaped )
		{
			// if quote chars are provided and the char isn't protected
		
			if ( !quoted )  // if not quoted, set state to open quote and set the quote character
			{
				quoted   = true;				
				add_char = false;  // don't add the quote-char to the token

				// the start of a quotation should behave like a delimiter:
				if(!token.IsEmpty())
					result.Add( token );
				token = "";
			}
			else // if quote is open already
			{
				// close quote and reset the quote character
				quoted   = false;
				add_char = false; // don't add the quote-char to the token


				while(TokenQuoted.GetSize() < result.GetSize())
					TokenQuoted.Add(FALSE);

				// the end of a quotation should behave like a delimiter:
				result.Add( token );  // token may be empty!
				TokenQuoted.Add(TRUE);
				token = "";

				
				
			}
		}


		// ... if the delimiter isn't preserved
		if ( !delimiters.IsEmpty() && !escaped && !quoted )
		{
			// if a delimiter is provided and the char isn't protected by quote or escape char
			
			if ( delimiters.Find(ch) > -1 )
			{
				// if ch is a delimiter and the token string isn't empty then the token is complete
				if ( !token.IsEmpty() ) // BUGFIX: 2006-03-04
					token_complete = true;
				
				// don't add the delimiter to the token
				add_char = false;
			}
		}


		// ... if the delimiter is preserved - add it as a token
		bool add_delimiter = false;
		if ( !delimiters_preserve.IsEmpty() && !escaped && !quoted )
		{
			// if a delimiter which will be preserved is provided and the
			// char isn't protected by quote or escape char
			if ( delimiters_preserve.Find(ch) > -1)
			{
				// if ch is a delimiter and the token string isn't empty then the token is complete
				if ( !token.IsEmpty() ) // BUGFIX: 2006-03-04
					token_complete = true;
				
				add_char = false; // don't add the delimiter to the token

				// add the delimiter as a new token
				delimiter = ch;
				add_delimiter = true;
			}
		}


		// add the character to the token
		if ( add_char )
			token += ch;
	

		// add the token if it is complete
		if ( token_complete && !token.IsEmpty() )
		{
			result.Add( token );
			token = "";	// clear the contents
			token_complete = false; // build the next token
		}

		// add the delimiter
		if ( add_delimiter )
		{
			// the next token is the delimiter
			result.Add( CString(delimiter) );
		}

		// repeat for the next character
		++pos;
	} // while


	// add the final token
	if ( !token.IsEmpty() )
		result.Add( token );
	
	while(TokenQuoted.GetSize() < result.GetSize())
		TokenQuoted.Add(FALSE);

	
}

  */