// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// TextDisplay.h
//
////////////////////////////////////////////////////////////////////////////////


#if !defined(AFX_TEXTDISPLAY_H_)
#define AFX_TEXTDISPLAY_H_

#if _MSC_VER > 1000
#pragma once
#endif 

#include "HtmlConvert.h"
#include "MultiLineTip.h"

class CTextDisplay : public CWnd
{
public:
	CTextDisplay();
	virtual ~CTextDisplay();

public:
	BOOL IsStopChar(TCHAR s);
	
	BOOL Create(CWnd* pParent, CRect rcPos);
	CString GetText() const { return m_pszData; };
	void SetText(LPCTSTR szText, const std::vector<LinkInfo>* pLinks = NULL);
	void SelectAll();
	int GetLineFromPoint(POINT pt) const;
	int GetCharPosFromPoint(POINT pt, int* pCurPos = NULL);
	int LineFromChar(int nIndex) const;
	void EnsureSelectionVisible(BOOL bUpdate = TRUE);
	void GetSel(int& nStartChar, int& nEndChar) const
	{ ::SendMessage(m_hWnd, EM_GETSEL, (WPARAM)&nStartChar,(LPARAM)&nEndChar); }

	void SetSel(int nStartChar, int nEndChar, BOOL bNoScroll = FALSE)
	{ 
		SendMessage(EM_SETSEL, nStartChar, nEndChar);
		if (!bNoScroll)
		EnsureSelectionVisible(); 
	}

    CString GetSelectedText() const;
	
	void SetHoverLinkCursor(HCURSOR hCursor)
	{	m_hHoverLinkCursor = hCursor;	}	

	COLORREF m_TextColor;
	COLORREF m_BkgndColor;

	//{{AFX_VIRTUAL(CTextDisplay)
	public:
	virtual BOOL PreTranslateMessage(MSG* pMsg);
	//}}AFX_VIRTUAL

protected:
	void OpenHyperlink(CString stLink);
	void InitMouseWheel();
	
	void CopySelToClipboard();
	void UpdateSelection(POINT point);

	// TFC Start
	void CopyLinkAddress();
	void LinkAction(int nLink, BOOL bHyperlink = TRUE);

	CPoint	m_pointRClick;
	// TFC End

	int GetLinkFromPoint(POINT pt);
	int GetLineHeight();
	HFONT CreateLinkFont(HFONT Font);

	void ExtractLinks(bool clearLinks = false);
	static int CheckMailChar(TCHAR c);
	static TCHAR lcase(TCHAR c);
	static int FindNoCase(LPCTSTR szStr, LPCTSTR szFind, int nStrStart = 0);

	static LRESULT CALLBACK WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	BOOL RegisterWndClass();

	void OnFontChanged(BOOL bUpdate);
		
	void PageDown();
	void PageUp();
	void LineDown();
	void LineUp();
	void UpdateOnScroll(BOOL bRepaintWindow = TRUE);
	void SetScrollbarInfo();

	enum {	cstOffsetX = 10, 
			cstOffsetY = 8   };	

	enum BLOCKTYPE
	{
		blText,
		blLink,
		blSelectedText,
		blSelectedLink,
	};

	enum SELTYPE
	{
		selNoSel,
		selFullSel,
		selSelStarts,
		selSelEnds,
		selSelInside,
	};

	SELTYPE GetSelType(LPCTSTR szStart, int Len);
	int CalculateLines();
	void StartTimer(UINT nInterval);
	void StopTimer();

	
	static BOOL	m_bRegistered;

	LPTSTR	m_pszData;
	HFONT	m_hFont;
	HFONT	m_hFontLink;
	int		m_nLineHeight;
	int		m_nVisibleLines;
	int		m_nTopIdx;
	int		m_nWidth;
	int		m_nAccumDelta;
	int		m_nDeltaPerLine;
	BOOL	m_bScrollPage;

	int		m_nSelStart; // first character selected
	int		m_nSelEnd;   // first non-selected character after selection
	int		m_nSelPointer;
	UINT	m_nTimerInterval;
	BOOL	m_bMenuVisible;

	HCURSOR m_hHoverLinkCursor;
 
	BOOL	m_bMouseDown; // we need this flag to make sure that a mouse up event is used only 
						  // when there has been a mouse down before!

	
	//{{AFX_MSG(CTextDisplay)
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg void OnPaint();
	afx_msg void OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
	afx_msg BOOL OnMouseWheel(UINT nFlags, short zDelta, CPoint pt);
	afx_msg BOOL OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message);
	afx_msg void OnLButtonDown(UINT nFlags, CPoint point);
	afx_msg void OnLButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnMouseMove(UINT nFlags, CPoint point);
	afx_msg BOOL OnEraseBkgnd(CDC* pDC);
	afx_msg void OnTimer(UINT nIDEvent);
	afx_msg void OnRButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnNcLButtonDown(UINT nHitTest, CPoint point);
	afx_msg void OnSettingChange(UINT uFlags, LPCTSTR lpszSection);
	afx_msg void OnLButtonDblClk(UINT nFlags, CPoint point);
	afx_msg void OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);
	//}}AFX_MSG


	struct StrInfo {
		StrInfo(LPCTSTR StrStart = NULL, DWORD StrLen = 0) : pszStrStart(StrStart), dwStrLen(StrLen) {}
		LPCTSTR		pszStrStart;
		DWORD		dwStrLen;
	};

    struct LinkStrInfo : public StrInfo{
		LinkStrInfo(LPCTSTR StrStart = NULL, DWORD StrLen = 0, LPCTSTR szLink = NULL) : StrInfo(StrStart, StrLen), m_Link(szLink) {}
		CString m_Link;
	};

    template <class T>
	class CStrList
	{
		public:
			CStrList(int InitSize = 200, int IncreaseSizeBy = 100) 
			{
				m_pData = new T[InitSize]; 
				m_nMaxItems = InitSize;
				m_nIncreaseSizeBy = IncreaseSizeBy;
				m_nNextIdx = 0;
			}

			~CStrList() {
				delete [] m_pData;
			}

			void AddItem(T& Item) 
			{
				if(m_nNextIdx >= m_nMaxItems)
				{
					m_nMaxItems += m_nIncreaseSizeBy;
					T* pNewArray = new T[m_nMaxItems];
					
					for(int n = 0; n < m_nNextIdx; n++)
						pNewArray[n] = m_pData[n];

					delete [] m_pData;
					m_pData = pNewArray;
				}

				m_pData[m_nNextIdx++] = Item;
			}

			int GetCount()   const { return m_nNextIdx; }
			void RemoveAll() { m_nNextIdx = 0; }

			T& operator[](int nIndex) const
			{
				if(nIndex >= 0 && nIndex < m_nNextIdx)
					return m_pData[nIndex];
				else
				{
					ASSERT(FALSE);
					return m_pData[0];
				}
			}
		

			void Sort()
			{
				long k, i, j;
				T h;
				k = m_nNextIdx / 2;
				while(k > 0)
				{
					for(i = 0; i < m_nNextIdx - k; i++)
					{
						j = i;
						while(j >= 0 && m_pData[j].pszStrStart > m_pData[j+k].pszStrStart)
						{
							h = m_pData[j];
							m_pData[j] = m_pData[j+k];
							m_pData[j+k] = h;
							j -= k;
						}
					}
					k /= 2;
				}
			}

		private:
			T*  m_pData;
			int		  m_nMaxItems;
			int		  m_nIncreaseSizeBy;
			int		  m_nNextIdx;
	};


	CStrList<StrInfo> m_Lines;
	CStrList<LinkStrInfo> m_Links;

    CMultiLineTip m_LinkToolTip;
    bool m_bTimerLink;


	BOOL IsInLink(LPCTSTR szP, int& nIdxLink);
	void GetLineSections(LPCTSTR szLine, int nLineLen, CStrList<StrInfo>& SecList);

    static CString word_boundaries;

	DECLARE_MESSAGE_MAP()
};


//{{AFX_INSERT_LOCATION}}

#endif // AFX_TEXTDISPLAY_H_
