// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// TextDisplay.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "resource.h"
#include "TextDisplay.h"
#include "RegKey.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//  i18nComment("Text View")

TCHAR szClassName[] = _T("PopmanTextDiplay");

BOOL CTextDisplay::m_bRegistered = FALSE;
const int TimerID_Selection  = 1;
const int TimerID_LinkTip    = 2;

/////////////////////////////////////////////////////////////////////////////
// CTextDisplay

BEGIN_MESSAGE_MAP(CTextDisplay, CWnd)
	//{{AFX_MSG_MAP(CTextDisplay)
	ON_WM_SIZE()
	ON_WM_PAINT()
	ON_WM_VSCROLL()
	ON_WM_MOUSEWHEEL()
	ON_WM_SETCURSOR()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_ERASEBKGND()
	ON_WM_TIMER()
	ON_WM_RBUTTONUP()
	ON_WM_NCLBUTTONDOWN()
	ON_WM_SETTINGCHANGE()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_EDIT_SELECT_ALL, SelectAll)
	ON_COMMAND(ID_EDIT_COPY, CopySelToClipboard)
		// TFC START
	ON_COMMAND(ID_COPY_LINK_ADDRESS, CopyLinkAddress)
		// TFC END
END_MESSAGE_MAP()

CTextDisplay::CTextDisplay() : m_Links(20, 10)
{
	m_pszData = NULL;
	m_nVisibleLines = 0;
	m_nTopIdx = 0;
	m_nWidth = 0;
	
	m_nAccumDelta = 0;
	m_nDeltaPerLine = 40;
	m_bScrollPage = FALSE;

	m_hFont = 0;
	m_hFontLink = 0;
	m_nLineHeight = 0;

	m_nSelStart = -1; 
	m_nSelEnd = -1;
	m_nSelPointer = -1;
	m_nTimerInterval = 0;
	m_bMenuVisible = FALSE;

	m_TextColor = GetSysColor(COLOR_WINDOWTEXT);
	m_BkgndColor = GetSysColor(COLOR_WINDOW);

	m_hHoverLinkCursor = ::LoadCursor(NULL, IDC_ARROW);

	m_bMouseDown = FALSE;

    m_bTimerLink = false;

//	m_bCachePaint = FALSE;
}

CTextDisplay::~CTextDisplay()
{
	if(m_pszData) free((void*)m_pszData);
	if(m_hFontLink)
		::DeleteObject(m_hFontLink);
}


BOOL CTextDisplay::Create(CWnd *pParent, CRect rcPos)
{
	InitMouseWheel();	

	if(m_bRegistered == FALSE)
		m_bRegistered = RegisterWndClass();

	if(m_bRegistered == FALSE)
		return FALSE;

	return CWnd::CreateEx(WS_EX_CLIENTEDGE, szClassName, NULL, WS_CHILD | WS_VISIBLE | WS_VSCROLL, rcPos, pParent, 0, this); 
}


void CTextDisplay::OnSettingChange(UINT uFlags, LPCTSTR lpszSection) 
{
	CWnd::OnSettingChange(uFlags, lpszSection);
	InitMouseWheel();	
}


void CTextDisplay::InitMouseWheel()
{
	int nScrollLines = 3;
	SystemParametersInfo(SPI_GETWHEELSCROLLLINES, 0, &nScrollLines, 0);
	if(nScrollLines == 0)
		m_nDeltaPerLine = 0;
	else if(nScrollLines == WHEEL_PAGESCROLL || nScrollLines == 0xFFFF)
		m_bScrollPage = TRUE;
	else
		m_nDeltaPerLine = WHEEL_DELTA / nScrollLines;
}


void CTextDisplay::SetText(LPCTSTR szText, const std::vector<LinkInfo>* pLinks)
{
	if(m_pszData) free((void*)m_pszData);
    m_pszData = _tcsdup(szText);

    m_Links.RemoveAll();

    if(pLinks) {
        const std::vector<LinkInfo> links = *pLinks;
        for(size_t i = 0; i < links.size(); ++i) {
            if(links[i].szLink[0] != _T('#'))
                m_Links.AddItem(LinkStrInfo(m_pszData + links[i].idxStart, links[i].len, links[i].szLink));
        }
    }
	
	ExtractLinks();
	
	m_nSelStart = -1;
	m_nSelEnd = -1;
	m_nSelPointer = -1;
	m_nTopIdx = 0;

	if(m_hWnd)
	{
		CalculateLines();
		Invalidate();
		UpdateWindow();
	}
}


int CTextDisplay::CalculateLines()
{
	CSize	Size;
	BOOL	bPara  = TRUE;
	LPTSTR	pBegin = NULL;
	LPTSTR	pEnd   = NULL; 
	LPTSTR	pText  = m_pszData;
	
	//TRACE("CalculateLines()\n");

	if(m_hWnd == 0)
		return 0;

	RECT	rect;
	GetClientRect(&rect);
	rect.left += cstOffsetX;
	
	LONG	nWndWidth = rect.right - rect.left;
	
	CClientDC DC(this);
	m_Lines.RemoveAll();

	CFont* pFont = GetFont();
	CFont* pFontOld = (CFont*)DC.SelectObject(pFont);

	CSize S	 = DC.GetTabbedTextExtent(_T("\t"), 1, 0, NULL);
	int nTabPos = S.cx;

	while(pText && *pText)
	{
		if ((bPara == FALSE) && (*pText == _T(' ')))    // skip over first leading space, if we don't start a new Absatz
			pText++;
	 
		pBegin = pText ;          // set pointer to char at beginning of line
		
		do                        // until the line is known
		{
			pEnd = pText;         // set pointer to char at end of line

			while (*pText != _T('\0') && *pText != _T('\r') && *pText++ != _T(' '));
	

			if ((*pText == _T('\0'))  ||  (*pText == _T('\r')))	
			{
				Size = DC.GetTabbedTextExtent(pBegin, pText - pBegin, 1, &nTabPos);
				if (Size.cx < nWndWidth)
				{
					pEnd = pText; 
					bPara = (*pText == _T('\r'));
					goto FoundEndOfLine;
				}
				break;
			}
					
			bPara = FALSE;
           
			Size = DC.GetTabbedTextExtent(pBegin, pText - pBegin - 1, 1, &nTabPos);  // after each space encountered, calculate extents
		}
		while(Size.cx < nWndWidth);
		
		if (pEnd == pBegin)	// wenn kein gltiges Zeilenende gefunden
			pEnd = pText;    

		// prfen, ob ermittelte Zeile nicht zu lang, ggf. korrigieren:
		Size = DC.GetTabbedTextExtent(pBegin, pEnd - pBegin, 1, &nTabPos);
		if(Size.cx > nWndWidth)
		{
			bPara = FALSE;
			do {
				--pEnd;
				Size = DC.GetTabbedTextExtent(pBegin, pEnd - pBegin, 1, &nTabPos);
			} while(Size.cx > nWndWidth);
		}

		bPara = (*pEnd == _T('\r'));

FoundEndOfLine:
		
		// pEnd zeigt auf das ungltige Zeichen
		m_Lines.AddItem(StrInfo(pBegin, pEnd - pBegin));

		pText = pEnd;

		if(bPara)
		{	// move pointer behind linebreak:
			++pText;
			++pText;
		}
	}

	SetScrollbarInfo();

	DC.SelectObject(pFontOld);

	return m_Lines.GetCount();
}



void CTextDisplay::SetScrollbarInfo()
{
	if(m_nLineHeight == 0)
		m_nLineHeight = GetLineHeight();

	RECT rect;
	GetClientRect(&rect);

	m_nVisibleLines = (int)((rect.bottom - rect.top - cstOffsetY) / (float)m_nLineHeight);
	int nInvisibleLines = m_Lines.GetCount() - m_nVisibleLines;

	SCROLLINFO ScrollInfo;
	ScrollInfo.cbSize = sizeof(ScrollInfo);
	ScrollInfo.fMask = SIF_PAGE | SIF_RANGE | SIF_DISABLENOSCROLL;
	ScrollInfo.nPage = m_nVisibleLines;//* nInvisibleLines / (m_Lines.GetCount() == 0 ? 1 : m_Lines.GetCount());
	ScrollInfo.nMin = 0;
	ScrollInfo.nMax = m_Lines.GetCount() - 1;

	SetScrollInfo(SB_VERT, &ScrollInfo);
	

	if(nInvisibleLines > 0)
		EnableScrollBar(SB_VERT, ESB_ENABLE_BOTH);
	else
		EnableScrollBar(SB_VERT, ESB_DISABLE_BOTH);


	UpdateOnScroll(FALSE);
}


void CTextDisplay::OnSize(UINT nType, int cx, int cy) 
{
	CWnd::OnSize(nType, cx, cy);

	Invalidate();

	if(m_nWidth != cx)
	{
		m_nWidth = cx;
		if(cx > 0)
			CalculateLines();
	}
	else
	{
		SetScrollbarInfo();
	}
}

void CTextDisplay::OnPaint() 
{
	if(m_hFontLink == NULL)
		m_hFontLink = CreateLinkFont(m_hFont);
	
	if(m_nLineHeight == 0)
		m_nLineHeight = GetLineHeight();

	CStrList<StrInfo> SecList(8, 8);
	CPaintDC dc(this);

//	CDC CacheDC;
//	CDC& dc = m_bCachePaint ? CacheDC : *((CDC*)&PaintDC);
//	CBitmap CacheBitmap;
//	CBitmap* pOldBitmap = NULL;
	CRect rcClient;
	RECT rect;
	GetClientRect(&rcClient);
/*
	if(m_bCachePaint)
	{
		VERIFY(CacheDC.CreateCompatibleDC(&PaintDC));
		VERIFY(CacheBitmap.CreateCompatibleBitmap(&PaintDC, rcClient.Width(), rcClient.Height())); 
		pOldBitmap = CacheDC.SelectObject(&CacheBitmap);
		CacheDC.FillSolidRect(&rcClient, GetSysColor(COLOR_WINDOW));
	}
*/
	CFont* pFont = GetFont();
	CFont FontLink;
	FontLink.Attach(m_hFontLink);

	CFont* pFontOld = (CFont*)dc.SelectObject(GetFont());
	
	CSize Size	 = dc.GetTabbedTextExtent(_T("\t"), 1, 0, NULL);
	int nTabPos = Size.cx;

	int nStartY = cstOffsetY;


	dc.SetTextAlign(TA_LEFT | TA_TOP);
	
	COLORREF TextColorSel = GetSysColor(COLOR_HIGHLIGHTTEXT);
	COLORREF BkColorSel = GetSysColor(COLOR_HIGHLIGHT);


	for(int n = m_nTopIdx; n < m_nTopIdx + m_nVisibleLines && n < m_Lines.GetCount(); n++)
	{
		int posX = cstOffsetX;
		const StrInfo& CurrLine = m_Lines[n];

        //if(n == 0x31) {
        //    TRACE("Start!\n");
        //}
		GetLineSections(CurrLine.pszStrStart, CurrLine.dwStrLen, SecList);
		for(int x = 0; x < SecList.GetCount(); x++)
		{
            const StrInfo& sec = SecList[x];
			BLOCKTYPE  BlockType = (BLOCKTYPE)LOWORD(sec.dwStrLen);
			unsigned short BlockLen = HIWORD(sec.dwStrLen);
			switch(BlockType)
			{
			case BLOCKTYPE::blText:
				dc.SelectObject(pFont);
				dc.SetTextColor(m_TextColor);
				dc.SetBkColor(m_BkgndColor);
				break;

			case BLOCKTYPE::blLink:
				dc.SelectObject(&FontLink);
				dc.SetTextColor(RGB(0, 0, 200));
				dc.SetBkColor(m_BkgndColor);
				break;

			case BLOCKTYPE::blSelectedText:
				dc.SelectObject(pFont);
				dc.SetTextColor(TextColorSel);
				dc.SetBkColor(BkColorSel);
				break;

			case BLOCKTYPE::blSelectedLink:
				dc.SelectObject(&FontLink);
				dc.SetTextColor(TextColorSel);
				dc.SetBkColor(BkColorSel);
				break;
			}

			Size = dc.TabbedTextOut(posX, nStartY, sec.pszStrStart, BlockLen, 1, &nTabPos, cstOffsetX);
		
			posX += Size.cx;
		}

		
		rect.left = posX;
		rect.right = rcClient.right;
		rect.top = nStartY;
		rect.bottom = nStartY + Size.cy;
		dc.FillSolidRect(&rect, m_BkgndColor);

		nStartY += m_nLineHeight;
	}

	rect.left = 0;
	rect.right = rcClient.right;
	rect.top = nStartY;
	rect.bottom = rcClient.bottom;
	dc.FillSolidRect(&rect, m_BkgndColor);

	rect.left = 0;
	rect.right = rcClient.right;
	rect.top = 0;
	rect.bottom = cstOffsetY;
	dc.FillSolidRect(&rect, m_BkgndColor);

	rect.left = 0;
	rect.right = cstOffsetX;
	rect.top = cstOffsetY;
	rect.bottom = rcClient.bottom;
	dc.FillSolidRect(&rect, m_BkgndColor);

	
	dc.SelectObject(pFontOld);
	FontLink.Detach();
/*
	if(m_bCachePaint)
	{
		PaintDC.BitBlt(rcClient.left, rcClient.top, rcClient.Width(), rcClient.Height(), &CacheDC, 0, 0, SRCCOPY);
		CacheDC.SelectObject(pOldBitmap);
		CacheDC.DeleteDC();
		CacheBitmap.DeleteObject();
		m_bCachePaint = FALSE;
	}
*/
}

void CTextDisplay::GetLineSections(LPCTSTR szLine, int nLineLen, CStrList<StrInfo>& SecList)
{
	SecList.RemoveAll();

	if(nLineLen == 0)
		return;

	unsigned short BlockLen = 0;
	BLOCKTYPE	BlockType;
	int IdxLink = 0;
	StrInfo Block(szLine);

	BlockType = IsInLink(szLine, IdxLink) ? blLink : blText;
	BOOL bLoop = TRUE;

	while(bLoop)
	{
		const unsigned short MaxBlockLen = static_cast<unsigned short>(szLine + nLineLen - Block.pszStrStart);

		if(BlockType == blText)
		{	
			BlockLen = (IdxLink > -1) ?  static_cast<unsigned short>(m_Links[IdxLink].pszStrStart - Block.pszStrStart) : MaxBlockLen;
			if(BlockLen >= MaxBlockLen)
			{
				BlockLen = MaxBlockLen;
				bLoop = FALSE;
			}

			Block.dwStrLen = MAKELONG(BlockType, BlockLen);
			SecList.AddItem(Block);
			BlockType = blLink;
			Block.pszStrStart = (IdxLink > -1) ? m_Links[IdxLink].pszStrStart : NULL;
		}
		else if(BlockType == blLink) // blLink
		{
			const StrInfo& Link = m_Links[IdxLink];
			
			if(Link.pszStrStart < szLine)
				BlockLen = (unsigned short)(Link.pszStrStart + Link.dwStrLen - szLine);
			else
				BlockLen = (unsigned short)Link.dwStrLen; 

			if(BlockLen >= MaxBlockLen)
			{
				BlockLen = MaxBlockLen;
				bLoop = FALSE;
			}
			
			Block.dwStrLen = MAKELONG(BlockType, BlockLen);
			SecList.AddItem(Block);
			BlockType = blText;
			Block.pszStrStart = Link.pszStrStart + Link.dwStrLen;
			if(++IdxLink >= m_Links.GetCount())
				IdxLink = -1;
		}
	}


	if(GetSelType(szLine, nLineLen) == selNoSel)
		return;

	LPCTSTR pSelStart = m_pszData + m_nSelStart;
	LPCTSTR pSelEnd = m_pszData + m_nSelEnd;
	int nCount = SecList.GetCount();
	for(int n = 0; n < nCount; n++)
	{
		int NewBlockLen;
		BLOCKTYPE NewBlockType;
		StrInfo& Block = SecList[n];
		LPCTSTR pBlockStart = Block.pszStrStart;
		int BlockLen = HIWORD(Block.dwStrLen);
		BLOCKTYPE BlockType = (BLOCKTYPE)LOWORD(Block.dwStrLen);

		switch(GetSelType(pBlockStart, BlockLen))
		{
		case selFullSel:
			BlockType = (BlockType == blText ? blSelectedText : blSelectedLink);
			Block.dwStrLen = MAKELONG(BlockType, BlockLen);
			break;

		case selSelInside:
			NewBlockLen = pBlockStart + BlockLen - pSelEnd;
			BlockLen = pSelStart - pBlockStart;
			Block.dwStrLen = MAKELONG(BlockType, BlockLen);
			
			NewBlockType = (BlockType == blText ? blSelectedText : blSelectedLink);
			SecList.AddItem(StrInfo(pSelStart, MAKELONG(NewBlockType, pSelEnd-pSelStart)));

			SecList.AddItem(StrInfo(pSelEnd, MAKELONG(BlockType, NewBlockLen)));
			goto End;

		case selSelStarts:
			NewBlockLen = pBlockStart + BlockLen - pSelStart;
			BlockLen = pSelStart - pBlockStart;
			Block.dwStrLen = MAKELONG(BlockType, BlockLen);

			BlockType = (BlockType == blText ? blSelectedText : blSelectedLink);
			SecList.AddItem(StrInfo(pSelStart, MAKELONG(BlockType, NewBlockLen)));
			break;

		case selSelEnds:
			NewBlockLen = pBlockStart + BlockLen - pSelEnd;
			BlockLen = pSelEnd - pBlockStart;
			NewBlockType = (BlockType == blText ? blSelectedText : blSelectedLink);
			Block.dwStrLen = MAKELONG(NewBlockType, BlockLen);

			SecList.AddItem(StrInfo(pSelEnd, MAKELONG(BlockType, NewBlockLen)));
			goto End;
		}
	}
End: SecList.Sort();

}

CTextDisplay::SELTYPE CTextDisplay::GetSelType(LPCTSTR szStart, int Len)
{
	if(m_nSelStart == -1 || m_nSelEnd == -1)
		return selNoSel;

	int IdxStart = szStart - m_pszData;
	int IdxEnd = IdxStart + Len;

	if(m_nSelEnd <= IdxStart || m_nSelStart >= IdxEnd)
		return selNoSel;

	if(IdxStart >= m_nSelStart && IdxEnd <= m_nSelEnd)
		return selFullSel;

	if(m_nSelStart > IdxStart && m_nSelEnd < IdxEnd)
		return selSelInside;

	return m_nSelStart > IdxStart ? selSelStarts : selSelEnds;
}


BOOL CTextDisplay::IsInLink(LPCTSTR szP, int &nIdxLink)
{
	for(int n = 0; n < m_Links.GetCount(); n++)
	{
        const LinkStrInfo& link = m_Links[n];
		if(link.pszStrStart <= szP)
		{
			if(szP < link.pszStrStart + link.dwStrLen)
			{
				nIdxLink = n;
				return TRUE;
			}
		}
		else
		{
			nIdxLink = n;
			return FALSE;
		}
	}
	nIdxLink = -1;
	return FALSE;
}

void CTextDisplay::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	
	switch (nSBCode) 
	{
		case SB_LINEUP :   LineUp();	break;
		case SB_LINEDOWN : LineDown();	break;
		case SB_PAGEUP :   PageUp();	break;
		case SB_PAGEDOWN : PageDown();	break;

		case SB_THUMBPOSITION : 
		case SB_THUMBTRACK : 
					m_nTopIdx = nPos;
					UpdateOnScroll();
					break;
	}

	CWnd::OnVScroll(nSBCode, nPos, pScrollBar);
}


void CTextDisplay::UpdateOnScroll(BOOL bRepaintWindow)
{
	//TRACE("UpdateOnScroll\n");

	if(m_nTopIdx > m_Lines.GetCount() - m_nVisibleLines)
		m_nTopIdx = m_Lines.GetCount() - m_nVisibleLines;	
	
	if(m_nTopIdx < 0)
		m_nTopIdx = 0;

	SetScrollPos(SB_VERT, m_nTopIdx);

	if(bRepaintWindow)
	{
		Invalidate();
		UpdateWindow();
	}
}

void CTextDisplay::LineUp()
{
	if(m_nTopIdx > 0)
	{
		--m_nTopIdx;
		UpdateOnScroll();
	}
}

void CTextDisplay::LineDown()
{
	if(m_nTopIdx < m_Lines.GetCount() - m_nVisibleLines)
	{
		++m_nTopIdx;
		UpdateOnScroll();
	}
}

void CTextDisplay::PageUp()
{
	if(m_nTopIdx > 0)
	{
		m_nTopIdx -= (m_nVisibleLines-1);
		UpdateOnScroll();
	}
}

void CTextDisplay::PageDown()
{
	if(m_nTopIdx < m_Lines.GetCount() - m_nVisibleLines)
	{
		m_nTopIdx += (m_nVisibleLines-1);
		UpdateOnScroll();
	}
}

BOOL CTextDisplay::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
	if(m_bScrollPage)
	{
		SendMessage(WM_VSCROLL, zDelta > 0 ? SB_PAGEUP : SB_PAGEDOWN, 0);
	}
	else if(m_nDeltaPerLine > 0)
	{
		m_nAccumDelta += zDelta;

		while(m_nAccumDelta >= m_nDeltaPerLine)
		{
			SendMessage(WM_VSCROLL, SB_LINEUP, 0);
			m_nAccumDelta -= m_nDeltaPerLine;
		}

		while(m_nAccumDelta <= -m_nDeltaPerLine)
		{
			SendMessage(WM_VSCROLL, SB_LINEDOWN, 0);
			m_nAccumDelta += m_nDeltaPerLine;
		}
		
		return 1;
	}
		
	return CWnd::OnMouseWheel(nFlags, zDelta, pt);
}


BOOL CTextDisplay::RegisterWndClass()
{
	WNDCLASSEX wc;

	wc.cbSize        = sizeof(WNDCLASSEX);
	wc.style         = CS_DBLCLKS | CS_PARENTDC;
	wc.lpfnWndProc   = WndProc;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = 0;
	wc.hInstance     = ::GetModuleHandle(NULL);
	wc.hIcon         = NULL;
	wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
	wc.lpszMenuName  = NULL;
	wc.lpszClassName = szClassName;
	wc.hIconSm = NULL;
	
	return ::RegisterClassEx(&wc) == 0 ? FALSE : TRUE;
}

LRESULT CALLBACK CTextDisplay::WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	CTextDisplay* pTx = (CTextDisplay*)::GetWindowLong(hwnd, GWL_USERDATA);
	
	switch(msg)
	{
	case WM_CREATE:
		{
			CREATESTRUCT* lpCS = (LPCREATESTRUCT)lParam;
			::SetWindowLong(hwnd, GWL_USERDATA, (LONG)lpCS->lpCreateParams);
		}
		break;

	case WM_SETFONT:
		pTx->m_hFont = (HFONT)wParam;
		pTx->OnFontChanged((BOOL)lParam);
		break;

	case WM_GETFONT:
		return (LRESULT)pTx->m_hFont;

	case WM_COPY:
		pTx->CopySelToClipboard();
		break;

	case EM_GETSEL:
	{
		int nSelStart = pTx->m_nSelStart;
		int nSelEnd = pTx->m_nSelEnd;

		if(wParam)
			*((DWORD*)wParam) = nSelStart == -1 ? 0 : nSelStart;

		if(lParam)
			*((DWORD*)lParam) = nSelStart == -1 ? 0 : nSelEnd;

		if(nSelStart > 0xFFFF || nSelEnd > 0xFFFF)
			return -1;

		return MAKELONG(nSelStart, nSelEnd);
	}

	case EM_SETSEL:
	{
		UINT nStart = wParam;
		LONG nEnd = lParam;

		if(nStart == 0 && nEnd == -1)
		{
			pTx->SelectAll();
			return 0;
		}

		if(nStart == -1)
		{
			pTx->m_nSelStart = -1;
			pTx->m_nSelEnd = -1;
		}
		else 
		{
			pTx->m_nSelStart = min((signed)nStart, nEnd);
			pTx->m_nSelEnd = max((signed)nStart, nEnd);

			if(pTx->m_nSelStart < -1 || pTx->m_nSelEnd < -1) {
				pTx->m_nSelStart = -1;
				pTx->m_nSelEnd = -1;
			}

			if(pTx->m_nSelStart == pTx->m_nSelEnd) {
				pTx->m_nSelStart = -1;
				pTx->m_nSelEnd = -1;
			}
		}
		pTx->Invalidate();
		pTx->UpdateWindow();
		
		break;
	}

	default:
		return ::DefWindowProc(hwnd, msg, wParam, lParam);
	}

	return 0;
}


void CTextDisplay::OnFontChanged(BOOL bUpdate)
{
	if(m_hFontLink)
		::DeleteObject(m_hFontLink);
	m_hFontLink = CreateLinkFont(m_hFont);

	m_nLineHeight = GetLineHeight();

	CalculateLines();
	EnsureSelectionVisible(bUpdate);
	if(bUpdate)
	{
		Invalidate();
		UpdateWindow();
	}
}

BOOL CTextDisplay::PreTranslateMessage(MSG* pMsg) 
{
	if (pMsg->message == WM_KEYDOWN)
	{
		switch(pMsg->wParam)
		{
		case VK_APPS:  SendMessage(WM_RBUTTONUP, 0, MAKELONG(12, 12));  break;
		case VK_DOWN:  LineDown();	break;
		case VK_UP:	   LineUp();	break;
		case VK_NEXT:  PageDown();	break;
		case VK_PRIOR: PageUp();	break;

		case 'C': 
		case VK_INSERT:
			if(GetKeyState(VK_CONTROL) & 8000)
			{
				CopySelToClipboard();
				return TRUE;
			}
			return CWnd::PreTranslateMessage(pMsg);
			
		case 'A': 
		{
			if(GetKeyState(VK_CONTROL) & 8000)
			{
				SelectAll();
				return TRUE;
			}
			return CWnd::PreTranslateMessage(pMsg);
		}
				
		default:
			return CWnd::PreTranslateMessage(pMsg);
		}
		return TRUE;
	}
	
	return CWnd::PreTranslateMessage(pMsg);
}


void CTextDisplay::ExtractLinks(bool bClearLinks)
{
	struct {
		LPCTSTR szItem;
		int		nIdxFound;
	} SearchItems[] = { 
		{ _T("https://"), -1 },
		{ _T("http://"), -1 },
		{ _T("www."),    -1 },
		{ _T("ftp://"),  -1 },
		{ _T("mailto:"), -1 },
	};

	int nOffset = 0;
	int n = 0;
	
    if(bClearLinks)
	    m_Links.RemoveAll();

	while(1)
	{
		for(n = 0; n < sizeof(SearchItems)/sizeof(SearchItems[0]); n++ )
			SearchItems[n].nIdxFound = FindNoCase(m_pszData, SearchItems[n].szItem, nOffset);
		

		int IdxMin = -1;
		int idxItem = -1;
		
		for(n = 0; n < sizeof(SearchItems)/sizeof(SearchItems[0]); n++ )
		{
			if(SearchItems[n].nIdxFound  > -1)
			{
				if(IdxMin == -1) {
					IdxMin = SearchItems[n].nIdxFound;
					idxItem = n;
				}
				else if(SearchItems[n].nIdxFound < IdxMin) {
					IdxMin = SearchItems[n].nIdxFound;
					idxItem = n;
				}
			}
		}

		if(IdxMin == -1)
			break;

		int IdxEnd = IdxMin;
		while(!IsStopChar(m_pszData[IdxEnd]))
			IdxEnd++;

		if(_tcslen(SearchItems[idxItem].szItem) >= (unsigned)IdxEnd-IdxMin) {
			nOffset = IdxEnd;
			continue;
		}

		switch(m_pszData[IdxEnd-1])
		{
		case _T('.'):
		case _T('!'):
		case _T('?'):
		case _T(','):
		case _T(';'):
		case _T('-'):
		case _T(':'):
			--IdxEnd;
		}


		nOffset = IdxEnd;

		m_Links.AddItem(LinkStrInfo(m_pszData + IdxMin, IdxEnd - IdxMin));
	}

	nOffset = 0;

	while(1)
	{
		LPCTSTR pAt = _tcschr(m_pszData + nOffset, _T('@'));
		if(pAt == NULL)
			break;
		
		nOffset = pAt - m_pszData + 1;

		LPCTSTR pR = pAt;
		int res = 0;

		while((res = CheckMailChar(*++pR)) == 1);
		if(res == 3)  // invalid char
			continue;

		LPCTSTR pL = pAt - 1;
		while(pL >= m_pszData && (res = CheckMailChar(*pL)) == 1)
			pL--;
		if(res == 3)  // invalid char
			continue;

		pL++;
	
		if((pL < pAt) && pR - pAt > 2) 
        {
            while(pR[-1] == _T('.'))
                pR--;

			m_Links.AddItem(LinkStrInfo(pL, pR - pL));
        }
	}

	// sort links:

	m_Links.Sort();

//	for(int p = 0; p < m_Links.GetCount(); p++)
//		TRACE("%d\n", m_Links[p].pszStrStart);

}



inline BOOL CTextDisplay::IsStopChar(TCHAR s)
{
	unsigned int c = (unsigned int)s;

	if(c <= _T(' '))
		return TRUE;

	if( c == _T('(')   ||
		c == _T(')')   ||
		c == _T('[')   ||
		c == _T(']')   ||
		c == _T('{')   ||
		c == _T('}')   ||
		c == _T('<')   ||
		c == _T('>')   ||
		c == _T('\"')  ||
		c == _T('\'') 
	   )
		return TRUE; 

	return FALSE;
}

inline int CTextDisplay::CheckMailChar(TCHAR s)
{
	unsigned int c = (unsigned int)s;

	if( c >= _T('a') && c <= _T('z')	||
		c >= _T('A') && c <= _T('Z')	||
		c >= _T('0') && c <= _T('9')	||
		c == _T('-')					||
		c == _T('_')					||
		c == _T('.')
	   )
		return 1; // valid char

	if(c <= _T(' ') || c == _T('<') || c == _T('>') || c == _T(';') || c == _T(','))
		return 2;	// white space

	return 3;		// invalid char

}

inline int CTextDisplay::FindNoCase(LPCTSTR szStr, LPCTSTR szFind, int nStrStart)
{
	if(szStr == NULL || szFind == NULL)
		return -1;

	LPCTSTR szStart = szStr;
	szStr += nStrStart;
	
	TCHAR szFirst = lcase(*szFind);
	
	//while(szStr < szStart + nStrLen*sizeof(TCHAR))
	while(*szStr)
	{
		if(lcase(*szStr) == szFirst)
		{
			LPCTSTR a = szStr + 1;
			LPCTSTR b = szFind + 1;
			
			while(*b)
				if(lcase(*a++) != lcase(*b++)) goto Jump;

			return (szStr - szStart); 
		}
Jump:	szStr++;

	}
	return -1;
}

inline TCHAR CTextDisplay::lcase(TCHAR c)
{
	if (c >= _T('A') && c <= _T('Z'))
		return (TCHAR)(_T('a') + c - _T('A'));
	else
		return c;
}

HFONT CTextDisplay::CreateLinkFont(HFONT hFont)
{
	LOGFONT lf;
	::GetObject(hFont, sizeof(LOGFONT), &lf); 
	lf.lfUnderline = TRUE;
	return (HFONT)::CreateFontIndirect(&lf);
}

int CTextDisplay::GetCharPosFromPoint(POINT pt, int* pCurPos)
{
	if(m_pszData == 0)
	{
		if(pCurPos) *pCurPos = 0;
		return -1;
	}

	int nLine = GetLineFromPoint(pt);
	if(nLine >= m_Lines.GetCount())
	{
		if(pCurPos) *pCurPos = _tcslen(m_pszData);
		return -1;
	}

	if(nLine < 0)
	{
		if(pCurPos) *pCurPos = 0;
		return -1;
	}

	StrInfo Line = m_Lines[nLine];
	if(Line.dwStrLen == 0)
	{
		if(pCurPos) *pCurPos = Line.pszStrStart - m_pszData;
		return -1;
	}

	CClientDC DC(this);
	CFont* pFontOld = DC.SelectObject(GetFont());
	CSize Size = DC.GetTabbedTextExtent(_T("\t"), 1, 0, NULL);
	int nTabPos = Size.cx;
	int res = -1;
	pt.x -= cstOffsetX;

	Size = DC.GetTabbedTextExtent(Line.pszStrStart, Line.dwStrLen, 1, &nTabPos);
	if(0 < pt.x && pt.x < Size.cx)
	{
		int CharWidth = Size.cx / Line.dwStrLen;
		int Idx = pt.x / CharWidth;
		if(Idx <= 0) Idx = 1;
		if(Idx > (signed)Line.dwStrLen) Idx = Line.dwStrLen;

		Size = DC.GetTabbedTextExtent(Line.pszStrStart, Idx, 1, &nTabPos);
		if(Size.cx < pt.x)
		{
			do {
				Idx++;
				Size = DC.GetTabbedTextExtent(Line.pszStrStart, Idx, 1, &nTabPos);
			} while(Size.cx < pt.x);
			--Idx;
		}
		else
		{
			do {
				Idx--;
				if(Idx == 0) break;
				Size = DC.GetTabbedTextExtent(Line.pszStrStart, Idx, 1, &nTabPos);
			} while(Size.cx > pt.x);
		}
		res = Line.pszStrStart - m_pszData + Idx;
	}
	else
	{
		if(pt.x <= 0) {
			if(pCurPos) *pCurPos = Line.pszStrStart - m_pszData; 
		}
		else
			if(pCurPos) *pCurPos = Line.pszStrStart + Line.dwStrLen - m_pszData;

		DC.SelectObject(pFontOld);
		return -1;
	}

	if(pCurPos)
	{
		int nLen = res - (Line.pszStrStart - m_pszData);
		Size = DC.GetTabbedTextExtent(Line.pszStrStart, nLen, 1, &nTabPos);
		int PosMiddleChar = Size.cx;
		Size = DC.GetTabbedTextExtent(Line.pszStrStart, nLen + 1, 1, &nTabPos);
		PosMiddleChar = PosMiddleChar + ((Size.cx - PosMiddleChar) / 2);
		*pCurPos = (pt.x > PosMiddleChar) ? res + 1 : res;
	}

	DC.SelectObject(pFontOld);
	return res;
}


int CTextDisplay::GetLineFromPoint(POINT pt) const
{
	RECT rect;
	GetClientRect(&rect);
	return m_nTopIdx + ((pt.y - cstOffsetY - 2) / m_nLineHeight);
}

int CTextDisplay::GetLineHeight()
{
	CClientDC dc(this);
	CFont* pFontOld = dc.SelectObject(GetFont());
	SIZE Size = dc.GetTabbedTextExtent(_T("A"), 1, 0, 0);
	dc.SelectObject(pFontOld);
	return Size.cy < 1 ? 1 : Size.cy;
}

int CTextDisplay::GetLinkFromPoint(POINT pt)
{
	int CharPos = GetCharPosFromPoint(pt);
	if(CharPos == -1)
		return -1;

	LPCTSTR pChar = m_pszData + CharPos;
	for(int n = 0; n < m_Links.GetCount(); n++)
	{
		if(pChar >= m_Links[n].pszStrStart)
		{
			if(pChar < m_Links[n].pszStrStart + m_Links[n].dwStrLen)
				return n;
		} else
			return -1;
	}
	return -1;
}




BOOL CTextDisplay::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
	CPoint point;
	::GetCursorPos(&point);
	ScreenToClient(&point);

	if(m_bMenuVisible)
	{
		::SetCursor(::LoadCursor(NULL, IDC_ARROW));
		return TRUE;
	}
		
    int nLink = GetLinkFromPoint(point);
    if(nLink > -1) {

		::SetCursor(m_hHoverLinkCursor);

        if(m_LinkToolTip.m_hWnd == 0 && !m_bTimerLink) {
            m_bTimerLink = (SetTimer(TimerID_LinkTip, 500, NULL) != 0);
        }

    }
	else
	{
		if(GetCharPosFromPoint(point) == -1)
			::SetCursor(::LoadCursor(NULL, IDC_ARROW));
		else
			::SetCursor(::LoadCursor(NULL, IDC_IBEAM));
	}

	return TRUE;
}

void CTextDisplay::OnLButtonDown(UINT nFlags, CPoint point) 
{
	SetFocus();

	if(m_nSelStart > -1)
	{
		m_nSelStart = -1;
		m_nSelEnd = -1;
		m_nSelPointer = -1;
		Invalidate();
		UpdateWindow();
	}

	SetCapture();
	GetCharPosFromPoint(point, &m_nSelPointer);

	m_bMouseDown = TRUE;

	CWnd::OnLButtonDown(nFlags, point);
}


void CTextDisplay::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if(m_bMouseDown) {
		int nLink = GetLinkFromPoint(point);
			// TFC Start
		LinkAction(nLink);
			// TFC End	
	}
	StopTimer();
	m_nSelPointer = -1;
	ReleaseCapture();	
	
	CWnd::OnLButtonUp(nFlags, point);
}

CString CTextDisplay::word_boundaries(_T(" \t\r\n.?!,:;@<>'$&|`\"()[]{}+-*\\/~%="));

void CTextDisplay::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
    int CharPos = GetCharPosFromPoint(point);
	if(CharPos != -1)
    {
        int left = CharPos;
        int right = CharPos;

        while(left >= 0 && word_boundaries.Find(m_pszData[left]) == -1)
            --left;
	
        ++left;

        while(m_pszData[right] && word_boundaries.Find(m_pszData[right]) == -1)
            ++right;

        if(left < right)
        {
            m_nSelStart = left;
		    m_nSelEnd = right;

            Invalidate();
		    UpdateWindow();
        }
    }

	CWnd::OnLButtonDblClk(nFlags, point);
}

void CTextDisplay::OnMouseMove(UINT nFlags, CPoint point) 
{
  	if(((nFlags & MK_LBUTTON) == 0) || m_nSelPointer == -1) 
	{
		CWnd::OnMouseMove(nFlags, point);	
		return;
	}

	if(GetLinkFromPoint(point) > -1)
	{
		if(m_nSelStart > -1)
			::SetCursor(::LoadCursor(NULL, IDC_IBEAM));
		else
			::SetCursor(m_hHoverLinkCursor);
	}
    else {
		::SetCursor(::LoadCursor(NULL, IDC_IBEAM));	
    }
	

	UpdateSelection(point);
	
	int nLine = GetLineFromPoint(point);
	if(nLine < m_nTopIdx || nLine >= m_nTopIdx + m_nVisibleLines)
	{
		RECT rect;
		GetClientRect(&rect);
		int nClientHeight = rect.bottom - rect.top;

		if(point.y < -m_nLineHeight || point.y > nClientHeight + m_nLineHeight)
			StartTimer(15);
		else
			StartTimer(90);
	}
	else
		StopTimer();


	CWnd::OnMouseMove(nFlags, point);
}


void CTextDisplay::OnTimer(UINT nIDEvent) 
{
	CWnd::OnTimer(nIDEvent);

    if(nIDEvent == TimerID_Selection)
    {
	    POINT point;
	    ::GetCursorPos(&point);
	    ScreenToClient(&point);
	    int nLine = GetLineFromPoint(point);
	    
	    if(nLine < m_nTopIdx)
		    LineUp();	
	    else if(nLine >= m_nTopIdx + m_nVisibleLines)
		    LineDown();
	    
	    UpdateSelection(point);
    }
    else if(nIDEvent == TimerID_LinkTip)
    {
        KillTimer(TimerID_LinkTip);
        m_bTimerLink = false;

        POINT point;
	    ::GetCursorPos(&point);
        ScreenToClient(&point);

        int nLink = GetLinkFromPoint(point);
        if(nLink > -1) {
        
            LinkStrInfo Link = m_Links[nLink];
            if(!Link.m_Link.IsEmpty()) {
                m_LinkToolTip.Show(this, _T(""), Link.m_Link, true, true);
            }
        }
    }
}


void CTextDisplay::UpdateSelection(POINT point)
{
	if(m_nSelPointer < 0)
		return;


	if(point.y < cstOffsetY)
	{
		point.y = cstOffsetY + m_nLineHeight / 2;
	}
	else
	{
		int Ymax = cstOffsetY + (m_nVisibleLines * m_nLineHeight) - (m_nLineHeight / 2);
		if(point.y > Ymax)
			point.y = Ymax;
	}


	int nCurPos = 0;
	GetCharPosFromPoint(point, &nCurPos);
	
	int oldSelStart = m_nSelStart;
	int oldSelEnd = m_nSelEnd;

	if(m_nSelPointer < nCurPos)
	{
		m_nSelStart = m_nSelPointer;
		m_nSelEnd = nCurPos;
	}
	else if(nCurPos < m_nSelPointer)
	{
		m_nSelStart = nCurPos;
		m_nSelEnd = m_nSelPointer;
	}
	else
	{
		m_nSelStart = -1;
		m_nSelEnd = -1;
	}

	if(oldSelStart != m_nSelStart || oldSelEnd != m_nSelEnd)
	{
		Invalidate(FALSE);
		UpdateWindow();
	}
}


BOOL CTextDisplay::OnEraseBkgnd(CDC* pDC) 
{
	return TRUE;
}


void CTextDisplay::StartTimer(UINT nInterval)
{
	if(m_nTimerInterval != nInterval)
		m_nTimerInterval = SetTimer(TimerID_Selection, nInterval, NULL) > 0 ? nInterval : 0;
}

void CTextDisplay::StopTimer()
{
	if(m_nTimerInterval != 0)
		KillTimer(TimerID_Selection);

	m_nTimerInterval = 0;
}

CString CTextDisplay::GetSelectedText() const
{
	if(m_nSelStart <  0 || m_nSelEnd < 0 || m_nSelEnd <= m_nSelStart )
		return _T("");

    const int len = m_nSelEnd - m_nSelStart;
    CString res;
    LPTSTR pStr = res.GetBuffer(len + 1);
    
    _tcsncpy(pStr, m_pszData + m_nSelStart, len);

    res.ReleaseBuffer(len);
    return res;
}

void CTextDisplay::CopySelToClipboard()
{
    CString selText = GetSelectedText();
	if(selText.GetLength() == 0)
		return;

	BOOL bOK = FALSE;
	
	if(OpenClipboard())
	{
		EmptyClipboard();
		HGLOBAL hData = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE, (selText.GetLength() + 1) * sizeof(TCHAR));
		if(hData != NULL)
		{
			LPTSTR pszData = (LPTSTR)::GlobalLock(hData);
			_tcsncpy(pszData, selText, selText.GetLength());
			pszData[selText.GetLength()] = _T('\0');
			GlobalUnlock(hData);
#ifndef _UNICODE
			bOK = SetClipboardData(CF_TEXT, hData) != NULL;
#else
			bOK = SetClipboardData(CF_UNICODETEXT, hData) != NULL;
#endif
		}
		CloseClipboard();
	}
	
	return;
}

void CTextDisplay::SelectAll()
{
	if(m_nSelPointer > -1)
		return;

	int nLen = m_pszData == 0 ? 0 : _tcslen(m_pszData);
	if(nLen == 0)
		return;

	int oldSelStart = m_nSelStart;
	int oldSelEnd = m_nSelEnd;

	m_nSelStart = 0;
	m_nSelEnd = nLen;

	if(oldSelStart != m_nSelStart || oldSelEnd != m_nSelEnd)
	{
		Invalidate(FALSE);
		UpdateWindow();
	}
}

void CTextDisplay::OnRButtonUp(UINT nFlags, CPoint point) 
{
	CWnd::OnRButtonUp(nFlags, point);

	CString stMenu;
	CMenu mnContext;

	mnContext.CreatePopupMenu();

	mnContext.AppendMenu(MF_STRING, ID_EDIT_COPY, i18n("C&opy\tCtrl+C"));
	mnContext.AppendMenu(MF_SEPARATOR);

		// TFC START
		// TFC Store point for right click menu
	m_pointRClick = point;
	int nLink = GetLinkFromPoint(point);
	if(nLink > -1 && m_nSelStart == -1)
		{
		mnContext.AppendMenu(MF_STRING, ID_COPY_LINK_ADDRESS, i18n("Copy Link Address"));
		mnContext.AppendMenu(MF_SEPARATOR);
		}
	// TFC END

	mnContext.AppendMenu(MF_STRING, ID_EDIT_SELECT_ALL, i18n("Select &All\tCtrl+A"));
	mnContext.EnableMenuItem(ID_EDIT_COPY, (m_nSelStart > -1 ? MF_ENABLED : MF_GRAYED));
	ClientToScreen(&point);
	m_bMenuVisible = TRUE;
	mnContext.TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, point.x, point.y, this);
	m_bMenuVisible = FALSE; 
}

int CTextDisplay::LineFromChar(int nIndex) const
{
	if(nIndex == -1)
		nIndex = m_nSelStart;

	if(nIndex == -1)
		return 0;
	
	for(int n = 0; n < m_Lines.GetCount(); n++)
	{
		const StrInfo Line = m_Lines[n];
		if(Line.pszStrStart - m_pszData <= nIndex)
		{
			if(signed(Line.pszStrStart - m_pszData + Line.dwStrLen) > nIndex)
				return n;
		}
		else
			return n - 1;
	}
	return 0;
}

void CTextDisplay::EnsureSelectionVisible(BOOL bUpdate)
{
	if(m_nSelStart == -1)
		return;

	int nLine = LineFromChar(m_nSelStart);

	if(nLine >= m_nTopIdx && nLine < m_nTopIdx + m_nVisibleLines)
		return;

	if(nLine < m_nTopIdx)
	{
		m_nTopIdx = nLine;	
	}
	else
	{
		int nLineEndSel = LineFromChar(m_nSelEnd-1);
		m_nTopIdx = nLineEndSel - m_nVisibleLines + 1;
	}

	UpdateOnScroll(bUpdate);
}

void CTextDisplay::OnNcLButtonDown(UINT nHitTest, CPoint point) 
{
	SetFocus();
	CWnd::OnNcLButtonDown(nHitTest, point);
}


// TFC Start

void CTextDisplay::CopyLinkAddress()
{
		// m_pointRClick stored  in ::OnRButtonUp
	int nLink = GetLinkFromPoint(m_pointRClick);

		// FALSE, we copy to clipboard
	LinkAction(nLink, FALSE);
	
	StopTimer();
	m_nSelPointer = -1;
	ReleaseCapture();	
	
}

void CTextDisplay::LinkAction(int nLink, BOOL bHyperlink)
{
	if(nLink > -1 && m_nSelStart == -1)
	{
		LinkStrInfo Link = m_Links[nLink];
		CString stLink;

        if(Link.m_Link.IsEmpty())
        {
		    LPTSTR pLink = stLink.GetBuffer(Link.dwStrLen);
		    _tcsncpy(pLink, Link.pszStrStart, Link.dwStrLen);
		    stLink.ReleaseBuffer(Link.dwStrLen);
	    
		    if(stLink.Find(_T('@')) > -1)
		    {
			    stLink.TrimLeft();
			    if(FindNoCase(stLink, _T("http://")) != 0 && FindNoCase(stLink, _T("ftp://")) != 0 && FindNoCase(stLink, _T("https://")) != 0)
			    {
				    if(FindNoCase(stLink, _T("mailto:")) != 0)
					    stLink = _T("mailto:") + stLink;
			    }
		    }
        }
        else
            stLink = Link.m_Link;
        

        if(bHyperlink) 
			OpenHyperlink(stLink);
		//	::ShellExecute(NULL, _T("open"), stLink, NULL, NULL, SW_SHOWNORMAL);
		else
		{
			if(OpenClipboard())
			{
				TCHAR	*buffer;
				CString	strBuffer;
				HGLOBAL	hgClipbuffer;
				int		nPosn;

				EmptyClipboard();

					// TFC remove the 'mailto:' prefix?
				strBuffer = stLink;
				strBuffer.MakeLower();
				nPosn = strBuffer.Find(_T("mailto:"));
				if(nPosn != -1)
					stLink = stLink.Mid(7);	// 'mailto:' length magic number

				hgClipbuffer = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE, (stLink.GetLength()+1)*sizeof(TCHAR));
				buffer = (TCHAR*)GlobalLock(hgClipbuffer);

				_tcscpy(buffer, LPCTSTR(stLink));
				GlobalUnlock(hgClipbuffer);

	#ifndef _UNICODE
				SetClipboardData(CF_TEXT, hgClipbuffer);
	#else
				SetClipboardData(CF_UNICODETEXT, hgClipbuffer);
	#endif

				CloseClipboard();
			}
		}
	}
}
// TFC End

void CTextDisplay::OpenHyperlink(CString stLink)
{
/*
    static CString stBrowser;

	if(FindNoCase(stLink, _T("mailto:")) == 0) // if the link starts with "mailto:"
	{	
		::ShellExecute(NULL, _T("open"), stLink, NULL, NULL, SW_SHOWNORMAL);
		return;
	}

	if(stBrowser.IsEmpty())
	{
		CRegistryKey Key;
		if(Key.Open(HKEY_CLASSES_ROOT, _T("http\\shell\\open\\command"), KEY_READ))
			Key.QueryValue(NULL, stBrowser);

        TRACE("QueryValue: %s \n", (LPCTSTR)stBrowser);
	
		if (stBrowser.GetLength() > 0)
		{
			// Strip the full path from the string
			int nStart = stBrowser.Find('"');
			int nEnd = stBrowser.Find('"', nStart+1);
			// Do we have either quote?
			// If so, then the path contains spaces
			if (nStart >= 0 && nEnd >= 0 && nEnd-nStart > 3)
				stBrowser = stBrowser.Mid(nStart + 1, nEnd - nStart - 1);
			else
			{
				// We may have a pathname with spaces but
				// no quotes (Netscape), e.g.:
				//   C:\PROGRAM FILES\NETSCAPE\COMMUNICATOR\PROGRAM\NETSCAPE.EXE -h "%1"
				// Look for the last backslash
				int nIndex = stBrowser.ReverseFind('\\');
				// Success?
				if (nIndex > 0)
				{
					// Look for the next space after the final
					// backslash
					int nSpace = stBrowser.Find(' ', nIndex);
					// Do we have a space?
					if (nSpace > 0)
						stBrowser = stBrowser.Left(nSpace);    
				}
			}

			CFileStatus status;
			if(FALSE == CFile::GetStatus(stBrowser, status) || (status.m_attribute & CFile::Attribute::directory))
				stBrowser.Empty();
		}
	}


	bool bSuccess = false;

    if(!stBrowser.IsEmpty()) {
        TRACE("ShellExecute: %s \n", (LPCTSTR)stBrowser);
		//bSuccess = (32 < (LRESULT)::ShellExecute(NULL, NULL, stBrowser, stLink, NULL, SW_SHOWNORMAL));
    }

	if(!bSuccess)
    */
		::ShellExecute(NULL, _T("open"), stLink, NULL, NULL, SW_SHOWNORMAL);
       
}

void CTextDisplay::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    if(nChar == VK_HOME)
    {
        m_nTopIdx = 0;
		UpdateOnScroll();
    }
    else if(nChar == VK_END)
    {
        m_nTopIdx = m_Lines.GetCount() - m_nVisibleLines;
        if(m_nTopIdx < 0) m_nTopIdx = 0;
        UpdateOnScroll();
    }
    else 
    {
        CWnd::OnKeyDown(nChar, nRepCnt, nFlags);
    }
}
