// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// SyncSocket.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "SyncTlsSocket.h"
#include <malloc.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

bool IsCertAcceptable(PCCERT_CONTEXT pCertContext, const bool trusted, const bool matchingName)
{
	return trusted && matchingName;
}

// This will get called once, or twice, the first call with "Required" false, which can return any
// certificate it likes, or none at all. If it returns one, that will be sent to the server.
// If that call did not return an acceptable certificate, the procedure may be called again if the server requests a 
// client certificate, whatever is returned on the first call (including null) is sent to the server which gets to decide
// whether or not it is acceptable. If there is a second call (which will have "Required" true and may have 
// pIssuerListInfo non-NULL) it MUST return a certificate or the handshake will fail.

SECURITY_STATUS SelectClientCertificate(PCCERT_CONTEXT & pCertContext, SecPkgContext_IssuerListInfoEx * pIssuerListInfo, bool Required)
{
	return SEC_E_CERT_UNKNOWN;
}

/////////////////////////////////////////////////////////////////////////////
// CSyncTlsSocket

CSyncTlsSocket::~CSyncTlsSocket()
{
	if (m_pEventSink) {
		delete m_pEventSink;
		m_pEventSink = nullptr;
	}
}

void CSyncTlsSocket::Cancel()
{
	Close();
}

BOOL CSyncTlsSocket::Connect(LPCTSTR lpszHostAddress, UINT nHostPort)
{
	BOOL ret = m_Socket.Connect(lpszHostAddress, nHostPort);

	if (!ret) {
		return FALSE;
	}

	if (!m_bUseTLS) {
		return TRUE;
	}

	m_pTLS = UniquePtr<CSSLClient>(new CSSLClient(&m_Socket));

	m_pTLS->ServerCertAcceptable = IsCertAcceptable;
	// m_pTLS->SelectClientCertificate = SelectClientCertificate;

	HRESULT hr = m_pTLS->Initialize(lpszHostAddress);
	if (SUCCEEDED(hr))
	{
		return TRUE;
	}
	else
	{
		m_pTLS->Close();
		return FALSE;
	}
}

void CSyncTlsSocket::Close()
{
	if (m_bUseTLS) {
		if (m_pTLS) {
			m_pTLS->Close();
		}
	}
	else {
		m_Socket.Close();
	}
}

int CSyncTlsSocket::Receive(void* lpBuf, int nBufLen, int nFlags)
{
	if (m_bUseTLS) {
		if (!m_pTLS) return SOCKET_ERROR;
		return m_pTLS->RecvPartial(lpBuf, nBufLen);
	}
	else {
		return m_Socket.Receive(lpBuf, nBufLen, nFlags);
	}
}

int CSyncTlsSocket::Send(const void* lpBuf, int nBufLen, int nFlags )
{
	if (m_bUseTLS) {
		if (!m_pTLS) return SOCKET_ERROR;
		return m_pTLS->SendPartial(lpBuf, nBufLen);
	}
	else {
		return m_Socket.Send(lpBuf, nBufLen, nFlags);
	}
}


BOOL CSyncTlsSocket::ReceiveStr(CString& stBuf, DWORD nHowManyChars)
{
	if (nHowManyChars == 0)
		return FALSE;

#ifndef _UNICODE

	char* pBuf = stBuf.GetBuffer(nHowManyChars+1);

	int iBytesReceived = Receive(pBuf, nHowManyChars);
	
	if(iBytesReceived <= 0)
	{
		stBuf.ReleaseBuffer(0);	
		return FALSE;
	}

	stBuf.ReleaseBuffer(iBytesReceived);
	return TRUE;

#else

	char* pBuf = (char*)alloca(nHowManyChars);

	int iBytesReceived = Receive(pBuf, nHowManyChars);
	
	if(iBytesReceived <= 0)
		return FALSE;
	
	WCHAR* pwBuf = stBuf.GetBuffer(iBytesReceived+1);
	//MultiByteToWideChar(CP_ACP, 0, pBuf, iBytesReceived, pwBuf, iBytesReceived);
	for(int n = 0; n < iBytesReceived; ++n)
		pwBuf[n] = (WCHAR)(unsigned char)pBuf[n];

	stBuf.ReleaseBuffer(iBytesReceived);

	if (m_pEventSink) {
		m_pEventSink->OnReceivedData(pBuf, iBytesReceived);
	}

	return TRUE;

#endif
	
}

BOOL CSyncTlsSocket::SendStr(const CString &stSend)
{
	if(stSend.IsEmpty())
		return FALSE;

#ifndef _UNICODE
	
	return (SOCKET_ERROR != Send(stSend, stSend.GetLength()));

#else

	int nBufLen = stSend.GetLength();
	char* szBuf = (char*)alloca(nBufLen);
	//WideCharToMultiByte(CP_ACP, 0, (LPCWSTR)stSend, nBufLen, szBuf, nBufLen, NULL, NULL);
	for(int n = 0; n < nBufLen; ++n)
		szBuf[n] = (char)(stSend[n] & 0xFF);	
	
	const int res = Send(szBuf, nBufLen);

	if (m_pEventSink) {
		m_pEventSink->OnSentData(szBuf, nBufLen);
	}

	return (SOCKET_ERROR != res);
	
#endif

}
