// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// StrFunctions.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "StrFunctions.h"


static LPTSTR SearchString(LPCTSTR szString, LPCTSTR szSearch, BOOL bMatchCase);
static LPTSTR SearchStringReverse(LPCTSTR szString, int iStringLen, LPCTSTR szSearch, BOOL bMatchCase);

static BOOL DecodeUTF8Char(LPCTSTR pS, LPCTSTR pE, WCHAR& Char0, WCHAR& Char1, int& nOffset);

static WCHAR ACharToWChar(char chAnsi);

static BOOL decode_base64(LPCTSTR aIn, size_t aInLen, LPTSTR aOut, size_t aOutSize, size_t* aOutLen);
static int isbase64(int a);

static CString MapUnicode2StringAndDelete(LPWSTR pUnicode);

#ifndef _UNICODE
#define CharToTCHAR(ch) (ch)
#else
#define CharToTCHAR(ch) ACharToWChar(ch)
#endif

// Sucht St in Src
// - wenn gefunden: der String links von St wird zurckgegeben; Src wird auf den String rechts von St gesetzt
// - wenn nicht gefunden: Src wird zurckgegeben; Src wird auf "" gesetzt 
CString Parse(CString& Src, LPCTSTR St)
{
	int i = Src.Find(St);
	if(i == -1)
	{
		CString temp(Src);
		Src = _T("");
		return temp;
	}
	else 
	{
		CString temp(Src.Left(i));
		//Src = Src.Mid(i + _tcslen(St));
        Src.Delete(0, i + _tcslen(St));
		return temp;
	}
}

CString ParseReverse(CString& Src, LPCTSTR St)
{
	LPCTSTR pFound = SearchStringReverse(Src, Src.GetLength(), St, TRUE);
	if(pFound == NULL)
	{
		CString temp(Src);
		Src = _T("");
		return temp;
	}
	else 
	{
		int i = pFound - (LPCTSTR)Src;
		CString temp(Src.Mid(i+1));
		Src.Delete(i, Src.GetLength() - i);
		return temp;
	}
}

CString ParseNoCase(CString& Src, CString St)
{
	int i = FindNoCase(Src, St);
	if(i == -1)
	{
		CString temp(Src);
		Src = "";
		return temp;
	}
	else 
	{
		CString temp(Src.Left(i));
		Src = Src.Mid(i + lstrlen(St));
		return temp;
	}
}

CString ParseConst(const CString& Src, LPCTSTR St, int& nStartPos)
{
	int i = Src.Find(St, nStartPos);
	if(i == -1)
	{
		int Idx = nStartPos;
		nStartPos = -1;
		return Src.Mid(Idx);
	}
	else 
	{
		int Idx = nStartPos;
		nStartPos = i + _tcslen(St);
		return Src.Mid(Idx, i - Idx);
	}
}

/*
CString QPDecode(CString Data, BOOL bReplaceUnderscoreWithSpace) 
{
	if(bReplaceUnderscoreWithSpace)
		Data.Replace(_T('_'), _T(' '));

	int i = Data.Find(_T('='));
	while(i > -1) 
	{
		CString stEnc(Data.Mid(i+1, 2));
		if(stEnc == _T("\r\n"))
		{
			Data.Replace(Data.Mid(i, 3), _T(""));
			i = Data.Find(_T('='), i);
		} 
		else
		{
			TCHAR Char = CharToTCHAR((char)_tcstoul(stEnc, NULL, 16));
			if(Char == 0)
			{
				Char = 0;
			}
			if(Char != 0 || stEnc == _T("00")) {
				Data.Delete(i, 3);
				Data.Insert(i, Char);
			}
			i = Data.Find(_T('='), i + 1);
		}
	}
	return Data; 
}

*/

bool isHexChar(TCHAR a)
{
	return (_T('A') <= a && a <= _T('F'))
        || (_T('a') <= a && a <= _T('f'))
        || (_T('0') <= a && a <= _T('9'));
}

static CString QPDecodeIntern(LPCTSTR szData, BOOL bReplaceUnderscoreWithSpace, TCHAR charStart)
{
	CString stBuf;
	LPTSTR pBuf = stBuf.GetBuffer(_tcslen(szData)+1);
	LPTSTR pTrg  = pBuf;

	TCHAR HexBuf[4] = _T("");

	for(LPCTSTR pSrc = szData; *pSrc != _T('\0'); ++pSrc)
	{
		if(bReplaceUnderscoreWithSpace && *pSrc == _T('_'))
		{
			*pTrg++ = _T(' ');
			continue;
		}

		if(*pSrc == charStart)
		{
			if(pSrc[1] == _T('\r') && pSrc[2] == _T('\n'))
			{
				pSrc += 2;
				continue;
			}

			HexBuf[0] = pSrc[1];
			HexBuf[1] = pSrc[2];

			if(!isHexChar(HexBuf[0]) || !isHexChar(HexBuf[1])) {
				*pTrg++ = *pSrc;
				continue;
			}

			TCHAR Char = (TCHAR)_tcstoul(HexBuf, NULL, 16);

			if(Char != 0 || (HexBuf[0] == _T('0') && HexBuf[1] == _T('0')))
				*pTrg++ = Char;

			pSrc += 2;

		}
		else
			*pTrg++ = *pSrc;
	}
	
	stBuf.ReleaseBuffer(pTrg-pBuf);
	return stBuf;
}

CString QPDecode2(LPCTSTR szData, BOOL bReplaceUnderscoreWithSpace)
{
	return QPDecodeIntern(szData, bReplaceUnderscoreWithSpace, _T('%'));
}

CString QPDecode(LPCTSTR szData, BOOL bReplaceUnderscoreWithSpace)
{
	return QPDecodeIntern(szData, bReplaceUnderscoreWithSpace, _T('='));
}

CString Base64Encode(const CString& Src)
{

#ifndef _UNICODE
	LPCSTR pSrc = Src;
#else
	int nBufLen = Src.GetLength();
	char* szBuf = (char*)alloca(nBufLen+1);
	WideCharToMultiByte(CP_ACP, 0, (LPCWSTR)Src, nBufLen, szBuf, nBufLen, NULL, NULL);
	LPCSTR pSrc = szBuf;
#endif	

	CString stRes;
	TCHAR   szEnc[] = _T("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/");
	unsigned char  c1, c2, c3;
		
	int w1, w2, w3, w4;

	for(int n = 0; n < Src.GetLength(); n += 3)
	{
		c1 = (unsigned char)pSrc[n];   
		if(Src.GetLength() > n + 1) c2 = (unsigned char)pSrc[n+1]; else c2 = _T('\0');  
		if(Src.GetLength() > n + 2) c3 = (unsigned char)pSrc[n+2]; else c3 = _T('\0');  
		
		w1 = c1 >> 2;
		w2 = ((c1 & 0x3) << 4) | (c2 >> 4);
		if(Src.GetLength() > n + 1)  w3 = ((c2 & 0xF) << 2) | (c3 >> 6);  else  w3 = -1;
		if(Src.GetLength() > n + 2)  w4 = (c3 & 0x3F);  else  w4 = -1;
		
		if(w1 >= 0) stRes += szEnc[w1]; else stRes += _T('=');
		if(w2 >= 0) stRes += szEnc[w2]; else stRes += _T('=');
		if(w3 >= 0) stRes += szEnc[w3]; else stRes += _T('=');
		if(w4 >= 0) stRes += szEnc[w4]; else stRes += _T('=');
 	}
	return stRes;
}

CString Base64Encode(const unsigned char * pData, size_t len)
{
	CString stRes;
	TCHAR   szEnc[] = _T("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/");
	unsigned char  c1, c2, c3;

	int w1, w2, w3, w4;

	for (size_t n = 0; n < len; n += 3)
	{
		c1 = (unsigned char)pData[n];
		if (len > n + 1) c2 = (unsigned char)pData[n + 1]; else c2 = _T('\0');
		if (len > n + 2) c3 = (unsigned char)pData[n + 2]; else c3 = _T('\0');

		w1 = c1 >> 2;
		w2 = ((c1 & 0x3) << 4) | (c2 >> 4);
		if (len > n + 1)  w3 = ((c2 & 0xF) << 2) | (c3 >> 6);  else  w3 = -1;
		if (len > n + 2)  w4 = (c3 & 0x3F);  else  w4 = -1;

		if (w1 >= 0) stRes += szEnc[w1]; else stRes += _T('=');
		if (w2 >= 0) stRes += szEnc[w2]; else stRes += _T('=');
		if (w3 >= 0) stRes += szEnc[w3]; else stRes += _T('=');
		if (w4 >= 0) stRes += szEnc[w4]; else stRes += _T('=');
	}
	return stRes;
}


CString Base64Decode(const CString& Src)
{
	CString stRes;

	int srcLen = Src.GetLength();
    LPCTSTR srcBuf = Src;
    int destSize = srcLen;

    // Allocate destination buffer
    LPTSTR destBuf = stRes.GetBuffer(destSize);

    // Encode source to destination
    size_t destLen = 0;
    decode_base64(srcBuf, srcLen, destBuf, destSize, &destLen);
    stRes.ReleaseBuffer(destLen);
    
	return stRes;
}


static char base64idx[128] = {
    '\377','\377','\377','\377','\377','\377','\377','\377',
    '\377','\377','\377','\377','\377','\377','\377','\377',
    '\377','\377','\377','\377','\377','\377','\377','\377',
    '\377','\377','\377','\377','\377','\377','\377','\377',
    '\377','\377','\377','\377','\377','\377','\377','\377',
    '\377','\377','\377',    62,'\377','\377','\377',    63,
        52,    53,    54,    55,    56,    57,    58,    59,
        60,    61,'\377','\377','\377','\377','\377','\377',
    '\377',     0,     1,     2,     3,     4,     5,     6,
         7,     8,     9,    10,    11,    12,    13,    14,
        15,    16,    17,    18,    19,    20,    21,    22,
        23,    24,    25,'\377','\377','\377','\377','\377',
    '\377',    26,    27,    28,    29,    30,    31,    32,
        33,    34,    35,    36,    37,    38,    39,    40,
        41,    42,    43,    44,    45,    46,    47,    48,
        49,    50,    51,'\377','\377','\377','\377','\377'
};

static BOOL decode_base64(LPCTSTR aIn, size_t aInLen, LPTSTR aOut, size_t aOutSize, size_t* aOutLen)
{
    if (!aIn || !aOut || !aOutLen)
        return FALSE;

    size_t inLen = aInLen;
    LPTSTR out = aOut;
    size_t outSize = ( ( inLen + 3 ) / 4 ) * 3;
    if (aOutSize < outSize)
        return FALSE;
    /* Get four input chars at a time and decode them. Ignore white space
     * chars (CR, LF, SP, HT). If '=' is encountered, terminate input. If
     * a char other than white space, base64 char, or '=' is encountered,
     * flag an input error, but otherwise ignore the char.
     */
    int isErr = 0;
    int isEndSeen = 0;
    int a1, a2, a3, a4;
    size_t inPos = 0;
    size_t outPos = 0;

    while (inPos < inLen) {
        a1 = a2 = a3 = a4 = 0;
        while (inPos < inLen) {
            a1 = aIn[inPos++] & 0xFF;
            if (isbase64(a1)) {
                break;
            }
            else if (a1 == _T('=')) {
                isEndSeen = 1;
                break;
            }
            else if (a1 != _T('\r') && a1 != _T('\n') && a1 != _T(' ') && a1 != _T('\t')) {
                isErr = 1;
            }
        }
        while (inPos < inLen) {
            a2 = aIn[inPos++] & 0xFF;
            if (isbase64(a2)) {
                break;
            }
            else if (a2 == _T('=')) {
                isEndSeen = 1;
                break;
            }
            else if (a2 != _T('\r') && a2 != _T('\n') && a2 != _T(' ') && a2 != _T('\t')) {
                isErr = 1;
            }
        }
        while (inPos < inLen) {
            a3 = aIn[inPos++] & 0xFF;
            if (isbase64(a3)) {
                break;
            }
            else if (a3 == _T('=')) {
                isEndSeen = 1;
                break;
            }
            else if (a3 != _T('\r') && a3 != _T('\n') && a3 != _T(' ') && a3 != _T('\t')) {
                isErr = 1;
            }
        }
        while (inPos < inLen) {
            a4 = aIn[inPos++] & 0xFF;
            if (isbase64(a4)) {
                break;
            }
            else if (a4 == _T('=')) {
                isEndSeen = 1;
                break;
            }
            else if (a4 != _T('\r') && a4 != _T('\n') && a4 != _T(' ') && a4 != _T('\t')) {
                isErr = 1;
            }
        }
        if (isbase64(a1) && isbase64(a2) && isbase64(a3) && isbase64(a4)) {
            a1 = base64idx[a1] & 0xFF;
            a2 = base64idx[a2] & 0xFF;
            a3 = base64idx[a3] & 0xFF;
            a4 = base64idx[a4] & 0xFF;
            out[outPos++] = (TCHAR) (((a1 << 2) & 0xFC) | ((a2 >> 4) & 0x03));
            out[outPos++] = (TCHAR) (((a2 << 4) & 0xF0) | ((a3 >> 2) & 0x0F));
            out[outPos++] = (TCHAR) (((a3 << 6) & 0xC0) | ( a4       & 0x3F));
        }
        else if (isbase64(a1) && isbase64(a2) && isbase64(a3) && a4 == '=') {
            a1 = base64idx[a1] & 0xFF;
            a2 = base64idx[a2] & 0xFF;
            a3 = base64idx[a3] & 0xFF;
            out[outPos++] = (TCHAR) (((a1 << 2) & 0xFC) | ((a2 >> 4) & 0x03));
            out[outPos++] = (TCHAR) (((a2 << 4) & 0xF0) | ((a3 >> 2) & 0x0F));
            break;
        }
        else if (isbase64(a1) && isbase64(a2) && a3 == _T('=') && a4 == _T('=')) {
            a1 = base64idx[a1] & 0xFF;
            a2 = base64idx[a2] & 0xFF;
            out[outPos++] = (TCHAR) (((a1 << 2) & 0xFC) | ((a2 >> 4) & 0x03));
            break;
        }
        else {
            break;
        }
        if (isEndSeen) {
            break;
        }
    } /* end while loop */

    *aOutLen = outPos;
    return (isErr ? FALSE : TRUE);
}

inline int isbase64(int a) {
    return (_T('A') <= a && a <= _T('Z'))
        || (_T('a') <= a && a <= _T('z'))
        || (_T('0') <= a && a <= _T('9'))
        || a == _T('+') || a == _T('/');
}

int mimedecode(const CString& Src)
{
	static CString Alpha(_T("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"));
	if(Src.GetLength() == 0)
		return -1;
	return Alpha.Find(Src);
}


int FindNoCase(LPCTSTR szSrc, LPCTSTR szFind, int nStart)
{
	if(szSrc == NULL || szFind == NULL || nStart < 0)
		return -1;
	
	LPCTSTR szStart = szSrc;
	
	TCHAR szFirst = lcase(*szFind);
	szSrc += nStart;
	while(*szSrc)
	{
		if(lcase(*szSrc) == szFirst)
		{
			LPCTSTR a = szSrc + 1;
			LPCTSTR b = szFind + 1;
			
			while(*b)
				if(lcase(*a++) != lcase(*b++)) goto Jump;

			return (szSrc - szStart); 
		}
Jump:	szSrc++;

	}
	return -1;
}


TCHAR lcase(TCHAR c)
{
	if (c < 128) {
		if (c >= _T('A') && c <= _T('Z'))
			return c + 32; //return (TCHAR)(_T('a') + c - _T('A'));
		else
			return c;
	}
	return tolower(c);
/* slow:
    TCHAR buffer[2] = {c, _T('\0')};
    _tcslwr(buffer);
    return buffer[0];
*/
}



int AdvancedSearch(BOOL bSearchDown, LPCTSTR szString, LPCTSTR szSearch, int iStartPos, BOOL bMatchCase, BOOL bWholeWord)
{
	LPCTSTR p = szString;
	
	if(bSearchDown)
	{
		p += iStartPos;
		p = SearchString(p, szSearch, bMatchCase);
	}
	else
		p = SearchStringReverse(p, iStartPos + 1, szSearch, bMatchCase);

	if(bWholeWord)
	{
		while(p)
		{
			TCHAR c = _T('\0');

			if(p == szString) 
			{ /* Wort am Anfang */
				c = p[lstrlen(szSearch)]; /* rechts */
				if(c == _T(' ') || c == _T(',') || c == _T('.') || c == _T('!') || c == _T('?') || c == _T(';') || c == _T(')')  || c == _T('"') || c == _T(':') || c == _T('\r') || c == _T('\0'))
					break;
			}
			
			if(*(p-1) == _T(' ') || *(p-1) == _T('\n') || *(p-1) == _T('(') || *(p-1) == _T('"'))  /* links  Wort mittendrin oder am Ende */
			{
				c = p[lstrlen(szSearch)]; /* rechts */
				if(c == _T(' ') || c == _T(',') || c == _T('.') || c == _T('!') || c == _T('?') || c == _T(';') || c == _T(')')  || c == _T('"') || c == _T(':') || c == _T('\r') || c == _T('\0'))
					break;
			}
							
			if(bSearchDown)
				p = SearchString((p+1), szSearch, bMatchCase);
			else
			{
				p = SearchStringReverse(szString, iStartPos + 1, szSearch, bMatchCase);
				iStartPos--;
			}

		}
	}
	return (p ? p - szString : -1);

}



static LPTSTR SearchString(LPCTSTR szString, LPCTSTR szSearch, BOOL bMatchCase)
{
	if(szString == 0 || szSearch==0)
		return 0;
	
	if(bMatchCase)
		return (LPTSTR)_tcsstr(szString, szSearch);
	
	TCHAR szFirst = lcase(*szSearch);
	while(*szString)
	{
		if(lcase(*szString) == szFirst)
		{
			LPCTSTR a = szString + 1;
			LPCTSTR b = szSearch + 1;
			
			while(*b)
				if(lcase(*a++) != lcase(*b++)) goto Jump;

			return (LPTSTR)szString;					
		}
Jump:	szString++;
	}
	return 0;
}



static LPTSTR SearchStringReverse(LPCTSTR szString, int iStringLen, LPCTSTR szSearch, BOOL bMatchCase)
{
	int iStringStart = (int)szString - 1;
	int iSearchStart = (int)szSearch - 1;
	int iSearchLen;
	
	if(szString == 0 || szSearch==0 || iStringLen==0)
		return 0;
	
	iSearchLen = _tcslen(szSearch);
	
	szString += iStringLen - 1;
	szSearch += iSearchLen - 1;
	
	if(bMatchCase)
	{
			TCHAR szFirst = *szSearch;
			while((int)szString > iStringStart)
			{
				if(*szString == szFirst)
				{
					LPCTSTR a = szString - 1;
					LPCTSTR b = szSearch - 1;
					
					while((int)b > iSearchStart)
						if(*a-- != *b--) goto Jump1;
					
					return (LPTSTR)(szString - iSearchLen + 1);					
				}
			Jump1:	szString--;
			}
	}
	else
	{
			TCHAR szFirst = lcase(*szSearch);
			while((int)szString > iStringStart)
			{
				if(lcase(*szString) == szFirst)
				{
					LPCTSTR a = szString - 1;
					LPCTSTR b = szSearch - 1;
					
					while((int)b > iSearchStart)
						if(lcase(*a--) != lcase(*b--)) goto Jump2;
					
					return (LPTSTR)(szString - iSearchLen + 1);
				}
			Jump2:	szString--;
			}
	}
	return 0;
}


CString ConvertLineBreaks(const CString& stSrc)
{
	CString stRes;

	int iCR = 0, iLF = 0;
	
	for(int n = 0; n < stSrc.GetLength(); n++)
	{
		if(stSrc[n] == _T('\r'))
			iCR++;
		else if(stSrc[n] == _T('\n'))
			iLF++;
	}

	LPTSTR pszTrg = stRes.GetBuffer(stSrc.GetLength() + iCR + iLF);
	
	TCHAR *pT = pszTrg; 
	const TCHAR *pS = (LPCTSTR)stSrc;
	TCHAR cC = 0, cP = 0;

	while(*pS)
	{
		cC = *pS++;
		if(cC != _T('\r') && cC != _T('\n'))
			*pT++ = cC;
		else 
		{
			if(cC != _T('\n') || cP != _T('\r')) 
			{
				*pT++ = _T('\r');	
				*pT++ = _T('\n');
			}
		}
		cP = cC;
	}
	*pT = 0;

	stRes.ReleaseBuffer();
	
	return stRes;
}



CString Crypt(CString stData, BOOL bEncode)
{
	TCHAR szKey[] = _T("MonTueWedThuFriSatSun");
	CString stResult;
	int idxKey = 0;
	
	if(!bEncode)
		stData = Base64Decode(stData);

	for(int n = 0; n < stData.GetLength(); n++, idxKey++)
	{
		if(idxKey >= sizeof(szKey)/sizeof(TCHAR)-1)
			idxKey = 0;
			
		TCHAR chKey = szKey[idxKey];	
		stResult += (TCHAR)(stData[n] ^ chKey);
	}

	if(bEncode)
		stResult = Base64Encode(stResult);

	return stResult;
}



CString DecodeUTF8(LPCTSTR szSrc)
{
	LPCTSTR pSrc = szSrc;
	const int LenSrc = _tcslen(szSrc);
	LPCTSTR pSrcEnd = pSrc + LenSrc - 1;
	
	WCHAR * const pUnicode = new WCHAR[LenSrc+1];
	WCHAR * pDest = pUnicode;

	while(*pSrc)
	{
		const char C = (char)(*pSrc & 0xFF);

		if((C & 0x80) == 0x00)
		{
			*pDest++ = C;
			pSrc++;
		}
		else
		{
			WCHAR Char = 0;
			WCHAR Char2 = 0;
			int   nOffset = 0;
			if (DecodeUTF8Char(pSrc, pSrcEnd, Char, Char2, nOffset))
			{
				*pDest++ = Char;
				if (Char2 != 0) {
					*pDest++ = Char2;
				}
				pSrc += nOffset;
			}
			else {
				*pDest++ = ' ';
				pSrc++;
			}			
		}
	}
	*pDest = 0x0000;

	return MapUnicode2StringAndDelete(pUnicode);
}


static BOOL DecodeUTF8Char(LPCTSTR pS, LPCTSTR pE, WCHAR& Char, WCHAR& Char1, int& nOffset)
{
	if(pS == NULL || pE == NULL)
		return FALSE;

	if( (*pS & 0x80) == 0x00 ) nOffset = 1;
	else if( (*pS & 0xE0) == 0xC0 ) nOffset = 2;
	else if( (*pS & 0xF0) == 0xE0 ) nOffset = 3;
	else if( (*pS & 0xF8) == 0xF0 ) nOffset = 4;
	else if( (*pS & 0xFC) == 0xF8 ) nOffset = 5;
	else if( (*pS & 0xFE) == 0xFC ) nOffset = 6;
	else return FALSE;

	if(pS + nOffset - 1 > pE)
		return FALSE;

	unsigned char lowC  = 0;
	unsigned char highC = 0;

	switch (nOffset)
	{
	case 1:
		Char = (WCHAR)(*pS & 0xFF);
		return TRUE;

	case 2:
		if ((pS[1] & 0xC0) != 0x80) return FALSE;

		highC = ((pS[0] & 0x1C) >> 2);
		lowC = ((pS[0] & 0x03) << 6) | (pS[1] & 0x3F);

		Char = (WCHAR)MAKEWORD(lowC, highC);
		return TRUE;

	case 3:
		if ((pS[1] & 0xC0) != 0x80) return FALSE;
		if ((pS[2] & 0xC0) != 0x80) return FALSE;

		highC = ((pS[0] & 0x0F) << 4) | ((pS[1] & 0x3C) >> 2);
		lowC = ((pS[1] & 0x03) << 6) | (pS[2] & 0x3F);

		Char = (WCHAR)MAKEWORD(lowC, highC);
		return TRUE;

	case 4: {
		const int c = pS[0] & 0x07;
		const int c1 = pS[1] & 0x3F;
		const int c2 = pS[2] & 0x3F;
		const int c3 = pS[3] & 0x3F;

		if ((c1 | c2 | c3) >= 0) {
			const int r = (c << 18) | (c1 << 12) | (c2 << 6) | c3;
			if (r >= 65536 && r <= 1114111) {

				const int x = r - 0x10000;
				const int low = 0xDC00 | (x & 0x3FF);
				const int hig = 0xD800 | ((x & 0xFFC00) >> 10);

				Char1 = low;
				Char  = hig;
				return TRUE;
			}
		}
	}

	default:
		Char = _T('?');
		return TRUE;		
	}
}

inline static WCHAR ACharToWChar(char chAnsi)
{
	WCHAR chwTrg;
	MultiByteToWideChar(CP_ACP, 0, &chAnsi, 1, &chwTrg, 1);
	return chwTrg;
}

BOOL SplitExeCmdParam(LPCTSTR szPath, CString& stExe, CString& stCmdParam)
{
	CString stFileName = szPath;
	stFileName.TrimLeft();
	if(stFileName.IsEmpty())
		return FALSE;

	int nIdx = stFileName.Find(_T('\"'), 1);

	if(stFileName[0] == _T('\"') && nIdx > 0)
	{
		if(stFileName.GetLength() > nIdx+1) 
			stCmdParam = stFileName.Mid(nIdx+1); 
		stExe = stFileName.Mid(1, nIdx - 1);
	} 
	else
	{
		int nIdx = stFileName.Find(_T(' '));
		if(nIdx == -1)
		{
			stCmdParam = _T("");
			stExe = stFileName;
		}
		else
		{
			stCmdParam = stFileName.Mid(nIdx+1); 
			stExe = stFileName.Left(nIdx);
		}
	} 

	stCmdParam.TrimLeft(); stCmdParam.TrimRight();

	return TRUE;
}


BOOL IsDouble(LPCTSTR szFormat)
{
	int nLen = _tcslen(szFormat);
	TCHAR Type = szFormat[nLen-1];
	switch(Type)
	{
	case _T('e'):
	case _T('E'):
	case _T('f'):
	case _T('g'):
	case _T('G'):
		return TRUE;

	default:
		return FALSE;
	}
}

int WriteStr(LPTSTR pDest, LPCTSTR pSrc, int nCount)
{
	LPTSTR pDestStart = pDest;
	TCHAR chLastChar = _T('\0');
	for(; nCount > 0; nCount--)
	{
		TCHAR Char = *pSrc++;
		if(Char == _T('%'))
			*pDest++ = _T('%');
		else if(Char == _T('{'))
		{
			if(chLastChar == _T('{')) {
				chLastChar = _T('\0');
				continue;
			}
		}
		*pDest++ = Char;
		chLastChar = Char;
	}
	return pDest - pDestStart;
}

CString StrFormat(LPCTSTR szFormat, ...)
{
	va_list argList;
	va_start(argList, szFormat);
	
	int nInParamCount = 0;
	int nOutParamCount = 0;
	
	LPCTSTR pS = szFormat;
	for(; *pS != _T('\0'); pS++)
	{
		if(*pS != _T('{'))
			continue;

		if(*++pS == _T('{'))
			continue;

		TCHAR FirstDigit = *pS;
		if(FirstDigit < _T('1') || FirstDigit > _T('9'))
			continue;

		if(*++pS == _T('}'))
		{
			if(FirstDigit - _T('0') > nInParamCount)
				nInParamCount = FirstDigit - _T('0');
			++nOutParamCount;
			continue;
		}

		TCHAR SecondDigit = *pS;
		if(SecondDigit < _T('0') || SecondDigit > _T('9'))
			continue;

		if(*++pS == _T('}'))
		{
			int x = 10 * (FirstDigit - _T('0'))  +  (SecondDigit - _T('0'));
			if(x > nInParamCount)
				nInParamCount = x;
			++nOutParamCount;
		}		
	}

	struct PARAM  {
		LPCTSTR szFormat;
		BOOL	bDbl;
		union {
			DWORD	nValue;
			double  dDouble; 
		};
	};
	PARAM* InParams   =  new PARAM[nInParamCount];
	PARAM* OutParams  =  new PARAM[nOutParamCount];
	memset(OutParams, 0, sizeof(PARAM)*nOutParamCount);


	va_list argListRead = argList;
	for(int i = 0; i < nInParamCount; i++)
	{
		InParams[i].szFormat = va_arg(argListRead, LPCTSTR);
		InParams[i].bDbl = IsDouble(InParams[i].szFormat);
		if(InParams[i].bDbl)
			InParams[i].dDouble = va_arg(argListRead, double);
		else
			InParams[i].nValue = va_arg(argListRead, DWORD);
	}


	LPTSTR szFinalFormat = new TCHAR[_tcslen(szFormat)+10*nOutParamCount+1];
	LPTSTR pTrg = szFinalFormat;
	int nParamIdx = 0;
	int nParamValue = 0;
	LPCTSTR pStart = szFormat;
	LPCTSTR pEnd = szFormat;
	int nItemLen = 0;
	for(pS = szFormat; *pS != _T('\0'); pS++)
	{
		if(*pS != _T('{'))
			continue;

		if(*++pS == _T('{'))
			continue;

		TCHAR FirstDigit = *pS;
		if(FirstDigit < _T('1') || FirstDigit > _T('9'))
			continue;

		if(*++pS == _T('}'))
		{
			nParamValue = FirstDigit - _T('0');
			pEnd = pS - 2;
			nItemLen = 3;
		}
		else
		{
			TCHAR SecondDigit = *pS;
			if(SecondDigit < _T('0') || SecondDigit > _T('9'))
				continue;

			if(*++pS != _T('}'))
				continue;
			
			nParamValue = 10 * (FirstDigit - _T('0'))  +  (SecondDigit - _T('0'));
			pEnd = pS - 3;
			nItemLen = 4; 
		}

		OutParams[nParamIdx++] = InParams[nParamValue-1];
		pTrg += WriteStr(pTrg, pStart, pEnd-pStart);
		*pTrg++ = _T('%');
		_tcscpy(pTrg, InParams[nParamValue-1].szFormat);
		pTrg += _tcslen(InParams[nParamValue-1].szFormat);
		pStart = pEnd + nItemLen;
	}
	pTrg += WriteStr(pTrg, pStart, _tcslen(pStart));
	*pTrg = _T('\0');

	delete [] InParams;

	char * pArgs = new char[nOutParamCount*sizeof(double)];
	va_list argListWrite = (va_list)pArgs;
	for(int n = 0; n < nOutParamCount; n++)
	{
		if(OutParams[n].bDbl)
			va_arg(argListWrite, double) = OutParams[n].dDouble ;
		else
			va_arg(argListWrite, DWORD) = OutParams[n].nValue;
	}

	delete [] OutParams;
		
	CString stRes;
	stRes.FormatV(szFinalFormat, (va_list)pArgs);
	delete [] szFinalFormat;
	delete [] pArgs;
	va_end(argList);
	return stRes;
}

int FindInStrArray(CStringArray& StrArray, const CString& stFind, int nPos)
{
	for(int n = nPos; n < StrArray.GetSize(); n++)
	{
		if(StrArray[n] == stFind)
			return n;
	}
	return -1;
}

CString word_boundaries(_T(" \t\r\n.?!,:;@<>'$&|`\"()[]{}+-*\\/~%="));

bool IsWordBoundary(const CString& str, int idx)
{
    if(idx < 0 || idx >= str.GetLength()) 
        return true;

    TCHAR c = str[idx];
    return (word_boundaries.Find(c) >= 0);
}




CString TranslateCodepage(const CString& input, WCHAR codepage[]) 
{
	const int InputLen = input.GetLength();
	if(InputLen == 0 || codepage == NULL)
		return input;

	WCHAR * pUnicode = new WCHAR[InputLen+1];

	for(int n = 0; n < InputLen; ++n) 
	{
		const unsigned int idx = (input[n] & 0xFF);
		pUnicode[n] = codepage[idx];
	}
	pUnicode[InputLen] = 0x00;

	return MapUnicode2StringAndDelete(pUnicode);
}

CString MapUnicode2StringAndDelete(LPWSTR pUnicode)
{
	if(pUnicode == NULL)
		return _T("");

#ifdef _UNICODE
	
	CString res(pUnicode);

#else

	const int OutputBuffSize = 2 * (wcslen(pUnicode)+1);
	CString res;
	LPSTR pDest = res.GetBuffer(OutputBuffSize+1);

	//if CP_ACP not supports Cyrillic -> Map Cyrillic to Latin transcript
	
	WideCharToMultiByte(CP_ACP, 0, pUnicode, -1, pDest, OutputBuffSize, "?", NULL);
	
	res.ReleaseBuffer();

#endif

	delete [] pUnicode;

	return res;
}

bool ContainsEncoding(const CString& str, LPCTSTR encoding)
{
	const int idx = FindNoCase(str, encoding);
	if(idx < 0)
		return false;

	return IsWordBoundary(str, idx-1) && IsWordBoundary(str, idx + _tcslen(encoding));
}

struct CP_Entry 
{
	LPCTSTR Name;
	WCHAR *Codepage;
};


#include "GeneratedCodepages.h"

CString DecodeGBK(LPCTSTR szSrc);

void ConvertEncodedText(CString& text, CString strContainingEncoding)
{
	strContainingEncoding.Replace(_T("-"), _T(""));

	if (ContainsEncoding(strContainingEncoding, _T("utf8")))
	{
		text = DecodeUTF8(text);
	}
	else if (ContainsEncoding(strContainingEncoding, _T("gbk"))    ||
		     ContainsEncoding(strContainingEncoding, _T("gb2312")) || 
		     ContainsEncoding(strContainingEncoding, _T("cp936"))  ||
			 ContainsEncoding(strContainingEncoding, _T("ms936"))  ||
		     ContainsEncoding(strContainingEncoding, _T("gb18030"))
		    ) {
		text = DecodeGBK(text);
	}
	else 
	{
		for(int n = 0; n < sizeof(CP_Mappings)/sizeof(CP_Entry); ++n)
		{
			const CP_Entry& entry = CP_Mappings[n];
			
			if(ContainsEncoding(strContainingEncoding, entry.Name))
			{
				text = TranslateCodepage(text, entry.Codepage);
				return;
			}
		}
	}
}

#include "GBK/gbk.h"

CString DecodeGBK(LPCTSTR szSrc)
{
	LPCTSTR pSrc = szSrc;
	const size_t N = _tcslen(szSrc);

	unsigned char * pIn = new unsigned char[N + 1];
	unsigned char * const pIn_Start = pIn;
	while (*pSrc) {
		*pIn++ = (unsigned char)*pSrc++;
	}
	*pIn = 0;
	pIn = pIn_Start;

	WCHAR * pW = new WCHAR[N + 1];
	WCHAR * const pWStart = pW;

	size_t remaining = N;
	while (*pIn) {
		ucs4_t unicode = 0;
		const int ret = ces_gbk_mbtowc(NULL, &unicode, pIn, remaining);
		if (ret <= 0) {
			break;
		}
		remaining -= ret;
		pIn += ret;
		*pW++ = unicode;
	}
	*pW = 0;

	delete[] pIn_Start;

	return MapUnicode2StringAndDelete(pWStart);
}

