// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Settings.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "Settings.h"
#include "StrFunctions.h"
#include <map>

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif



CSettingsKey::CSettingsKey()
{
	m_bOpen = FALSE;
	m_bRegistry = FALSE;
	m_pFileKey = NULL;
}

CSettingsKey::~CSettingsKey()
{
	Close();
}

BOOL CSettingsKey::Close()
{
	if(!m_bOpen)
		return TRUE;
	
	BOOL bRes = TRUE;
	if(m_bRegistry)
		bRes = m_RegKey.Close();
	
	m_bOpen = FALSE;
	return bRes;
}


BOOL CSettingsKey::OpenSubKey(LPCTSTR szSubKey)
{
	if(!m_bOpen || szSubKey == NULL || szSubKey[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.Open(m_RegKey, szSubKey);
	else
	{
		CFileKey* pFileKey = GetSubKeyByName(m_pFileKey, szSubKey);
		if(pFileKey == NULL) return FALSE;
		m_pFileKey = pFileKey;
		return TRUE;
	}
}


BOOL CSettingsKey::CreateSubKey(LPCTSTR szSubKey)
{
	if(!m_bOpen || szSubKey == NULL || szSubKey[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.Create(m_RegKey, szSubKey);
	else
	{
		CFileKey* pFileKey = GetSubKeyByName(m_pFileKey, szSubKey, TRUE);
		if(pFileKey == NULL) return FALSE;
		m_pFileKey = pFileKey;
		return TRUE;
	}
}


 
BOOL CSettingsKey::OpenSubKey(LPCTSTR szSubKey, CSettingsKey& SubKey)
{
	if(!m_bOpen || szSubKey == NULL || szSubKey[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
	{
		BOOL bRes = SubKey.m_RegKey.Open(m_RegKey, szSubKey);
		if(bRes)
		{
			SubKey.m_bOpen = TRUE;
			SubKey.m_bRegistry = TRUE;
		}
		return bRes;
	}
	else
	{
		CFileKey* pFileKey = GetSubKeyByName(m_pFileKey, szSubKey);
		if(pFileKey == NULL) return FALSE;
		SubKey.m_pFileKey = pFileKey;
		SubKey.m_bOpen = TRUE;
		SubKey.m_bRegistry = FALSE;
		return TRUE;
	}
}


BOOL CSettingsKey::CreateSubKey(LPCTSTR szSubKey, CSettingsKey& SubKey)
{
	if(!m_bOpen || szSubKey == NULL || szSubKey[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
	{
		BOOL bRes = SubKey.m_RegKey.Create(m_RegKey, szSubKey);
		if(bRes)
		{
			SubKey.m_bOpen = TRUE;
			SubKey.m_bRegistry = TRUE;
		}
		return bRes;
	}
	else
	{
		CFileKey* pFileKey = GetSubKeyByName(m_pFileKey, szSubKey, TRUE);
		if(pFileKey == NULL) return FALSE;
		SubKey.m_pFileKey = pFileKey;
		SubKey.m_bOpen = TRUE;
		SubKey.m_bRegistry = FALSE;
		return TRUE;
	}
}




BOOL CSettingsKey::DeleteSubKey(LPCTSTR szSubKey)
{
	if(!m_bOpen || szSubKey == NULL || szSubKey[0] == _T('\0'))
		return FALSE;
	
	if(m_bRegistry)
		return m_RegKey.RecurseDeleteKey(szSubKey);
	else
	{
		CFileKey* pParentKey = NULL;
		CFileKey* pFileKey = GetSubKeyByName(m_pFileKey, szSubKey, FALSE, &pParentKey);
		if(pFileKey == NULL || pParentKey == NULL) return FALSE;

		POSITION pos = pParentKey->m_SubKeys.Find(pFileKey);
		if(pos == NULL) return FALSE;
		pParentKey->m_SubKeys.RemoveAt(pos);
		delete pFileKey;

		return TRUE;
	}
}


BOOL CSettingsKey::DeleteValue(LPCTSTR szValueName)
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.DeleteValue(szValueName);
	else
		return (0 != m_pFileKey->m_Values.RemoveKey(szValueName));
}



BOOL CSettingsKey::QueryValue(LPCTSTR szValueName, CString& stValue) const
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	BOOL bRes;

	if(m_bRegistry)
		bRes = m_RegKey.QueryValue(szValueName, stValue);
	else
		bRes = (0 != m_pFileKey->m_Values.Lookup(szValueName, stValue));

	if(!bRes) return FALSE;

	
	if(stValue.Find(_T('')) > -1) {   // unmask linebreaks
		CString stRes;
		bool bMask = false;
		for(int i = 0; i < stValue.GetLength(); ++i) {
			TCHAR cur = stValue[i];
			if(bMask) {
				if(cur == _T(''))
					stRes += _T('');
				else if(cur == _T('r'))
					stRes += _T('\r');
				else if(cur == _T('n'))
					stRes += _T('\n');
				else {
					stRes += _T('');
					stRes += cur;
				}
				bMask = false;
			}
			else {
				if(cur == _T(''))
					bMask = true;
				else
					stRes += cur;
			}
		}

		stValue = stRes;
	}

	return TRUE;
}



BOOL CSettingsKey::QueryValue(LPCTSTR szValueName, DWORD& dwValue) const
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.QueryValue(szValueName, dwValue);
	else
	{
		CString stValue;
		if(0 == m_pFileKey->m_Values.Lookup(szValueName, stValue))
			return FALSE;
		dwValue = _tcstoul(stValue, NULL, 10);
		return TRUE;
	}
}


BOOL CSettingsKey::QueryValue(LPCTSTR szValueName, UINT& nValue) const
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.QueryValue(szValueName, nValue);
	else
	{
		CString stValue;
		if(0 == m_pFileKey->m_Values.Lookup(szValueName, stValue))
			return FALSE;
		nValue = _tcstoul(stValue, NULL, 10);
		return TRUE;
	}
}

BOOL CSettingsKey::QueryValue(LPCTSTR szValueName, int& nValue) const
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.QueryValue(szValueName, nValue);
	else
	{
		CString stValue;
		if(0 == m_pFileKey->m_Values.Lookup(szValueName, stValue))
			return FALSE;
		long res = _tcstol(stValue, NULL, 10);
		if(res == LONG_MAX)
			nValue = -1;
		else
			nValue = res;
		return TRUE;
	}
}

BOOL CSettingsKey::QueryValue(LPCTSTR szValueName, bool& bValue) const
{
    int value = 0;
    if(QueryValue(szValueName, value)) {
        bValue = (value == 0 ? false : true);
        return TRUE;
    }
    return FALSE;
}

BOOL CSettingsKey::QueryValue(LPCTSTR szValueName, long& lValue) const
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.QueryValue(szValueName, lValue);
	else
	{
		CString stValue;
		if(0 == m_pFileKey->m_Values.Lookup(szValueName, stValue))
			return FALSE;
		lValue = _tcstol(stValue, NULL, 10);
		if(lValue == LONG_MAX)
			lValue = -1;

		return TRUE;
	}
}


BOOL CSettingsKey::SetValue(LPCTSTR szValueName, const CString& stValue)
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	CString stRes;
	if(stValue.FindOneOf(_T("\r\n")) > -1) {  // mask linebreaks
		for(int i = 0; i < stValue.GetLength(); ++i) {
			TCHAR cur = stValue[i];
			if(cur == '')
				stRes += _T("");
			else if(cur == '\r')
				stRes += _T("r");
			else if(cur == '\n')
				stRes += _T("n");
			else
				stRes += cur;
		}
	}

	const CString& stVal = stRes.IsEmpty() ? stValue : stRes;

	if(m_bRegistry)
		return m_RegKey.SetValue(szValueName, stVal);
	else
	{
		m_pFileKey->m_Values[szValueName] = stVal;
		return TRUE;
	}
}

BOOL CSettingsKey::SetValue(LPCTSTR szValueName, DWORD dwValue)
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.SetValue(szValueName, dwValue);
	else
	{
		CString stValue;
		stValue.Format(_T("%u"), dwValue);
		m_pFileKey->m_Values[szValueName] = stValue;
		return TRUE;
	}
}


BOOL CSettingsKey::SetValue(LPCTSTR szValueName, UINT nValue)
{
	return SetValue(szValueName, (DWORD)nValue);
}


BOOL CSettingsKey::SetValue(LPCTSTR szValueName, int nValue)
{
	return SetValue(szValueName, (long)nValue);
}

BOOL CSettingsKey::SetValue(LPCTSTR szValueName, bool bValue)
{
    return SetValue(szValueName, bValue? 1 : 0);
}

BOOL CSettingsKey::SetValue(LPCTSTR szValueName, long nValue)
{
	if(!m_bOpen || szValueName == NULL || szValueName[0] == _T('\0'))
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.SetValue(szValueName, nValue);
	else
	{
		CString stValue;
		stValue.Format(_T("%d"), nValue);
		m_pFileKey->m_Values[szValueName] = stValue;
		return TRUE;
	}
}

BOOL CSettingsKey::EnumKey(DWORD dwIndex, CString& stName)
{
	if(!m_bOpen)
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.EnumKey(dwIndex, stName);
	else
	{
		POSITION pos = m_pFileKey->m_SubKeys.FindIndex(dwIndex);
		if(pos == NULL) return FALSE;
		stName = m_pFileKey->m_SubKeys.GetAt(pos)->m_stName;
		return TRUE;
	}
}

BOOL CSettingsKey::EnumValue(DWORD dwIndex, CString& stName)
{
	if(!m_bOpen)
		return FALSE;

	if(m_bRegistry)
		return m_RegKey.EnumValue(dwIndex, stName);
	else
	{
		if(m_pFileKey->m_Values.GetCount() <= (signed)dwIndex) return FALSE;
		unsigned int i = 0;
		CString stValue;
		POSITION pos = m_pFileKey->m_Values.GetStartPosition();
		while(pos != NULL)
		{
			m_pFileKey->m_Values.GetNextAssoc(pos, stName, stValue);
			if(i == dwIndex)
				return TRUE;
			++i;
		}
		return FALSE;
	}
}



CFileKey* CSettingsKey::GetSubKeyByName(CFileKey* pFileKey, LPCTSTR szName, BOOL bCreate, CFileKey** pParentKey)
{
	CString stFullName = szName;
	CString stName = Parse(stFullName, _T("\\"));

	CFileKey* pThisKey = NULL;
	POSITION pos = pFileKey->m_SubKeys.GetHeadPosition();
	while(pos != NULL)
	{
		CFileKey* pKey = pFileKey->m_SubKeys.GetNext(pos);
		if(pKey->m_stName.CompareNoCase(stName) == 0)
		{
			pThisKey = pKey;
			break;
		}
	}

	if(pThisKey == NULL)
	{
		if(!bCreate) return NULL;

		pThisKey = new CFileKey(stName);
		pFileKey->m_SubKeys.AddTail(pThisKey);
	}

	if(stFullName.IsEmpty())
	{
		if(pParentKey) *pParentKey = pFileKey;
		return pThisKey;
	}

	return GetSubKeyByName(pThisKey, stFullName, bCreate, pParentKey);
}

int	CSettingsKey::CountSubKeys()
{
	if(!m_bOpen)
		return 0;

	if(m_bRegistry)
	{
		DWORD dwNum = 0;
		m_RegKey.CountSubKeys(dwNum);
		return dwNum;
	}
	else
		return m_pFileKey->m_SubKeys.GetCount();
}

/////////////////////////////////////////////////////////////////////////////////////



CSettings::CSettings() : m_RootKey(_T("")) 
{
	m_bOpen = FALSE;
	m_bRegistry = TRUE;
}

CSettings::~CSettings()
{
	Close();
}

BOOL CSettings::InitRegistry(LPCTSTR szRootKey)
{
	if(m_bOpen) return FALSE;
	if(szRootKey == NULL || szRootKey[0] == _T('\0'))
		return FALSE;
	
	m_bOpen = TRUE;
	m_bRegistry = TRUE;
	m_stRootKey = szRootKey;

	return TRUE;
}

BOOL CSettings::InitIniFile(LPCTSTR szFileName)
{
	if(m_bOpen) return FALSE;
	if(szFileName == NULL || szFileName[0] == _T('\0'))
		return FALSE;
		
	if(!ReadIniFile(szFileName))
		return FALSE;

	m_bOpen = TRUE;
	m_stIniFile = szFileName;
	m_bRegistry = FALSE;
	return TRUE;
}


BOOL CSettings::Close()
{
	if(!m_bOpen) return TRUE;

	BOOL bRes = TRUE;
	if(!m_bRegistry)
		bRes = SaveIniFile(m_stIniFile);
	
	m_RootKey.Reset();
	m_bOpen = FALSE;
	return bRes;
}

BOOL CSettings::ReadIniFile(LPCTSTR szFileName)
{
	char* pCharData = NULL;

	try
	{
		CFile File(szFileName, CFile::modeRead);
        unsigned int nFileLen = static_cast<unsigned int>(File.GetLength());
		pCharData = new char[nFileLen + 1];
		File.Read(pCharData, nFileLen);
		pCharData[nFileLen] = (char)0;
		File.Close();
	}
	catch(CFileException* e)
	{
		e->Delete();
		return FALSE;
	}

	USES_CONVERSION;
	LPTSTR pData = A2T(pCharData);

	
	LPTSTR pNextLine = pData;
	CFileKey* pKey = NULL;

	while(pNextLine)
	{
		LPTSTR pLine = pNextLine;
		pNextLine = _tcsstr(pNextLine, _T("\r\n"));
		if(pNextLine) 
		{
			*pNextLine = _T('\0');
			pNextLine += 2;
		}

		while(*pLine == _T(' ') || *pLine == _T('\t'))
			++pLine;

		if(*pLine == _T('['))
		{
			++pLine;
			LPTSTR pEnd = _tcschr(pLine, _T(']'));
			if(pEnd == NULL || pEnd == pLine) continue;
			*pEnd = _T('\0');

			pKey = new CFileKey(pLine);
			m_RootKey.m_SubKeys.AddTail(pKey);
			continue;
		}

		LPTSTR pAssign = _tcschr(pLine, _T('='));
		if(pAssign != NULL && pAssign > pLine)
		{
			*pAssign++ = _T('\0');
			if(pKey)
				pKey->m_Values[pLine] = pAssign;
			else
				m_RootKey.m_Values[pLine] = pAssign;
			continue;
		}
	}

#ifndef _UNICODE
	delete [] pData;
#endif

	NormalizeKey(&m_RootKey);

	if (pCharData != NULL) {
		delete [] pCharData;
	}
	
	return TRUE;
}

void CSettings::NormalizeKey(CFileKey* pFileKey)
{
	if(pFileKey == NULL) return;

	POSITION pos = pFileKey->m_SubKeys.GetHeadPosition();
	while(pos != NULL)
	{
		POSITION currentPos = pos;

		CFileKey* pKey = pFileKey->m_SubKeys.GetNext(pos);
		if(pKey->m_stName.Find(_T('\\')) == -1) continue;

		CString stName = Parse(pKey->m_stName, _T("\\"));

		CFileKey* pParentKey = NULL;
		POSITION pos2 = pFileKey->m_SubKeys.GetHeadPosition();
		while(pos2 != NULL)
		{
			CFileKey* pKeyLoop = pFileKey->m_SubKeys.GetNext(pos2);
			if(pKeyLoop == pKey) continue;
			if(pKeyLoop->m_stName.CompareNoCase(stName) == 0)
			{
				pParentKey = pKeyLoop;
				break;
			}
		}

		if(pParentKey == NULL)
		{
			pParentKey = new CFileKey(stName);
			pFileKey->m_SubKeys.AddTail(pParentKey);
		}

		pParentKey->m_SubKeys.AddTail(pKey);
		pFileKey->m_SubKeys.RemoveAt(currentPos);
	}

	pos = pFileKey->m_SubKeys.GetHeadPosition();
	while(pos != NULL)
	{
		CFileKey* pKey = pFileKey->m_SubKeys.GetNext(pos);
		NormalizeKey(pKey);
	}
}


BOOL CSettings::SaveIniFile(LPCTSTR szFileName)
{
	CString stRes;

	SaveKey(&m_RootKey, stRes, NULL);
	
	try
	{
		CFile File(szFileName, CFile::modeCreate | CFile::modeWrite);
		
		USES_CONVERSION;
		LPCSTR pData = T2A((LPTSTR)(LPCTSTR)stRes);

		File.Write(pData, strlen(pData));

		File.Close();
	}
	catch(CFileException* e)
	{
		e->Delete();
		return FALSE;
	}


	return TRUE;
}


void CSettings::SaveKey(CFileKey* pKey, CString& stOut, LPCTSTR szParentTree)
{
	if(pKey == NULL) return;

	CString stMyName;
	if(!pKey->m_stName.IsEmpty())
	{
		if(szParentTree && szParentTree[0] != _T('\0'))
		{
			stMyName = szParentTree;
			stMyName += _T('\\');
		}
		stMyName += pKey->m_stName;
	}


	if(!stMyName.IsEmpty() && (pKey->m_Values.GetCount() > 0 || pKey->m_SubKeys.GetCount() == 0))
	{
		stOut +=  _T('[');
		stOut += stMyName;
		stOut += _T("]\r\n");
	}

	std::map<CString,CString> map; // will store keys lexicographically

	CString stName;
	CString stValue;
	POSITION pos = pKey->m_Values.GetStartPosition();
	while (pos)
	{
		pKey->m_Values.GetNextAssoc(pos, stName, stValue);
		map[stName] = stValue;
	}

	for (auto it = map.begin(); it != map.end(); ++it) {
		stOut += it->first;
		stOut += _T('=');
		stOut += it->second;
		stOut += _T("\r\n");
	}

	if (pKey->m_Values.GetCount() > 0 || pKey->m_SubKeys.GetCount() == 0) 
		stOut += _T("\r\n");


	pos = pKey->m_SubKeys.GetHeadPosition();
	while(pos)
	{
		CFileKey* pSubKey = pKey->m_SubKeys.GetNext(pos);
		SaveKey(pSubKey, stOut, stMyName);
	}
}


BOOL CSettings::GetRootKey(CSettingsKey& Key)
{
	if(!m_bOpen) return FALSE;

	Key.m_bRegistry = m_bRegistry;
	
	BOOL bRes = TRUE;
	if(m_bRegistry)
		bRes = Key.m_RegKey.Create(HKEY_CURRENT_USER, m_stRootKey);
	else
		Key.m_pFileKey = &m_RootKey;

	Key.m_bOpen = bRes;
	return bRes;
}

BOOL CSettings::OpenSubKey(CSettingsKey& Key, LPCTSTR szSubKey)
{
	if(!GetRootKey(Key))
		return FALSE;
	
	BOOL bRes = Key.OpenSubKey(szSubKey);
	if(!bRes)
		Key.Close();

	return bRes;
}


BOOL CSettings::CreateSubKey(CSettingsKey& Key, LPCTSTR szSubKey)
{
	if(!GetRootKey(Key))
		return FALSE;

	BOOL bRes = Key.CreateSubKey(szSubKey);
	if(!bRes)
		Key.Close();

	return bRes;
}


BOOL CSettings::Flush()
{
	if(!m_bOpen) return FALSE;

	BOOL bRes = TRUE;
	if(!m_bRegistry)
		bRes = SaveIniFile(m_stIniFile);
	
	return bRes;
}