// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Rule.h
//
////////////////////////////////////////////////////////////////////////////////

#if !defined(AFX_RULE_H_)
#define AFX_RULE_H_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <afxtempl.h>
#include "Tokenizer.h"

class CMail;


class CPredicatedString {
public:
    CPredicatedString(const CString& str) : m_String(str) {}
    virtual bool Equals(const CString& stText) const;
    virtual bool IsContainedIn(const CString& stText) const = 0;
    virtual bool IsPrefixOf(const CString& stText) const = 0;
    virtual bool IsPostfixOf(const CString& stText) const = 0;
    virtual ~CPredicatedString() {}
    const CString& GetString() const { return m_String; }
    virtual CPredicatedString* Clone() const = 0;
    virtual CString GetStringRepresentation() const = 0;
protected:
    const CString m_String;
};


class CPredicatedStrings : public CList<const CPredicatedString*, const CPredicatedString*> 
{
public:

    CPredicatedStrings(bool bDestroyContent) : m_bDestroyContent(bDestroyContent) {}
    virtual ~CPredicatedStrings()
    {
        if(m_bDestroyContent) 
        {
            POSITION posList = GetHeadPosition();
            while(posList)
                delete GetNext(posList);
        }
    }
private:
    bool m_bDestroyContent;
};

// instances of this class are used instead of CPredicatedStrings* 
// when the ownership of the pointer is unclear. Clients just
// have to treat it as if they own the pointer, that means
// release() it when they no longer need it
class CPredicatedStrings_Ptr {
public:
    CPredicatedStrings_Ptr(CPredicatedStrings* ptr, bool own) : 
          m_Ptr(ptr), m_bOwn(own) {}

    CPredicatedStrings_Ptr(CPredicatedStrings* ptr) : 
          m_Ptr(ptr), m_bOwn(true) {}

    CPredicatedStrings_Ptr() : 
          m_Ptr(NULL), m_bOwn(true) {}
    
    const CPredicatedStrings* ptr() const { return m_Ptr; }
    bool is_null() const { return m_Ptr == NULL; }
    
    void release()
    {
        if(m_bOwn && m_Ptr)
            delete m_Ptr;
        m_Ptr = NULL;
    }

    const CPredicatedStrings* operator->() const { return m_Ptr; }
    CPredicatedStrings& operator*()  const { return *m_Ptr; }

private:
    CPredicatedStrings* m_Ptr;
    bool m_bOwn;
};



typedef bool (CPredicatedString::*FNStrPredicate)(const CString& stText) const;

class Predicate {
public:
    Predicate() : m_fnPredicate(NULL) {}

    Predicate(FNStrPredicate predicate, LPCTSTR szDescription) : 
        m_fnPredicate(predicate), m_Description(szDescription) {}
    
    bool Apply(const CPredicatedString* pStr, const CString& str) const {
        return (pStr->*m_fnPredicate)(str);
    }

    bool is_null() const { return m_fnPredicate == NULL; }
    const CString& GetDescription() const { return m_Description; }

    FNStrPredicate fn() const { return m_fnPredicate; }

private:
    FNStrPredicate m_fnPredicate;
    CString m_Description;
};



class CParsable 
{
public:

    class CVariableResolver {
    public:
        virtual CPredicatedStrings* GetListFromName(const CString& name) const = 0;
        virtual const CPredicatedString* GetStringFromName(const CString& name) const = 0;
        virtual const CString* GetName(const CPredicatedStrings* list) const = 0;
        virtual const CString* GetName(const CPredicatedString* string) const = 0;
        virtual bool IsVarDefined(const CString& name) const {
            return GetListFromName(name) || GetStringFromName(name);
        }
    };

    CParsable( const CVariableResolver& varResolver,
               unsigned int fileposStart,
               unsigned int fileposEnd,
               LPCTSTR szStr, 
               LPCTSTR delimiters, 
               LPCTSTR delimiters_preserve,
               const TCHAR    esc = '\\',
               const TCHAR    comment = 0 );

    virtual ~CParsable() {}

    virtual bool Parse() = 0;
    int GetParseErrorLine();

    unsigned int GetFileposStart() const { return m_FileposStart; }
    unsigned int GetFileposEnd()   const { return m_FileposEnd; }
    void SetFileposStart(unsigned int pos) { m_FileposStart = pos; }
    void SetFileposEnd(unsigned int pos)   { m_FileposEnd = pos; } 

protected:
    CString GetNextToken();
    void RestoreLastToken();
    bool LastTokenIsSingleQuoted();
    bool LastTokenIsDoubleQuoted();
    bool LastTokenIsQuoted() { return LastTokenIsDoubleQuoted() || LastTokenIsSingleQuoted(); }

    bool IsCurrTokenString(const CString& currToken);
    const CPredicatedString*  ReadString();
    CPredicatedStrings_Ptr ReadStringList(bool bAcceptSingleString);

    const CVariableResolver& m_VarResolver;

private:
    Tokenizer m_Tokenizer;
    bool m_bRestoreToken;
	int m_LineNo;

    unsigned int m_FileposStart;
    unsigned int m_FileposEnd;
};

class Info;

class CRule : public CParsable
{
public:
	bool Apply(CMail* pMail, bool bImmune, bool& bNew, bool& bDelete, Info& info, Info& actions, bool bMuteMode) const;
	
	CRule(const CVariableResolver& varResolver,  
          unsigned int fileposStart,
          unsigned int fileposEnd, 
          LPCTSTR szStr);

    virtual ~CRule();
    virtual bool Parse();
    CString GetName() const { return m_Name; }
    void SetName(const CString& name) { m_Name = name; }

    static CStringArray m_SoundCache;

protected:

	class CCondition {
	public:
		bool eval(const CMail* pMail, Info& info) const;
        virtual ~CCondition();

		enum CONJUNCTION {
			None,
			Or,
			And
		};

		enum QUANTOR {
			NoQuantor,
			AnyOf,
			AllOf,
			NoneOf
		};	
		
        struct Area 
        {
        public:
            Area(const CString& str, bool bRaw) : m_Str(str), m_bRaw(bRaw) {}
            Area() : m_bRaw(false) {}
            bool IsRaw() const { return m_bRaw; }
            const CString& String() const { return m_Str; }
        private:
            CString m_Str;
            bool m_bRaw;
        };

        typedef CList<Area,const Area&> CAreas;
        CAreas m_Areas;

        enum CONJUNCTION_AREA {
			areaNone,
            areaOr,
			areaAnd,
            areaConcatenation
		};
	    CONJUNCTION_AREA m_areaConjunction;
        CONJUNCTION_AREA m_translatedAreaConjunction; // used when evaluating the condition, maybe faster (e.g. use OR or AND instead of Concatenation)

        CCondition(bool bNot, const CVariableResolver& varResolver) : 
                m_areaConjunction(CONJUNCTION_AREA::areaNone),
                m_translatedAreaConjunction(CONJUNCTION_AREA::areaNone),
                m_bNot(bNot), 
                m_pArg(NULL),
				m_Conjunction(CONJUNCTION::None),
				m_Quantor(QUANTOR::NoQuantor),  
                m_bTestIfEmpty(false),
                m_VarResolver(varResolver) {}


		bool	m_bNot;
        
        Predicate m_Predicate; // member function pointer

		const CPredicatedString* m_pArg; // only used if there is no list of args (m_ArgList.GetSize() == 0)
		
		CONJUNCTION m_Conjunction; // defines how this condition is connected to the following condition (or, and)
		QUANTOR m_Quantor;

	    CPredicatedStrings_Ptr m_pArgList;

        bool m_bTestIfEmpty; // true if condition is empty test, e.g. "Subject is empty -> ..."
        
        void ParseComplete(); 

    private:
        
        const CString& GetAreaTextOfMail(const Area& Area, const CMail* pMail, CString& tmp) const;
        bool EvalArea(const CString& area, Info& info) const;
        CONJUNCTION_AREA TryTranslateConcatenation2Logical(CONJUNCTION_AREA areaConjunction) const;
        CString GetShortListRepresentation(CPredicatedStrings_Ptr pList) const;
        CString GetStringRepresentation(const CPredicatedString* pStr) const;
        bool MustEvalAllArgsToBecomeTrue() const;

        const CString& ToString() const;
        mutable CString m_stToString;

        const CVariableResolver& m_VarResolver;
	};


	class CAction {
	public:

		enum ACTION {
			Unknown,
			MarkForDelete,
			NotNew,
			Read,
			Delete,
			PlaySoundFile,
            PlaySingleSoundFile,
			Open,
			PrependSubject,
			Protect,
			Color,
			Print
		};

		CAction(ACTION Action) : m_Action(Action) { }

		ACTION m_Action;

		CStringArray m_Params;
	};

private:
    bool ParseAttributes();
	bool ParseCondition();
    bool ParseConditionArgs(CCondition& condition);
	bool ParseActions();
	bool ParseParams(CStringArray& result, int MinParam, int MaxParam);

    CString m_Name;
    CPredicatedStrings_Ptr m_pAccounts; // the name of the accounts to which this rule applies (if is empty -> apply to all accounts)

	CList<CCondition*, CCondition*>	m_Conditions;
	CList<CAction*, CAction*>		m_Actions;

};


class NamedString : public CParsable {
public:

    NamedString(const CVariableResolver& varResolver, 
                unsigned int fileposStart,
                unsigned int fileposEnd,
                LPCTSTR szStr);

    virtual ~NamedString() { 
        //TRACE("~NamedString %X %s \n", this, (LPCSTR)m_Name);
        delete m_pString ;
    } 

    virtual bool Parse();

    const CString& GetName() const { return m_Name; }
    const CPredicatedString& GetString() const { return *m_pString; }

private:
    CString m_Name;
    const CPredicatedString* m_pString;
};


class NamedStringList : public CParsable {    
public:

    NamedStringList(const CVariableResolver& varResolver,
                    unsigned int fileposStart,
                    unsigned int fileposEnd,
                    LPCTSTR szStr);

    virtual ~NamedStringList() { 
        //TRACE("~NamedStringList %X %s \n", this, (LPCSTR)m_Name);
        m_pStringList.release(); 
    } 

    virtual bool Parse();

    const CString& GetName() const { return m_Name; }
    CPredicatedStrings& GetList() const { return *m_pStringList; }

private:
    CString m_Name;

    CPredicatedStrings_Ptr m_pStringList;
};


class CRulesManager : protected CParsable::CVariableResolver
{
public:
	bool Apply(CMail* pMail, bool bImmune, bool& bNew, bool& bDelete, bool bMuteMode);
	
	CRulesManager() {}
	virtual ~CRulesManager();

	bool LoadFromFile(LPCTSTR szFile);
	bool CheckReload();

    enum StringType {
        stringAny,
        stringWord,
        stringRegEx        
    };

    int CountDefinedLists() const { return m_NamedLists.GetCount(); }
    void GetDefinedListNames(CStringArray& names);
    bool AddItemToList(const CString& szListName, const CString& item, StringType type, CString& errMsg);

	const CString& GetFileName() const { return m_stFile; }

    bool SoundPlayed() const { return CRule::m_SoundCache.GetSize() > 0; }

protected:
    virtual CPredicatedStrings* GetListFromName(const CString& name) const;
    virtual const CPredicatedString*  GetStringFromName(const CString& name) const;
    virtual const CString* GetName(const CPredicatedStrings* list) const;
    virtual const CString* GetName(const CPredicatedString* string) const;

private:
    NamedStringList* GetNamedList(const CString& name) const;

    const CPredicatedString* MakeString(const CString& str, StringType type);

	void Clear();

	CString m_stFile;
	CTime m_LastModified;

    CList<CRule*, CRule*> m_Rules;
    CList<NamedString*, NamedString*> m_NamedStrings;
    CList<NamedStringList*, NamedStringList*> m_NamedLists;
};





#endif // !defined(AFX_RULE_H_)
