// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Rule.cpp
//
////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "StrFunctions.h"
#include "Rule.h"
#include "Account.h"
#include <mmsystem.h> // needed to play sound file
#include <regex>
#include "PrintMail.h"
#include "PopMan.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif
  
//  i18nComment("Rules")

class CFullTextString : public CPredicatedString {
public:
    CFullTextString(const CString& str) : CPredicatedString(str) {}
    virtual bool IsContainedIn(const CString& stText) const;
    virtual bool IsPrefixOf(const CString& stText) const;
    virtual bool IsPostfixOf(const CString& stText) const;
    virtual CPredicatedString* Clone() const {
        return new CFullTextString(*this);
    }
    virtual CString GetStringRepresentation() const;
};

class CWordString : public CPredicatedString {
public:
    CWordString(const CString& str) : CPredicatedString(str) {}
    virtual bool IsContainedIn(const CString& stText) const;
    virtual bool IsPrefixOf(const CString& stText) const;
    virtual bool IsPostfixOf(const CString& stText) const;
    virtual CPredicatedString* Clone() const {
        return new CWordString(*this);
    }
    virtual CString GetStringRepresentation() const;
private:
    //bool IsWordBoundary(const CString& str, int idx) const;
    bool IsInMailAddress(const CString& str, int idx, int& leftBorder, int& rightBorder) const;
    bool AcceptAsWord(const CString& stText, int idx, int len) const;
    static CString boundaries;
};



#ifdef _UNICODE
    typedef std::wregex REGEX_TYPE;
#else
    typedef std::regex REGEX_TYPE;
#endif



class CRegExpString : public CPredicatedString {
public:
    CRegExpString(const CString& str) : 
        CPredicatedString(str), m_Pattern((LPCTSTR)str), m_pPostfixPattern(NULL) { }
    virtual ~CRegExpString();
    virtual bool IsContainedIn(const CString& stText) const;
    virtual bool Equals(const CString& stText) const;
    virtual bool IsPrefixOf(const CString& stText) const;
    virtual bool IsPostfixOf(const CString& stText) const;
    virtual CPredicatedString* Clone() const {
        return new CRegExpString(*this); // copy constructor redefined below
    }
    virtual CString GetStringRepresentation() const;

    CRegExpString(const CRegExpString& regexStr) : CPredicatedString(regexStr.m_String) {
        m_Pattern = regexStr.m_Pattern;
        if(regexStr.m_pPostfixPattern != NULL)
            m_pPostfixPattern = new REGEX_TYPE(*m_pPostfixPattern);
        else
            m_pPostfixPattern = NULL;
    }

private:
    
    REGEX_TYPE  m_Pattern;
    mutable REGEX_TYPE* m_pPostfixPattern;
};



CStringArray CRule::m_SoundCache;
const TCHAR constEscapeChar = _T('~');
const TCHAR constCommentChar = _T('#');
const LPCTSTR constRegexIdentifier = _T("regex:");
const LPCTSTR constVariableIndicator = _T("$");



static CString encode(const CString& str, TCHAR badChar)
{
    CString res;
    res.GetBuffer(str.GetLength()+4);

    for(int i = 0; i < str.GetLength(); ++i) {
        TCHAR c = str[i];
        if(c == badChar || c == constEscapeChar)
            res += constEscapeChar;
        res += c;
    }
    return res;
}

bool CPredicatedString::Equals(const CString& stText) const 
{
    return stText.CompareNoCase(m_String) == 0;
}

bool CFullTextString::IsContainedIn(const CString& stText) const
{
    return FindNoCase(stText, m_String) > -1;
}

bool CFullTextString::IsPrefixOf(const CString& stText) const 
{
    return FindNoCase(stText, m_String) == 0;
}

bool CFullTextString::IsPostfixOf(const CString& stText) const
{
    int start = stText.GetLength() - m_String.GetLength();
	if(start < 0) return false;
	return FindNoCase(stText, m_String, start) == start;
}


CString CFullTextString::GetStringRepresentation() const
{
    const TCHAR quote = _T('\"');

    CString res(quote);
    res += encode(m_String, quote);
    res += quote;

    return res;
}

/*
CString CWordString::boundaries(_T(" \t\r\n.?!,:;@<>'$&|`\"()[]{}+-*\\/~%="));

bool CWordString::IsWordBoundary(const CString& str, int idx) const
{
    if(idx < 0 || idx >= str.GetLength()) 
        return true;

    TCHAR c = str[idx];
    return (boundaries.Find(c) >= 0);
}
*/
static inline bool IsMailCharacter(TCHAR s)
{
	unsigned int c = (unsigned int)s;

	if( c >= _T('a') && c <= _T('z')	||
		c >= _T('A') && c <= _T('Z')	||
		c >= _T('0') && c <= _T('9')	||
		c == _T('-')					||
		c == _T('_')					||
		c == _T('.')                    ||
        c == _T('@')
	   )
		return true; // valid char

    return false;	// invalid char
}

bool CWordString::IsInMailAddress(const CString& str, int idx, int& leftBorder, int& rightBorder) const
{
    leftBorder = idx;
    rightBorder = idx;

    while(leftBorder >= 0)
    {
        if(!IsMailCharacter(str[leftBorder]))
            break;
        --leftBorder;
    }
    ++leftBorder;

    while(rightBorder < str.GetLength())
    {
        if(!IsMailCharacter(str[rightBorder]))
            break;
        ++rightBorder;
    }

    while(rightBorder-1 > leftBorder && str[rightBorder-1] == _T('.'))
        --rightBorder;


    int atPos;

    for(atPos = leftBorder; atPos < rightBorder; ++atPos)
        if(str[atPos] == _T('@'))
            break;

    if(atPos > leftBorder && atPos < rightBorder-1)
    {
        ++atPos;
        for(; atPos < rightBorder; ++atPos)
            if(str[atPos] == _T('@'))
                return false; // we've found two @ in one "mail address" !

        const TCHAR leftChar = str[leftBorder];

        if(leftChar == _T('_') || leftChar == _T('-') ||  leftChar == _T('.'))
            return false;
    
        const TCHAR rightChar = str[rightBorder-1];

        if(rightChar == _T('_') || rightChar == _T('-'))
            return false;

        return true;
    }

    return false;
}

bool CWordString::AcceptAsWord(const CString& stText, int idx, int len) const
{
    if(IsWordBoundary(stText, idx-1) && IsWordBoundary(stText, idx+len))
    {
        int left, right;
        if(IsInMailAddress(stText, idx, left, right))
            return (idx == left && idx+len == right);

        if(IsInMailAddress(stText, idx+len-1, left, right))
            return (idx == left && idx+len == right);

        return true;
    }
    else
    {
        return false;
    }
}

bool CWordString::IsContainedIn(const CString& stText) const
{
    int idx = 0;
    int len = m_String.GetLength();

    while(true) 
    {
        idx = FindNoCase(stText, m_String, idx);
        if(idx < 0) return false;
        if(AcceptAsWord(stText, idx, len)) 
            return true;
        ++idx;
    }    
}

bool CWordString::IsPrefixOf(const CString& stText) const
{
    return FindNoCase(stText, m_String) == 0 && AcceptAsWord(stText, 0, m_String.GetLength());
}

bool CWordString::IsPostfixOf(const CString& stText) const
{
    int start = stText.GetLength() - m_String.GetLength();
	if(start < 0) return false;
	return FindNoCase(stText, m_String, start) == start && AcceptAsWord(stText, start, m_String.GetLength());
}

CString CWordString::GetStringRepresentation() const
{
    const TCHAR quote = _T('\'');

    CString res(quote);
    res += encode(m_String, quote);
    res += quote;

    return res;
}

bool CRegExpString::Equals(const CString& stText) const 
{
    return std::regex_match((LPCTSTR)stText, m_Pattern); 
}

bool CRegExpString::IsContainedIn(const CString& stText) const
{
    return std::regex_search((LPCTSTR)stText, m_Pattern, std::regex_constants::match_any);
}

bool CRegExpString::IsPrefixOf(const CString& stText) const 
{
    return std::regex_search((LPCTSTR)stText, m_Pattern, std::regex_constants::match_continuous);
}

bool CRegExpString::IsPostfixOf(const CString& stText) const
{
    if(!m_pPostfixPattern) 
    {
        const CString tmp = m_String + _T("\\z");
        m_pPostfixPattern = new REGEX_TYPE(tmp);
    }

    return std::regex_search((LPCTSTR)stText, *m_pPostfixPattern, std::regex_constants::match_any);
}

CRegExpString::~CRegExpString()
{
    if(m_pPostfixPattern)
        delete m_pPostfixPattern;
}

CString CRegExpString::GetStringRepresentation() const
{
    const TCHAR quote = _T('\"');

    CString res(constRegexIdentifier);
    res += quote;
    res += encode(m_String, quote);
    res += quote;

    return res;
}



CParsable::CParsable( const CVariableResolver& varResolver,
                      unsigned int fileposStart,
                      unsigned int fileposEnd,
                      LPCTSTR szStr, 
                      LPCTSTR delimiters, 
                      LPCTSTR delimiters_preserve,
                      const TCHAR esc,
                      const TCHAR comment )
               : 
                 m_VarResolver(varResolver),
                 m_FileposStart(fileposStart),
                 m_FileposEnd(fileposEnd),
                 m_Tokenizer(szStr,  
                             delimiters,             // non preserving delimiters
                             delimiters_preserve,    // preserving delimiters; keep \n for line counting!
                             esc,                    // escape character
                             comment  ),
                m_LineNo(1),
                m_bRestoreToken(false)
{
    //TRACE("CParsable::CParsable: start: %d, end: %d \n", fileposStart, fileposEnd);
}



const CPredicatedString* CParsable::ReadString() 
{
    CString str = GetNextToken();

    if(!LastTokenIsQuoted()) 
    {
        if(str.CompareNoCase(constRegexIdentifier) == 0) 
        {
            str = GetNextToken();
            if(LastTokenIsQuoted() && !str.IsEmpty())
            {
                try {
                    return new CRegExpString(str);
                }
                catch(...) {}
            }
            return NULL;
        }
        else
        {
            CString varName = str;
            if(str.CompareNoCase(constVariableIndicator) == 0)
            {
                str = GetNextToken();
            }

            const CPredicatedString* pStr = m_VarResolver.GetStringFromName(varName);
            if(pStr)
                return pStr->Clone(); // we return a copy because the caller is responsible for deleting the object pointer!
            else
                return NULL;            
        } 
    }
    else
    {
        if(LastTokenIsSingleQuoted())
            return new CWordString(str);
        else
            return new CFullTextString(str);
    }
}

bool CParsable::IsCurrTokenString(const CString& currToken)
{
    return LastTokenIsQuoted() || (currToken.CompareNoCase(constRegexIdentifier) == 0);
}

CPredicatedStrings_Ptr CParsable::ReadStringList(bool bAcceptSingleString) 
{
    CString start = GetNextToken();

    if(IsCurrTokenString(start)) 
    {
        RestoreLastToken();

        if(!bAcceptSingleString)
            return CPredicatedStrings_Ptr(NULL);

        const CPredicatedString* pStr = ReadString();
        if(!pStr)
            return CPredicatedStrings_Ptr(NULL);

        CPredicatedStrings* pList = new CPredicatedStrings(true);
        pList->AddTail(pStr); 

        return CPredicatedStrings_Ptr(pList, true); // create owning "pointer"  		
    }

    if(start == _T("{")) 
    {
       // TRACE("Expecting native list\n");

        CString delim;
        CPredicatedStrings* strings = new CPredicatedStrings(true);
    
	    while(true)
	    {
            const CPredicatedString* pStr = ReadString();
            if(pStr != NULL)
                strings->AddTail(pStr);
            else
                break; // failed

		    delim = GetNextToken();
		    if(LastTokenIsQuoted()) 
                break;  // failed
        
		    if(delim == _T(","))
			    continue;
		    else if(delim == _T("}"))
			    return CPredicatedStrings_Ptr(strings, true); // create owning "pointer"  
		    else
			    break; // failed
	    }

        // we failed to read a valid list, clean up:

        delete strings; // destructor will delete containing pointers too!
    
        return CPredicatedStrings_Ptr(NULL);

    }
    else // expecting name of (list or string) variable, possibly prepended by $
    {
        //TRACE("Expecting name of list\n");
        CString varName;

        if(start == constVariableIndicator)
            varName = GetNextToken();
        else
            varName = start;

        CPredicatedStrings* pList = m_VarResolver.GetListFromName(varName);
        if(pList)
            return CPredicatedStrings_Ptr(pList, false);  // create non-owning "pointer"  

        if(bAcceptSingleString)
        {
            const CPredicatedString* pStr = m_VarResolver.GetStringFromName(varName);
            if(pStr) {
                CPredicatedStrings* pList = new CPredicatedStrings(false); // Create a list that will NOT delete its elements
                pList->AddTail(pStr);  // We don't have to Clone() the string
                return CPredicatedStrings_Ptr(pList, true);
            }            
        }

        return CPredicatedStrings_Ptr(NULL);        
    }
}

CString CParsable::GetNextToken()
{
    if(m_bRestoreToken) 
    {
        m_bRestoreToken = false;
        return m_Tokenizer.Token();
    }
    else 
    {
        if(m_Tokenizer.ReadNext()) {

            if(m_Tokenizer.Token() == _T("\n")) {
                ++m_LineNo;
                return GetNextToken();
            }

            return m_Tokenizer.Token();
        }
        else {
            return _T("");
        }
    }
}

void CParsable::RestoreLastToken()
{
    m_bRestoreToken = true;
}

bool CParsable::LastTokenIsSingleQuoted()
{
    return m_Tokenizer.IsSingleQuoted();
}

bool CParsable::LastTokenIsDoubleQuoted()
{
    return m_Tokenizer.IsDoubleQuoted();
}

int CParsable::GetParseErrorLine()
{
	return m_LineNo;
}


/////////////////////////////////////////////////////////////////////////////////

class Info {
public:
    Info(int initSize) { if (initSize > 0) m_stInfo.GetBuffer(initSize); }

    void AppendLineBreak() { *this += _T("\r\n"); }
    const Info& operator+=(const CString& str) { m_stInfo += str;  return *this; }
    const Info& operator+=(LPCTSTR pStr)       { m_stInfo += pStr; return *this; }
    const Info& operator+=(const Info& info)   { m_stInfo += info.GetString(); return *this; }

    int  GetResetMark() const { return m_stInfo.GetLength(); }
    void Reset() { Reset(0); }
	void Reset(int resetPoint) {
		if (resetPoint >= 0 && resetPoint<m_stInfo.GetLength())
			m_stInfo.GetBufferSetLength(resetPoint);
	}
    bool IsEmpty() const { return m_stInfo.IsEmpty(); }
    CString GetString() const { return m_stInfo; }

private:	
    CString m_stInfo;
};

/////////////////////////////////////////////////////////////////////////////////

CRule::CRule(const CVariableResolver& varResolver,  
             unsigned int fileposStart,
             unsigned int fileposEnd, 
             LPCTSTR szStr)     
  : CParsable(varResolver, fileposStart, fileposEnd, szStr, _T(" \t\r"), _T( "\n(),{}[]=+" ), constEscapeChar, constCommentChar)         
{
	//TRACE("CRule::CRule() \n");
	m_Conditions.RemoveAll();
	m_Actions.RemoveAll();
}

CRule::~CRule()
{
	//TRACE("CRule::~CRule() %s \n", m_Name);

	POSITION pos = m_Conditions.GetHeadPosition();
	while(pos) delete m_Conditions.GetNext(pos);
	m_Conditions.RemoveAll();

	pos = m_Actions.GetHeadPosition();
	while(pos) delete m_Actions.GetNext(pos);
	m_Actions.RemoveAll();

    m_pAccounts.release();
}

bool CRule::Parse()
{
    if(!ParseAttributes())
        return false;

	while(true) 
	{
		if(!ParseCondition())
			return false;

		CString next = GetNextToken();
		
		if(next == _T("->")) {
			if(!ParseActions())
				return false;
			if(!GetNextToken().IsEmpty())
				return false;

			return true;
		}

		next.MakeLower();
		if(next == _T("and"))
			m_Conditions.GetTail()->m_Conjunction = CCondition::CONJUNCTION::And;
		else if(next == _T("or"))
			m_Conditions.GetTail()->m_Conjunction = CCondition::CONJUNCTION::Or;
		else
			return false;
	}
	return false;
}

bool CRule::ParseAttributes()
{
    if(GetNextToken() != _T('[')) {
        RestoreLastToken();
        return true;
    }
    else if(LastTokenIsQuoted()) {
        return false;
    }
    else
    {
        CStringArray accounts;

        while(true)
        {
            CString attribute = GetNextToken();
            attribute.MakeLower();

            if(GetNextToken() != _T("=") || LastTokenIsQuoted())
                return false;

            if(attribute == _T("name")) {   
                
                if(!m_Name.IsEmpty())
                    return false;

                const CPredicatedString* pStr = ReadString();
                if(!pStr)
                    return false;
                m_Name = pStr->GetString();
                delete pStr;
                if(m_Name.IsEmpty())
                    return false;
            }
            else if(attribute == _T("account")) {

                if(accounts.GetSize() > 0)
                    return false;

                m_pAccounts = ReadStringList(true);
                if(m_pAccounts.is_null())
                    return false;
            }
            else
                return false;

            CString sep = GetNextToken();
            if(sep == _T("]") && !LastTokenIsQuoted())
                break;
            else if(sep == _T(",") && !LastTokenIsQuoted())
                continue;
            else
                return false;
        }

        return true;
    }    
}


bool CRule::ParseCondition()
{
    CRule::CCondition::CAreas Areas;

    CRule::CCondition::CONJUNCTION_AREA areaConjunc = CRule::CCondition::CONJUNCTION_AREA::areaNone;
    while(true)
    {
        CString strArea = GetNextToken();
	    if(strArea.GetLength() < 2) return false;

	    bool bRawArea = false;
	    
	    if(strArea[0] == _T('<') && strArea[strArea.GetLength()-1] == _T('>')) 
        {
		    if(strArea.GetLength() < 4) return false;
		    bRawArea = true;
		    strArea = strArea.Mid(1, strArea.GetLength()-2);
	    }

        Areas.AddTail(CRule::CCondition::Area(strArea, bRawArea));

        CString conj = GetNextToken();
        if(LastTokenIsQuoted()) return false;

        if(conj == _T("+")) {

            if(areaConjunc == CRule::CCondition::CONJUNCTION_AREA::areaNone)
                areaConjunc = CRule::CCondition::CONJUNCTION_AREA::areaConcatenation;
            else if(areaConjunc != CRule::CCondition::CONJUNCTION_AREA::areaConcatenation)
                return false;
        }
        else if(conj.CompareNoCase(_T("and")) == 0) {

            if(areaConjunc == CRule::CCondition::CONJUNCTION_AREA::areaNone)
                areaConjunc = CRule::CCondition::CONJUNCTION_AREA::areaAnd;
            else if(areaConjunc != CRule::CCondition::CONJUNCTION_AREA::areaAnd)
                return false;
        }
        else if(conj.CompareNoCase(_T("or")) == 0) {

            if(areaConjunc == CRule::CCondition::CONJUNCTION_AREA::areaNone)
                areaConjunc = CRule::CCondition::CONJUNCTION_AREA::areaOr;
            else if(areaConjunc != CRule::CCondition::CONJUNCTION_AREA::areaOr)
                return false;
        }
        else {
            RestoreLastToken();
            break;
        }
    }
    

    Predicate predicate;
	CString stPred = GetNextToken();
	if(LastTokenIsQuoted()) return false;
	stPred.MakeLower();

    

	bool bNot = false;
	if(stPred == "not") 
	{
		bNot = true;
		stPred = GetNextToken();
		stPred.MakeLower();
	}

    bool bTestIfEmpty = false;

    // TRACE("stPred: %s \n", (LPCTSTR)stPred);

	if(stPred == "contains")
        predicate = Predicate(&CPredicatedString::IsContainedIn, _T("contains"));
	else if(stPred == "equals")
		predicate = Predicate(&CPredicatedString::Equals, _T("equals"));
	else if(stPred == "begins")
	{
		stPred = GetNextToken();
		stPred.MakeLower();
		if(stPred == "with")
			predicate = Predicate(&CPredicatedString::IsPrefixOf, _T("begins with"));
		else
			return false;
	}
	else if(stPred == "ends") 
	{
		stPred = GetNextToken();
		stPred.MakeLower();
		if(stPred == "with")
			predicate = Predicate(&CPredicatedString::IsPostfixOf, _T("ends with"));
		else
			return false;
	}
    else if(stPred == "is") 
	{
		stPred = GetNextToken();
		stPred.MakeLower();
		if(stPred == "empty")
			bTestIfEmpty = true;
		else
			return false;
	}
	else
		return false;

    if( areaConjunc == CRule::CCondition::CONJUNCTION_AREA::areaConcatenation &&
        !bTestIfEmpty &&
        predicate.fn() !=  &CPredicatedString::IsContainedIn)
        return false; // concatenation only makes sense for "is empty" and "contains" predicate
        
    CCondition* pCondition = new CCondition(bNot, m_VarResolver);
    pCondition->m_Predicate = predicate;
    pCondition->m_bTestIfEmpty = bTestIfEmpty;
    pCondition->m_areaConjunction = areaConjunc;

    POSITION pos = Areas.GetHeadPosition();
    while(pos)
        pCondition->m_Areas.AddTail(Areas.GetNext(pos));
	
    if(!bTestIfEmpty) 
    {
        if(!ParseConditionArgs(*pCondition))
        {
            delete pCondition;
            return false;
        }

        if(pCondition->m_bNot && pCondition->m_Quantor == CCondition::QUANTOR::NoneOf) 
        {   // not none of == any of !
            pCondition->m_bNot = false;
            pCondition->m_Quantor = CCondition::QUANTOR::AnyOf;
        }
    }

    pCondition->ParseComplete();
	m_Conditions.AddTail(pCondition);
	return true;
}


bool CRule::ParseConditionArgs(CCondition& condition)
{
	CString stQuantor = GetNextToken();
	stQuantor.MakeLower();

   // TRACE("ParseConditionArgs\n");
				
	if(stQuantor == "any" && !LastTokenIsQuoted())
		condition.m_Quantor = CCondition::QUANTOR::AnyOf;
	else if(stQuantor == "all" && !LastTokenIsQuoted())
		condition.m_Quantor = CCondition::QUANTOR::AllOf;
	else if(stQuantor == "none" && !LastTokenIsQuoted())
		condition.m_Quantor = CCondition::QUANTOR::NoneOf;
    else {
        RestoreLastToken();
        // there was no quantor, therefore expecting single string
        if( NULL == (condition.m_pArg = ReadString()) )
		    return false;
        else
            return true;
    }

	if(GetNextToken().CompareNoCase(_T("of")) != 0 || LastTokenIsQuoted())
		return false;

//    TRACE("Reading List\n");

    condition.m_pArgList = ReadStringList(false);
    if(condition.m_pArgList.is_null())
        return false;
	
    return true;
}

bool CRule::ParseActions()
{
	CString stAction;

	do 
	{
		stAction = GetNextToken();
		stAction.MakeLower();
		CStringArray params;
		CAction::ACTION Action = CAction::ACTION::Unknown;
		if(stAction == "markfordelete")
			Action = CAction::ACTION::MarkForDelete;

		else if(stAction == "notnew")
			Action = CAction::ACTION::NotNew;
		else if(stAction == "read")
			Action = CAction::ACTION::Read;

		else if(stAction == "delete")
			Action = CAction::ACTION::Delete;

		else if(stAction == "playsound") {
			Action = CAction::ACTION::PlaySoundFile;
			if(!ParseParams(params, 1, 1))
				return false;
		}
        else if(stAction == "playsinglesound") {
			Action = CAction::ACTION::PlaySingleSoundFile;
			if(!ParseParams(params, 1, 1))
				return false;
		}
		else if(stAction == "open") {
			Action = CAction::ACTION::Open;
			if(!ParseParams(params, 1, 2))
				return false;
		}
        else if(stAction == "prependsubject") //SNH
		{
			Action = CAction::ACTION::PrependSubject;
			if(!ParseParams(params, 1, 1))
				return false;
		}
		else if (stAction == "color")
		{
			Action = CAction::ACTION::Color;
			if (!ParseParams(params, 1, 1))
				return false;
		}
		else if (stAction == "print")
		{
			Action = CAction::ACTION::Print;
		}
		else if(stAction == "protect" )	//PB
			Action = CAction::ACTION::Protect;
		else
			return false;

		CAction* pAction = new CAction(Action);
		pAction->m_Params.Copy(params);
		m_Actions.AddTail(pAction);

		stAction = GetNextToken();
	}
	while(stAction == ",");

	if(!stAction.IsEmpty())
		return false;

	return true;
}


bool CRule::ParseParams(CStringArray& result, int MinParam, int MaxParam)
{
	CString item = GetNextToken();
	if(item != "(") return false;

	for(int i = 1; i <= MaxParam; ++i) 
	{
		item = GetNextToken();
		if(item.IsEmpty()) return false;
		result.Add(item);

		item = GetNextToken();
		if(i < MinParam) {
			if(item != ",") return false;
		}
		else if(item == ")")
			return true;
		else if(item != ",")
			return false;

	}

	return false;
}

COLORREF ParseColor(CString color) {	
	
	color.Trim();
	color.MakeLower();

	if (color.IsEmpty()) return RGB(0, 0, 0);
	
	if (color[0] == '#') color.Delete(0);
	if (color.GetLength() != 6) return RGB(0, 0, 0);

	char buff[3];
	buff[0] = (char)color[0];
	buff[1] = (char)color[1];
	buff[2] = 0;
	auto r = std::stoul(buff, nullptr, 16);

	buff[0] = (char)color[2];
	buff[1] = (char)color[3];
	buff[2] = 0;
	auto g = std::stoul(buff, nullptr, 16);

	buff[0] = (char)color[4];
	buff[1] = (char)color[5];
	buff[2] = 0;
	auto b = std::stoul(buff, nullptr, 16);
	
	return RGB(r, g, b);
}

inline int HeightFromPoints(int iPoints)
{
	HDC hDC = ::GetDC(NULL);
	int iHeight = -MulDiv(iPoints, ::GetDeviceCaps(hDC, LOGPIXELSY), 72);
	::ReleaseDC(NULL, hDC);
	return iHeight;
}

bool CRule::Apply(CMail* pMail, bool bImmune, bool& bNew, bool& bDelete, Info& info, Info& actions, bool bMuteMode) const
{

	if(!m_pAccounts.is_null() && !m_pAccounts->IsEmpty()) // if rule is specific to certain account(s)
    {
        bool bFound = false;
        POSITION pos = m_pAccounts->GetHeadPosition();
        while(pos)
        {
            const CPredicatedString& accName = *m_pAccounts->GetNext(pos);

            if( accName.Equals(pMail->GetAccount()->m_stName) )
            {
                bFound = true;
                break;
            }
        }
        if(!bFound) return false;
    }
    

    bool bVal = false;

    // Eval all conditions
    // OR has a stronger binding than AND !
    // Therefore: a or b and c or d == (a or b) and (c or d) !

	CCondition::CONJUNCTION conjunction = CCondition::CONJUNCTION::None;
	POSITION pos = m_Conditions.GetHeadPosition();
	while(pos) 
	{
        if(conjunction == CCondition::CONJUNCTION::And) {
            info += _T(" and");
            info.AppendLineBreak();
        }

		const CCondition& condition = *m_Conditions.GetNext(pos);

        int resetMark = info.GetResetMark();
        bool con = condition.eval(pMail, info);
        if(!con) info.Reset(resetMark);		

        //TRACE("Condition result: %s \n", (con? "true" : "false"));

		switch(conjunction)
		{
		case CCondition::CONJUNCTION::None:
			bVal = con;
            //TRACE("First bVal == %d ! \n", (int)bVal);
			break;

		case CCondition::CONJUNCTION::Or:
			bVal = bVal || con;
            //TRACE("Or bVal == %d ! \n", (int)bVal);
			break;

		case CCondition::CONJUNCTION::And:
			bVal = bVal && con;
            //TRACE("And bVal == %d ! \n", (int)bVal);
			break;
		}		

		conjunction = condition.m_Conjunction;

        if(bVal == false && conjunction == CCondition::CONJUNCTION::And) {
            //TRACE("AND shortcircuting! \n");
			return false; // shortcircuiting
        }

        if(bVal == true && conjunction == CCondition::CONJUNCTION::Or) {
            //TRACE("OR shortcircuting! \n");
			
	        while(conjunction != CCondition::CONJUNCTION::And && pos) 
	        {
		        const CCondition& condition = *m_Conditions.GetNext(pos);
                conjunction = condition.m_Conjunction;
            }
        }
	}


	if(bVal) 
	{
        //TRACE("Rule success! \n");
        info.AppendLineBreak();
        info += _T("-> ");

        bool bNewOriginal = bNew;
        int resetPoint = 0;

		POSITION pos = m_Actions.GetHeadPosition();
		while(pos) 
		{
			CAction& action = *m_Actions.GetNext(pos);
            bool bActionIsEffective = false;

			switch(action.m_Action) 
			{
			case CAction::ACTION::MarkForDelete:
                if(!bImmune) 
                {
                    actions += _T("MarkForDelete");
 //                   TRACE(" Action: MarkForDelete \n");
				    pMail->SetMarkedForDelete(TRUE, TRUE);
                    bActionIsEffective = true;
                }
				break;

			case CAction::ACTION::NotNew:
                if(!bImmune) 
                {
                    actions += _T("NotNew");
//				    TRACE(" Action: NotNew \n");
				    bNew = false;
                    bActionIsEffective = true;
                }
				break;

			case CAction::ACTION::Read:
                if(!bImmune) 
                {
                    actions += _T("Read");
//				    TRACE(" Action: Read \n");
				    pMail->SetRead();
                    bActionIsEffective = true;
                }
				break;

			case CAction::ACTION::Delete:
                if(!bImmune) 
                {
                    actions += _T("Delete");
//				    TRACE(" Action: Delete \n");
				    bDelete = true;
                    bActionIsEffective = true;
                }
				break;

			case CAction::ACTION::PlaySoundFile:
			{
//				TRACE(" Action: PlaySound \n");
				bool alreadyPlayed = false;
				for(int i = 0; i < m_SoundCache.GetSize(); ++i) {
					if(m_SoundCache[i] == action.m_Params[0]) {
						alreadyPlayed = true;
						break;
					}
				}

				if(bNewOriginal && !alreadyPlayed && !bMuteMode) {

                    actions += _T("PlaySoundFile(\"");
                    actions += action.m_Params[0];
                    actions += _T("\")");

					::PlaySound(action.m_Params[0], 0, SND_FILENAME | SND_ASYNC);
					m_SoundCache.Add(action.m_Params[0]);
                    bActionIsEffective = true;
				}
				break;
			}
            case CAction::ACTION::PlaySingleSoundFile:
			{
//				TRACE(" Action: PlaySingleSound \n");
			
				if(bNewOriginal && m_SoundCache.GetSize() == 0 && !bMuteMode) {

                    actions += _T("PlaySingleSoundFile(\"");
                    actions += action.m_Params[0];
                    actions += _T("\")");

					::PlaySound(action.m_Params[0], 0, SND_FILENAME | SND_ASYNC);
					m_SoundCache.Add(action.m_Params[0]);
                    bActionIsEffective = true;
				}
				break;
			}
			case CAction::ACTION::Open:
//				TRACE(" Action: Open \n");
                if(bNewOriginal)
                {
                    actions += _T("Open(\"");
                    actions += action.m_Params[0];
                    actions += _T("\")");

					::ShellExecute(NULL, 
						_T("open"), 
						action.m_Params[0], 
						(action.m_Params.GetSize()>1 ? (LPCTSTR)action.m_Params[1] : NULL),
						NULL, 
						SW_SHOWNORMAL);
                    bActionIsEffective = true;
                }
				break;

            case CAction::ACTION::PrependSubject:
                actions += _T("PrependSubject(\"");
				actions += action.m_Params[0];
				actions += _T("\")");
//                  TRACE(" Action: PrependSubject \n");
				pMail->SetSubject(action.m_Params[0] + pMail->GetSubject());
                bActionIsEffective = true;
				break;

			case CAction::ACTION::Protect:
				pMail->SetMarkedForDelete(FALSE, TRUE);
				pMail->SetAlwaysProtected(TRUE);
				bActionIsEffective = true;
				bDelete = false;
				break;

			case CAction::ACTION::Color: {

				actions += _T("Color(\"");
				actions += action.m_Params[0];
				actions += _T("\")");

				CString color = action.m_Params[0];
				bActionIsEffective = true;
				pMail->m_bCustomColor = true;
				pMail->m_CustomColor = ParseColor(color);
				break;
			}

			case CAction::ACTION::Print:

				actions += _T("Print");
				bActionIsEffective = true;

				CString  stFaceName(_T("Courier New"));
				int  	 nPoints = 10;
				BOOL	 bBold = FALSE;
				BOOL	 bItalic = FALSE;

				const TCHAR szFaceNameValue[] = _T("FaceName");
				const TCHAR szPointsValue[] = _T("Points");
				const TCHAR szBoldValue[] = _T("Bold");
				const TCHAR szItalicValue[] = _T("Italic");
				const TCHAR szForeColorValue[] = _T("ForeColor");

				CSettingsKey Key;
				if (OpenSettingsKey(Key, szMsgWindowKey))
				{
					Key.QueryValue(szFaceNameValue, stFaceName);
					Key.QueryValue(szPointsValue, nPoints);
					Key.QueryValue(szBoldValue, bBold);
					Key.QueryValue(szItalicValue, bItalic);
				}

				CFont font;
				font.CreateFont(HeightFromPoints(nPoints), 0, 0, 0, bBold ? FW_BOLD : FW_NORMAL,
					(BYTE)bItalic, (BYTE)FALSE, 0, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS,
					CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, stFaceName);

				OnPrintMail(pMail, FALSE, &font);

				break;
			}
            
            if(pos && bActionIsEffective) {
                resetPoint = actions.GetResetMark();
                actions += _T(", ");
            }
            if(!pos && !bActionIsEffective) actions.Reset(resetPoint);
		}
        
	}

    info += actions;

    return bVal;
}

const CString& CRule::CCondition::GetAreaTextOfMail(const CRule::CCondition::Area& Area, const CMail* pMail, CString& tmp) const
{
	if (Area.String().CompareNoCase(_T("message")) == 0) // the extracted message
		return pMail->GetExtractedMsg();
	else if (!Area.IsRaw() && Area.String().CompareNoCase(_T("subject")) == 0)	// the subject
		return pMail->GetSubject();
	else if (!Area.IsRaw() && Area.String().CompareNoCase(_T("from")) == 0)	// from
		return pMail->GetEntireFrom();
	else if (Area.String().CompareNoCase(_T("header")) == 0)	// the raw header
		return pMail->GetHeader();
	else if (Area.String().CompareNoCase(_T("mail")) == 0)	// the entire raw mail
		return pMail->GetSource();
	else if (Area.String().CompareNoCase(_T("html")) == 0)	// the entire raw mail
		return pMail->GetHtml();
    else {	
		tmp = pMail->GetProperty(Area.String(), !Area.IsRaw());
        return tmp;
    }
}

CString CRule::CCondition::GetShortListRepresentation(CPredicatedStrings_Ptr pList) const
{
    if(pList.is_null()) return _T("");
    const CString* pName = m_VarResolver.GetName(pList.ptr());

    if(pName && !pName->IsEmpty())
        return *pName;
    else 
    {
        CString result;
        result.GetBuffer(64);
        bool bIsLongList = (pList->GetCount() > 4);

        result = _T("{");
        int count = 1;
        POSITION pos = pList->GetHeadPosition();
        while(pos) 
        {
            if(bIsLongList && count > 2)
            {
                result += _T("..., ");
                result += GetStringRepresentation(pList->GetTail());
                break;
            }
            else
            {
                const CPredicatedString* pStr = pList->GetNext(pos);
                result += GetStringRepresentation(pStr);
                if(pos) result += _T(", ");                
            }

            count++;            
        }
        result += _T("}");
       
        return result;
    }
}

CString CRule::CCondition::GetStringRepresentation(const CPredicatedString* pStr) const
{
    if(!pStr) return _T("");
    const CString* pName = m_VarResolver.GetName(pStr);

    if(pName && !pName->IsEmpty())
        return *pName;
    else
        return pStr->GetStringRepresentation();
}

bool CRule::CCondition::EvalArea(const CString& txtArea, Info& info) const
{
    bool result = false;

    if(m_bNot) info += _T("not ");

    if(m_bTestIfEmpty) 
    {
        info += _T("is empty");

        bool isEmpty = true;

        for(int i = 0; isEmpty && i < txtArea.GetLength(); ++i)
        {
            TCHAR c = txtArea[i];
            if( c!=_T(' ') && c!=_T('\t') && c!=_T('\r') && c!=_T('\n') )
                isEmpty = false;
        }
        result = isEmpty;
    }
    else 
    {
	    if(m_Quantor == QUANTOR::NoQuantor) 
        {
            info += m_Predicate.GetDescription();
            info += _T(" ");
            info += GetStringRepresentation(m_pArg);
            result = m_Predicate.Apply(m_pArg, txtArea);
        }
	    else 
	    {
            info += m_Predicate.GetDescription();
            info += _T(" ");

            POSITION pos = m_pArgList->GetHeadPosition();

            switch(m_Quantor)
            {
            case QUANTOR::AllOf:

                result = true;
                while(pos) 
                {
                    const CPredicatedString* pArg = m_pArgList->GetNext(pos);
                    if(!m_Predicate.Apply(pArg, txtArea)) {
                        result = false;
                        info += GetStringRepresentation(pArg);
                        break;
                    }
                }
                if(result) {
                    info += _T("all of ");
                    info += GetShortListRepresentation(m_pArgList);
                }
                break;

            case QUANTOR::NoneOf:
            case QUANTOR::AnyOf:

                result = false;
                while(pos) 
                {
                    const CPredicatedString* pArg = m_pArgList->GetNext(pos);
                    if(m_Predicate.Apply(pArg, txtArea)) {
                        result = true;
                        info += GetStringRepresentation(pArg);
                        break;
                    }
                }
                if(!result) {
                    info += m_Quantor == QUANTOR::AnyOf ? _T("any of ") : _T("none of ");
                    info += GetShortListRepresentation(m_pArgList);
                }

                // the special case "not none of" can not happen because it was simplified to "any of" when the condition was parsed!

                break;
            }
			    
		    if(m_Quantor == QUANTOR::NoneOf) // none of <=> not any of
			    result = !result;
	    }
    }

    return m_bNot ? !result : result;
}


bool CRule::CCondition::eval(const CMail* pMail, Info& info) const
{
	if(m_Predicate.is_null() && !m_bTestIfEmpty) return false;

    if(m_translatedAreaConjunction == CONJUNCTION_AREA::areaConcatenation) 
    {
        //TRACE("Eval Concatenation! \n");
        CString strUnion;

        POSITION pos = m_Areas.GetHeadPosition();
        while(pos)
        {
            CString tmp;
            const Area& area = m_Areas.GetNext(pos);
            const CString& txtArea = GetAreaTextOfMail(area, pMail, tmp);
            strUnion += txtArea + _T("\n\n");

            info += area.String();
            if(pos) info += _T("+"); else info += _T(" ");
        }
    
        //TRACE("strUnion: %s \n", (LPCTSTR)strUnion.Mid(0, 40));

        return EvalArea(strUnion, info);
    }
    else
    {
        //TRACE("Eval Logical! \n");

        bool res = false;

        int completeResetMark = info.GetResetMark();

        POSITION pos = m_Areas.GetHeadPosition();
        while(pos)
        {
            CString tmp;
            const Area& area = m_Areas.GetNext(pos);
            const CString& txtArea = GetAreaTextOfMail(area, pMail, tmp);

            int resetMark = info.GetResetMark();
            info += area.String();          
            info += _T(" ");
            bool areaResult = EvalArea(txtArea, info);
            if(!areaResult) info.Reset(resetMark);

            //TRACE("  -> %s \n", (areaResult ? "true" : "false") );

            if(m_translatedAreaConjunction == CONJUNCTION_AREA::areaAnd) {
                if(!areaResult) return false;
                if(pos) { info += _T(" and"); info.AppendLineBreak(); }
            }
            else if(m_translatedAreaConjunction == CONJUNCTION_AREA::areaOr) {
                if(areaResult) return true;
            }

            res = areaResult;
        }

        // use a more compact representation if possible:
        if(res && m_translatedAreaConjunction == CONJUNCTION_AREA::areaAnd && 
           MustEvalAllArgsToBecomeTrue()) 
        {
            info.Reset(completeResetMark);
            info += ToString();
        }

        return res;
    }
}

inline bool CRule::CCondition::MustEvalAllArgsToBecomeTrue() const
{
    return (m_Quantor == QUANTOR::NoneOf && !m_bNot) || // "NoneOf"
           (m_Quantor == QUANTOR::AnyOf  &&  m_bNot) || // "not AnyOf" equals "NoneOf"
           (m_Quantor == QUANTOR::AllOf  && !m_bNot) || // "AllOf"
           (m_Quantor == QUANTOR::NoQuantor          || // no quantor (just one arg)
            m_bTestIfEmpty) ;           
}

const CString& CRule::CCondition::ToString() const
{
    CString& str = m_stToString;
    if(!str.IsEmpty()) return str;

    str.GetBuffer(60);

    POSITION pos = m_Areas.GetHeadPosition();
    while(pos)
    {
        const Area& area = m_Areas.GetNext(pos);
        str += area.String();
        if(pos) {
            switch(m_translatedAreaConjunction) {
            case CONJUNCTION_AREA::areaAnd:
                str += _T(" and ");
                break;
            
            case CONJUNCTION_AREA::areaOr:
                str += _T(" or ");
                break;

            case CONJUNCTION_AREA::areaConcatenation:
                str += _T("+");
                break;
            default: break;
            }
        }
    }
    str += _T(" ");
    if(m_bNot) str += _T("not ");
    if(m_bTestIfEmpty)
        str += _T("is empty");
    else {
        str += m_Predicate.GetDescription();
        str += _T(" ");
        switch(m_Quantor) {
        case QUANTOR::NoQuantor:  
            str += GetStringRepresentation(m_pArg);
            break;
        case QUANTOR::AnyOf:
            str += _T("any of ");
            str += GetShortListRepresentation(m_pArgList);
            break;
        case QUANTOR::AllOf:
            str += _T("all of ");
            str += GetShortListRepresentation(m_pArgList);
            break;
        case QUANTOR::NoneOf:
            str += _T("none of ");
            str += GetShortListRepresentation(m_pArgList);
            break;
        default: break;
        }
    }

    return str;
}


void CRule::CCondition::ParseComplete()
{
    m_translatedAreaConjunction = TryTranslateConcatenation2Logical(m_areaConjunction);
}

CRule::CCondition::CONJUNCTION_AREA CRule::CCondition::TryTranslateConcatenation2Logical(CONJUNCTION_AREA areaConjunction) const
{
    if(areaConjunction == CONJUNCTION_AREA::areaConcatenation)
    {
        // Try to transform concatenation operator to a logical equivalent (which can be evaluated more efficiently):
        // Note: concatenation operator is only permitted for "is empty" and "contains in" predicate!

        if(m_bTestIfEmpty) 
        {
            areaConjunction = CONJUNCTION_AREA::areaAnd;
        }
        else
        {
            switch(m_Quantor) {
            case QUANTOR::NoQuantor:
                areaConjunction = CONJUNCTION_AREA::areaOr;
                break;

            case QUANTOR::AnyOf:
                areaConjunction = CONJUNCTION_AREA::areaOr;
                break;

            case QUANTOR::NoneOf:
                areaConjunction = CONJUNCTION_AREA::areaAnd;
                break;

            case QUANTOR::AllOf:
                areaConjunction = CONJUNCTION_AREA::areaConcatenation; // can not transform to equivalent logical operator!
                break;
            }

            if(m_bNot)
            {
                if(areaConjunction == CONJUNCTION_AREA::areaOr)
                    areaConjunction = CONJUNCTION_AREA::areaAnd;
                else if(areaConjunction == CONJUNCTION_AREA::areaAnd)
                    areaConjunction = CONJUNCTION_AREA::areaOr;
            }
        }
    }

    return areaConjunction;
}


CRule::CCondition::~CCondition() 
{
	if(m_pArg) delete m_pArg;
    m_pArgList.release();
}


/////////////////////////////////////////////////////////////////////////////////


NamedString::NamedString(const CVariableResolver& varResolver, 
                         unsigned int fileposStart,
                         unsigned int fileposEnd,
                         LPCTSTR szStr)
  : CParsable(varResolver, fileposStart, fileposEnd, szStr, _T(" \t\r"), _T( "\n,=" ), constEscapeChar, constCommentChar), m_pString(NULL)      
{
	//TRACE("NamedString::NamedString() %X \n", this);
}

bool NamedString::Parse()
{
    if(GetNextToken().CompareNoCase(_T("define")) != 0)
        return false;

    if(GetNextToken().CompareNoCase(_T("string")) != 0)
        return false;
 
    m_Name = GetNextToken();
    if(m_Name.IsEmpty()) return false;

    if(m_VarResolver.IsVarDefined(m_Name))
        return false;

    if(GetNextToken() != _T("="))
        return false;

    m_pString = ReadString();
    if(!m_pString)
        return false;

    //TRACE("NamedString::Parse() %X %s \n", this, (LPCTSTR)m_Name);
    
    return true;
}

/////////////////////////////////////////////////////////////////////////////////


NamedStringList::NamedStringList(const CVariableResolver& varResolver, 
                                 unsigned int fileposStart,
                                 unsigned int fileposEnd,
                                 LPCTSTR szStr)
  : CParsable(varResolver, fileposStart, fileposEnd, szStr, _T(" \t\r"), _T( "\n,{}=" ), constEscapeChar, constCommentChar)         
{
	//TRACE("NamedStringList::NamedStringList() %X \n", this);
}

bool NamedStringList::Parse()
{
    if(GetNextToken().CompareNoCase(_T("define")) != 0)
        return false;

    if(GetNextToken().CompareNoCase(_T("list")) != 0)
        return false;
 
    m_Name = GetNextToken();
    if(m_Name.IsEmpty()) return false;

    if(m_VarResolver.IsVarDefined(m_Name))
        return false;

    if(GetNextToken() != _T("="))
        return false;

    m_pStringList = ReadStringList(false);
    if(m_pStringList.is_null())
        return false;

    //TRACE("NamedStringList::Parse() %X %s \n", this, (LPCTSTR)m_Name);
    
    return true;
}

/////////////////////////////////////////////////////////////////////////////////

CRulesManager::~CRulesManager() 
{
	Clear();
}


void CRulesManager::Clear()
{
	POSITION pos = m_Rules.GetHeadPosition();
	while(pos) delete m_Rules.GetNext(pos);
	m_Rules.RemoveAll();

	CRule::m_SoundCache.RemoveAll();

    pos = m_NamedStrings.GetHeadPosition();
    while(pos)
        delete m_NamedStrings.GetNext(pos);
    m_NamedStrings.RemoveAll();

    pos = m_NamedLists.GetHeadPosition();
    while(pos)
        delete m_NamedLists.GetNext(pos);
    m_NamedLists.RemoveAll();
}

struct Segment 
{
public:
    Segment(const CString& str, int lineStart, unsigned int filePosStart, unsigned int filePosEnd) : 
        m_Str(str), m_LineStart(lineStart), m_FilePosFirstLine(filePosStart), m_FilePosLastLine(filePosEnd), m_MaxErrorLine(0) {}
    Segment() : m_LineStart(0), m_FilePosFirstLine(0), m_FilePosLastLine(0), m_MaxErrorLine(0) {}

    int LineStart() const { return m_LineStart; }
    const CString& String() const { return m_Str; }
    unsigned int FilePosFirstLine() const { return m_FilePosFirstLine; }
    unsigned int FilePosLastLine()  const { return m_FilePosLastLine; }
    
    void SetMaxErrorLine(unsigned int errLine) { m_MaxErrorLine = errLine; }
    unsigned int GetMaxErrorLine() const { return m_MaxErrorLine; }
private:
    CString m_Str;
    int m_LineStart;
    unsigned int m_FilePosFirstLine;
    unsigned int m_FilePosLastLine;
    unsigned int m_MaxErrorLine;
};


template<class T> bool Load(CParsable::CVariableResolver* pResolve, CList<T*, T*>* pItems, const CString& str, unsigned int fileposStart, unsigned int fileposEnd, int* pMaxErrline = NULL)
{
    T* pItem = new T(*pResolve, fileposStart, fileposEnd, str);
    if(!pItem->Parse()) {
        if(pMaxErrline) *pMaxErrline = max(*pMaxErrline, pItem->GetParseErrorLine());
        delete pItem;
        return false;
    }
    else {
        pItems->AddTail(pItem);
        return true;
    }
 }

bool IsUTF8(LPCTSTR szFile) {
	try {
		CFile File(szFile, CFile::modeRead);
		unsigned char buf[3];
		const UINT N = File.Read(buf, 3);		
		if (N < 3) return false;
		const wchar_t c0 = buf[0];
		const wchar_t c1 = buf[1];
		const wchar_t c2 = buf[2];
		return c0 == 0xEF && c1 == 0xBB && c2 == 0xBF;
	}
	catch (CFileException* e)
	{
		e->Delete();
		return false;
	}
}

bool ReadLine(CFile& f, CString& line) {

	line.Empty();

	unsigned char buf[2];
	buf[1] = 0;

	while (true) {

		UINT read = f.Read(buf, 1);
		if (read == 0) return !line.IsEmpty();
		if (buf[0] == '\r') {
			read = f.Read(buf, 1);
			if (read == 0 || buf[0] == '\n') return true;
			f.Seek(-1, CFile::current);
			return true;
		}
		if (buf[0] == '\n') {
			return true;			
		}
		const wchar_t c = buf[0];
		line.AppendChar(c);
	}

	return true;
}

#include "StrFunctions.h"

bool CRulesManager::LoadFromFile(LPCTSTR szFile)
{
	Clear();

	if(!szFile) return false;
	m_stFile = szFile;  

    CList<Segment, Segment&> UnresolvedSrc;

	const bool utf8 = IsUTF8(szFile);

	try 
	{
		CFile File(szFile, CFile::modeRead);
		
		CFileStatus status;
		File.GetStatus(status);
		m_LastModified = status.m_mtime;

		CString stLine;
		CString stItem;
		bool eof = false;
        int line = 0;
        unsigned int fileCursorPosOfLineStart = 0;

        int line_ItemStart = 0; // line number of the start
        unsigned int filepos_ItemStart = 0; // file pos of the start of the first line        
        unsigned int filepos_ItemEnd = 0;   // file pos of the start of the last line        
        
		do
		{
            ++line;

			eof = !ReadLine(File, stLine);

			if (line == 1 && utf8) {
				stLine.Delete(0, 3); // Remove UTF8 BOM
			}

			if (utf8) {
				stLine = DecodeUTF8(stLine);
			}

			stLine.TrimLeft();
			stLine.TrimRight();

            
            //TRACE("---> Line=%d LineStart=%d \n", line, fileCursorPosOfLineStart);
			
			if(stLine.IsEmpty() || eof)
			{
                if(!stItem.IsEmpty()) 
                {
                    if( 
                        !Load<CRule>           (this, &m_Rules,        stItem, filepos_ItemStart, filepos_ItemEnd) &&
                        !Load<NamedStringList> (this, &m_NamedLists,   stItem, filepos_ItemStart, filepos_ItemEnd) && 
                        !Load<NamedString>     (this, &m_NamedStrings, stItem, filepos_ItemStart, filepos_ItemEnd)
                       )
                        UnresolvedSrc.AddTail(Segment(stItem, line_ItemStart, filepos_ItemStart, filepos_ItemEnd));

				    stItem = _T("");
                }
			}
            else
            {
                if(stItem.IsEmpty()) {
				    line_ItemStart = line;
                    filepos_ItemStart = fileCursorPosOfLineStart;
                }
			    else
				    stItem += _T('\n');

			    if(stLine[0] != constCommentChar)
                {
                    if(stItem.GetAllocLength() < stItem.GetLength() + stLine.GetLength()+1) 
                    {  
                        stItem.GetBuffer( max(8*1024, 2*stItem.GetAllocLength()) );
                    }

				    stItem += stLine;
                    filepos_ItemEnd = fileCursorPosOfLineStart;
                }
            }
            
            fileCursorPosOfLineStart = static_cast<unsigned int>(File.GetPosition());

		} 
		while(!eof);

		File.Close();

	}
	catch(CFileException* e)
	{
		if(e->m_cause != CFileException::fileNotFound)
			e->ReportError();

		e->Delete();
        return false;
	}
   	
	
    // There may be still some unresolved code, possible reasons:
    // 1. syntax error
    // 2. reference to a named list or string that is defined after the current item

    while(UnresolvedSrc.GetCount() > 0)
    {
        int countBefore = UnresolvedSrc.GetCount();

        POSITION pos = UnresolvedSrc.GetHeadPosition();
        while(pos)
        {
            POSITION prevPos = pos;
            Segment& seg = UnresolvedSrc.GetNext(pos);
            int errLine = 0;

            if( Load<CRule>            (this, &m_Rules,        seg.String(), seg.FilePosFirstLine(), seg.FilePosLastLine(), &errLine) ||
                Load<NamedStringList>  (this, &m_NamedLists,   seg.String(), seg.FilePosFirstLine(), seg.FilePosLastLine(), &errLine) || 
                Load<NamedString>      (this, &m_NamedStrings, seg.String(), seg.FilePosFirstLine(), seg.FilePosLastLine(), &errLine)    )
            {
                UnresolvedSrc.RemoveAt(prevPos);
            }
            else 
            {
                seg.SetMaxErrorLine(errLine);
            }
        }

        if(UnresolvedSrc.GetCount() >= countBefore) // if no further item could be resolved there must be an error!
        {   
            CString stMsg;

            POSITION pos = UnresolvedSrc.GetHeadPosition();
            while(pos)
            {
                const Segment& seg = UnresolvedSrc.GetNext(pos);
		        stMsg += StrFormat(i18n("Error while parsing rules file at line {1}!"), _T("d"), (seg.LineStart() + seg.GetMaxErrorLine() - 1) );
                if(pos) stMsg += _T("\n");
            }

		    AfxMessageBox(stMsg, MB_OK | MB_ICONEXCLAMATION);
            Clear();
            return false;
        }
    }


    // sort rules according to their order in the file

    POSITION posHead = m_Rules.GetHeadPosition();
    while(posHead)
    {
        int minFilepos = -1;
        POSITION posMin = posHead;
        POSITION posFind = posHead;
        while(posFind)
        {
            POSITION posFindPrev = posFind;
            const CRule* pRule = m_Rules.GetNext(posFind);
            if(minFilepos == -1 || pRule->GetFileposStart() < (unsigned)minFilepos) {
                posMin = posFindPrev;
                minFilepos = pRule->GetFileposStart();
            }
        }
        if(posMin != posHead) { // swap
            CRule* pRuleMin = m_Rules.GetAt(posMin);
            m_Rules.SetAt(posMin, m_Rules.GetAt(posHead));
            m_Rules.SetAt(posHead, pRuleMin);
        }
        m_Rules.GetNext(posHead);
    }


    // Set default name for unnamed rules:

    int ruleNo = 1;
    POSITION pos = m_Rules.GetHeadPosition();
    while(pos)
    {
        CRule* pRule = m_Rules.GetNext(pos);
        if(pRule->GetName().IsEmpty()) 
        {
            CString name;
            name.Format(_T("Rule #%d"), ruleNo);
            pRule->SetName(name);
        }
        ruleNo++;

        //TRACE("Rule Name: %s \n", (LPCTSTR)pRule->GetName());
    }

	return true;
}

bool CRulesManager::CheckReload()
{
	CRule::m_SoundCache.RemoveAll(); 

	CFileStatus status;
	CFile::GetStatus(m_stFile, status);
	if(m_LastModified != status.m_mtime) {
		//TRACE("reloading Rules file!  \n ");
		return LoadFromFile(m_stFile);
	}
    return true;
}

bool CRulesManager::Apply(CMail* pMail, bool bImmune, bool& bIsNew, bool& bDelete, bool bMuteMode)
{
    //TRACE("\n\nApplying Mail: %s \n", (LPCTSTR)pMail->GetSubject());

    bool bRet = false;
    bool bReturnNotNew = false;
 
	bDelete = false;
	
	Info info(512);
	Info actions(60);

	POSITION pos = m_Rules.GetHeadPosition();
	while(pos) 
	{
		const CRule& Rule = *m_Rules.GetNext(pos);
		bool bNew = bIsNew;

        if(Rule.Apply(pMail, bImmune, bNew, bDelete, info, actions, bMuteMode))
        {
            bRet = true;
            pMail->m_RuleInfos.AddTail(CMail::RuleInfo(Rule.GetName(), info.GetString(), actions.GetString()));

		    if(bIsNew && !bNew)
			    bReturnNotNew = true;  // reset the "new" state if at least one rule wants this!
        }
        info.Reset();
        actions.Reset();
	}

	if(bReturnNotNew)
		bIsNew = false;

    return bRet;
}

NamedStringList* CRulesManager::GetNamedList(const CString& name) const
{
    POSITION pos = m_NamedLists.GetHeadPosition();
    while(pos)
    {
        NamedStringList* pList = m_NamedLists.GetNext(pos);
        if(pList && pList->GetName().CompareNoCase(name) == 0)
            return pList;
    }

    return NULL; 
}

CPredicatedStrings* CRulesManager::GetListFromName(const CString& name) const
{
    NamedStringList* pList = GetNamedList(name);
    if(pList)
        return &pList->GetList();
    else
        return NULL;
}

const CPredicatedString* CRulesManager::GetStringFromName(const CString& name) const
{
    POSITION pos = m_NamedStrings.GetHeadPosition();
    while(pos)
    {
        NamedString* pString = m_NamedStrings.GetNext(pos);
        if(pString && pString->GetName().CompareNoCase(name) == 0)
            return &pString->GetString();
    }

    return NULL; 
}

const CString* CRulesManager::GetName(const CPredicatedStrings* list) const
{
    if(!list) return NULL;
    POSITION pos = m_NamedLists.GetHeadPosition();
    while(pos)
    {
        const NamedStringList* pList = m_NamedLists.GetNext(pos);
        if(&pList->GetList() == list)
            return &pList->GetName();
    }
    return NULL;
}

const CString* CRulesManager::GetName(const CPredicatedString* string) const 
{
    if(!string) return NULL;
    POSITION pos = m_NamedStrings.GetHeadPosition();
    while(pos)
    {
        const NamedString* pString = m_NamedStrings.GetNext(pos);
        const CString& s1 = pString->GetString().GetString();
        const CString& s2 = string->GetString();
        if(s1.GetLength() == s2.GetLength() && s1.Compare(s2) == 0)
            return &pString->GetName();
    }
    return NULL; 
}

void CRulesManager::GetDefinedListNames(CStringArray& names)
{
    names.RemoveAll();
    POSITION pos = m_NamedLists.GetHeadPosition();
    while(pos) 
    {
        const NamedStringList* pList = m_NamedLists.GetNext(pos);
        names.Add(pList->GetName());
    }
}

const CPredicatedString* CRulesManager::MakeString(const CString& str, StringType type)
{
    switch(type)
    {
    case StringType::stringAny:
        return new CFullTextString(str);

    case StringType::stringWord:
        return new CWordString(str);

    case StringType::stringRegEx:
        try {
            return new CRegExpString(str);
        }
        catch(...) { return NULL;}

    default:
        return NULL;
    }
}

template<class T> void IncreaseOffsets(CList<T*, T*>& list, unsigned int increase)
{
    POSITION pos = list.GetHeadPosition();
    while(pos)
    {
        CParsable* pItem = list.GetNext(pos);
        if(pItem) {
            pItem->SetFileposStart(pItem->GetFileposStart() + increase);
            pItem->SetFileposEnd(pItem->GetFileposEnd() + increase);
        }
    }
}

char* ToUTF8(CString& s) {
	const int Len = WideCharToMultiByte(CP_UTF8, 0, s.GetString(), s.GetLength(), NULL, 0, NULL, NULL);
	char * data = new char[Len+1];
	WideCharToMultiByte(CP_UTF8, 0, s.GetString(), s.GetLength(), data, Len, NULL, NULL);
	data[Len] = 0;
	return data;
}

bool CRulesManager::AddItemToList(const CString& szListName, const CString& newItem, StringType type, CString& errMsg)
{
    if(!CheckReload()) {
        return false;
    }

    NamedStringList* pNamedList = GetNamedList(szListName);
    if(!pNamedList) return false;
    CPredicatedStrings& list = pNamedList->GetList();
    
    if(newItem.IsEmpty())
        return false;

	const bool utf8 = IsUTF8(m_stFile);

    CFile file;
    CFileException fileException;
    if ( !file.Open( m_stFile, CFile::modeReadWrite, &fileException) )
    {
        fileException.GetErrorMessage(errMsg.GetBuffer(255), 255);
        errMsg.ReleaseBuffer();
        return false;
    }


    const CPredicatedString* pNewItem = MakeString(newItem, type);
    if(!pNewItem) return false;

    // check if new string is already contained in list
    POSITION pos = list.GetHeadPosition();
    while(pos) 
    {
        const CPredicatedString* pItem = list.GetNext(pos);
        if(pItem->GetString().CompareNoCase(newItem) == 0) {

            if(typeid(pItem) == typeid(pNewItem)) {
                delete pNewItem;
                errMsg = i18n("The item is already contained in the list!");
                return false;
            }
        }
    }

    list.AddTail(pNewItem);

    // write item to rules file

    CString encodedItem = _T(",\r\n") + pNewItem->GetStringRepresentation();

	USES_CONVERSION;
	char* pEncodedItem = utf8 ? ToUTF8(encodedItem) : T2A((LPTSTR)(LPCTSTR)encodedItem);
	const int Len = strlen(pEncodedItem);

    unsigned int additionalSpace = Len;

    try 
    {
        file.Seek( pNamedList->GetFileposEnd(), CFile::begin );
    
        unsigned int fileLen = static_cast<unsigned int>(file.GetLength());
        unsigned int readSize = fileLen - pNamedList->GetFileposEnd();
        unsigned int buffSize = additionalSpace + readSize + 1;
    
        char* pBuf = new char[buffSize];
        pBuf[buffSize-1] = '\0';
        pBuf[0] = '\0';

        file.Read(pBuf+additionalSpace, readSize);

        char* const pStart = pBuf+additionalSpace;

        char* pScan = pStart;
        while(*pScan && *pScan!='\n')
            ++pScan;
        --pScan;

        while(*pScan && *pScan!='}')
            --pScan;

        memcpy(pBuf, pStart, pScan-pStart);

        char* pWrite = pScan-additionalSpace;
        for(int i = 0; i < Len; ++i)
            *pWrite++ = pEncodedItem[i];
    
        file.Seek( pNamedList->GetFileposEnd(), CFile::begin );
        file.Write(pBuf, buffSize-1);
        file.Flush();
        file.Close();

	    delete [] pBuf;
    }
	catch(CFileException* e)
	{
		e->GetErrorMessage(errMsg.GetBuffer(255), 255);
        errMsg.ReleaseBuffer();

		e->Delete();
        return false;
	}

    IncreaseOffsets<CRule>(m_Rules, additionalSpace);
    IncreaseOffsets<NamedString>(m_NamedStrings, additionalSpace);
    IncreaseOffsets<NamedStringList>(m_NamedLists, additionalSpace);

	if (utf8) {
		delete[] pEncodedItem;
	}

    return true;
}

