// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// PopManView.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "PopManView.h"
#include "PopManDoc.h"
#include "MailList.h"
#include "Settings.h"
#include "MessageView.h"
#include "MainFrm.h"
#include "DlgInput.h"
#include "StrFunctions.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//  i18nComment("Main View")

const TCHAR szMultiTabValue[]   = _T("MultiTab");
const TCHAR szMultilineTabsValue[] = _T("MultiLineTabs");
const TCHAR szHideMarkedValue[] = _T("HideMarked");
const TCHAR szTrashTabValue[]   = _T("TrashTab");
const TCHAR szAllKey[]			= _T("All");
const TCHAR szTabbingKey[]		= _T("Tabs");
const TCHAR szCaptionValue[]	= _T("Caption");
const TCHAR szAutoCaptionValue[]= _T("AutoCaption");

const int IDC_MULTI_TAB = 1000;


const TCHAR szFaceNameValue[]   = _T("FaceName");
const TCHAR szPointsValue[]     = _T("Points");
const TCHAR szBoldValue[]       = _T("Bold");
const TCHAR szItalicValue[]     = _T("Italic");
const TCHAR szForeColorValue[]  = _T("ForeColor");


IMPLEMENT_DYNCREATE(CPopManView, CFormView)


BEGIN_MESSAGE_MAP(CPopManView, CFormView)
	//{{AFX_MSG_MAP(CPopManView)
	ON_NOTIFY(TCN_SELCHANGE, IDC_MULTI_TAB, OnTabSelchange)
	ON_NOTIFY(TCN_SELCHANGING, IDC_MULTI_TAB, OnTabSelchanging)
	ON_NOTIFY(NM_RCLICK, IDC_MULTI_TAB, OnTabRclick)
	ON_WM_DESTROY()
	ON_WM_SIZE()
	ON_COMMAND(ID_EXTRAS_FONT, OnFont)
	ON_COMMAND(ID_EXTRAS_FONT_COLOR, OnFontColor)
	ON_COMMAND(ID_GOTO_NEXT_TAB, OnGotoNextTab)
	ON_COMMAND(ID_GOTO_PREV_TAB, OnGotoPrevTab)
	ON_COMMAND(ID_TAB_RENAME, OnTabRename)
	ON_WM_MENUSELECT()
	ON_COMMAND(ID_VIEW_HIDE_MARKED, OnHideMarked)
	ON_UPDATE_COMMAND_UI(ID_VIEW_HIDE_MARKED, OnUpdateHideMarked)
	ON_COMMAND(ID_VIEW_TRASHTAB, OnTrashTab)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TRASHTAB, OnUpdateTrashTab)
	ON_WM_LBUTTONDOWN()
	ON_WM_INITMENUPOPUP()
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_MAIL_READ, OnMailRead)
	ON_COMMAND(ID_TAB_SPLIT, OnTabSplit)
	ON_COMMAND(ID_TAB_ACC_REMOVE, OnTabAccRemove)
	ON_COMMAND(ID_TAB_JOIN, OnJoinTabs)
	ON_COMMAND(ID_TAB_CHECK_MAIL, CheckCurrentTab)
	ON_COMMAND(ID_APP_UPDATE_GUI, OnUpdateGUI)
END_MESSAGE_MAP()


static BOOL IsMail (const CMail& Mail)                  { return true; }
static BOOL IsMarkedMail (const CMail& Mail)            { return Mail.IsMarkedForDelete(); }
static BOOL IsUnmarkedMail (const CMail& Mail)          { return !Mail.IsMarkedForDelete(); }
static BOOL IsUnreadMail (const CMail& Mail)            { return !Mail.IsRead(); }
static BOOL IsUnreadAndMarkedMail (const CMail& Mail)   { return !Mail.IsRead() && Mail.IsMarkedForDelete(); }
static BOOL IsUnreadAndUnmarkedMail (const CMail& Mail) { return !Mail.IsRead() && !Mail.IsMarkedForDelete(); }


CPopManView::CPopManView()
	: CFormView(CPopManView::IDD)
{
	//{{AFX_DATA_INIT(CPopManView)
	//}}AFX_DATA_INIT
	m_bMultiTab = TRUE;
    m_bMultilineTabs = FALSE;
	m_bHideMarkedMails = FALSE;
	m_bStartUp = TRUE;
	m_Tab.m_pView = this;
	m_nTabContextMenuIdx = 0;
	m_bTabContextMenu = false;
    m_bTrashTab = TRUE;
    m_bTrashTabIsVisible = false;
    m_LastTabIdx = 0;
}

CPopManView::~CPopManView()
{
}

#ifdef _DEBUG
void CPopManView::AssertValid() const
{
	CFormView::AssertValid();
}

void CPopManView::Dump(CDumpContext& dc) const
{
	CFormView::Dump(dc);
}
#endif 

void CPopManView::Refresh()
{
	m_List.Refresh();
}

void CPopManView::OnInitialUpdate() 
{
	CFormView::OnInitialUpdate();

	SetScrollSizes(MM_TEXT, CSize(1, 1));

	RECT rect = {0, 0, 0, 0};
	m_List.SetDocument(GetDocument());
	m_List.Create(LVS_REPORT | WS_VISIBLE, rect, this, 1);
	
	m_ImageList.Create(GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON), ILC_MASK | ILC_COLOR24, 4, 1); 

	LoadImageList();
	
	LoadSettings();

	m_bStartUp = FALSE;
}

void CPopManView::LoadImageList()
{
	m_ImageList.SetImageCount(2);
	m_ImageList.Replace(0, CPopManApp::GetIcon(IDI_TAB_MAIL));
	m_ImageList.Replace(1, CPopManApp::GetIcon(IDI_TAB_UNREAD_MAIL));
}

void CPopManView::OnUpdateGUI()
{
	LoadImageList();
    if(m_Tab.m_hWnd)
    {
        UpdateAllTabIconsAndText();
	    m_Tab.Invalidate();
	    m_Tab.UpdateWindow();
    }
}

void CPopManView::OnDestroy() 
{
	SaveSettings();	
	CFormView::OnDestroy();	
}

BOOL CPopManView::LoadSettings()
{
	CSettingsKey Key;
	if(OpenSettingsKey(Key, szViewKey)) {
		Key.QueryValue(szMultiTabValue, m_bMultiTab);
        Key.QueryValue(szMultilineTabsValue, m_bMultilineTabs);
		Key.QueryValue(szHideMarkedValue, m_bHideMarkedMails);
        Key.QueryValue(szTrashTabValue, m_bTrashTab);
	}
		
	Key.Close();
	m_TabArray.SetSize(0, 16);

	SetTabbing(m_bMultiTab);

	HWND hwnd = m_List.GetSafeHwnd();
	if(hwnd == NULL) hwnd = GetSafeHwnd(); 

	HFONT hFont = (HFONT)::SendMessage(hwnd, WM_GETFONT, 0, 0); 
	LOGFONT lf;
	::GetObject(hFont, sizeof(LOGFONT), &lf);
		
	CString  stFaceName(lf.lfFaceName);
	int  	 nPoints = PointsFromHeight(lf.lfHeight);
	BOOL	 bBold   = (lf.lfWeight > FW_NORMAL);
	BOOL	 bItalic = lf.lfItalic;
	COLORREF TextColor = m_List.GetTextColor();

	if(OpenSettingsKey(Key, szMainWindowKey))
	{
		Key.QueryValue(szFaceNameValue, stFaceName);
		Key.QueryValue(szPointsValue,   nPoints);
		Key.QueryValue(szBoldValue,     bBold);
		Key.QueryValue(szItalicValue,   bItalic);
		Key.QueryValue(szForeColorValue, TextColor);
	}

	m_Font.CreateFont(HeightFromPoints(nPoints), 0, 0, 0, bBold ? FW_BOLD : FW_NORMAL,
					 (BYTE)bItalic, (BYTE)FALSE, 0, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, 
					 CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, stFaceName);	

	m_List.SetFont(&m_Font);
	m_List.SetTextColor(TextColor);
	
	return TRUE;
}

void CPopManView::SaveSettings()
{
	CSettingsKey Key;

	LOGFONT lf;
	m_Font.GetLogFont(&lf);
		
	CString stFaceName(lf.lfFaceName);
	int  	nPoints = PointsFromHeight(lf.lfHeight);
	BOOL	bBold   = (lf.lfWeight > FW_NORMAL);
	BOOL	bItalic = lf.lfItalic;
	COLORREF TextColor = m_List.GetTextColor();
	
	
	if(CreateSettingsKey(Key, szMainWindowKey))
	{
		Key.SetValue(szFaceNameValue, stFaceName);
		Key.SetValue(szPointsValue,   nPoints);
		Key.SetValue(szBoldValue,     bBold);
		Key.SetValue(szItalicValue,   bItalic);
		Key.SetValue(szForeColorValue, TextColor);
	}

	Key.Close();
	if(CreateSettingsKey(Key, szViewKey)) {
		Key.SetValue(szMultiTabValue, m_bMultiTab);
        Key.SetValue(szMultilineTabsValue, m_bMultilineTabs);
		Key.SetValue(szHideMarkedValue, m_bHideMarkedMails);
        Key.SetValue(szTrashTabValue, m_bTrashTab);
	}
		
	Key.Close();

	SaveLayoutAndReset();
}

inline int CPopManView::PointsFromHeight(int iHeight)
{
	HDC hDC = ::GetDC(NULL);
	int iPoints = -MulDiv(iHeight, 72, ::GetDeviceCaps(hDC, LOGPIXELSY));
	::ReleaseDC(NULL, hDC);
	return iPoints;
}

inline int CPopManView::HeightFromPoints(int iPoints)
{
	HDC hDC = ::GetDC(NULL);
	int iHeight = -MulDiv(iPoints, ::GetDeviceCaps(hDC, LOGPIXELSY), 72);
	::ReleaseDC(NULL, hDC);
	return iHeight;
}

bool CPopManView::ShowMarkedMails(int tabIdx) const
{
    return (GetTrashTabIdx() == tabIdx)                      ||
           ( m_bMultiTab&&!m_bTrashTab&&!m_bHideMarkedMails) ||
           (!m_bMultiTab&&!m_bHideMarkedMails);
}

bool CPopManView::ShowUnmarkedMails(int tabIdx) const
{
    return (GetTrashTabIdx() != tabIdx);
}

void CPopManView::OnTabSelchanging(NMHDR* pNMHDR, LRESULT* pResult) 
{
	int idx = m_Tab.GetCurSel();
	if(idx >= 0 && idx < m_TabArray.GetSize())
	{
        m_LastTabIdx = idx;
		TabInfo& tab = m_TabArray[idx];
		tab.TopIdx = m_List.GetTopIndex();
		tab.pSelectedMail = m_List.GetSelectedMail();
        tab.nSelectedPos = m_List.GetSelectedItem();
	}
	
	*pResult = 0;
}

void CPopManView::OnTabSelchange(NMHDR* pNMHDR, LRESULT* pResult) 
{
	int idx = m_Tab.GetCurSel();
	if(idx >= 0 && idx < m_TabArray.GetSize())
	{
		TabInfo& tab = m_TabArray[idx];
		m_List.m_pAccounts = &tab.Accounts;
        m_List.m_bShowMarkedMail   = ShowMarkedMails(idx);
        m_List.m_bShowUnmarkedMail = ShowUnmarkedMails(idx);
		m_List.Refresh();
		m_List.SetTopIndex(tab.TopIdx);
        if(m_List.SelectMail(tab.pSelectedMail) < 0)
            m_List.SelectItem(tab.nSelectedPos);
    }

	*pResult = 0;
}

void CPopManView::OnTabRclick(NMHDR* pNMHDR, LRESULT* pResult) 
{
	*pResult = 0;
	
	POINT point;
	GetCursorPos(&point);

	POINT clientPoint = point;
	m_Tab.ScreenToClient(&clientPoint);
	TCHITTESTINFO info;
	info.pt = clientPoint;
	info.flags = 0;
	int idx = m_Tab.HitTest(&info);

	if(idx >= 0 && idx < m_Tab.GetItemCount() && idx != GetTrashTabIdx())
	{
		if(m_Tab.GetCurSel() != idx)
			SwitchTab(idx);

		CString stAll = i18n("Separate account | All");
		CString stSeparateAcc = Parse(stAll, _T("|"));
		stSeparateAcc.TrimRight();
		stAll.TrimLeft();


		CMenu mnAccsTab;
		mnAccsTab.CreatePopupMenu();

		mnAccsTab.AppendMenu(MF_STRING, ID_TAB_SPLIT, stAll);
		mnAccsTab.AppendMenu(MF_SEPARATOR);

		CAccounts& Accounts = m_TabArray[idx].Accounts;
		POSITION pos = Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			CAccount* pAcc = Accounts.GetNext(pos);
			CString stName = pAcc->m_stName;
			stName.Replace(_T("&"), _T("&&"));
			mnAccsTab.AppendMenu(MF_STRING, ID_TAB_ACC_REMOVE, stName);
		}

		bool bMultiAccs = (Accounts.GetCount() > 1);


		CMenu mnTabs;
		mnTabs.CreatePopupMenu();
		for(int i = 0; i < m_Tab.GetItemCount() && i != GetTrashTabIdx(); ++i) 
		{
			CString stTabCaption = m_TabArray[i].Caption;
			stTabCaption.Replace(_T("&"), _T("&&"));
			if(i != idx)
				mnTabs.AppendMenu(MF_STRING, ID_TAB_JOIN, stTabCaption);
		}

		bool bMultiTabs = (m_Tab.GetItemCount() > 1);

		CMenu mnContext;
		mnContext.CreatePopupMenu();

	
		mnContext.AppendMenu(MF_STRING, ID_TAB_CHECK_MAIL, i18n("Check accounts\tF7"));
		mnContext.AppendMenu(MF_SEPARATOR);
		mnContext.AppendMenu(MF_STRING | MF_POPUP | (bMultiAccs ? MF_ENABLED : MF_GRAYED), (UINT)mnAccsTab.m_hMenu, stSeparateAcc);
		mnContext.AppendMenu(MF_STRING | MF_POPUP | (bMultiTabs ? MF_ENABLED : MF_GRAYED), (UINT)mnTabs.m_hMenu, i18n("Join with tab"));
		mnContext.AppendMenu(MF_SEPARATOR);
		mnContext.AppendMenu(MF_STRING, ID_TAB_RENAME, i18n("Rename tab\tF2"));
        mnContext.AppendMenu(MF_SEPARATOR);

        mnContext.AppendMenu(MF_STRING, ID_TAB_MARKALL_READ,   i18n("Mark All Mails Read"));
        mnContext.AppendMenu(MF_STRING, ID_TAB_MARKALL_UNREAD, i18n("Mark All Mails Unread"));

		m_bTabContextMenu = true;
		mnContext.TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, point.x, point.y, this);
		m_bTabContextMenu = false;
	}
}

void CPopManView::CheckCurrentTab()
{
	if(!IsTabbingEffective()) return;

	int Idx = m_Tab.GetCurSel();

    BOOL hardReload = (GetKeyState(VK_CONTROL) & 8000) != 0;

    if(Idx != GetTrashTabIdx())
		GetDocument()->ListMailsFromAccs(m_TabArray[Idx].Accounts, TRUE, hardReload);
}

void CPopManView::OnTabAccRemove()
{
	int idx = m_Tab.GetCurSel();
	if(idx < 0) return;
	CAccounts& Accounts = m_TabArray[idx].Accounts;
	POSITION pos = Accounts.FindIndex(m_nTabContextMenuIdx-2); // -2 because the first item is "All", the second is the separator!
	if(pos == NULL) return;

	CAccount* pAcc = Accounts.GetAt(pos);

	Accounts.RemoveAt(pos);
	m_List.Refresh();

	UpdateTabIconAndText(idx);

	AddTab(pAcc, idx+1);
}

void CPopManView::OnJoinTabs()
{
	int idx = m_Tab.GetCurSel();

	JoinTabs(idx, m_nTabContextMenuIdx + (idx <= m_nTabContextMenuIdx ? 1 : 0));
}

BOOL CPopManView::SetTabbing(BOOL bTabbing)
{
	if(!m_bStartUp)
		SaveLayoutAndReset();

	CSettingsKey Key;
	OpenSettingsKey(Key, szViewKey);

	const CMail* pSelectedMail = m_List.GetDisplayedMail();

    m_bMultiTab = bTabbing;

	if(!bTabbing || GetDocument()->m_Accounts.GetCount() == 0)
	{
		Key.OpenSubKey(szAllKey);
		m_List.OnInitialUpdate(Key);
		m_List.m_pAccounts = &GetDocument()->m_Accounts;
        m_List.m_bShowMarkedMail   = !m_bHideMarkedMails;
        m_List.m_bShowUnmarkedMail = TRUE;
		m_List.Refresh();
		m_List.SelectMail(pSelectedMail);
        m_bTrashTabIsVisible = false;
	}
	else
	{
		Key.OpenSubKey(szTabbingKey);
		
		m_List.OnInitialUpdate(Key);
		

		RECT rect = {0, 0, 0, 0};
        m_Tab.Create(WS_VISIBLE | WS_CHILD | TCS_FOCUSNEVER | (m_bMultilineTabs ? TCS_MULTILINE : TCS_SINGLELINE), rect, this, IDC_MULTI_TAB); 
		m_Tab.SetFont(GetFont());
		m_Tab.SetImageList(&m_ImageList);

		CString stTabNum;
		int n = 1;
		int nAccCount = 0;
		while(true)
		{
			stTabNum.Format(_T("Tab%d"), n++);
			CSettingsKey SubKey;
			if(!Key.OpenSubKey(stTabNum, SubKey))
				break;

			nAccCount += AddTab(SubKey);		
		
			SubKey.Close();
		}

		// Are there any accounts which are not represented on a tab?
		if(nAccCount < GetDocument()->m_Accounts.GetCount())
		{
            if(nAccCount == 0 && GetDocument()->m_Accounts.GetCount() > 1)
            {
                AddTab(GetDocument()->m_Accounts, false, i18n("Mail Accounts"), 0);
            }
            else
            {
			    CAccounts& Accounts = GetDocument()->m_Accounts;
			    POSITION pos = Accounts.GetHeadPosition();
			    while(pos != NULL)
			    {
				    bool bFound = false;
				    CAccount* pAcc = Accounts.GetNext(pos);
				    for(int i = 0; i < m_TabArray.GetSize(); ++i)
				    {
					    if(m_TabArray[i].Accounts.Find(pAcc)) {
						    bFound = true;
						    break;
					    }
				    }
				    if(!bFound)	AddTab(pAcc);
			    }
            }
		}

        SetTrashTab();

		// find tab, which holds the previous selected mail:
        int IdxTab = pSelectedMail ? GetTabIdxContainingMail(pSelectedMail) : 0;
		SetActiveTab(IdxTab);
		m_List.SelectMail(pSelectedMail);

        UpdateAllTabIconsAndText();        
	}

	Key.Close();	

	ResizeControls();

	return TRUE;
}

void CPopManView::SetTrashTab()
{
    if(NeedTrashTab())
    {
        if(m_bTrashTabIsVisible) return;
        m_bTrashTabIsVisible = true;
        AddTab(GetDocument()->m_Accounts, false, _T("")); // add to the end; the last tab is the trahs tab if m_bTrashTabIsVisible == true; Caption is set in UpdateTabIconAndText!
    }
    else
    {
        if(!m_bTrashTabIsVisible) return;
        if(GetTrashTabIdx() == m_Tab.GetCurSel()) SwitchTab(m_LastTabIdx);
        RemoveTab(GetTrashTabIdx(), 0);
        m_bTrashTabIsVisible = false;
        m_List.SetFocus();
    }

    UpdateAllTabIconsAndText();
}

bool CPopManView::NeedTrashTab() const
{
    if(!m_bTrashTab || !IsTabbingEffective()) return false;
    return GetDocument()->m_Accounts.ExistsMail(IsMarkedMail);
}

int CPopManView::GetTrashTabIdx() const
{
    if(m_bTrashTabIsVisible)
        return m_TabArray.GetSize()-1; // Trash Tab is always the last one
    else
        return -1;
}

void CPopManView::UpdateAllTabIconsAndText()
{
    if(IsTabbingEffective())
    {
        for(int i = 0; i < m_Tab.GetItemCount(); ++i)
            UpdateTabIconAndText(i);
    }
}

void CPopManView::UpdateActiveTab()
{
    if(IsTabbingEffective())
    {
        int currIdx = m_Tab.GetCurSel();
        SwitchTab(currIdx);
        UpdateTabIconAndText(currIdx);
    }
}

void CPopManView::SaveLayoutAndReset()
{	
	CSettingsKey Key;
	CreateSettingsKey(Key, szViewKey);

	if(!IsTabbingEffective()) 
	{
		Key.CreateSubKey(szAllKey);
		m_List.SaveSettings(Key);
	}
	else
	{
		Key.CreateSubKey(szTabbingKey);
		m_List.SaveSettings(Key);
		
		CString stTabNum;
		int i = 0;
		while(i < m_TabArray.GetSize() && i != GetTrashTabIdx())
		{
			stTabNum.Format(_T("Tab%d"), i+1);
			CSettingsKey SubKey;
			Key.CreateSubKey(stTabNum, SubKey);

			// Save accounts:
			CString stAcc;
			int n = 1;

			const CAccounts& Accounts = m_TabArray[i].Accounts;
			POSITION pos = Accounts.GetHeadPosition();
			while(pos != NULL) {
				const CAccount* pAcc = Accounts.GetNext(pos);
				stAcc.Format(_T("Acc%d"), n++);
				SubKey.SetValue(stAcc, pAcc->m_stName);
			}

			// delete possible former accounts:
			CString stData;
			while(true)
			{
				stAcc.Format(_T("Acc%d"), n++);
				if(!SubKey.QueryValue(stAcc, stData))
					break;

				SubKey.DeleteValue(stAcc);
			}

			// Save tab caption:
			SubKey.SetValue(szCaptionValue, m_TabArray[i].Caption);

			SubKey.SetValue(szAutoCaptionValue, m_TabArray[i].bAutoCaption);
			
			SubKey.Close();

			++i;
		}

		// delete obsolete tab settings:
		while(true)
		{
			CSettingsKey SubKey;
			stTabNum.Format(_T("Tab%d"), ++i);
			if(!Key.OpenSubKey(stTabNum, SubKey))
				break;

			SubKey.Close();
			Key.DeleteSubKey(stTabNum);
		}

		m_Tab.DestroyWindow();
	}

	m_TabArray.SetSize(0, 16);
}


void CPopManView::OnSize(UINT nType, int cx, int cy) 
{
	CFormView::OnSize(nType, cx, cy);
	
	ResizeControls();
}

void CPopManView::ResizeControls()
{
	RECT rect;
	GetClientRect(&rect);
	int cx = rect.right;
	int cy = rect.bottom;

	if(m_List.m_hWnd == NULL)
		return;
	
	if(!IsTabbingEffective()) 
	{
		RECT rect;
		GetClientRect(&rect);
		m_List.MoveWindow(&rect); 
	}
	else
	{
		if(m_Tab.m_hWnd == NULL)
			return;   

		RECT tabRect = {0,0,0,0};
		m_Tab.GetItemRect(0, &tabRect);
		m_Tab.SetWindowPos(&wndBottom, 0, 0, cx, tabRect.bottom+4, SWP_SHOWWINDOW);

        m_Tab.GetItemRect(0, &tabRect);
		m_Tab.SetWindowPos(&wndBottom, 0, 0, cx, tabRect.bottom+4, SWP_SHOWWINDOW);
	
		m_List.SetWindowPos(&wndTop, 0, tabRect.bottom, cx, cy-tabRect.bottom, SWP_SHOWWINDOW);
	}
}

void CPopManView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
	if(!IsTabbingEffective())
    {
		m_List.OnUpdate(pSender, lHint, pHint);

        if(m_bMultiTab && lHint == CPopManDoc::NOTIFICATIONS::ntNewAccount)
            SetTabbing(TRUE);
    }
	else
    {
		switch(lHint)
		{
			case CPopManDoc::NOTIFICATIONS::ntNewAccount:
			{
				CAccount* pAcc = (CAccount*)pHint;
				AddTab(pAcc);
                
                if(IsTrashTabVisible())
                    m_TabArray[GetTrashTabIdx()].Accounts.AddTail(pAcc);

				break;
			}
			case CPopManDoc::NOTIFICATIONS::ntAccountRemoved:
			{

				CAccount* pAcc = (CAccount*)pHint;
				for(int i = 0; i < m_TabArray.GetSize(); ++i)
				{
					POSITION pos = m_TabArray[i].Accounts.Find(pAcc);
					if(pos != NULL)
					{
						m_TabArray[i].Accounts.RemoveAt(pos);
						if(m_TabArray[i].Accounts.GetCount() == 0) 
						{
                            RemoveTab(i, 0);
                            --i;
        
							if(m_TabArray.GetSize() == 0)
								SetTabbing(TRUE);
						}
						else
						{
							m_List.Refresh();
							UpdateTabIconAndText(i);
						}
					}
				}
				break;
			}
			case CPopManDoc::NOTIFICATIONS::ntAccountChanged:
			{
				CPopManDoc::AccountChangedParam* param = (CPopManDoc::AccountChangedParam*)pHint;
				CAccount* pAcc = param->m_pOldAccount;
				for(int i = 0; i < m_TabArray.GetSize(); ++i)
				{
					POSITION pos = m_TabArray[i].Accounts.Find(pAcc);
					if(pos != NULL)
					{
						if(param->m_pNewAccount != param->m_pOldAccount)
							m_TabArray[i].Accounts.GetAt(pos) = param->m_pNewAccount;

						UpdateTabIconAndText(i);
					}
				}
				break;
			}
			case CPopManDoc::NOTIFICATIONS::ntEnsureVisible:
			{
				CMail* pMail = (CMail*)pHint;
				if(pMail) {
					int idx = GetTabIdxContainingMail(pMail);
                    if(m_Tab.GetCurSel() != idx)
					    SetActiveTab(idx);
                    m_List.SelectMail(pMail);
				}
				break;
			}
			default:
			{
				m_List.OnUpdate(pSender, lHint, pHint);
				if(lHint == CPopManDoc::NOTIFICATIONS::ntNewMail)
				{
					const CMail* pMail = (CMail*)pHint;
					int Idx = GetTabIdxContainingMail(pMail);
					if(Idx > -1) 
						UpdateTabIconAndText(Idx);
				}
				else if(lHint == CPopManDoc::NOTIFICATIONS::ntClear)
				{
					UpdateAllTabIconsAndText();
				}
				else if(lHint == CPopManDoc::NOTIFICATIONS::ntMailsDeleted)
				{
					UpdateAllTabIconsAndText();
				}
			}
		}
    }
}

int CPopManView::GetTabIdxContainingMail(const CMail* pMail)
{
    if(!pMail) return -1;

    if(pMail->IsMarkedForDelete() && IsTrashTabVisible())
        return GetTrashTabIdx();

    CAccount* pAcc = pMail->GetAccount();
    for(int i = 0; i < m_TabArray.GetSize(); ++i)
	{
		if(m_TabArray[i].Accounts.Find(pAcc) != NULL)
			return i;
	}
	return -1;
}

void CPopManView::RemoveTab(int idx, int idxNewCurrent)
{
    if(idx < 0 || idx >= m_TabArray.GetSize()) return;

    bool bSetNewCurr     = (idx == m_Tab.GetCurSel());
    bool bReinitAccounts = (idx <  m_Tab.GetCurSel());

	m_TabArray.RemoveAt(idx);
	m_Tab.DeleteItem(idx);

    if(bReinitAccounts) // we need to update the pointer to the accounts, because the accounts have moved!!!
        m_List.m_pAccounts = &m_TabArray[m_Tab.GetCurSel()].Accounts;
    else if(bSetNewCurr)
        SetActiveTab(idxNewCurrent);
}

int CPopManView::AddTab(const CAccounts& accounts, bool bAutoCaption, const CString& stCaption, int Idx)
{
	TabInfo newTab;
	newTab.bAutoCaption = bAutoCaption;
	newTab.Accounts = accounts;
    newTab.Caption = stCaption;

    if(Idx < 0) Idx = m_TabArray.GetSize();

 	m_TabArray.InsertAt(Idx, newTab); // this may reallocate all tab infos
    // therefore the pointer to the Accounts list may be invalid!!!
    
    int currIdx = m_Tab.GetCurSel();
    if(currIdx >= 0 && currIdx < m_TabArray.GetSize()) 
        m_List.m_pAccounts = &m_TabArray[currIdx].Accounts;
    

	m_Tab.InsertItem(Idx, stCaption);

	UpdateTabIconAndText(Idx);
    return Idx;
}

int CPopManView::AddTab(CAccount* pAcc, int Idx)
{ 
	if(!pAcc || Idx > m_TabArray.GetSize()) 
		return -1;

    CAccounts accounts;
    accounts.AddTail(pAcc);

    if( Idx<0 || Idx==m_TabArray.GetSize() )
        Idx = GetTrashTabIdx();             // add to the end if there is no trash tab

    return AddTab(accounts, true, _T(""), Idx); 
}

UINT CPopManView::AddTab(const CSettingsKey& settings)
{
    CAccounts accounts;

	CString stAccNum, stAccVal;
	int n = 1;
	while(true)
	{
		stAccNum.Format(_T("Acc%d"), n++);
		if(!settings.QueryValue(stAccNum, stAccVal))
			break;

		CAccount* pAcc = GetDocument()->m_Accounts.FindByName(stAccVal);
		if(pAcc)
			accounts.AddTail(pAcc);
	}

    if(accounts.GetCount() == 0) // if the tab has no valid accounts, don't add it!
		return 0;

    bool bAutoCaption = true;
	settings.QueryValue(szAutoCaptionValue, bAutoCaption);

    CString stCaption;
	if(!bAutoCaption)
		settings.QueryValue(szCaptionValue, stCaption);

    AddTab(accounts, bAutoCaption, stCaption); 

    return accounts.GetCount();
}

void CPopManView::OnFont() 
{
	LOGFONT lgFont;
	m_Font.GetLogFont(&lgFont);
	
	CFontDialog fDlg(&lgFont, CF_SCREENFONTS);
	if(IDOK != fDlg.DoModal())
		return;
		
	fDlg.GetCurrentFont(&lgFont);
	m_Font.DeleteObject();
	m_Font.CreateFontIndirect(&lgFont);

	m_List.SetFont(&m_Font);
}

void CPopManView::OnFontColor() 
{
	CColorDialog cDlg(m_List.GetTextColor());
	if(IDOK != cDlg.DoModal())
		return;

	m_List.SetTextColor(cDlg.GetColor());
	m_List.Invalidate();
	m_List.UpdateWindow();
}


BOOL CPopManView::OnCmdMsg(UINT nID, int nCode, void* pExtra, AFX_CMDHANDLERINFO* pHandlerInfo) 
{
	if(CFormView::OnCmdMsg(nID, nCode, pExtra, pHandlerInfo))
		return TRUE;

	// route unhandeled Commands to the MailList:
	return m_List.OnCmdMsg(nID, nCode, pExtra, pHandlerInfo);
}

void CPopManView::OnGotoNextTab() 
{
	if(!IsTabbingEffective()) return;

	int idx = 1 + m_Tab.GetCurSel();

	if(idx >= m_Tab.GetItemCount())
		idx = 0;

	SwitchTab(idx);
}

void CPopManView::OnGotoPrevTab() 
{
	if(!IsTabbingEffective()) return;

	int idx = m_Tab.GetCurSel() - 1;
	if(idx < 0)
		idx = m_Tab.GetItemCount()-1;

	SwitchTab(idx);
}

void CPopManView::SetMultiTabbing(BOOL bMultiTab)
{
    /*
	if(bMultiTab && GetDocument()->m_Accounts.GetCount() == 0)
		AfxMessageBox(i18n("You have to add at least one account before you can switch to tab view."), MB_OK | MB_ICONINFORMATION);
	else
		SetTabbing(bMultiTab);
    */

    SetTabbing(bMultiTab);
}

BOOL CPopManView::GetMultiTabbing()
{
	return m_bMultiTab;
}

void CPopManView::SetActiveTab(int Idx)
{
	if(Idx >= 0 && Idx < m_TabArray.GetSize() && IsTabbingEffective())
	{
		m_Tab.SetCurSel(Idx);
		TabInfo& tab = m_TabArray[Idx];
		m_List.m_pAccounts = &tab.Accounts;
        m_List.m_bShowMarkedMail   = ShowMarkedMails(Idx);
        m_List.m_bShowUnmarkedMail = ShowUnmarkedMails(Idx);
		m_List.Refresh();
	}
}

void CPopManView::SwitchTab(int Idx)
{
	if(!IsTabbingEffective()) return;
	LRESULT res = 0;
	OnTabSelchanging(NULL, &res);
	m_Tab.SetCurSel(Idx);
	OnTabSelchange(NULL, &res);
}

CString CPopManView::GetAutoTabCaption(const CAccounts& Accounts)
{
	CString stCaption;
	POSITION pos = Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		const CAccount* pAcc = Accounts.GetNext(pos);
		stCaption += pAcc->m_stName;
		if(pos) stCaption += _T(" | ");
	}
	return stCaption;
}


void CPopManView::OnTabRename() 
{
	if(!IsTabbingEffective()) return;
	int Idx = m_Tab.GetCurSel();
    if(Idx == GetTrashTabIdx()) return;

	CString stCaption = m_TabArray[Idx].Caption;

	CDlgInput input(this);
	input.m_stHint = i18n("Caption:");
	input.m_stMessage = i18n("Please enter the caption of the tab:");
	input.m_stTitle = i18n("Rename tab");
	input.m_bAcceptEmptyInput = FALSE;
	input.m_stValue = stCaption;

	if(input.DoModal() == IDOK && !input.m_stValue.IsEmpty())
	{
		m_TabArray[Idx].Caption = input.m_stValue;
		if(input.m_stValue != stCaption)
			m_TabArray[Idx].bAutoCaption = FALSE;
		UpdateTabIconAndText(Idx);
	}	
}



void CPopManView::UpdateTabIconAndText(int Idx)
{
	if(Idx < 0 || Idx >= m_TabArray.GetSize() || !IsTabbingEffective())
		return;

    const CAccounts& Accounts = m_TabArray[Idx].Accounts;

	UINT nMails = 0; 
    UINT nUnreadMails = 0;

    if(!IsTrashTabVisible())
    {
        nMails = Accounts.CountMails(IsMail);
        nUnreadMails = Accounts.CountMails(IsUnreadMail);
    }
    else 
    {
        if(Idx == GetTrashTabIdx()) {
            nMails = Accounts.CountMails(IsMarkedMail);
            nUnreadMails = Accounts.CountMails(IsUnreadAndMarkedMail);
        }
        else {
            nMails = Accounts.CountMails(IsUnmarkedMail);
            nUnreadMails = Accounts.CountMails(IsUnreadAndUnmarkedMail);
        }
    }

	TCITEM item;
	item.mask = TCIF_IMAGE;
	item.iImage = GetIconIdx(nMails, nUnreadMails);
	m_Tab.SetItem(Idx, &item);	


	if(m_TabArray[Idx].bAutoCaption)
		m_TabArray[Idx].Caption = GetAutoTabCaption(m_TabArray[Idx].Accounts);

    if(Idx == GetTrashTabIdx())
        m_TabArray[Idx].Caption = i18n("Trash");
	
	CString stCaption = m_TabArray[Idx].Caption;

	if(nMails > 0)
	{
		CString stFormat;
		if(nUnreadMails > 0)
			stFormat.Format(_T(" - %d/%d"), nUnreadMails, nMails);
		else
			stFormat.Format(_T(" - %d"), nMails);
		stCaption += stFormat;
	}
	
	m_Tab.SetItemCaption(Idx, stCaption);
}

int CPopManView::GetIconIdx(UINT nMails, UINT nUnreadMails)
{
	if(nUnreadMails > 0) 
		return 1;

	if(nMails > 0) 
		return 0;
	
	return -1;
}

/*
int CPopManView::GetUpdatedTabIconIdx(int TabIdx)
{
	if(TabIdx < 0 || TabIdx >= m_TabArray.GetSize())
		return -1;

	bool bMails = false;
	bool bUnreadMails = false;

	CAccounts& Accounts = m_TabArray[TabIdx].Accounts;
	
	
	POSITION pos1 = Accounts.GetHeadPosition();
	while(pos1 != NULL && !bUnreadMails)
	{
		const CAccount* pAcc = Accounts.GetNext(pos1);
		const CMails& Mails = pAcc->m_Mails;
		POSITION pos2 = Mails.GetHeadPosition();
		while(pos2 != NULL)
		{
			bMails = true;
			const CMail* pMail = Mails.GetNext(pos2);
			if(!pMail->IsRead()) {
				bUnreadMails = true;
				break;
			}
		}
	}

	return GetIconIdx(bMails?1:0, bUnreadMails?1:0);
}
*/

LRESULT CPopManView::OnMailRead(WPARAM wParam, LPARAM lParam)
{
	if(!IsTabbingEffective()) return 1;

	CMail* pMail = NULL;

	try {
		pMail = dynamic_cast<CMail*>((CObject*)wParam);
		if(pMail == 0) return 0;
	} 
	catch (...)	{
		return 0;
	}

	int Idx = GetTabIdxContainingMail(pMail);
	UpdateTabIconAndText(Idx);

	return 1;
}

void CPopManView::JoinTabs(int Master, int Slave)
{
    if(Master == GetTrashTabIdx() || Slave == GetTrashTabIdx()) return;
    
    if(Master == Slave) return;
	if(Master < 0 || Master >= m_TabArray.GetSize()) return;
	if(Slave < 0  || Slave >= m_TabArray.GetSize()) return;


	CAccounts& Accounts = m_TabArray[Slave].Accounts;
	POSITION pos = Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = Accounts.GetNext(pos);
		m_TabArray[Master].Accounts.AddTail(pAcc);
	}
	
    int newActiveTab = Master - (Slave < Master ? 1 : 0);

    RemoveTab(Slave, newActiveTab);

	SetActiveTab(newActiveTab);
	UpdateTabIconAndText(newActiveTab);
}

void CPopManView::SplitTab(int Idx)
{
    if(Idx == GetTrashTabIdx()) return;

	if(Idx < 0  || Idx >= m_TabArray.GetSize()) return;
	if(m_TabArray[Idx].Accounts.GetCount() < 2) return;

	CAccounts& Accounts = m_TabArray[Idx].Accounts;
	POSITION pos = Accounts.GetHeadPosition();
	CAccount* pFirstAcc = Accounts.GetNext(pos);
	int i = Idx;
	while(pos != NULL)
	{
		CAccount* pAcc = Accounts.GetNext(pos);
		AddTab(pAcc, ++i);
	}
	m_TabArray[Idx].Accounts.RemoveAll();
	m_TabArray[Idx].Accounts.AddTail(pFirstAcc);
	m_TabArray[Idx].bAutoCaption = TRUE;

	SetActiveTab(Idx);
	UpdateTabIconAndText(Idx);
}

void CPopManView::OnTabSplit() 
{
	SplitTab(m_Tab.GetCurSel());
}

void CPopManView::OnMoveTab(int from, int to)
{
    if(from == GetTrashTabIdx() || to == GetTrashTabIdx()) return;

	if(from < 0  || from >= m_TabArray.GetSize()) return;
	if(to < 0    || to >= m_TabArray.GetSize()) return;
	if(from == to) return;

	TCITEM item;
	item.mask = TCIF_IMAGE;
	m_Tab.GetItem(from, &item);	
	CString stCaption;
	m_Tab.GetItemCaption(from, stCaption);
	TabInfo info;
	info = m_TabArray[from];

	m_Tab.DeleteItem(from);
	m_TabArray.RemoveAt(from);

	m_Tab.InsertItem(to, stCaption, item.iImage);
	m_TabArray.InsertAt(to, info);
	
	SetActiveTab(to);
}

void CPopManView::OnMenuSelect(UINT nItemID, UINT nFlags, HMENU hSysMenu) 
{
	CFormView::OnMenuSelect(nItemID, nFlags, hSysMenu);

    //TRACE("OnMenuSelect %d \n", nItemID); 

	if(nItemID != ID_TAB_ACC_REMOVE && nItemID != ID_TAB_JOIN) return;

    //TRACE("OnMenuSelect Remove or Join %d \n", nItemID); 

	MENUITEMINFO info;
	info.cbSize = sizeof(info);
	info.fMask = MIIM_STATE;

	int count = ::GetMenuItemCount(hSysMenu);
	for(int i=0; i < count; ++i) 
	{
		if(GetMenuItemInfo(hSysMenu, i, TRUE, &info)) {
			if(info.fState & MFS_HILITE) {
				m_nTabContextMenuIdx = i;
                //TRACE("m_nTabContextMenuIdx: %d \n", m_nTabContextMenuIdx); 
				break;
			}
		}
	}
}

void CPopManView::OnShowTabToolTip(int TabIdx)
{	
	if(TabIdx < 0  || TabIdx >= m_TabArray.GetSize() || TabIdx == GetTrashTabIdx()) return;

	if(m_TipAcc.m_hWnd != NULL || m_bTabContextMenu) return;

	CString stAccs;

	CAccounts& Accounts = m_TabArray[TabIdx].Accounts;
	POSITION pos = Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = Accounts.GetNext(pos);
		stAccs += _T(" ") + pAcc->m_stName;
		if(pos != NULL) stAccs += _T("\r\n");
	}

	m_TipAcc.Show(NULL, CString(i18n("Tab accounts:")) + _T(" "), stAccs, true);
}


void CPopManView::OnHideMarked() 
{
	m_bHideMarkedMails = !m_bHideMarkedMails;

    if(IsTabbingEffective())
        UpdateActiveTab();
    else {
        CMail* pSelectedMail = m_List.GetSelectedMail();
        m_List.m_bShowMarkedMail = !m_bHideMarkedMails;
		m_List.Refresh();
		m_List.SelectMail(pSelectedMail);
    }
}
void CPopManView::OnUpdateHideMarked(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(m_bHideMarkedMails);	
}

void CPopManView::OnTrashTab() 
{
	m_bTrashTab = !m_bTrashTab;
    SetTrashTab();
    UpdateActiveTab();
    UpdateAllTabIconsAndText();
}
void CPopManView::OnUpdateTrashTab(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck(m_bTrashTab);		
}


void CPopManView::OnMailMarkedChange()
{
	SetTrashTab();
    m_List.OnMailMarkedChange();
    if(IsTabbingEffective())
    {
        if(m_Tab.GetCurSel() == GetTrashTabIdx())
        {
            UpdateAllTabIconsAndText();
        }
        else
        {
            UpdateTabIconAndText(m_Tab.GetCurSel());
            UpdateTabIconAndText(GetTrashTabIdx());
        }
    }
}



/////////////////////////////////////////////////////////////////////////////
//// CMyTabCtrl  ////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////


BEGIN_MESSAGE_MAP(CMyTabCtrl, CTabCtrl)
	//{{AFX_MSG_MAP(CMyTabCtrl)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_SETCURSOR()
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CMyTabCtrl::InsertItem( int nItem, LPCTSTR lpszItem )
{
	CString stCaption = lpszItem;
	stCaption.Replace(_T("&"), _T("&&"));
	return CTabCtrl::InsertItem(nItem, stCaption);
}

BOOL CMyTabCtrl::InsertItem( int nItem, LPCTSTR lpszItem, int nImage )
{
	CString stCaption = lpszItem;
	stCaption.Replace(_T("&"), _T("&&"));
	return CTabCtrl::InsertItem(nItem, stCaption, nImage);
}

void CMyTabCtrl::OnLButtonDown(UINT nFlags, CPoint point) 
{
	if(nFlags & MK_CONTROL)
	{
		m_pView->JoinTabs(GetCurSel(), GetTabIdxBelowCursor());
		return;
	}

    if(m_pView->IsTabDraggable(GetTabIdxBelowCursor()))
    {
	    ::GetCursorPos(&m_ptDragStart);
	    ScreenToClient(&m_ptDragStart);
	    SetCapture();
	    m_bCapture = true;
    }

	CTabCtrl::OnLButtonDown(nFlags, point);
}

void CMyTabCtrl::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if(m_bCapture) ReleaseCapture();
	m_bCapture = false;

	if(!m_bDragging)
		CTabCtrl::OnLButtonUp(nFlags, point);

	if(m_bDragging && GetTabIdxBelowCursor() != GetCurSel())
		m_pView->OnMoveTab(GetCurSel(), GetTabIdxBelowCursor());

	m_bDragging = false;
}


void CMyTabCtrl::OnMouseMove(UINT nFlags, CPoint point) 
{
	if(m_bCapture && !m_bDragging) 
	{
		point -= m_ptDragStart;
		if(abs(point.x) > 5 || abs(point.y) > 5)
			m_bDragging = true;
	}
	
	if(m_bDragging)
	{
		if(m_curDragMove == NULL)	m_curDragMove = CPopManApp::GetCursor(IDC_DRAG_MOVE);
		if(m_curNoDrop == NULL)		m_curNoDrop   = CPopManApp::GetCursor(IDC_NO_DROP);
	
        int tabBelowCursor = GetTabIdxBelowCursor();

		if(tabBelowCursor > -1 && m_pView->IsTabDropable(tabBelowCursor))
			::SetCursor(m_curDragMove);
		else
			::SetCursor(m_curNoDrop);
		return;
	}

	if(!m_bCapture && !m_bDragging)
	{
		m_currTabIdx = GetTabIdxBelowCursor();
		if(m_currTabIdx > -1) 
		{
			if(m_TimerID > 0) {
				KillTimer(m_TimerID);
				m_TimerID = 0;
			}

			m_TimerID = SetTimer(1, 1200, NULL);
		}
	}

	CTabCtrl::OnMouseMove(nFlags, point);
}

void CMyTabCtrl::OnTimer(UINT nIDEvent) 
{
	KillTimer(m_TimerID);
	m_TimerID = 0;
	CTabCtrl::OnTimer(nIDEvent);

	if(m_currTabIdx == GetTabIdxBelowCursor() && !m_bCapture)
		m_pView->OnShowTabToolTip(m_currTabIdx);
}

void CMyTabCtrl::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
/*	if(nFlags & MK_CONTROL)
	{
		if(GetTabIdxBelowCursor() == GetCurSel())
			m_pView->SplitTab(GetCurSel());
	}
	else 
	{
    */
        m_pView->CheckCurrentTab();
	//}

	CTabCtrl::OnLButtonDblClk(nFlags, point);
}

int CMyTabCtrl::GetTabIdxBelowCursor()
{
	POINT point;
	::GetCursorPos(&point);
	ScreenToClient(&point);
	TCHITTESTINFO info;
	info.pt = point;
	info.flags = 0;
	return HitTest(&info);
}


BOOL CMyTabCtrl::GetItemCaption(int Idx, CString& stCaption)
{
	if(GetItemCount() <= Idx)
		return FALSE;

	TCITEM item; 
	const int nBufferSize = 250;
	LPTSTR pCaption = stCaption.GetBuffer(nBufferSize);
	item.mask = TCIF_TEXT;
	item.pszText = pCaption;
	item.cchTextMax = nBufferSize;

	BOOL bRes = GetItem(Idx, &item);
	stCaption.ReleaseBuffer();
	stCaption.Replace(_T("&&"), _T("&"));
	return bRes;
}

BOOL CMyTabCtrl::SetItemCaption(int Idx, CString stCaption)
{
	if(GetItemCount() <= Idx)
		return FALSE;

	stCaption.Replace(_T("&"), _T("&&"));

	TCITEM item; 
	item.mask = TCIF_TEXT;
	item.pszText = (LPTSTR)(LPCTSTR)stCaption;

	return SetItem(Idx, &item);
}


void CPopManView::OnInitMenuPopup(CMenu* pPopupMenu, UINT nIndex, BOOL bSysMenu) 
{
	CFormView::OnInitMenuPopup(pPopupMenu, nIndex, bSysMenu);
	
    // Send WM_INITMENUPOPUP message to main window because only MainFrames handle this message in order to trigger ON_UPDATE_COMMAND_UIs
    AfxGetMainWnd()->SendMessage(WM_INITMENUPOPUP, (UINT)pPopupMenu->m_hMenu, MAKELONG(nIndex, bSysMenu));	
}
