// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// PopManDoc.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "PopMan.h"
#include <ctype.h>
#include "PopManDoc.h"
#include "MainFrm.h"
#include "DlgAccounts.h"
#include "DlgDownload.h"
#include "DlgPass.h"
#include "DlgSave.h"
#include "RegKey.h"
#include "StrFunctions.h"
#include "SyncSocket.h"
#include "DlgAccImport.h"
#include "FileDlgEx.h"
#include "MessageFrm.h"
#include "AutoUpdate.h"
#include "DlgMailAddressList.h"
#include "POP3Account.h"
#include "PluginAccount.h"
#include "Plugin.h"
#include "MessageBoxDialog.h"
#include "NTKbdLites.h"
#include "DlgRulesItem.h"

#include <mmsystem.h>
#include <wininet.h>        
#include <mapi.h>
#include <vector>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


CPopManDoc* CPopManDoc::m_pDoc = NULL;

//  i18nComment("Document")

/////////////////////////////////////////////////////////////////////////////
// CPopManDoc

IMPLEMENT_DYNCREATE(CPopManDoc, CDocument)

BEGIN_MESSAGE_MAP(CPopManDoc, CDocument)
	//{{AFX_MSG_MAP(CPopManDoc)
	ON_UPDATE_COMMAND_UI(ID_FILE_CANCEL, OnUpdateCancel)
	ON_COMMAND(ID_FILE_ACCOUNTS, OnAccounts)
	ON_UPDATE_COMMAND_UI(ID_FILE_LIST_MAILS, OnUpdateListMails)
	ON_COMMAND(ID_FILE_LIST_MAILS, OnListMails)
	ON_COMMAND(ID_EXTRAS_IMPORT_ACCOUNTS, OnImportAccounts)
	ON_COMMAND(ID_RELOAD_MAILS, OnReloadMails)
	ON_COMMAND(ID_APP_AUTOCHECK, OnAutocheckEnable)
	ON_UPDATE_COMMAND_UI(ID_APP_AUTOCHECK, OnUpdateAutocheckEnable)
	ON_COMMAND(ID_APP_EMAIL_CLIENT, OnLaunchEmailClient)
	ON_COMMAND(ID_EXTRAS_BLACKLIST, OnShowBlacklist)
	ON_COMMAND(ID_FILE_EMPTY_TRASH, OnEmptyTrash)
	ON_UPDATE_COMMAND_UI(ID_FILE_EMPTY_TRASH, OnUpdateEmptyTrash)
	ON_COMMAND(ID_EXTRAS_RULES, OnRules)
	ON_COMMAND(ID_EXTRAS_WHITELIST, OnShowWhitelist)
	ON_COMMAND(ID_FILE_CANCEL, OnCancel)
	ON_COMMAND(ID_APP_MUTE_MODE, OnMuteMode)
	ON_UPDATE_COMMAND_UI(ID_APP_MUTE_MODE, OnUpdateMuteMode)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPopManDoc Konstruktion/Destruktion

CPopManDoc::CPopManDoc() : m_TrayIcon(1)
{
	TRACE(_T(" CPopManDoc::CPopManDoc\n"));
	m_pDoc = this;

	m_TrayIcon.m_fnCallback = OnTrayAction;

	m_pDlgDownload = NULL;
	m_nUpdateInterval = 10;
	m_nTimerIDAutoCheck = 0;
	m_nTimerCheckOnStartup = 0;
	m_bNotification = TRUE;
	m_bSoundNotify = TRUE;
	m_bMarkUnread = TRUE;

	m_bLastLogFromServer = FALSE;
	
	m_bCheckMailOnStartup = TRUE;

	m_stAppDataPath  = CPopManApp::GetAppDataPath();
	if(m_stAppDataPath.Right(1) != _T("\\")) m_stAppDataPath += _T('\\');

	m_SaveMailAs = SAVEFILETYPES::frmEmlFile;
	
	m_dwLastUpdateCheck = 0;

	m_currentAction = ACTIONS::Idle;

	m_bError = FALSE;

	m_bAutoCheckMinimized = TRUE;
	m_nNotify = NOTIFYSTATES::AutoChecked;
	m_bAutoChecked = FALSE;

	m_bSuspendAutoCheck = FALSE;
	m_nSuspendFrom = 20 * 3600;
	m_nSuspendTo = 8 * 3600;
	m_nStartCheckDelay = 60;

	m_nReplyOption = REPLYOPTIONS::MAPI;
	m_bHTMLReply = FALSE;
	m_bEnforceAddress = FALSE;
	m_stReplyParam = _T("mailto:[TO]?subject=[SUBJECT]");

	m_actionLeftClick = TRAYACTIONS::ShowMailInfo;
	m_actionMiddleClick = TRAYACTIONS::CheckMail;
	m_actionRightClick = TRAYACTIONS::PopUpMenu;
	m_actionDoubleClick = TRAYACTIONS::ShowWindow;

	m_ClientLaunchAction = CLIENTLAUNCHACTION::HideToTray;

	m_bConfirmDeleteMain = TRUE;
	m_bConfirmDeleteMsg = TRUE;
	m_numMails = NUMMAILSTRAY::Unread;
	m_bNumMailsInTray = TRUE;
	m_bRotateMailNum = FALSE;
	m_nRotationInterval = 25;	// == 2.5 seconds
	m_bHighColorTray = TRUE;
	m_bAdvancedTrayInfo = FALSE;
	m_bShowMainWnd = FALSE;

	m_bShowNewMailTip = TRUE;
	m_bDetailedNewTip = TRUE;
	m_bBallonTip = FALSE;
	m_bNotifyProtectedOnly = FALSE;

    SetMaxLogSize(750 * 1024);
    EnableLog(true);

    SetMaxRulesLogSize(250 * 1024);
    EnableRulesLog(true);

	m_bCompactLogging = TRUE;

	m_stCurrStatusText = i18n("Ready");
	m_LastCheckTime.SetStatus(COleDateTime::DateTimeStatus::null);
    m_LastSuccessfulCheckTime.SetStatus(COleDateTime::DateTimeStatus::null);

	m_bCustomAutoCheck = TRUE;
	m_nTimerIDKeyboard = 0;
	m_KeyboardLight = KEYBOARDLIGHT::None;
	m_bKeyLightToggled = false;
	m_hKeyboardDev = INVALID_HANDLE_VALUE; 

    m_bMuteMode = false;

	m_bMarkDelete = TRUE;
	m_bShowMarkDeleteInfo = TRUE;
	m_bShowEmptyTrashWarning = TRUE;
	    
    m_nNewMailTipDuration = 900;
    m_bMarkBlackListedMail = FALSE;

    m_bDisableDeferCheck = true;
 }

CPopManDoc::~CPopManDoc()
{
	m_pDoc = NULL;

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
		delete m_Accounts.GetNext(pos);
	
	m_Accounts.RemoveAll();


	pos = m_ProtocolPlugins.GetHeadPosition();
	while(pos != NULL)
		delete m_ProtocolPlugins.GetNext(pos);
	
	m_ProtocolPlugins.RemoveAll();


	TRACE(_T("~CPopManDoc\n"));
}


/////////////////////////////////////////////////////////////////////////////
// CPopManDoc Diagnose

#ifdef _DEBUG
void CPopManDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CPopManDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG



void CPopManDoc::Timer_AutoCheck(HWND hwnd, UINT uMSG, UINT idEvent, DWORD dwTime)
{
	if(m_pDoc == NULL)
		return;

    m_pDoc->SaveMailCache();
	
	if(m_pDoc->m_bNotification == FALSE)
		return;

	if(m_pDoc->m_bAutoCheckMinimized) {
		if( !AfxGetMainWnd()->IsIconic() && AfxGetMainWnd()->IsWindowVisible() )
			return;
	}

	if(m_pDoc->m_bSuspendAutoCheck)
	{
		CTime Now = CTime::GetCurrentTime();
		UINT nSeconds = Now.GetHour() * 3600 + Now.GetMinute() * 60 + Now.GetSecond();
		
		if(m_pDoc->m_nSuspendFrom <= m_pDoc->m_nSuspendTo)
		{
			if(nSeconds >= m_pDoc->m_nSuspendFrom && nSeconds <= m_pDoc->m_nSuspendTo)
				return;
		}
		else
		{
			if(nSeconds >= m_pDoc->m_nSuspendFrom || nSeconds <= m_pDoc->m_nSuspendTo)
				return;
		}
	}


	static COleDateTime lastAutoCheckedTime = COleDateTime::GetCurrentTime();

	const COleDateTimeSpan diff = COleDateTime::GetCurrentTime() - lastAutoCheckedTime;
    if(!m_pDoc->m_bDisableDeferCheck && diff.GetTotalMinutes() > 5)
    {
        // if we've got no autocheck timer event for more than 5 minutes,
        // assume that we are resumed from standby -> don't auto check immediately
        // because network connection is most probably not ready yet.
        m_pDoc->m_bDisableDeferCheck = true;
        return; // abort this auto check
    }

	m_pDoc->OnAutoCheck();

    lastAutoCheckedTime = COleDateTime::GetCurrentTime();
}


void CPopManDoc::OnAutoCheck()
{
	m_bAutoChecked = TRUE;
	
	CAccounts Accs;

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
		if(pAcc->IsBusy() || !pAcc->m_bActive)
			continue;

        const COleDateTime now = COleDateTime::GetCurrentTime();
		const COleDateTimeSpan diff = now - pAcc->GetLastCheckTime();
		const UINT nCheckInterval = (m_bCustomAutoCheck && pAcc->m_bCustomAutoCheckInterval ? pAcc->m_nCustomAutoCheckInterval : m_nUpdateInterval);

		//CString stLog;
		//stLog.Format("\nOnAutoCheck-Acc: %s %s: %u, %u ", now.Format(), pAcc->m_stName, (int)diff.GetTotalMinutes(), nCheckInterval);
		//Log(stLog);

		if(diff.GetTotalMinutes() >= nCheckInterval && nCheckInterval > 0) 
        {
			Accs.AddTail(pAcc);
		}
	}
	
	if(Accs.GetCount() > 0)
		ListMailsFromAccs(Accs, FALSE, FALSE);

	m_bAutoChecked = FALSE;
}

void CPopManDoc::SetUpdateInterval(UINT UpdateInterval)
{
	m_nUpdateInterval = UpdateInterval;
}


void CPopManDoc::StartTimer()
{
	StopTimer();
	m_nTimerIDAutoCheck = ::SetTimer(0, 0, 60000, Timer_AutoCheck);
}

void CPopManDoc::StopTimer()
{
	if(m_nTimerIDAutoCheck != 0) {
		::KillTimer(0, m_nTimerIDAutoCheck);
		m_nTimerIDAutoCheck = 0;
	}
}


BOOL CPopManDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	LoadPlugins();

	CString stAccStr;
	
	CSettingsKey Key;
	if(OpenSettingsKey(Key, szAccountsKey))
	{
		// Konten laden:
		int countAccs = Key.CountSubKeys();
		for(int n = 1; n <= countAccs; ++n)
		{
			CAccount* pAcc = NULL;

			stAccStr.Format(_T("Acc%d"), n);

			if(LoadAccount(Key, stAccStr, pAcc))
				m_Accounts.AddTail(pAcc);
			else
				delete pAcc;	
		} 
		Key.Close();
	}
	
	if(!OpenSettingsKey(Key, szSettingsKey) || !Key.QueryValue(szDUNValue, m_bUseDUN))
	{
		DWORD dwState = 0;
		::InternetGetConnectedState(&dwState, 0);
		m_bUseDUN = dwState & INTERNET_CONNECTION_MODEM;
	}


	UINT UpdateInterval = m_nUpdateInterval;
	Key.QueryValue(szUpdateInterval, UpdateInterval);
	SetUpdateInterval(UpdateInterval);

	Key.QueryValue(szNotification, m_bNotification);
	Key.QueryValue(szSoundNotify, m_bSoundNotify);
	if(!Key.QueryValue(szSoundFile, m_stSoundFile))
		m_stSoundFile = GetStdNotifySound();

	Key.QueryValue(szMarkUnread, m_bMarkUnread);
	Key.QueryValue(szShowMarkDeleteInfo, m_bShowMarkDeleteInfo);
	Key.QueryValue(szShowEmptyTrashWarning, m_bShowEmptyTrashWarning);
	

    bool logEnabled;
	if(Key.QueryValue(szLoggingEnabled, logEnabled))
        EnableLog(logEnabled);

    if(Key.QueryValue(szRulesLoggingEnabled, logEnabled))
        EnableRulesLog(logEnabled);

	Key.QueryValue(szCompactLogging, m_bCompactLogging);

    int maxSize = 0;
	if(Key.QueryValue(szMaxLogSize, maxSize))
	{
        SetMaxLogSize(maxSize);
	}
    else {
		EnableLog(true);
    }

    if(Key.QueryValue(szRulesMaxLogSize, maxSize))
        SetMaxRulesLogSize(maxSize);


	CString stVersion;
	Key.QueryValue(szLatestReleaseVersion, stVersion);
	m_LatestReleaseVersion.ReadFromString(stVersion);
	stVersion.Empty();
	Key.QueryValue(szLatestBetaVersion, stVersion);
	m_LatestBetaVersion.ReadFromString(stVersion);
	m_LatestBetaVersion.SetBeta(TRUE);

	Key.QueryValue(szCheckMailOnStartup, m_bCheckMailOnStartup);
	
	Key.QueryValue(szSaveFolder, m_stSaveFolder);
	int nSaveMailAs = 0;
	if(Key.QueryValue(szSaveAs, nSaveMailAs))
		m_SaveMailAs = ((SAVEFILETYPES)nSaveMailAs) == SAVEFILETYPES::frmTxtFile ? SAVEFILETYPES::frmTxtFile : SAVEFILETYPES::frmEmlFile;

	Key.QueryValue(szFileTemplate, m_stFileTemplate);
	Key.QueryValue(szLastUpdateCheck, m_dwLastUpdateCheck);
	Key.QueryValue(szAutoCheckMinimized, m_bAutoCheckMinimized);
	Key.QueryValue(szCustomAutoCheck, m_bCustomAutoCheck);

	
	DWORD nNotify = (DWORD)m_nNotify;
	Key.QueryValue(szNotifyCode, nNotify);
	m_nNotify = (NOTIFYSTATES)nNotify;

	Key.QueryValue(szSuspendAutoCheck, m_bSuspendAutoCheck);
	Key.QueryValue(szSuspendFrom, m_nSuspendFrom);
	Key.QueryValue(szSuspendTo, m_nSuspendTo);
	Key.QueryValue(szStartCheckDelay, m_nStartCheckDelay);
	
	if(!Key.QueryValue(szEmailClient, m_stEmailClient))
		m_stEmailClient = GetStdEmailClient();

	DWORD nReplyOption = (DWORD)m_nReplyOption;
	Key.QueryValue(szReplyOption, nReplyOption);
	m_nReplyOption = (REPLYOPTIONS)nReplyOption;

	Key.QueryValue(szReplyParam, m_stReplyParam);

	Key.QueryValue(szHTMLReply, m_bHTMLReply);
	Key.QueryValue(szEnforceAddress, m_bEnforceAddress);

	DWORD nClientAction = (DWORD)m_ClientLaunchAction;
	Key.QueryValue(szClientLaunchAction, nClientAction);
	m_ClientLaunchAction = (CLIENTLAUNCHACTION)nClientAction;

	Key.QueryValue(szConfirmDeleteMain, m_bConfirmDeleteMain);
	Key.QueryValue(szConfirmDeleteMsg, m_bConfirmDeleteMsg);
	Key.QueryValue(szMarkDelete, m_bMarkDelete);
	
	DWORD nAction = 0;
	if(Key.QueryValue(szActionLeftClick, nAction))
		m_actionLeftClick = (TRAYACTIONS)nAction;

	if(Key.QueryValue(szActionRightClick, nAction))
		m_actionRightClick = (TRAYACTIONS)nAction;

	if(Key.QueryValue(szActionMiddleClick, nAction))
		m_actionMiddleClick = (TRAYACTIONS)nAction;

	if(Key.QueryValue(szActionDoubleClick, nAction))
		m_actionDoubleClick = (TRAYACTIONS)nAction;


	DWORD nNum = 0;
	if(Key.QueryValue(szNumMails, nNum))
		 m_numMails = (NUMMAILSTRAY)nNum;
	
	Key.QueryValue(szNumMailsInTray, m_bNumMailsInTray);
	Key.QueryValue(szRotateMailNum, m_bRotateMailNum);
	
	Key.QueryValue(szRotationInterval, m_nRotationInterval);
	Key.QueryValue(szHighColorTray, m_bHighColorTray);
	Key.QueryValue(szAdvancedTrayInfo, m_bAdvancedTrayInfo);
	Key.QueryValue(szShowMainWnd, m_bShowMainWnd);
	Key.QueryValue(szDoubleClickDelay, m_TrayIcon.m_bDoubleClickDelay);
	
	DWORD dwVal = 0;
	if(Key.QueryValue(szKeyboardLight, dwVal))
		m_KeyboardLight = (KEYBOARDLIGHT)dwVal;
	
	Key.QueryValue(szBallonTip, m_bBallonTip);
	Key.QueryValue(szDetailedNewTip, m_bDetailedNewTip);
	Key.QueryValue(szNotifyProtectedOnly, m_bNotifyProtectedOnly);
	if(!Key.QueryValue(szShowNewMailTip, m_bShowNewMailTip))
		m_bBallonTip = FALSE;

    Key.QueryValue(szNewTipDurationValue, m_nNewMailTipDuration);
    Key.QueryValue(szMarkBlackListedMailValue, m_bMarkBlackListedMail);

    Key.QueryValue(szLastDestinationList, m_stLastSelectedRulesDestList);
    Key.QueryValue(szRulesFileNameSetting, m_stRulesName);
    if(m_stRulesName.IsEmpty())
        m_stRulesName = szRulesFile;
    
	Key.QueryValue(szCheckForUpdates, CPopManApp::m_bCheckForUpdates);

	Key.Close();

	if(m_bNotification)
		StartTimer();

	LoadMailCache();
    m_BlackList.Load(m_stAppDataPath + szBlackListFile);
    m_WhiteList.Load(m_stAppDataPath + szWhiteListFile);

    m_LogFile.Init    (m_stAppDataPath + szLogFile);
    m_RuleLogFile.Init(m_stAppDataPath + szRuleLogFile);

	if(m_bHighColorTray && !m_TrayIcon.HighColorTrayEnabled())
		m_TrayIcon.EnableHighColorTray();
	
	UpdateTrayIcon();


	CString stPath = CPopManApp::GetAppDataPath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += m_stRulesName;

	m_RuleManager.LoadFromFile(stPath);

	return TRUE;
}


void CPopManDoc::OnCloseDocument() 
{
	SaveSettings();

	m_LogFile.Close();
    m_RuleLogFile.Close();

	StopTimer();

	ClearDirectory(CPopManApp::GetAppTempPath());

	CDocument::OnCloseDocument();
}



void CPopManDoc::SaveSettings()
{
	CSettingsKey Key;

	int n = 0;
	CString stAccStr;

	if(OpenSettingsKey(Key, szAccountsKey))
	{
		if(Key.CountSubKeys() > m_Accounts.GetCount())
		{
			n = m_Accounts.GetCount();
			
			// Konten in der Registry lschen:
			do	{

				stAccStr.Format(_T("Acc%d"), ++n);

			} while(Key.DeleteSubKey(stAccStr));
		}
		Key.Close();
	}
	
	// Konten abspeichern:
	if(CreateSettingsKey(Key, szAccountsKey))
	{
		n = 0;
		POSITION pos = m_Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			n++;
			stAccStr.Format(_T("Acc%d"), n);
			CAccount* pAcc = m_Accounts.GetNext(pos);
			SaveAccount(Key, stAccStr, *pAcc);
		}	
	}


	if(CreateSettingsKey(Key, szSettingsKey))
	{
		Key.SetValue(szDUNValue, m_bUseDUN);
		Key.SetValue(szUpdateInterval, m_nUpdateInterval);
		Key.SetValue(szNotification, m_bNotification);
		Key.SetValue(szSoundNotify, m_bSoundNotify);
		Key.SetValue(szSoundFile, m_stSoundFile);
		Key.SetValue(szMarkUnread, m_bMarkUnread);
		
		Key.SetValue(szLoggingEnabled, IsLogEnabled() );
		Key.SetValue(szMaxLogSize, GetMaxLogSize());
        
        Key.SetValue(szRulesLoggingEnabled, IsRulesLogEnabled() );
		Key.SetValue(szRulesMaxLogSize, GetMaxRulesLogSize());

		Key.SetValue(szCompactLogging, m_bCompactLogging);
		Key.SetValue(szBallonTip, m_bBallonTip);
		Key.SetValue(szNotifyProtectedOnly, m_bNotifyProtectedOnly);

		Key.SetValue(szLatestReleaseVersion, m_LatestReleaseVersion.GetString());
		Key.SetValue(szLatestBetaVersion, m_LatestBetaVersion.GetString());

		Key.SetValue(szCheckMailOnStartup, m_bCheckMailOnStartup);

		Key.SetValue(szSaveFolder, m_stSaveFolder);
		Key.SetValue(szSaveAs, m_SaveMailAs);
		Key.SetValue(szFileTemplate, m_stFileTemplate);
		Key.SetValue(szLastUpdateCheck, m_dwLastUpdateCheck);
		Key.SetValue(szAutoCheckMinimized, m_bAutoCheckMinimized);
		Key.SetValue(szCustomAutoCheck, m_bCustomAutoCheck);

		Key.SetValue(szNotifyCode, (DWORD)m_nNotify);

		Key.SetValue(szSuspendAutoCheck, m_bSuspendAutoCheck);
		Key.SetValue(szSuspendFrom, m_nSuspendFrom);
		Key.SetValue(szSuspendTo, m_nSuspendTo);
		Key.SetValue(szStartCheckDelay, m_nStartCheckDelay);
		Key.SetValue(szEmailClient, m_stEmailClient);
		Key.SetValue(szReplyParam, m_stReplyParam);

		Key.SetValue(szReplyOption, (DWORD)m_nReplyOption);
		Key.SetValue(szHTMLReply, m_bHTMLReply);
		Key.SetValue(szEnforceAddress, m_bEnforceAddress);
		Key.SetValue(szClientLaunchAction, (DWORD)m_ClientLaunchAction);

		Key.SetValue(szConfirmDeleteMain, m_bConfirmDeleteMain);
		Key.SetValue(szConfirmDeleteMsg, m_bConfirmDeleteMsg);
		Key.SetValue(szMarkDelete, m_bMarkDelete);
		Key.SetValue(szShowMarkDeleteInfo, m_bShowMarkDeleteInfo);
		Key.SetValue(szShowEmptyTrashWarning, m_bShowEmptyTrashWarning);

		Key.SetValue(szActionLeftClick, m_actionLeftClick);
		Key.SetValue(szActionMiddleClick, m_actionMiddleClick);
		Key.SetValue(szActionRightClick, m_actionRightClick);
		Key.SetValue(szActionDoubleClick, m_actionDoubleClick);

		Key.SetValue(szNumMailsInTray, m_bNumMailsInTray);
		Key.SetValue(szRotateMailNum, m_bRotateMailNum);
		Key.SetValue(szNumMails, (DWORD)m_numMails);
		Key.SetValue(szRotationInterval, m_nRotationInterval);
		Key.SetValue(szHighColorTray, m_bHighColorTray);
		Key.SetValue(szAdvancedTrayInfo, m_bAdvancedTrayInfo);
		Key.SetValue(szShowMainWnd, m_bShowMainWnd);
		Key.SetValue(szDoubleClickDelay, m_TrayIcon.m_bDoubleClickDelay);
		Key.SetValue(szKeyboardLight, (DWORD)m_KeyboardLight);
		
		Key.SetValue(szDetailedNewTip, m_bDetailedNewTip);
		Key.SetValue(szShowNewMailTip, m_bShowNewMailTip);

		Key.SetValue(szCheckForUpdates, CPopManApp::m_bCheckForUpdates);

        Key.SetValue(szLastDestinationList, m_stLastSelectedRulesDestList);
        Key.SetValue(szRulesFileNameSetting, m_stRulesName);

        Key.SetValue(szNewTipDurationValue, m_nNewMailTipDuration);

        Key.SetValue(szMarkBlackListedMailValue, m_bMarkBlackListedMail);
	}
	
	SaveMailCache();

	ResetKeyboardLight();

	FlushSettings(); // make sure all settings get saved on Windows shutdown!
}



BOOL CPopManDoc::LoadAccount(CSettingsKey& AccountsKey, LPCTSTR szAccount, CAccount*& pAcc)
{
	CSettingsKey Key;
	
	if(!AccountsKey.OpenSubKey(szAccount, Key))
		return FALSE;

	CString stProtocol = _T("POP3");
	Key.QueryValue(szAccProtocol, stProtocol);

	if (stProtocol == _T("POP3"))
    {
		pAcc = new CPOP3Account(this, false);
        return pAcc->ReadSettings(Key);
    }
	else if (stProtocol == _T("POP3 TLS"))
	{
		pAcc = new CPOP3Account(this, true);
		return pAcc->ReadSettings(Key);
	}
	else 
	{
		CProtocol* pProtocol = GetProtocolFromName(stProtocol);
        bool bInvalidProtocol = (pProtocol == NULL);
        if(bInvalidProtocol)
            pProtocol = new CProtocol(NULL, stProtocol, 110, true); // this results in memory leaks, accept it because it only happens for rare error case!

		pAcc = new CPluginAccount(this, *pProtocol);
        BOOL res = pAcc->ReadSettings(Key);

        if(bInvalidProtocol) 
        {
            CString stPrompt = StrFormat(i18n("Failed to initialize account {1}: No plugin available to provide protocol {2}!"), _T("s"), (LPCTSTR)pAcc->m_stName, _T("s"), (LPCTSTR)stProtocol);
			AfxMessageBox(stPrompt, MB_OK | MB_ICONERROR);            
        }
        return res;
	}	
}


void CPopManDoc::SaveAccount(CSettingsKey& AccountsKey, LPCTSTR szAccount, const CAccount& Acc)
{
	CSettingsKey Key;
	
	if(AccountsKey.CreateSubKey(szAccount, Key))
	{
		Acc.SaveSettings(Key);

		Key.SetValue(szAccProtocol, Acc.GetProtocolName());
	}
}



void CPopManDoc::OnCancel() 
{
	CancelConnections();
}

void CPopManDoc::OnUpdateCancel(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(ConnectionsPending());
}

void CPopManDoc::CancelConnections()
{
	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
		pAcc->Cancel();
	}
}


BOOL CPopManDoc::ConnectionsPending() const
{
	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
//		TRACE(" Acc: %s  Busy=%d \n", pAcc->m_stName, pAcc->IsBusy());
		if(pAcc->IsBusy())
			return TRUE;
	}
	return FALSE;
}

void GetOneLineMailInfo(const CMail* pMail, CString& res)
{
    res.Empty();
    res.GetBuffer(255);

    res += COleDateTime::GetCurrentTime().Format(_T("%c"));
    res += _T("   ");
    res += pMail->GetAccount()->m_stName;

    int off = 7 - pMail->GetAccount()->m_stName.GetLength();
    while(off > 0) {
        res += _T(' ');
        --off;
    }

    res += _T("   \"");
    res += pMail->GetSubject();
    res += _T("\"   \"");

    CString tmp = pMail->GetEntireFrom();
    tmp.Remove(_T('"'));
    res += tmp;
    res += _T("\"   \"");

    tmp = pMail->GetTo();
    tmp.Remove(_T('"'));
    res += tmp;

    res += _T("\"   ");
    res += pMail->GetDate().Format(_T("%c"));
}

void CPopManDoc::OnNewMail(CMail* pMail, bool bIsNew, bool& bDeleteMail)
{
	if(pMail == NULL)
		return;

    const bool bImmune = m_WhiteList.Contains(pMail->GetFromAddress());

	bDeleteMail = false;

    bool bIsNewOriginal = bIsNew;
    CString stOneLineMailInfo;

    CString buffer;    

	if(m_RuleManager.Apply(pMail, bImmune, /* byRef */ bIsNew, /* byRef */ bDeleteMail, m_bMuteMode) && m_RuleLogFile.IsEnabled())
    {
        GetOneLineMailInfo(pMail, stOneLineMailInfo);

        buffer.GetBuffer(255);
        buffer += _T("\r\n\r\n");
        buffer += stOneLineMailInfo;

        POSITION pos = pMail->m_RuleInfos.GetHeadPosition();
        while(pos)
        {
            const CMail::RuleInfo& rule = pMail->m_RuleInfos.GetNext(pos);
            buffer += _T("\r\n");
            buffer += rule.Name;
            buffer += _T(":\r\n  ");

            CString indentedInfo;
            indentedInfo.GetBuffer(rule.Info.GetLength()+24);
            indentedInfo += rule.Info;
            indentedInfo.Replace(_T("\r\n"), _T("\r\n  "));

            buffer += indentedInfo;
        }

        m_RuleLogFile.Log(buffer);
    }

    if(bIsNewOriginal && IsLogEnabled() && m_bCompactLogging)
    {
        if(stOneLineMailInfo.IsEmpty())
            GetOneLineMailInfo(pMail, stOneLineMailInfo);

        buffer.GetBuffer(255);
        buffer.ReleaseBuffer(0); // "empty" without freeing unused buffer
        buffer += _T("\r\n");
        buffer += stOneLineMailInfo;

        if(pMail->m_RuleInfos.GetCount() > 0)
        {
            buffer += _T("    [");

            POSITION pos = pMail->m_RuleInfos.GetHeadPosition();
            while(pos)
            {
                const CMail::RuleInfo& rule = pMail->m_RuleInfos.GetNext(pos);
                buffer += rule.Name;
                buffer += _T(" -> ");
                buffer += rule.Actions;
                if(pos) buffer += _T("; ");
            }
            buffer += _T("]");
        }

        if(!bDeleteMail && m_BlackList.Contains(pMail->GetFromAddress()) && !bImmune)
        {
            buffer += _T("    Black Listed!");
        }

        if(bImmune)
        {
            buffer += _T("    White Listed!");
        }

        m_LogFile.Log(buffer);
    }

    

	if(bDeleteMail) 
    {
		TRACE(_T("Deleting Mail because of rule! \n"));
		return;
	}

	
	if(m_BlackList.Contains(pMail->GetFromAddress()) && !bImmune) 
    {
		TRACE(_T(" \"%s\" is on BlackList! \n"), pMail->GetFromAddress());
        if(m_bMarkBlackListedMail) {
            pMail->SetMarkedForDelete(TRUE, TRUE);
            pMail->SetRead();
            bIsNew = FALSE;
        }
        else {
		    bDeleteMail = true;
		    return;
        }
	}

	if(bIsNew && (m_bNotifyProtectedOnly == FALSE || (m_bNotifyProtectedOnly == TRUE && pMail->GetAlwaysProtected() == TRUE))) {
//		m_nNewMailCount++;
		m_NewMailList.Add(pMail->GetSignature());
		m_OnNewMailList.Add(pMail->GetSignature());
	}

	UpdateAllViews(NULL, ntNewMail, pMail);

    if(pMail->IsMarkedForDelete()) // we need to send the marked change notification AFTER the mail has been added to the view(s)
        AfxGetMainWnd()->SendMessage(WM_MAIL_DEL_MARKED, 0, 0);
}

void CPopManDoc::OnStatusChanged(const CAccount *pAccount)
{
	CString stError;
	CString stState;

	if(pAccount->GetState() == CAccount::STATES::stError)
	{
		stError = pAccount->m_stName + _T(":  ") + pAccount->GetErrorDescription();
		if(!pAccount->m_stServerErrMessage.IsEmpty())
		{
			stError += _T(" (");
			stError += pAccount->m_stServerErrMessage;
			stError += _T(")");
		}

		if(pAccount->m_nErrorLocation)
		{
			CString stErrLoc;
			stErrLoc.Format(_T("; ErrCode: %d"), pAccount->m_nErrorLocation);
			stError += stErrLoc;
		}

		DisplayError(stError);

		if(pAccount->GetLastError() != CAccount::ERRORS::erCanceled)
		{
			m_bError = TRUE;
			m_stErrorTip = FormatAccountErrTip(pAccount);
		}

/*
		if(IsLogEnabled() &&  m_bErrorLogging)
		{
			CString stFileName;
			stFileName = _T("ErrorLog-");
			COleDateTime tmNow = COleDateTime::GetCurrentTime();
			
			stFileName += tmNow.Format(_T("%X"));
			stFileName += _T(".log");

			int nPos;
			while(-1 != (nPos = stFileName.FindOneOf(_T("\\/:*?\"<>|"))))
				stFileName.SetAt(nPos, _T('-'));

			stFileName = m_stAppDataPath + stFileName;


			CFile ErrorFile;
			if(!ErrorFile.Open(stFileName, CFile::shareDenyNone | CFile::modeWrite | CFile::modeCreate))
			{
				CString stErr = StrFormat(i18n("The error log file \"{1}\" could not be created."), _T("s"), (LPCTSTR)stFileName);
				AfxMessageBox(stErr);
				return;
			}

            unsigned int size = 0;
            char* pBuf = m_LogFile.GetCopyOfContent(size);
			ErrorFile.Write(pBuf, size);
			ErrorFile.Close();
			delete [] pBuf;

		}
    */

	} 
	else if(pAccount->GetState() == CAccount::STATES::stIdle)
	{
		DisplayState(GetIdleStateInfo());
	}
	else
	{
		stState = pAccount->m_stName + _T(":  ") + pAccount->GetStateDescription();
		DisplayState(stState);
	}
}



void CPopManDoc::OnAccounts() 
{
	CDlgAccounts Dlg(this);
	if(Dlg.DoModal() == -1) //  -1 is returned when DoModal() exits because PopMan was terminated
		return;				//   the document is already destroyed in this case

	RepaintAllViews();
	UpdateTrayIcon();
}


BOOL CPopManDoc::ListMails(BOOL bGoOnline, BOOL bHardReload) 
{
	CAccounts Accs;

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);

		if(!pAcc->IsBusy() && (pAcc->m_bActive || pAcc->m_Mails.GetCount() > 0))
			Accs.AddTail(pAcc);
	}

	return ListMailsFromAccs(Accs, bGoOnline, bHardReload);
}

BOOL CPopManDoc::ListMailsFromAccs(CAccounts& Accs, BOOL bGoOnline, BOOL bHardReload)
{
	if(Accs.GetCount() < 1)
		return FALSE;

	if(ConnectionsPending())
		return FALSE;
	
	if(!CheckInetConnection(bGoOnline))
		return FALSE;

  //  CString stLog;
  //  CTime now = CTime::GetCurrentTime();
  //  stLog.Format("\n%u:%u:%u ListMailsFromAccs: %u\n", now.GetHour(), now.GetMinute(), now.GetSecond(), Accs.GetCount());
  //  Log(stLog);

    m_bDisableDeferCheck = false;

	DisplayError(_T(""));
	
	m_RuleManager.CheckReload();
    CheckReloadMailCache();

	OnActionChanged(ACTIONS::Checking);

	if(bHardReload)
	{
        CMails mailList;
	    POSITION pos = Accs.GetHeadPosition();
		while(pos != NULL)
        {
			const CAccount* pAcc = Accs.GetNext(pos);
            
            POSITION posMail = pAcc->m_Mails.GetHeadPosition();
            while(posMail != NULL)
            {
                CMail* pMail = pAcc->m_Mails.GetNext(posMail);
                mailList.AddTail(pMail);
            }           
        }

        UpdateAllViews(NULL, ntMailsDeleted, &mailList);
        mailList.RemoveAll();

		pos = Accs.GetHeadPosition();
		while(pos != NULL)
			(Accs.GetNext(pos))->ClearMails();

        UpdateAllViews(NULL, ntMailsDeleted, &mailList);

        AfxGetMainWnd()->SendMessage(WM_MAIL_DEL_MARKED, 0, 0);
	}

    if(!ListAccs(Accs, bHardReload))
        return FALSE;
/*
    int countSuccessful = 0;
    int countFailed = 0;
    bool bNoFailureAfterFirstSuccess = true;

	POSITION pos = Accs.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = Accs.GetNext(pos);

		m_stCheckingTip = StrFormat(i18n("Checking {1}..."), _T("s"), (LPCTSTR)pAcc->m_stName);
		UpdateTrayIcon();

		if(pAcc->ListMails(bHardReload))
		{
            ++countSuccessful;
		}
        else 
        {
            ++countFailed;

            if(countSuccessful > 0)
                bNoFailureAfterFirstSuccess = false;

			if(pAcc->IsCanceled())
			{
				OnActionChanged(ACTIONS::Idle);
				return FALSE;
			}
        }
	}
*/
	
	m_LastCheckTime = COleDateTime::GetCurrentTime();
    if (!GetErrorStatus()) m_LastSuccessfulCheckTime = COleDateTime::GetCurrentTime();
	
    OnActionChanged(ACTIONS::Idle);
		
	
//	if(m_nNewMailCount > 0)
//		OnNewMailEvent(m_nNewMailCount);

    if(m_OnNewMailList.GetSize() > 0)
		OnNewMailEvent(m_OnNewMailList.GetSize());

	// remove all "new" mails that are no longer found in any account
    // (because they have been deleted by another mail client in the mean time):
	for(int n = m_NewMailList.GetSize() - 1; n >= 0 ; n--)
	{
		BOOL bFound = FALSE;
		const CString& Signature = m_NewMailList[n];
		POSITION pos = m_Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			CAccount* pAcc = m_Accounts.GetNext(pos);
			if(pAcc->m_MailCache.FindSignature(Signature) > -1) 
			{
				bFound = TRUE;
				break;
			}
		}

		if(!bFound)
			m_NewMailList.RemoveAt(n);
	}

	if(m_NewMailList.GetSize() == 0)
		SetNewMailState(FALSE);
	else
	{
		UpdateNewMailTip();
		UpdateTrayIcon();
	}

    SaveMailCache();

    AfxGetMainWnd()->SendMessage(WM_MAIL_DEL_MARKED, 0, 0); // mails that were marked for delete might have been removed from the server
	
	return TRUE;
}

BOOL CPopManDoc::ListAccs(CAccounts& Accs, BOOL bHardReload, BOOL bRetryFailedAccs)
{
    CAccounts AccFailed;
    int countSuccessful = 0;
    bool bNoFailureAfterFirstSuccess = true;

	POSITION pos = Accs.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = Accs.GetNext(pos);

		m_stCheckingTip = StrFormat(i18n("Checking {1}..."), _T("s"), (LPCTSTR)pAcc->m_stName);
		UpdateTrayIcon();

		if(pAcc->ListMails(bHardReload))
		{
            ++countSuccessful;
		}
        else 
        {
            AccFailed.AddTail(pAcc);

            if(countSuccessful > 0)
                bNoFailureAfterFirstSuccess = false;

			if(pAcc->IsCanceled())
			{
				OnActionChanged(ACTIONS::Idle);
				return FALSE;
			}
        }
	}

    if(bRetryFailedAccs && AccFailed.GetCount() > 0 && countSuccessful > 0 && bNoFailureAfterFirstSuccess)
        return ListAccs(AccFailed, bHardReload, FALSE);
    else
        return TRUE;
}

BOOL CPopManDoc::MarkMailsAsSeen(CMails* pMailList)
{
	CAccounts  AccountList;
	CMail*     pMail = NULL;
	CAccount*  pAccount = NULL;
	POSITION   posAcc, posMail;

	DisplayError(_T(""));


	if (!CheckInetConnection(TRUE))
		return FALSE;

	OnActionChanged(ACTIONS::Marking);

	// Liste mit den verwendeten Accounts anlegen:
	posMail = pMailList->GetHeadPosition();
	while (posMail != NULL)
	{
		pMail = pMailList->GetNext(posMail);
		pAccount = pMail->GetAccount();

		if (AccountList.Find(pAccount) == NULL)
			AccountList.AddTail(pAccount);
	}

	// Liste der Accounts durchlaufen; fr jeden Account eine Liste 
	// mit dessen zu lschenden Mails anlegen und DeleteMails aufrufen:
	CMails MailList;

	posAcc = AccountList.GetHeadPosition();
	while (posAcc != NULL)
	{
		pAccount = AccountList.GetNext(posAcc);
		MailList.RemoveAll();
		posMail = pMailList->GetHeadPosition();
		while (posMail != NULL)
		{
			pMail = pMailList->GetNext(posMail);
			if (IsMailValid(pMail))
			{
				if (pAccount == pMail->GetAccount())
					MailList.AddTail(pMail);
			}
		}

		if (!pAccount->MarkMailsAsSeen(&MailList)) {
			OnActionChanged(ACTIONS::Idle);
			return FALSE;
		}
		else
			UpdateAllViews(NULL, ntMailsDeleted, &MailList);
	}

	//AfxGetMainWnd()->SendMessage(WM_MAIL_DEL_MARKED, 0, 0);
	OnActionChanged(ACTIONS::Idle);
	return TRUE;
}

BOOL CPopManDoc::DeleteMails(CMails* pMailList)
{
	CAccounts  AccountList;
	CMail*     pMail    = NULL;
	CAccount*  pAccount = NULL;
	POSITION   posAcc, posMail;
	
	DisplayError(_T(""));
	
	
	if(!CheckInetConnection(TRUE))
		return FALSE;

	OnActionChanged(ACTIONS::Deleting);

	// Liste mit den verwendeten Accounts anlegen:
	posMail = pMailList->GetHeadPosition();
	while(posMail != NULL)
	{
		pMail = pMailList->GetNext(posMail);
		pAccount = pMail->GetAccount();
		
		if(AccountList.Find(pAccount) == NULL)
			AccountList.AddTail(pAccount);
	}

	// Liste der Accounts durchlaufen; fr jeden Account eine Liste 
	// mit dessen zu lschenden Mails anlegen und DeleteMails aufrufen:
	CMails MailList;

	posAcc = AccountList.GetHeadPosition();
	while(posAcc != NULL)
	{
		pAccount = AccountList.GetNext(posAcc);
		MailList.RemoveAll();
		posMail = pMailList->GetHeadPosition();
		while(posMail != NULL)
		{
			pMail = pMailList->GetNext(posMail);
			if(IsMailValid(pMail))
			{
				if(pAccount == pMail->GetAccount())
					MailList.AddTail(pMail);
			}
		}
		if(!pAccount->DeleteMails(&MailList)) {
			OnActionChanged(ACTIONS::Idle);
			return FALSE;
		}
		else
			UpdateAllViews(NULL, ntMailsDeleted, &MailList);	


	}
	
	AfxGetMainWnd()->SendMessage(WM_MAIL_DEL_MARKED, 0, 0);
	OnActionChanged(ACTIONS::Idle);
	return TRUE;
}


BOOL CPopManDoc::DownloadMail(CMail *pMail, BOOL& bNoConnection, BOOL bKeepConnectionAlive)
{
	if(!IsMailValid(pMail))	
		return FALSE;

	DisplayError(_T(""));
	
	
	if(!CheckInetConnection(TRUE))
	{
		bNoConnection = TRUE;
		return FALSE;
	}

	BOOL bChangeAction = (m_currentAction != ACTIONS::Downloading);
	if(bChangeAction)
		OnActionChanged(ACTIONS::Downloading);

	bNoConnection = FALSE;

	m_pDlgDownload = new CDlgDownload(pMail);
	
	CAccount* pAcc = pMail->GetAccount();
	BOOL bRes = pAcc->DownloadMail(pMail, bKeepConnectionAlive);
	
	delete m_pDlgDownload;
	m_pDlgDownload = NULL;
	
	if(bChangeAction)
		OnActionChanged(ACTIONS::Idle);

	return bRes; 
}


void CPopManDoc::OnDataTraffic(BOOL bReceivedData, const void* pData, int nDataLen)
{
	if(m_pDlgDownload && bReceivedData)
		m_pDlgDownload->OnReceivedData(pData, nDataLen);

    if(!m_bCompactLogging)
    {
	    char szPass[] = "PASS ******\r\n";
	    if(bReceivedData)
	    {
		    if(m_bLastLogFromServer == FALSE)
			    Log(_T("\r\nServer: "));
	    }
	    else
	    {
		    Log(_T("\r\nClient: "));
		    if(nDataLen > 5)
		    {
			    const char* pD = (const char*)pData;
			    if(pD[0] == 'P' && pD[1] == 'A' && pD[2] == 'S' && pD[3] == 'S' && pD[4] == ' ')
			    {
				    pData = szPass;
				    nDataLen = strlen(szPass);
			    }
		    }
	    }

        m_LogFile.Log(pData, nDataLen);

	    m_bLastLogFromServer = bReceivedData;
    }
}


void CPopManDoc::Log(LPCTSTR szLog)
{
    if(!m_bCompactLogging)
        m_LogFile.Log(szLog);
}

void CPopManDoc::ViewMailExternally(CMail* pMail)
{
	if (!pMail->IsComplete())
	{
		BOOL bConnectionCanceled = FALSE;
		if (!DownloadMail(pMail, bConnectionCanceled, TRUE))
		{
			if (!bConnectionCanceled && !pMail->GetAccount()->IsCanceled())
				AfxMessageBox(i18n("Downloading email failed.\r\nPress Ctrl+F5 to update the email listing and try again."));
			return;
		}
	}

	CString stFolder = CPopManApp::GetAppTempPath();
	if (stFolder.Right(1) != _T("\\")) stFolder += _T('\\');
	CString stFileName = GetNewFileName(stFolder, _T("Mail"), _T(".eml"));

	try
	{
		CFile File(stFileName, CFile::modeCreate | CFile::modeWrite);

		const CString& stData = pMail->GetSource();
		const int Len = stData.GetLength();
		char * data = new char[Len];
		for (int i = 0; i < Len; ++i) {
			data[i] = (char)(unsigned char)stData.GetAt(i);
		}

		File.Write(data, Len);
		File.Close();
		delete[] data;

		INT_PTR res = (INT_PTR)ShellExecute(NULL, NULL, stFileName, NULL, NULL, SW_SHOWNORMAL);
		if (res <= 32) {
			CString msg = i18n("Failed to open message file. Make sure that files with extension 'eml' will be opened by an email viewer, e.g. Thunderbird Portable.");
			msg += _T("\n\n");
			msg += stFileName;
			AfxMessageBox(msg);
		}
	}
	catch (CFileException* e)
	{
		e->ReportError();
		e->Delete();
	}
}

BOOL CPopManDoc::SaveMail(CMails* pMailList)
{
	if(pMailList == NULL || pMailList->GetCount() == 0)
		return FALSE;
	
	if(pMailList->GetCount() > 1)
	{
		CDlgSave DlgSave;
		DlgSave.m_stFolder = m_stSaveFolder;
		DlgSave.m_SaveAs = m_SaveMailAs == SAVEFILETYPES::frmEmlFile ? CDlgSave::SAVEFORMAT::frmEmlFile : CDlgSave::SAVEFORMAT::frmTxtFile;
		if(m_stFileTemplate.IsEmpty())
		{
			m_stFileTemplate = _T("[");
			m_stFileTemplate += i18n("Subject");
			m_stFileTemplate += _T("]");
		}
		DlgSave.m_stTemplate = m_stFileTemplate;
		if(DlgSave.DoModal() != IDOK)
			return FALSE;
	
		m_stSaveFolder = DlgSave.m_stFolder;
		m_SaveMailAs = DlgSave.m_SaveAs == CDlgSave::SAVEFORMAT::frmEmlFile ? SAVEFILETYPES::frmEmlFile : SAVEFILETYPES::frmTxtFile;
		m_stFileTemplate = DlgSave.m_stTemplate;
	}


	CAccounts  AccountList;
	CMail*     pMail    = NULL;
	CAccount*  pAccount = NULL;
	POSITION   posAcc, posMail;

	// Liste mit den verwendeten Accounts anlegen:
	posMail = pMailList->GetHeadPosition();
	while(posMail != NULL)
	{
		pMail = pMailList->GetNext(posMail);
		pAccount = pMail->GetAccount();
		
		if(AccountList.Find(pAccount) == NULL)
			AccountList.AddTail(pAccount);
	}
	

	// Liste der Accounts durchlaufen; fr jeden Account eine Liste 
	// mit dessen zu speichernden Mails anlegen:
	CMails AccMails;

	posAcc = AccountList.GetHeadPosition();
	while(posAcc != NULL)
	{
		pAccount = AccountList.GetNext(posAcc);
		AccMails.RemoveAll();
		posMail = pMailList->GetHeadPosition();
		while(posMail != NULL)
		{
			pMail = pMailList->GetNext(posMail);
			if(pAccount == pMail->GetAccount())
				AccMails.AddTail(pMail);
		}

		// Liste der zu speichernden Emails des aktuellen Accounts durchlaufen und speichern:
		posMail = AccMails.GetHeadPosition();
		while(posMail != NULL)
		{
			pMail = AccMails.GetNext(posMail);

			if(!IsMailValid(pMail))
				return FALSE;

			BOOL bEML = TRUE;
			CString stFileName(pMail->GetSubject());
			int nPos;
			CString stInvalidChars; stInvalidChars.Format(IDS_INVALID_FILECHARS);
			while(-1 != (nPos = stFileName.FindOneOf(stInvalidChars)))
				stFileName.SetAt(nPos, _T('_'));
			
			if(pMailList->GetCount() == 1)
			{
				CFileDlgEx Dlg;
				Dlg.AddFilter(i18n("Email (*.eml)"), _T("*.eml"));
				Dlg.AddFilter(i18n("Text (*.txt)"),  _T("*.txt"));
				Dlg.m_ofn.lpstrInitialDir = m_stSaveFolder;
				bEML = m_SaveMailAs == SAVEFILETYPES::frmEmlFile;
				if (!Dlg.GetSaveFileName(stFileName, (bEML ? _T("eml") : _T("txt")),  (bEML ? 1 : 2) ))
					return FALSE;

				bEML = stFileName.Right(3) == _T("eml");
				m_SaveMailAs = bEML ? SAVEFILETYPES::frmEmlFile : SAVEFILETYPES::frmTxtFile;
			}
			else
			{
				stFileName = GetNameFromTemplate(m_stFileTemplate, pMail);
				bEML = m_SaveMailAs == SAVEFILETYPES::frmEmlFile;
				stFileName = GetNewFileName(m_stSaveFolder, stFileName, (bEML ? _T(".eml") : _T(".txt")));
			}

	
			if((bEML && !pMail->IsComplete()) || (!bEML && !pMail->IsComplete() && !CMail::IsMessageTextComplete(pMail->GetSource())))
			{
				BOOL bConnectionCanceled = FALSE;
				if(!DownloadMail(pMail, bConnectionCanceled, TRUE))
				{
					if(!bConnectionCanceled && !pMail->GetAccount()->IsCanceled())
						AfxMessageBox(i18n("Downloading email failed.\r\nPress Ctrl+F5 to update the email listing and try again."));

					return FALSE;
				}
			}


			// abspeichern...
			try 
			{
				CFile File(stFileName, CFile::modeCreate | CFile::modeWrite);
				
				const CString& stData = bEML ? pMail->GetSource() : pMail->GetTextMail();
				char * data = nullptr;
				int Len = 0;

				if (bEML) {
					Len = stData.GetLength();
					data = new char[Len];
					for (int i = 0; i < Len; ++i) {
						data[i] = (char)(unsigned char)stData.GetAt(i);
					}
				}
				else {
					Len = WideCharToMultiByte(CP_UTF8, 0, stData.GetString(), stData.GetLength(), NULL, 0, NULL, NULL);
					data = new char[Len];					
					WideCharToMultiByte(CP_UTF8, 0, stData.GetString(), stData.GetLength(), data, Len, NULL, NULL);

					unsigned char BOM[] = { 0xEF, 0xBB, 0xBF }; // UTF8 Byte Order Mark
					File.Write(BOM, 3);
				}

				File.Write(data, Len);

				File.Close();

				delete[] data;

			}
			catch(CFileException* e)
			{
				e->ReportError();
				e->Delete();
				pAccount->Close();
				return FALSE;
			}

		}
	
		pAccount->Close();
	}



	return TRUE;
}



CString CPopManDoc::GetNameFromTemplate(CString stTemplate, const CMail* pMail)
{
	if(pMail == NULL)
		return _T("");
	
	int nPos = 0;
	CString stInvalidChars; stInvalidChars.Format(IDS_INVALID_FILECHARS);
	
	CString stSubject = i18n("Subject");  stSubject = _T("[") + stSubject + _T("]");
	CString stFrom    = i18n("From");	  stFrom    = _T("[") + stFrom + _T("]");
	CString stTo	  = i18n("To");		  stTo      = _T("[") + stTo + _T("]");
	CString stDate	  = i18n("Date");	  stDate	= _T("[") + stDate + _T("]");
	CString stTime    = i18n("Time");	  stTime	= _T("[") + stTime + _T("]");

	stSubject.MakeLower();
	stFrom.MakeLower();
	stTo.MakeLower();
	stDate.MakeLower();
	stTime.MakeLower();


	BOOL bIn = FALSE;
	for(int n = 0; n < stTemplate.GetLength(); n++)
	{
		if(stTemplate[n] == _T('['))
			bIn = TRUE;

		if(stTemplate[n] == _T(']'))
			bIn = FALSE;

		if(bIn)
			stTemplate.SetAt(n, lcase(stTemplate[n]));
	}

	
	CString stDateRep = pMail->GetDate().Format(_T("%x"));
	while(-1 != (nPos = stDateRep.FindOneOf(stInvalidChars)))
		stDateRep.SetAt(nPos, _T('-'));

	CString stTimeRep = pMail->GetDate().Format(_T("%X"));
	while(-1 != (nPos = stTimeRep.FindOneOf(stInvalidChars)))
		stTimeRep.SetAt(nPos, _T('.'));

	stTemplate.Replace(stTime, stTimeRep);
	stTemplate.Replace(stDate, stDateRep);
	stTemplate.Replace(stTo, pMail->GetTo());
	stTemplate.Replace(stFrom, pMail->GetFrom());
	stTemplate.Replace(stSubject, pMail->GetSubject());

	while(-1 != (nPos = stTemplate.FindOneOf(stInvalidChars)))
		stTemplate.SetAt(nPos, _T('_'));

	return stTemplate;
}

void CPopManDoc::DisplayError(LPCTSTR pszError)
{
	CMainFrame* pFrame = (CMainFrame*)AfxGetMainWnd();
	if(pFrame) {
		CColorStatusBar& StatusBar = pFrame->GetStatusBar();
		StatusBar.SetPaneText(1, pszError);
	}
	if(pszError != NULL && pszError[0] != _T('\0'))
	{
		Log(_T("\r\n -->> ErrorMessage: "));
		Log(pszError);
	}
}

void CPopManDoc::DisplayState(LPCTSTR pszState)
{
	m_stCurrStatusText = pszState;

	CMainFrame* pFrame = (CMainFrame*)AfxGetMainWnd();
	if(pFrame) {
		CColorStatusBar& StatusBar = pFrame->GetStatusBar();
		StatusBar.SetPaneText(0, pszState);
	}
}



BOOL CPopManDoc::CheckInetConnection(BOOL bGoOnline)
{
	if(m_bUseDUN == FALSE)
		return TRUE;

	if(!IsOnline())
	{
		if(bGoOnline)
		{
			if(!ShowDialDialog())
				return FALSE;
		}
		else
			return FALSE;
	}
	return TRUE;
}



BOOL CPopManDoc::IsOnline()
{
	return ::InternetGetConnectedState(NULL, 0);
}

BOOL CPopManDoc::ShowDialDialog()
{
	static BOOL bBusy = FALSE;	
	if(bBusy)
		return FALSE;
	bBusy = TRUE;

	InternetAutodial(INTERNET_AUTODIAL_FORCE_ONLINE, 0);

	bBusy = FALSE;

	return IsOnline();
}


BOOL CPopManDoc::IsMailValid(const CMail* pMail) const
{
	if(pMail == NULL)
		return FALSE;

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
		if(pAcc->m_Mails.GetIdxByMail(pMail) > -1)
			return TRUE;
	}
	return FALSE;
}


void CPopManDoc::OnUpdateListMails(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!ConnectionsPending());	
}

void CPopManDoc::CheckReloadMailCache()
{
    CString stPath;
    GetMailCacheFileName(stPath);

    CFileStatus status;
    if(CFile::GetStatus(stPath, status)) 
    {
       if(m_LastModifiedMailCache != status.m_mtime)
       {
            LoadMailCache();

            // The "Read" state of the mails may have been changed:
            POSITION pos = m_Accounts.GetHeadPosition();
	        while(pos != NULL)
	        {
		        CAccount* pAcc = m_Accounts.GetNext(pos);
                POSITION posMail = pAcc->m_Mails.GetHeadPosition();
                while(posMail) {
                    CMail* pMail = pAcc->m_Mails.GetNext(posMail);
                    int idx = pAcc->m_MailCache.FindSignature(pMail->GetSignature());
                    if(idx >= 0)
                    {
                        BOOL bRead = pAcc->m_MailCache[idx].m_bRead;
                        if(bRead != pMail->IsRead()) {
                            pMail->SetRead(bRead);
                            AfxGetMainWnd()->SendMessageToDescendants(WM_MAIL_READ, (WPARAM)pMail, 0);
                        }
                    }
                }
            }

            AfxGetMainWnd()->RedrawWindow();
       }
    }
}

void CPopManDoc::GetMailCacheFileName(CString& fileName)
{
    fileName = m_stAppDataPath;
    fileName += szMailCacheFile;
}

void CPopManDoc::LoadMailCache()
{ 
    TRACE(_T("LoadMailCache \n"));

	CString stData;
	try 
	{
		CString stPath;
        GetMailCacheFileName(stPath);

		CFile File(stPath, CFile::modeRead);
        unsigned int nFileLen = static_cast<unsigned int>(File.GetLength());
		char* pData = new char[nFileLen + 1];
		File.Read(pData, nFileLen);
		pData[nFileLen] = (char)0;
		File.Close();

		LPWSTR pWstr = stData.GetBufferSetLength(nFileLen);
		MultiByteToWideChar(CP_ACP, 0, pData, nFileLen, pWstr, nFileLen);
		stData.ReleaseBufferSetLength(nFileLen);

		delete [] pData;

        CFileStatus status;
	    if(CFile::GetStatus(stPath, status))
            m_LastModifiedMailCache = status.m_mtime;

	}
	catch(CFileException* e)
	{
		if(e->m_cause != CFileException::fileNotFound)
			e->ReportError();

		e->Delete();
	}


    POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
        pAcc->m_MailCache.Clear();
    }


	CacheItem Item;
	CString stItem;
	CString stUser, stServer;
	int nIdx = 0;

	while(nIdx > -1)
	{
		stItem = ParseConst(stData, _T("\r\n"), nIdx);
		if(stItem.GetLength() == 0) 
			continue;
		
		int nTabPos = stItem.Find(_T('\t'));
		if(nTabPos <= 1)
			continue;

		if(stItem[0] != _T('#'))
		{
			Item.m_bRead = TRUE;
			Item.m_stSignature = stItem.Left(nTabPos);
		}
		else
		{
			Item.m_bRead = FALSE;
			Item.m_stSignature = stItem.Mid(1, nTabPos - 1);
		}

		stServer = stItem.Mid(nTabPos+1);
		stUser = Parse(stServer, _T("\t")); 	

		POSITION pos = m_Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			CAccount* pAcc = m_Accounts.GetNext(pos);
			if(pAcc->m_stUser == stUser)
			{
				BOOL bUnique = TRUE;
				POSITION posNext = pos;
				while(posNext != NULL) {
					CAccount* pAccNext = m_Accounts.GetNext(posNext);
					if(pAccNext->m_stUser == stUser) {
						bUnique = FALSE;
						break;
					}
				}
				if(bUnique || pAcc->m_stServer == stServer) {
                    if(pAcc->m_MailCache.FindSignature(Item.m_stSignature) == -1)
					    pAcc->m_MailCache.Add(Item);
					//break;
				}
			}
		}
	}

    ResetMailCacheChanged();
}

bool CPopManDoc::MailCacheChanged()
{
	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		const CAccount* pAcc = m_Accounts.GetNext(pos);
        if(pAcc->m_MailCache.IsChanged())
            return true;
    }
    return false;
}

void CPopManDoc::ResetMailCacheChanged()
{
	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
		pAcc->m_MailCache.ResetChanged();
    }
}

void CPopManDoc::SaveMailCache()
{
    if(!MailCacheChanged())
        return;

    TRACE(_T("SaveMailCache \n"));

	CString stData;
    stData.GetBuffer(16*1024);

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);

		const CacheItems& MailCache = pAcc->m_MailCache;

        if(stData.GetAllocLength() < stData.GetLength() + (MailCache.GetSize()*200)) 
        {  
            TRACE(_T(" # # # ReAlloc! # # # \n"));
            stData.GetBuffer( 2*stData.GetAllocLength() );
        }

		for(int n = 0; n < MailCache.GetSize(); n++)
		{
			if(MailCache[n].m_bRead == FALSE)
				stData += _T('#');
			stData += MailCache[n].m_stSignature;
			stData += _T('\t');
			stData += pAcc->m_stUser;
			stData += _T('\t');
			stData += pAcc->m_stServer;
			stData += _T("\r\n");
		}
	}	

	const int len = stData.GetLength();
	char* pData = new char[len];
	WideCharToMultiByte(CP_ACP, 0, stData.GetString(), stData.GetLength(), pData, len, NULL, NULL);

	try 
	{
        CString stPath;
        GetMailCacheFileName(stPath);		
		
		CFile File(stPath, CFile::modeCreate | CFile::modeWrite | CFile::modeNoTruncate);
		File.SetLength(0);
		File.Write(pData, len);
		File.Close();

        ResetMailCacheChanged();

        CFileStatus status;
	    if(CFile::GetStatus(stPath, status))
            m_LastModifiedMailCache = status.m_mtime;

	}
	catch(CFileException* e)
	{
		e->ReportError();
		e->Delete();
	}

	delete[] pData;
}


void CPopManDoc::RepaintAllViews()
{
	POSITION pos = GetFirstViewPosition();
	while(pos != NULL)
	{
		CView* pView = GetNextView(pos);
		pView->Invalidate();
		pView->UpdateWindow();
	}   
}

/*
BOOL CPopManDoc::OpenLogFile()
{
	CString stFileName;
	stFileName = m_stAppDataPath + szLogFile;

	if(!m_LogFile.Open(stFileName, CFile::shareDenyNone | CFile::modeReadWrite | CFile::modeCreate | CFile::modeNoTruncate))
	{
		m_bLoggingEnabled = FALSE;

		CString stErr = StrFormat(i18n("The log file \"{1}\" could not be opened.\r\nLogging disabled!"), _T("s"), (LPCTSTR)stFileName);
		AfxMessageBox(stErr);
		return FALSE;
	}

	return TRUE;
}
*/


void CPopManDoc::OnListMails() 
{
	CWaitCursor Wait;

    BOOL hardReload = (GetKeyState(VK_CONTROL) & 8000) != 0;

	if(!ListMails(TRUE, hardReload))
		return;

	if(ConnectionsPending() || !IsWindowVisible(AfxGetMainWnd()->GetSafeHwnd()) )
		return;

	COleDateTime Now = COleDateTime::GetCurrentTime();
	DWORD currDay = (DWORD)Now.GetDay();
	if(currDay == m_dwLastUpdateCheck)
		return;
	m_dwLastUpdateCheck = currDay;


	BOOL  bNewVersion = FALSE;
	CVersion NewVersion;
	CString stInfoLink;

	if(!CheckOnlineUpdate(FALSE, bNewVersion, NewVersion, stInfoLink))
		return;

	if(bNewVersion == FALSE)
		return;

	if(!cstAppVersion.IsBeta() && NewVersion.IsBeta())
		return;

	CVersion LatestVersion;
	if(NewVersion.IsBeta())
		LatestVersion = m_LatestBetaVersion;
	else
		LatestVersion = m_LatestReleaseVersion;

	if(NewVersion > LatestVersion)
	{
		if(NewVersion.IsBeta())
			m_LatestBetaVersion = NewVersion;
		else
			m_LatestReleaseVersion = NewVersion;

		GetNewVersion(NewVersion, stInfoLink);
		/*
		if(GetNewVersion(NewVersion, stInfoLink))
		{
			CDlgAutoUpdate DlgUpdate(0, 0);
			DlgUpdate.DoModal();
		}*/
	}
}

struct REPLYDATA {
    CMail* pMail;
    CPopManDoc* pDoc;
};

UINT ReplyThread(LPVOID lParam)
{
    REPLYDATA* pRepData = (REPLYDATA*)lParam;

    CMail *pMail = pRepData->pMail;
    CPopManDoc* pDoc = pRepData->pDoc;

    delete pRepData;

	CString stSubject = pMail->GetSubject();
	if(FindNoCase(stSubject, _T("Re:")) != 0)
		stSubject = _T("Re: ") + stSubject;

	CString stTo;
	if(pMail->GetFrom() != pMail->GetReplyTo() && !pMail->GetFrom().IsEmpty())
	{
		stTo += _T("\"");
		stTo += pMail->GetFrom();
		stTo += _T("\"");
		stTo += _T(" <");
		stTo += pMail->GetReplyTo();
		stTo +=  _T('>');
	} 
	else
		stTo += pMail->GetReplyTo();

	switch(pDoc->m_nReplyOption)
	{
    case CPopManDoc::REPLYOPTIONS::Eml:
		{
			CString stReply = _T("To: ");
			stReply += stTo;
			stReply += _T("\r\nSubject: ");
			stReply += stSubject;
			stReply += _T("\r\nX-Unsent: 1");
			stReply += _T("\r\n\r\n") + pMail->GetQuottedMsg();
			
			CString stFile = CPopManApp::GetAppTempPath();
			if(stFile.Right(1) != _T("\\")) stFile += _T('\\');
			stFile += szReplyTmpFile;

			try 
			{
				CFile File(stFile, CFile::modeCreate | CFile::modeWrite);
				USES_CONVERSION;
				File.Write(T2A((LPTSTR)(LPCTSTR)stReply), stReply.GetLength());
				File.Close();
			}
			catch(CFileException* e)
			{
				e->ReportError();
				e->Delete();
				return 1;
			}

			::ShellExecute(NULL, _T("open"), stFile, NULL, NULL, SW_SHOWNORMAL);
			
			break;
		}

        case CPopManDoc::REPLYOPTIONS::MAPI:
		{
				const HINSTANCE hMAPILib = ::LoadLibrary(_T("MAPI32.DLL"));

				if(hMAPILib == NULL)
				{
					AfxMessageBox(i18n("Unable to load mail system support."));
					return 1;
				}

				
				LPMAPISENDMAIL lpMAPISendMail = (LPMAPISENDMAIL)GetProcAddress(hMAPILib, "MAPISendMail");

				if(lpMAPISendMail != NULL)
				{
					USES_CONVERSION;

					static MapiMessage Msg;
					memset(&Msg, 0, sizeof(Msg));

					MapiRecipDesc Recip;
					memset(&Recip, 0, sizeof(Recip));

					Recip.ulRecipClass = MAPI_TO;
					Recip.lpszName     = T2A((LPTSTR)(LPCTSTR)(pDoc->m_bEnforceAddress ? pMail->GetReplyTo() : pMail->GetFrom()));   

                    CString sTmpReplyTo = "SMTP:" + pMail->GetReplyTo(); // prepend "SMTP:" to make Outlook happy!
					Recip.lpszAddress  = T2A((LPTSTR)(LPCTSTR)sTmpReplyTo);


					Msg.lpszSubject  = T2A((LPTSTR)(LPCTSTR)stSubject);
					CString stMsg = pMail->GetQuottedMsg();
					if(pDoc->m_bHTMLReply)
					{
						stMsg.Replace(_T("&"),    _T("&amp;"));
						stMsg.Replace(_T("<"),    _T("&lt;"));
						stMsg.Replace(_T(">"),    _T("&gt;"));
						stMsg.Replace(_T("\r\n"), _T("<br>"));
					}
					Msg.lpszNoteText = T2A((LPTSTR)(LPCTSTR)stMsg);
					Msg.nRecipCount = 1;
					Msg.lpRecips = &Recip;
					
					lpMAPISendMail(NULL, NULL, &Msg, (FLAGS) (MAPI_LOGON_UI | MAPI_DIALOG), 0);

				}
				else
				{
					AfxMessageBox(i18n("Mail system DLL is invalid."));
				}

				::FreeLibrary(hMAPILib);
				
				break;
		}

        case CPopManDoc::REPLYOPTIONS::MailTo:
		{
			CString stMailTo = _T("mailto:");

			if(stTo.Find(_T('?')) > -1 || stTo.Find(_T('&')) > -1)
				stMailTo += pMail->GetReplyTo();
			else
				stMailTo += stTo;

			stMailTo += _T("?subject=");
			CString stShellSubject = stSubject;
			stShellSubject.Replace(_T("?"), _T("%3F"));
			stShellSubject.Replace(_T("&"), _T("%26"));
			stMailTo += stShellSubject;

			::ShellExecute(NULL, _T("open"), stMailTo, NULL, NULL, SW_SHOWNORMAL);
		
			break;
		}

        case CPopManDoc::REPLYOPTIONS::CmdParam:
		{
			CString stExe, stCmdParam;
			if(!SplitExeCmdParam(pDoc->m_stEmailClient, stExe, stCmdParam))
				return 1;

			stCmdParam = pDoc->m_stReplyParam;
			stCmdParam.Replace(_T("[TO]"), pMail->GetReplyTo());
			stCmdParam.Replace(_T("[SUBJECT]"), stSubject);

			ShellExecute(NULL, _T("open"), stExe, stCmdParam, NULL, SW_SHOWNORMAL);	
		}
	}
    return 0;
}

void CPopManDoc::ReplyToMail(CMail *pMail)
{
	if(!pMail || pMail->GetReplyTo().IsEmpty())
		return;

    REPLYDATA* pData = new REPLYDATA;
    pData->pDoc = this;
    pData->pMail = pMail;

    AfxBeginThread(ReplyThread, pData, 0, 0, 0, 0);
}



BOOL CPopManDoc::CheckOnlineUpdate(BOOL bGoOnline, BOOL& bNewVersion, CVersion& Version, CString& stInfoLink)
{
	bNewVersion = FALSE;

	if (!CPopManApp::m_bCheckForUpdates)
		return FALSE;

	if (ConnectionsPending())
		return FALSE;

	if (!CheckInetConnection(bGoOnline))
		return FALSE;

	CSyncSocket Socket;
	Socket.SetTimeOut(10);

	if(!Socket.Create() || !Socket.Connect(_T("www.ch-software.de"), 80))
		return FALSE;

	CString stData;

	stData = _T("GET /popman/updateinfo.dat HTTP/1.0\r\n");
	stData += _T("Host: www.ch-software.de\r\n");

	TIME_ZONE_INFORMATION TZ_Info;
	BOOL bIsDayLight = (TIME_ZONE_ID_DAYLIGHT == ::GetTimeZoneInformation(&TZ_Info));
	int iLocalOffset = -TZ_Info.Bias/60 + (bIsDayLight ? 1 : 0);

	TCHAR szCountry[90] = _T("");
	GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SENGCOUNTRY, szCountry, sizeof(szCountry)/sizeof(TCHAR));

	CString stVersion;

	if(cstAppVersion.IsBeta()) 
		stVersion.Format(_T("%d.%d Beta %d"), cstAppVersion.GetMajor(), cstAppVersion.GetMinor(), cstAppVersion.GetSub());
	else
		stVersion = cstAppVersion.GetString();

	CString stFormat;
	stFormat.Format(_T("User-Agent: PopMan %s (%s %+.2d) \r\n"), (LPCTSTR)stVersion, szCountry, iLocalOffset);
	stData += stFormat;
	stData += _T("\r\n");

	if(!Socket.SendStr(stData))
	{
		TRACE(_T("SendStr failed; TimedOut= %s\n"), Socket.IsTimedOut() ? _T("Ja") : _T("Nein"));
		return FALSE;
	}

	stData.Empty();
	if(!Socket.ReceiveStr(stData, 4096))
	{
		TRACE(_T("ReceiveStr failed; TimedOut= %s\n"), Socket.IsTimedOut() ? _T("Ja") : _T("Nein"));
		return FALSE;
	}

	Socket.Close();

	CString stFirstLine = Parse(stData, _T("\r\n"));
	if(stFirstLine.Find(_T(" 200 ")) == -1 && stFirstLine.Find(_T(" OK")) == -1)
		return FALSE;

	Parse(stData, _T("\r\n\r\n"));



	CVersion vRelease, vBeta;
	CString  stReleaseInfo, stBetaInfo;
	if(!CVersion::ParseVersionInfo(stData, vRelease, stReleaseInfo, vBeta, stBetaInfo))
		return FALSE;


	if(vRelease > cstAppVersion) {
		bNewVersion = TRUE;
		Version = vRelease;
		stInfoLink = stReleaseInfo;
		return TRUE;
	}

	if(vBeta > cstAppVersion) {
		bNewVersion = TRUE;
		Version = vBeta;
		stInfoLink = stBetaInfo;
		return TRUE;
	}

	return TRUE;
}

void CPopManDoc::OnCheckUpdate() 
{

	CDlgAutoUpdate DlgUpdate(0, 0);
	DlgUpdate.DoModal();


/*

	DWORD dwMajor = 0;
	DWORD dwMinor = 0;
	DWORD dwSub   = 0;
	BOOL  bNewVersion = FALSE;

	if(!CheckOnlineUpdate(TRUE, bNewVersion, dwMajor, dwMinor, dwSub))
	{
		AfxMessageBox(i18n("Unable to detect if a new version is available."));
		return;
	}

	if(bNewVersion == FALSE)
	{
		AfxMessageBox(i18n("There is no update available."));
	}
	else
	{
		if(GetNewVersion(dwMajor, dwMinor, dwSub))
			ShellExecute(NULL, _T("open"), szPopManWebsite, NULL, NULL, SW_SHOWNORMAL);
	}
*/
}


BOOL CPopManDoc::GetNewVersion(const CVersion& Version, CString stInfoLink)
{
	CString stMsg;
	CString stVersion;

	stVersion.Format(_T("%d.%d"), Version.GetMajor(), Version.GetMinor());

	if(!Version.IsBeta())
	{
		if(Version.GetSub() > 0)
			stVersion = Version.GetString();
	
		stMsg = StrFormat(i18n("There is a new version {1} available!"), _T("s"), (LPCTSTR)stVersion);
	}
	else 		
		stMsg = StrFormat(i18n("There is a new {1} Beta version available!"), _T("s"), (LPCTSTR)stVersion);
	

	//stMsg += _T("\r\n"); 
	//stMsg += i18n("Do you want to start the update wizard and download the latest version?");

	return (IDYES == AfxMessageBox(stMsg, MB_OK | MB_ICONINFORMATION));
}

void CPopManDoc::OnUpdateCheckUpdate(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!ConnectionsPending());	
}



BOOL CPopManDoc::AutoStartEnabled()
{
	CRegistryKey Reg;
	const CString stPath = _T("Software\\Microsoft\\Windows\\CurrentVersion\\Run");
	
	if(!Reg.Open(HKEY_CURRENT_USER, stPath))
		return FALSE;

	CString stName;
	for(int n = 0; ; n++)
	{
		if(!Reg.EnumValue(n, stName))
			break;

		if(stName == _T("PopMan"))
			return TRUE;
	}
	return FALSE;
}


BOOL CPopManDoc::SetAutoStart(BOOL bAutoStart)
{
	CRegistryKey Reg;
	const CString stPath = _T("Software\\Microsoft\\Windows\\CurrentVersion\\Run");
	
	if(!Reg.Open(HKEY_CURRENT_USER, stPath))
		return FALSE;

	if(bAutoStart)
	{
		CString stPath = CPopManApp::GetAppPath();
		if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
		stPath += _T("PopMan.exe -minimize");

		return Reg.SetValue(_T("PopMan"), stPath);
	}
	else
	{
		CString stName;
		for(int n = 0; ; n++)
		{
			if(!Reg.EnumValue(n, stName))
				return FALSE;

			if(stName == _T("PopMan"))
			{
				Reg.DeleteValue(stName);
				return TRUE;
			}
		}
	}
}


BOOL CPopManDoc::GetAccountPass(const CAccount* pAcc, CString& stPass, BOOL& bRemember)
{
	if(pAcc == NULL)
		return FALSE;
	
	CDlgPass DlgPass;
	DlgPass.m_stAccount = pAcc->m_stName;
	DlgPass.m_stUser = pAcc->m_stUser;
	DlgPass.m_bRemember = bRemember;
	BOOL bRes = (DlgPass.DoModal() == IDOK);

	if(bRes)
	{
		stPass = DlgPass.m_stPass;
		bRemember = DlgPass.m_bRemember;
	}
	
	return bRes;
}

void CPopManDoc::OnImportAccounts() 
{
	CDlgAccImport Import(this, FALSE);
	Import.DoModal();	
}


CString CPopManDoc::GetNewFileName(CString stFolder, CString stFile, CString stExt)
{
	CString stFileName;
	CFileStatus Status;
	int nCount = 0;

	if(stFolder.Right(1) != _T("\\"))
		stFolder += _T("\\");

	if(stExt.Left(1) != _T("."))
		stExt = _T(".") + stExt;

	stFileName = stFolder + stFile + stExt;

	while (CFile::GetStatus(stFileName, Status) == TRUE)
	{
		CString stFormat;
		stFormat.Format(_T("_%d"), ++nCount);
		stFileName = stFolder + stFile + stFormat + stExt;
	}
	
	return stFileName;
}


void CPopManDoc::RemoveMails(const CMails *pMails)
{
	if(pMails == NULL) return;
	UpdateAllViews(NULL, ntMailsDeleted, (CObject*)pMails);
}

void CPopManDoc::OnReloadMails() 
{
	ListMails(TRUE, TRUE);	
}

BOOL CPopManDoc::ClearDirectory(LPCTSTR szDir)
{
	WIN32_FIND_DATA FileData;
	HANDLE hHandle = NULL;
	CStringArray Files;

	CString stSearch = szDir;
	if(stSearch.Right(1) != _T("\\")) stSearch += _T('\\');
	stSearch += _T("*");

	
	hHandle = FindFirstFile(stSearch, &FileData);
	if(hHandle == INVALID_HANDLE_VALUE)
		return FALSE;

	if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0)
		Files.Add(FileData.cFileName);

	while(FindNextFile(hHandle, &FileData) != 0)
		if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0)
			Files.Add(FileData.cFileName);

	FindClose(hHandle);
	
	stSearch.TrimRight(_T('*'));
	for(int n = 0; n < Files.GetSize(); n++)
	{
		CString stFile = stSearch + Files[n];
		if(DeleteFile(stFile) == FALSE)
			return FALSE;
	}

	return TRUE;
}



void CPopManDoc::OnMainWndRestore()
{
	ResetKeyboardLight();
}

void CPopManDoc::OnMainWndHide()
{
	SetNewMailState(FALSE);
	ResetKeyboardLight();
    SaveMailCache();
}

void CPopManDoc::OnMainWndMinimize()
{
	SetNewMailState(FALSE);
	ResetKeyboardLight();
    SaveMailCache();
}


void CPopManDoc::OnActionChanged(ACTIONS newAction)
{
	m_currentAction = newAction;

    CWnd* pMainWnd = AfxGetMainWnd();

	switch(newAction) {
	case ACTIONS::Checking:
	case ACTIONS::Deleting:
	case ACTIONS::Marking:
	case ACTIONS::Downloading:
		m_bError = FALSE;
//		m_nNewMailCount = 0;
		m_OnNewMailList.RemoveAll();


		if( !pMainWnd->IsIconic() && pMainWnd->IsWindowVisible() )
			SetNewMailState(FALSE);

		break;

	case ACTIONS::Idle:
		DisplayState(GetIdleStateInfo());
		break;
	}

	UpdateTrayIcon();



#ifndef WM_IDLEUPDATECMDUI
    #define WM_IDLEUPDATECMDUI 0x0363
#endif

    if(pMainWnd->IsWindowVisible()) // we need this to make the toolbar buttons refresh properly !?
        pMainWnd->SendMessageToDescendants(WM_IDLEUPDATECMDUI, (WPARAM)TRUE, 0, TRUE, TRUE);
}

void CPopManDoc::UpdateTrayIcon()
{
	switch(m_currentAction)
	{
	case ACTIONS::Checking:
		m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_ACTION), m_stCheckingTip, FALSE);
		break;
	
	case ACTIONS::Deleting:
		m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_ACTION), i18n("Deleting mail..."), FALSE);
		break;

	case ACTIONS::Marking:
		m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_ACTION), i18n("Marking mail as seen..."), FALSE);
		break;

	case ACTIONS::Downloading:
		m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_ACTION), i18n("Downloading mail..."), FALSE);
		break;

	case ACTIONS::Idle:
		
		if(IsNewMailAvailable() && !m_bNumMailsInTray)
		{
			m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_NEWMAIL), m_stNewMailTip, FALSE);
			break;
		}
		
		if(!m_bNotification)
		{
			m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_DISABLED), i18n("AutoCheck disabled"), FALSE);
			break;
		}

		if(m_bNumMailsInTray)
		{
			CString stTip;
			int nMailCount = GetMailCount(m_numMails, stTip);
			
			if(nMailCount > 0 && !m_bRotateMailNum)
			{
				HICON hIcon = CreateNumIcon(nMailCount, RGB(0, 0, 200));
				if(m_bError == FALSE)
				{
					m_TrayIcon.SetIcon(hIcon, stTip, TRUE);
					break;
				}
				else
				{
					CIcons Icons;
					TIPICON TipIcon;
					TipIcon.hIcon = hIcon;
					TipIcon.stTip = stTip;
					TipIcon.bDestroy = TRUE;
					Icons.Add(TipIcon);
					TipIcon.hIcon = CPopManApp::GetIcon(IDI_TRAY_ERROR); //(HICON)::LoadImage(AfxGetApp()->m_hInstance, MAKEINTRESOURCE(IDR_TRAY_ERROR), IMAGE_ICON, 0, 0, LR_SHARED);
					TipIcon.stTip = m_stErrorTip;
					TipIcon.bDestroy = FALSE;
					Icons.Add(TipIcon);
					m_TrayIcon.SetIcons(Icons, m_nRotationInterval * 100);
					break;
				}
			}

			CString stAllMailTip;
			int nAllMailCount = GetMailCount(NUMMAILSTRAY::All, stAllMailTip);
			
			if(m_bRotateMailNum)
			{
				if(nMailCount > 0 || m_bError || (nAllMailCount > 0 && m_bAdvancedTrayInfo))
				{
					if(nMailCount == 0)
						stTip = stAllMailTip;

					CIcons Icons;
					TIPICON TipIcon;

					POSITION pos = m_Accounts.GetHeadPosition();
					while(pos) 
					{	
						CAccount* pAcc = m_Accounts.GetNext(pos);
					
						if(pAcc->GetLastError() != CAccount::ERRORS::erNoError && pAcc->GetLastError() != CAccount::ERRORS::erCanceled)
						{
							TipIcon.hIcon = CreateErrIcon(pAcc->m_TrayColor);
							TipIcon.stTip = FormatAccountErrTip(pAcc);
							TipIcon.bDestroy = TRUE;
							Icons.Add(TipIcon);
							continue;
						}
						int nMailAcc = GetAccountMailNum(pAcc, m_numMails);
						if(nMailAcc > 0)
						{
							TipIcon.hIcon = CreateNumIcon(nMailAcc, pAcc->m_TrayColor);
							TipIcon.stTip = stTip;
							TipIcon.bDestroy = TRUE;
							Icons.Add(TipIcon);
							continue;
						}
						
						if(!pAcc->m_bActive)
							continue;

						if(m_bAdvancedTrayInfo)
						{
							TipIcon.stTip = stTip;
							TipIcon.bDestroy = TRUE;
							
							if(pAcc->m_Mails.GetCount() == 0)
								TipIcon.hIcon = CreateInnerRectIcon(RGB(255, 255, 255), pAcc->m_TrayColor);
							else
								TipIcon.hIcon = CreateInnerRectIcon(RGB(255, 255, 0), pAcc->m_TrayColor);

							Icons.Add(TipIcon);
							continue;
						}
					}

					m_TrayIcon.SetIcons(Icons, m_nRotationInterval * 100);
					break;
				}
			}
		}

		if(m_bError)
		{
			m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_ERROR), m_stErrorTip, FALSE);
			break;
		}

		CString stTip;
		int nMailCount = GetMailCount(NUMMAILSTRAY::All, stTip);

		if(nMailCount == 0)
			m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY_EMPTY), i18n("No messages"), FALSE);
		else
			m_TrayIcon.SetIcon(CPopManApp::GetIcon(IDI_TRAY), stTip, FALSE);
		
		break;
	}

}



void CPopManDoc::OnReadStateChanged()
{
	if(m_bNumMailsInTray && m_numMails == NUMMAILSTRAY::Unread)
		UpdateTrayIcon();
}


void CPopManDoc::OnNewMailEvent(int nNewMails)
{
	BOOL bIsMinimized = (AfxGetMainWnd()->IsIconic() || !AfxGetMainWnd()->IsWindowVisible());

	switch(m_nNotify)
	{
	case NOTIFYSTATES::AutoChecked:
		if(!bIsMinimized && !m_bAutoChecked)
			return;
		break;

	case NOTIFYSTATES::Minimized:
		if(!bIsMinimized)
			return;
		break;
	}

	SetNewMailState(TRUE);

	if(m_bSoundNotify && !m_bMuteMode && !m_RuleManager.SoundPlayed())
	{
		if(m_stSoundFile.GetLength() > 0)
            ::PlaySound(m_stSoundFile, 0, SND_FILENAME | SND_ASYNC);
		else
            ::MessageBeep(MB_ICONEXCLAMATION);
	}
 	
	if(m_bShowNewMailTip)
	{
		CNewTip::CLines Lines;
		CString stNewMailTip;

		POSITION pos = m_Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			const CAccount* pAcc = m_Accounts.GetNext(pos);	
			int nMailCount = 0;
			for(int n = 0; n < m_OnNewMailList.GetSize(); n++)
			{
				const CMail* pMail = pAcc->GetMailFromSignature(m_OnNewMailList[n]);
				if(pMail)
				{
					nMailCount++;
					if(m_bDetailedNewTip && Lines.GetSize() < 32)
					{
						const int SenderLen  = m_bBallonTip ? 16 : 22;
						const int SubjectLen = m_bBallonTip ? 28 : 46;

						CString stInfo = pMail->GetFrom();
						if(stInfo.GetLength() > SenderLen)
							stInfo = stInfo.Left(SenderLen) + _T("...");
						
						stNewMailTip += stInfo;
						stNewMailTip += _T('\t');
						
						stInfo = pMail->GetSubject();
						if(stInfo.GetLength() > SubjectLen)
							stInfo = stInfo.Left(SubjectLen) + _T("...");
						
						stNewMailTip += stInfo;
						stNewMailTip += _T("\r\n");

						CNewTip::LINE Line;
						Line.nColor = pAcc->m_CustomColor;
						Line.dwValue = (DWORD)pMail;
						Lines.Add(Line);
					}
				}
			}
			if(nMailCount > 0 && !m_bDetailedNewTip)
			{
				stNewMailTip += pAcc->m_stName;
				stNewMailTip += _T(":\t");
				CString stAcc;
				if(nMailCount == 1)
					stAcc = i18n("1 new message");
				else
					stAcc = StrFormat(i18n("{1} new messages"), _T("d"), nMailCount);
				stNewMailTip += stAcc;
				stNewMailTip += _T("\r\n");

				CNewTip::LINE Line;
				Line.nColor  = ::GetSysColor(COLOR_INFOTEXT);
				Line.dwValue = 0;
				Lines.Add(Line);
			}
		}

		stNewMailTip = TabJustify(stNewMailTip, m_MailInfo.m_Font);
		LPCTSTR szHeader = i18n("PopMan - New Mail");

		if(m_TrayIcon.BallonsEnabled() && m_bBallonTip)
			m_TrayIcon.ShowBalloonTip(stNewMailTip, szHeader, m_bDetailedNewTip ? 20000 : 8000);
		else
		{
			for(int n = 0; n < Lines.GetSize(); n++)
				Lines[n].stStr = Parse(stNewMailTip, _T("\r\n"));

			ShowNewMailTip(Lines, szHeader);
		}
	}

	if(m_bShowMainWnd)
		AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_TRAY_SHOW, 0);

	if(m_KeyboardLight != KEYBOARDLIGHT::None && m_nTimerIDKeyboard == 0) {
		m_bKeyLightToggled = false;
		m_nTimerIDKeyboard = ::SetTimer(NULL, 0, 400, Timer_ToggleKeyboardLight);
	}
}

void CPopManDoc::OnAutocheckEnable() 
{
	m_bNotification = !m_bNotification;
	UpdateTrayIcon();

	if(m_bNotification)
		StartTimer();
	else
		StopTimer();
}


void CPopManDoc::OnUpdateAutocheckEnable(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(m_bNotification ? 1 : 0);	
}

void CPopManDoc::OnStartUp()
{
	if(m_bCheckMailOnStartup)
	{
		if(m_nStartCheckDelay == 0)
			Timer_CheckOnStartup(0, 0, 0, 0);
		else
			m_nTimerCheckOnStartup = ::SetTimer(0, 0, m_nStartCheckDelay * 1000, Timer_CheckOnStartup);
	}
}


void CPopManDoc::Timer_CheckOnStartup(HWND hwnd, UINT uMSG, UINT idEvent, DWORD dwTime)
{
	if(m_pDoc == NULL)
		return;
	
	if(m_pDoc->m_nTimerCheckOnStartup > 0)
	{
		::KillTimer(0, m_pDoc->m_nTimerCheckOnStartup);
		m_pDoc->m_nTimerCheckOnStartup = 0;
	}
	
	m_pDoc->m_bAutoChecked = TRUE;
	m_pDoc->ListMails(TRUE);
	m_pDoc->m_bAutoChecked = FALSE;
}


CString CPopManDoc::GetStdNotifySound()
{
	CString stSound;
	GetWindowsDirectory(stSound.GetBuffer(MAX_PATH), MAX_PATH);
	stSound.ReleaseBuffer();
	if(stSound.Right(1) != _T("\\")) stSound += _T('\\');
	stSound += _T("Media\\notify.wav");
	CFileStatus FileStatus;
	if(0 == CFile::GetStatus(stSound, FileStatus))
		stSound.Empty();
	return stSound;
}


CString CPopManDoc::GetStdEmailClient()
{
	const CString stClientsRootPath = _T("Software\\Clients\\Mail");

	CRegistryKey Reg;
	if(!Reg.Open(HKEY_LOCAL_MACHINE, stClientsRootPath, KEY_READ))
		return _T("");

	CString stStdClientName;
	Reg.QueryValue(_T(""), stStdClientName);
	if(stStdClientName.IsEmpty())
		return _T("");

	Reg.Close();

	CString stOpenPath = stClientsRootPath;
	stOpenPath += _T('\\');
	stOpenPath += stStdClientName;
	stOpenPath += _T("\\shell\\open\\command");

	if(!Reg.Open(HKEY_LOCAL_MACHINE, stOpenPath, KEY_READ))
		return _T("");

	CString stStdClientPath;
	if(!Reg.QueryValue(_T(""), stStdClientPath))
		return _T("");

	TCHAR szExpandedPath[MAX_PATH] = _T("");
	if(ExpandEnvironmentStrings(stStdClientPath, szExpandedPath, sizeof(szExpandedPath)/sizeof(TCHAR)))
		stStdClientPath = szExpandedPath;

	return stStdClientPath;
}

void CPopManDoc::OnLaunchEmailClient() 
{
	CString stExe, stCmdParam;
	if(!SplitExeCmdParam(m_stEmailClient, stExe, stCmdParam))
		return;

	ShellExecute(NULL, _T("open"), stExe, stCmdParam, NULL, SW_SHOWNORMAL);

    SetNewMailState(FALSE);
	ResetKeyboardLight();

	if(m_ClientLaunchAction == CLIENTLAUNCHACTION::Quit)
		AfxGetMainWnd()->PostMessage(WM_COMMAND, ID_APP_EXIT, 0);
	else if(m_ClientLaunchAction == CLIENTLAUNCHACTION::HideToTray)
	{
		CMainFrame* pFrame = (CMainFrame*)AfxGetMainWnd();
		if(pFrame) {
			pFrame->HideToTray();
		}
	}    
}

bool CPopManDoc::IsNewMailAvailable() const 
{
    return m_NewMailList.GetSize() > 0;
}


void CPopManDoc::SetNewMailState(BOOL bNewMail)
{
	if(!bNewMail && !IsNewMailAvailable())
		return;
	
	if(bNewMail)
		UpdateNewMailTip();
	else
		m_NewMailList.RemoveAll();

	UpdateTrayIcon();
}


void CPopManDoc::UpdateNewMailTip()
{
	GetMailCount(NUMMAILSTRAY::New, m_stNewMailTip);
}

void CPopManDoc::OnTrayAction(int MouseAction)
{
	switch(MouseAction)
	{
	case WM_LBUTTONUP:
		m_pDoc->ExecuteAction(m_pDoc->m_actionLeftClick);
		break;

	case WM_RBUTTONUP:
		m_pDoc->ExecuteAction(m_pDoc->m_actionRightClick);
		break;

	case WM_MBUTTONUP:
		m_pDoc->ExecuteAction(m_pDoc->m_actionMiddleClick);
		break;

	case WM_LBUTTONDBLCLK:
		m_pDoc->ExecuteAction(m_pDoc->m_actionDoubleClick);
		break;
	}
}



static void MenuTextCmd(CMenu* pMenu, int Cmd, LPCTSTR szText)
{
	pMenu->ModifyMenu(Cmd, MF_BYCOMMAND|MF_STRING, Cmd, szText);
}

void CPopManDoc::ExecuteAction(TRAYACTIONS Action)
{
	m_MailInfo.DestroyWindow();
	m_NewTip.DestroyWindow();

	switch(Action)
	{
	case TRAYACTIONS::ShowWindow:
		AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_TRAY_SHOW, 0);
		// Ted Ferenc New Alterations 
		AfxGetMainWnd()->SetForegroundWindow(); 
		
        //AfxGetMainWnd()->ShowWindow(SW_SHOWNORMAL);
        // uncommented this last line because it prevents window from being restored maximized
		break;

	case TRAYACTIONS::CheckMail:
		OnListMails();
		break;

	case TRAYACTIONS::PopUpMenu:
	{
		CMenu Menu;
		if(!Menu.LoadMenu(IDR_TRAY))
			return;

		CMenu* pSubMenu = Menu.GetSubMenu(0);
		if(pSubMenu == NULL)
			return;
				
		pSubMenu->SetDefaultItem(0, TRUE);
		
		MenuTextCmd(pSubMenu, ID_TRAY_SHOW,				i18n("&Open PopMan"));
		MenuTextCmd(pSubMenu, ID_FILE_LIST_MAILS,		i18n("Check &Mail"));
		MenuTextCmd(pSubMenu, ID_FILE_CANCEL,			i18n("&Cancel"));
		MenuTextCmd(pSubMenu, ID_APP_EMAIL_CLIENT,		i18n("&Email Client"));
		MenuTextCmd(pSubMenu, ID_APP_AUTOCHECK,			i18n("A&utoCheck Enabled"));
        MenuTextCmd(pSubMenu, ID_APP_MUTE_MODE,			i18n("Mu&te Mode"));
		MenuTextCmd(pSubMenu, ID_EXTRAS_SETTINGS,		i18n("O&ptions..."));
		MenuTextCmd(pSubMenu, ID_APP_ABOUT,				i18n("&About..."));
		MenuTextCmd(pSubMenu, ID_APP_EXIT,				i18n("E&xit"));
		
		CPoint mouse;
		GetCursorPos(&mouse);
		AfxGetMainWnd()->SetForegroundWindow();  
		pSubMenu->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, mouse.x, mouse.y, AfxGetMainWnd(), NULL);
		
		break;
	}


	case TRAYACTIONS::RunEmailClient:
		OnLaunchEmailClient();
		break;

	case TRAYACTIONS::CheckAndShow:
		AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_TRAY_SHOW, 0);
		OnListMails();
		break;

	case TRAYACTIONS::ToggleWindow:
	{
		CMainFrame* pFrame = (CMainFrame*)AfxGetMainWnd();
		if(pFrame != NULL)
		{
			if(pFrame->IsIconic() || !pFrame->IsWindowVisible())
				pFrame->RestoreFromTray();
			else 
				pFrame->HideToTray();
		}
		break;
	}
		

	case TRAYACTIONS::ToggleAutoCheck:
		OnAutocheckEnable();
		break;

	case TRAYACTIONS::ShowMailInfo:
	{
		CString stTip;
		int nMails = GetMailCount(NUMMAILSTRAY::All, stTip);
		stTip.Empty();

		POSITION pos = m_Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			const CAccount* pAcc = m_Accounts.GetNext(pos);
			if(pAcc->m_Mails.GetCount() > 0 || (pAcc->m_bActive && nMails == 0))
			//if(pAcc->m_bActive && (pAcc->m_Mails.GetCount() > 0 || nMails == 0))
			{
				int nAllMailAcc = GetAccountMailNum(pAcc, NUMMAILSTRAY::All);
				int nUnreadMailAcc = GetAccountMailNum(pAcc, NUMMAILSTRAY::Unread);
				int nNewMailAcc = GetAccountMailNum(pAcc, NUMMAILSTRAY::New);
						
				stTip += pAcc->m_stName;
				stTip += _T(":\t");
				int nSizeKB = int(pAcc->GetSizeOfAllMails() / 1024);
				stTip += StrFormat(i18n("{1} msgs. ({2} unread, {3} new)\t{4} KB"), _T("d"), nAllMailAcc, _T("d"), nUnreadMailAcc, _T("d"), nNewMailAcc, _T("d"), nSizeKB);
				stTip += _T("\r\n");
			}
		}
		stTip.TrimRight(_T("\r\n"));

		if(m_Accounts.GetCount() == 0)
			stTip = i18n("No messages");

		stTip = TabJustify(stTip, m_MailInfo.m_Font);

		m_MailInfo.Show(NULL, i18n("PopMan - Mail Info"), stTip);
	}
		break;

	}
}




HICON CPopManDoc::CreateNumIcon(int nNum, COLORREF colorCircle)
{
	COLORREF colorText =  RGB(255, 255, 255);

	int R = (colorCircle & 0x000000FF);
	int G = (colorCircle & 0x0000FF00) >> 8;
	int B = (colorCircle & 0x00FF0000) >> 16;

	if(R+G+B > 400)
		colorText = 0;

	CString stNum;	
	stNum.Format(_T("%d"), nNum);

	HDC screenDC = ::GetDC(0);
	HDC bmpDC = ::CreateCompatibleDC(screenDC);
	HDC maskDC = ::CreateCompatibleDC(screenDC);

	HBITMAP Bmp     = ::CreateCompatibleBitmap(screenDC, 16, 16);
	HBITMAP maskBmp = ::CreateBitmap(16, 16, 1, 1, NULL);
	HBITMAP oldBmp     = (HBITMAP)::SelectObject(bmpDC, Bmp); 
	HBITMAP oldmaskBmp = (HBITMAP)::SelectObject(maskDC, maskBmp); 


	HICON hIcon = CPopManApp::GetIcon(IDI_TRAY_NUM); //(HICON)::LoadImage(GetModuleHandle(0), MAKEINTRESOURCE(IDR_TRAY_NUM), IMAGE_ICON, 16, 16, LR_DEFAULTCOLOR);
	::DrawIconEx(bmpDC,  0, 0, hIcon, 16, 16, 0, 0, DI_IMAGE);
	::DrawIconEx(maskDC, 0, 0, hIcon, 16, 16, 0, 0, DI_MASK);
	//::DestroyIcon(hIcon);

	PaintCircle(bmpDC,  1, 0, colorCircle);
	PaintCircle(maskDC, 1, 0, 0);

	CFont Font;
//	Font.CreatePointFont(nNum > 9 ? 72 : 85, _T("Arial"));
	LOGFONT logFont;
	memset(&logFont, 0, sizeof(LOGFONT));
	logFont.lfCharSet = DEFAULT_CHARSET;
	logFont.lfHeight = nNum > 9 ? -9 : -11;
	lstrcpyn(logFont.lfFaceName, _T("Arial"), sizeof(logFont.lfFaceName)/sizeof(logFont.lfFaceName[0]));

	Font.CreateFontIndirect(&logFont);
	
	HFONT oldFont = (HFONT)::SelectObject(bmpDC, Font.m_hObject);

	::SetBkMode(bmpDC, TRANSPARENT);
	::SetTextColor(bmpDC, colorText);
	::TextOut(bmpDC, (nNum > 9 ? 3 : 5), nNum > 9 ? 1 : 0, stNum, stNum.GetLength());
	::SelectObject(bmpDC, oldFont);
	Font.DeleteObject();

	::SelectObject(bmpDC, oldBmp);
	::SelectObject(maskDC, oldmaskBmp);

	ICONINFO IconInfo;
	memset(&IconInfo, 0, sizeof(ICONINFO));
	IconInfo.fIcon = TRUE;
	IconInfo.hbmColor = Bmp;
	IconInfo.hbmMask = maskBmp;
	
	HICON Icon = ::CreateIconIndirect(&IconInfo);
		
	::DeleteObject(Bmp);
	::DeleteObject(maskBmp);

	::DeleteDC(bmpDC);
	::DeleteDC(maskDC);
	::ReleaseDC(0, screenDC);

	return Icon;
}



HICON CPopManDoc::CreateErrIcon(COLORREF colorCircle)
{
	HDC screenDC = ::GetDC(0);
	HDC bmpDC = ::CreateCompatibleDC(screenDC);
	HDC maskDC = ::CreateCompatibleDC(screenDC);

	HBITMAP Bmp     = ::CreateCompatibleBitmap(screenDC, 16, 16);
	HBITMAP maskBmp = ::CreateBitmap(16, 16, 1, 1, NULL);
	HBITMAP oldBmp     = (HBITMAP)::SelectObject(bmpDC, Bmp); 
	HBITMAP oldmaskBmp = (HBITMAP)::SelectObject(maskDC, maskBmp); 

	HICON hIcon = CPopManApp::GetIcon(IDI_TRAY_NUM); //(HICON)::LoadImage(GetModuleHandle(0), MAKEINTRESOURCE(IDR_TRAY_NUM), IMAGE_ICON, 16, 16, LR_DEFAULTCOLOR);
	::DrawIconEx(bmpDC,  0, 0, hIcon, 16, 16, 0, 0, DI_IMAGE);
	::DrawIconEx(maskDC, 0, 0, hIcon, 16, 16, 0, 0, DI_MASK);
	//::DestroyIcon(hIcon);

	PaintCircle(bmpDC,  1, 0, colorCircle);
	PaintCircle(maskDC, 1, 0, 0);

	hIcon = CPopManApp::GetIcon(IDI_TRAY_EXCLAMATION); //(HICON)::LoadImage(GetModuleHandle(0), MAKEINTRESOURCE(IDR_TRAY_EXCLAMATION), IMAGE_ICON, 16, 16, LR_DEFAULTCOLOR);
	::DrawIconEx(bmpDC,  0, 0, hIcon, 16, 16, 0, 0, DI_NORMAL);
	//::DestroyIcon(hIcon);

	::SelectObject(bmpDC, oldBmp);
	::SelectObject(maskDC, oldmaskBmp);

	ICONINFO IconInfo;
	memset(&IconInfo, 0, sizeof(ICONINFO));
	IconInfo.fIcon = TRUE;
	IconInfo.hbmColor = Bmp;
	IconInfo.hbmMask = maskBmp;
	
	HICON Icon = ::CreateIconIndirect(&IconInfo);
		
	::DeleteObject(Bmp);
	::DeleteObject(maskBmp);

	::DeleteDC(bmpDC);
	::DeleteDC(maskDC);
	::ReleaseDC(0, screenDC);

	return Icon;
}

HICON CPopManDoc::CreateInnerRectIcon(COLORREF colorInner, COLORREF colorOuter)
{
	HDC screenDC = ::GetDC(0);
	HDC bmpDC = ::CreateCompatibleDC(screenDC);
	HDC maskDC = ::CreateCompatibleDC(screenDC);

	HBITMAP Bmp     = ::CreateCompatibleBitmap(screenDC, 16, 16);
	HBITMAP maskBmp = ::CreateBitmap(16, 16, 1, 1, NULL);
	HBITMAP oldBmp     = (HBITMAP)::SelectObject(bmpDC, Bmp); 
	HBITMAP oldmaskBmp = (HBITMAP)::SelectObject(maskDC, maskBmp); 

	HICON hIcon = CPopManApp::GetIcon(IDI_TRAY_NUM); //(HICON)::LoadImage(GetModuleHandle(0), MAKEINTRESOURCE(IDR_TRAY_NUM), IMAGE_ICON, 16, 16, LR_DEFAULTCOLOR);
	::DrawIconEx(bmpDC,  0, 0, hIcon, 16, 16, 0, 0, DI_IMAGE);
	::DrawIconEx(maskDC, 0, 0, hIcon, 16, 16, 0, 0, DI_MASK);
	//::DestroyIcon(hIcon);

	PaintCircle(bmpDC,  1, 0, colorOuter);
	PaintCircle(maskDC, 1, 0, 0);
	
	PaintInnerRect(bmpDC, 3, 4, 10, 6, colorInner);

	::SelectObject(bmpDC, oldBmp);
	::SelectObject(maskDC, oldmaskBmp);

	ICONINFO IconInfo;
	memset(&IconInfo, 0, sizeof(ICONINFO));
	IconInfo.fIcon = TRUE;
	IconInfo.hbmColor = Bmp;
	IconInfo.hbmMask = maskBmp;
	
	HICON Icon = ::CreateIconIndirect(&IconInfo);
		
	::DeleteObject(Bmp);
	::DeleteObject(maskBmp);

	::DeleteDC(bmpDC);
	::DeleteDC(maskDC);
	::ReleaseDC(0, screenDC);

	return Icon;

}


void CPopManDoc::PaintInnerRect(HDC dc, int xOffset, int yOffset, int Width, int Height, COLORREF Color)
{
	HPEN Pen = ::CreatePen(PS_SOLID, 1, Color);
	HBRUSH Brush = ::CreateSolidBrush(Color);
	
	HPEN oldPen = (HPEN)::SelectObject(dc, Pen);
	HBRUSH oldBrush = (HBRUSH)::SelectObject(dc, Brush);

	::Rectangle(dc, xOffset, yOffset, xOffset + Width, yOffset + Height);
	
	::SelectObject(dc, oldBrush);
	::SelectObject(dc, oldPen);
	::DeleteObject(Brush);
	::DeleteObject(Pen);
}

void CPopManDoc::PaintCircle(HDC dc, int xOffset, int yOffset, COLORREF Color)
{
	HPEN Pen = ::CreatePen(PS_SOLID, 1, Color);
	HBRUSH Brush = ::CreateSolidBrush(Color);
	
	HPEN oldPen = (HPEN)::SelectObject(dc, Pen);
	HBRUSH oldBrush = (HBRUSH)::SelectObject(dc, Brush);

	::Rectangle(dc, xOffset, yOffset + 5, xOffset + 14, yOffset + 9);
	::Rectangle(dc, xOffset + 5, yOffset, xOffset + 9, yOffset + 14);
	::Rectangle(dc, xOffset + 1, yOffset + 3, xOffset + 13, yOffset + 11);
	::Rectangle(dc, xOffset + 3, yOffset + 1, xOffset + 11, yOffset + 13);
	::Rectangle(dc, xOffset + 2, yOffset + 2, xOffset + 12, yOffset + 12);
	
	::SelectObject(dc, oldBrush);
	::SelectObject(dc, oldPen);
	::DeleteObject(Brush);
	::DeleteObject(Pen);
}

int CPopManDoc::GetAccountMailNum(const CAccount* pAcc, NUMMAILSTRAY numMails)
{
	int nMailCount = 0;
	if(numMails == NUMMAILSTRAY::All)
	{
		nMailCount = pAcc->m_Mails.GetCount();
	}
	else if(numMails == NUMMAILSTRAY::Unread)
	{
		nMailCount = pAcc->CountUnreadMails();
	}
	else if(numMails == NUMMAILSTRAY::New)
	{
		for(int n = 0; n < m_NewMailList.GetSize(); n++)
		{
			if(pAcc->m_MailCache.FindSignature(m_NewMailList[n]) > -1)
				nMailCount++;
		}
	}

	return nMailCount;
}

CString CPopManDoc::GetMessageCountStr(NUMMAILSTRAY numMails, int nMessages, int nAccs, const CString& stAccName)
{
	CString stMsg;

	if(numMails == NUMMAILSTRAY::All)
	{
		if(nMessages == 1)
			stMsg = i18n("1 message");
		else
			stMsg = StrFormat(i18n("{1} messages"), _T("d"), nMessages);
	}
	else if(numMails == NUMMAILSTRAY::Unread)
	{
		if(nMessages == 1)
			stMsg = i18n("1 unread message");
		else
			stMsg = StrFormat(i18n("{1} unread messages"), _T("d"), nMessages);  
	}
	else if(numMails == NUMMAILSTRAY::New)
	{
		if(nMessages == 1)
			stMsg = i18n("1 new message");
		else
			stMsg = StrFormat(i18n("{1} new messages"), _T("d"), nMessages); 
	}
	stMsg += _T(' ');
	
	CString stAcc;
	if(nAccs == 1)
		stAcc = StrFormat(i18n("in {1}"), _T("s"), (LPCTSTR)stAccName);
	else
		stAcc = StrFormat(i18n("in {1} accounts"), _T("d"), nAccs);

	stMsg += stAcc;
	return stMsg;
}

int CPopManDoc::GetMailCount(NUMMAILSTRAY numMails, CString& stTip)
{
	CString stAccName;
	int nMailCount = 0;
	int nAccs = 0;
	stTip.Empty();

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		const CAccount* pAcc = m_Accounts.GetNext(pos);
		int nMailAcc = GetAccountMailNum(pAcc, numMails);
		if(nMailAcc > 0)
		{
			nMailCount += nMailAcc;
			nAccs++;
			stAccName = pAcc->m_stName;
		}
	}

	stTip = GetMessageCountStr(numMails, nMailCount, nAccs, stAccName);
		
	return nMailCount;
}

CString CPopManDoc::FormatAccountErrTip(const CAccount* pAcc)
{
	CString stErrorTip;
	stErrorTip = pAcc->m_stName + _T(":  ") + pAcc->GetErrorDescription();
	if(!pAcc->m_stServerErrMessage.IsEmpty())
	{	
		stErrorTip += _T("  \"");
		stErrorTip += pAcc->m_stServerErrMessage;
		stErrorTip += _T("\"");
	}
	return stErrorTip;
}


CString CPopManDoc::TabJustify(CString stTabs, HFONT hFont)
{
 
	typedef std::vector<CString> CCols;
	typedef std::vector<CCols> CLines;

	CLines Lines;
	size_t nColCount = 0;

	while(stTabs.GetLength() > 0)
	{
		CString stLine = Parse(stTabs, _T("\r\n"));
		CCols Cols;
		while(stLine.GetLength() > 0)
			Cols.push_back(Parse(stLine, _T("\t")));
		Lines.push_back(Cols);
		if(Cols.size() > nColCount) nColCount = Cols.size();
	}

	size_t n = 0;
	for(n = 0; n < Lines.size(); n++)
	{
		while(Lines[n].size() < nColCount)
			Lines[n].push_back(_T(""));
	}


	HDC screenDC = ::GetDC(0);
	HDC hDC = ::CreateCompatibleDC(screenDC);
	::ReleaseDC(0, screenDC);

	HFONT oldFont = (HFONT)::SelectObject(hDC, hFont);
	
	for(n = 0; n < nColCount - 1 ; n++)
	{
		size_t m = 0;
		size_t nMaxColWidth = 0;
		for(m = 0; m < Lines.size(); m++)
		{
			const CString& St = Lines[m][n];
			const unsigned int nWidth = LOWORD(::GetTabbedTextExtent(hDC, St, St.GetLength(), 0, NULL));
			if(nWidth > nMaxColWidth) nMaxColWidth = nWidth;
		}

		for(m = 0; m < Lines.size(); m++)
		{
			CString& Col = Lines[m][n];
			while(LOWORD(::GetTabbedTextExtent(hDC, Col, Col.GetLength(), 0, NULL)) < nMaxColWidth)
				Col += _T(' ');
		}
	}

	::SelectObject(hDC, oldFont);
	::DeleteDC(hDC);

	CString stRes;
	for(n = 0; n < Lines.size(); n++)
	{
		CString stLine;
		for(size_t m = 0; m < Lines[n].size(); m++)
		{
			stLine += Lines[n][m];
			if(m + 1 < Lines[n].size()) stLine += _T('\t');
		}
		stRes += stLine;
		if(n + 1 < Lines.size()) stRes += _T("\r\n");
	}

	return stRes;
}


void CPopManDoc::ShowNewMailTip(const CNewTip::CLines& Lines, LPCTSTR szHeader)
{
	m_NewTip.m_fnCallback = OnNewTipEvent;
	m_NewTip.m_Font		= (HFONT)m_MailInfo.m_Font.m_hObject;
	m_NewTip.m_headFont = (HFONT)m_MailInfo.m_boldFont.m_hObject;

	m_NewTip.m_stHead = szHeader;

	m_NewTip.m_Lines.Copy(Lines);

	CNewTipButton Button;
	m_NewTip.m_Buttons.RemoveAll();

	Button.m_stCaption = i18n("Show");
	Button.m_dwID = NEWTIPBUTTON::ShowMail;
	m_NewTip.m_Buttons.Add(Button);

	Button.m_stCaption = i18n("Email Client");
	Button.m_dwID = NEWTIPBUTTON::EmailClient;
	m_NewTip.m_Buttons.Add(Button);

	if(Lines.GetSize() == 1 && Lines[0].dwValue != 0)
	{
		Button.m_stCaption = i18n("Delete");
		Button.m_dwID = NEWTIPBUTTON::DeleteMail;
		m_NewTip.m_Buttons.Add(Button);
	}

	Button.m_stCaption = i18n("Close");
	Button.m_dwID = NEWTIPBUTTON::CloseTip;
	m_NewTip.m_Buttons.Add(Button);

	m_NewTip.Show(m_nNewMailTipDuration);
}

void CPopManDoc::OnNewTipEvent(int nEvent, DWORD dwValue)
{
	CNewTip::NEWTIPEVENT Event = (CNewTip::NEWTIPEVENT)nEvent;

	if(m_pDoc)
		m_pDoc->ResetKeyboardLight();

	switch(Event)
	{
	case CNewTip::NEWTIPEVENT::ItemClicked:
	{
		m_pDoc->SetNewMailState(FALSE);

		CMail* pMail = (CMail*)dwValue;
		if(m_pDoc->IsMailValid(pMail))
			CMessageFrm* pMessage = new CMessageFrm(m_pDoc, pMail);
		break;
	}
		

	case CNewTip::NEWTIPEVENT::ButtonClicked:
	{
        m_pDoc->SetNewMailState(FALSE);

		switch((NEWTIPBUTTON)dwValue)
		{
		case NEWTIPBUTTON::EmailClient:
			m_pDoc->ExecuteAction(TRAYACTIONS::RunEmailClient);
			break;
		case NEWTIPBUTTON::ShowMail:
		{
			CMail* pMail = NULL;
			if(m_pDoc->m_OnNewMailList.GetSize() > 0) {
				const CString& stNewMail = m_pDoc->m_OnNewMailList[0];
				POSITION pos = m_pDoc->m_Accounts.GetHeadPosition();
				while(pos != NULL)
				{
					const CAccount* pAcc = m_pDoc->m_Accounts.GetNext(pos);	
					pMail = pAcc->GetMailFromSignature(stNewMail);
					if(pMail != NULL) break;
				}
			}

			if(pMail != NULL)
				m_pDoc->UpdateAllViews(NULL, ntEnsureVisible, pMail);

			m_pDoc->ExecuteAction(TRAYACTIONS::ShowWindow);
			break;
		}
		case NEWTIPBUTTON::CloseTip:
			
			// nothing to do, because NewTip destroys itself
			break;
		case NEWTIPBUTTON::DeleteMail:
			{
				CMail* pMail = (CMail*)m_pDoc->m_NewTip.m_Lines[0].dwValue;
				if(!m_pDoc->IsMailValid(pMail))
					return;
				

				bool bShift = (::GetKeyState(VK_SHIFT) & 0x8000) != 0;

				if(m_pDoc->m_bMarkDelete && !bShift) 
				{
					pMail->SetMarkedForDelete();
					m_pDoc->ShowMarkMessagInfo();
					pMail->SetRead(); // Mark unread if marked for delete, so it doesn't show up in the tray icon
					AfxGetMainWnd()->SendMessageToDescendants(WM_MAIL_READ, (WPARAM)pMail, 0);
					AfxGetMainWnd()->RedrawWindow();
				}
				else
				{
					CMails MailList;
					MailList.AddTail(pMail);

					if(m_pDoc->m_bConfirmDeleteMsg)
					{
						CString stMsg, stPrompt;
						stMsg = i18n("Are you sure you want to delete this email?");
						stMsg += _T("\r\n\r\n");
						stMsg += _T("\"%s\"");
						stPrompt.Format(stMsg, (LPCTSTR)pMail->GetSubject());
						if(IDNO == AfxMessageBox(stPrompt, MB_YESNO))
							return;
					}
					
					m_pDoc->m_NewTip.DestroyWindow();

					if(m_pDoc->DeleteMails(&MailList))
						return;

					AfxMessageBox(i18n("Deleting email failed!"), MB_ICONERROR);
				}
				break;
			}
		}
		break;		
	}
	}
}

CString CPopManDoc::GetIdleStateInfo() const
{
	COleDateTime::DateTimeStatus timeState = m_LastCheckTime.GetStatus();
	if(timeState != COleDateTime::DateTimeStatus::valid)
		return i18n("Ready");


	CString stMsg;
	int nMailCount = 0;

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL){
		const CAccount* pAcc = m_Accounts.GetNext(pos);
		nMailCount += pAcc->m_Mails.GetCount();
	}

	if(nMailCount == 1)
		stMsg = i18n("1 message");
	else
		stMsg = StrFormat(i18n("{1} messages"), _T("d"), nMailCount);

	stMsg += _T(' ');
	
	CString stTime = m_LastCheckTime.Format(VAR_TIMEVALUEONLY);

	int Idx = stTime.Find(m_LastCheckTime.Format(_T("%H:%M:%S")));
	if(Idx == -1)
		Idx = stTime.Find(m_LastCheckTime.Format(_T("%I:%M:%S")));
	if(Idx != -1)
		stTime.Delete(Idx+5, 3);
	
	stMsg += StrFormat(i18n("(last check: {1})"), _T("s"), (LPCTSTR)stTime);
	

	return stMsg;
}

void CPopManDoc::OnUpdateGUI()
{
	DisplayState(GetIdleStateInfo());
	UpdateTrayIcon();
}

void CPopManDoc::OnShowBlacklist() 
{
    CDlgMailAddressList dlg(CDlgMailAddressList::MailAddListTypes::BlackList, AfxGetMainWnd());

	dlg.m_Addresses.Copy(m_BlackList.GetAddresses());
	dlg.m_Domains.Copy(m_BlackList.GetDomains());
	if(dlg.DoModal() == IDOK)
	{
        m_BlackList.SetAddresses(dlg.m_Addresses);
        m_BlackList.SetDomains(dlg.m_Domains);
        m_BlackList.Save();
	}
}

void CPopManDoc::OnShowWhitelist() 
{
    CDlgMailAddressList dlg(CDlgMailAddressList::MailAddListTypes::WhiteList, AfxGetMainWnd());

	dlg.m_Addresses.Copy(m_WhiteList.GetAddresses());
	dlg.m_Domains.Copy(m_WhiteList.GetDomains());
	if(dlg.DoModal() == IDOK)
	{
        m_WhiteList.SetAddresses(dlg.m_Addresses);
        m_WhiteList.SetDomains(dlg.m_Domains);
        m_WhiteList.Save();
	}
}

void CPopManDoc::BlackListSenders(const CMails& list)
{
	POSITION pos = list.GetHeadPosition();
	while(pos != NULL) 
    {
		const CMail* pMail = list.GetNext(pos);
		const CString& sender = pMail->GetFromAddress();

        m_BlackList.AddAddress(sender);
	}

    m_BlackList.Save();
}

void CPopManDoc::WhiteListSenders(const CMails& list)
{
	POSITION pos = list.GetHeadPosition();
	while(pos != NULL) 
    {
		const CMail* pMail = list.GetNext(pos);
		const CString& sender = pMail->GetFromAddress();

        m_WhiteList.AddAddress(sender);
	}

    m_WhiteList.Save();
}


void CPopManDoc::LoadPlugins()
{
	m_ProtocolPlugins.RemoveAll();

	WIN32_FIND_DATA FileData;

	CString stDir = CPopManApp::GetAppPluginsPath();
	if(stDir.Right(1) != _T("\\")) stDir += _T('\\');
	
	HANDLE hHandle = FindFirstFile(stDir + _T("*.dll"), &FileData);
	if(hHandle == INVALID_HANDLE_VALUE)
		return;

	do 
	{
		if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0)
			continue;

		CProtocolPlugin* pPlugin = CProtocolPlugin::LoadProtocolPlugin(stDir + FileData.cFileName);
		if(pPlugin != NULL)
			m_ProtocolPlugins.AddTail(pPlugin);

	} while(FindNextFile(hHandle, &FileData) != 0);

	FindClose(hHandle);	
}

CProtocol* CPopManDoc::GetProtocolFromName(LPCTSTR szName)
{
	POSITION posPlugin = m_ProtocolPlugins.GetHeadPosition();
	while(posPlugin != NULL)
	{
		CProtocolPlugin& Plugin = *m_ProtocolPlugins.GetNext(posPlugin);
		const CProtocols& protocols = Plugin.Protocols();
		POSITION posProtocol = protocols.GetHeadPosition();
		while(posProtocol != NULL)
		{
			CProtocol& proto = *protocols.GetNext(posProtocol);
			if(proto.GetName().CompareNoCase(szName) == 0)
				return &proto;
		}
	}
	return NULL;
}

void CPopManDoc::GetAvailableProtocols(CProtocols& resProtocols)
{
	resProtocols.RemoveAll();

	POSITION posPlugin = m_ProtocolPlugins.GetHeadPosition();
	while(posPlugin != NULL)
	{
		CProtocolPlugin& Plugin = *m_ProtocolPlugins.GetNext(posPlugin);
		const CProtocols& protocols = Plugin.Protocols();
		POSITION posProtocol = protocols.GetHeadPosition();
		while(posProtocol != NULL)
			resProtocols.AddTail(protocols.GetNext(posProtocol));
	}
}


void CPopManDoc::ToggleKeyboardLight()
{
	int keyLight = KEYBOARD_SCROLL_LOCK;

	switch(m_KeyboardLight) {
		case KEYBOARDLIGHT::ScrollLock:	keyLight = KEYBOARD_SCROLL_LOCK; break;
		case KEYBOARDLIGHT::NumLock:	keyLight = KEYBOARD_NUM_LOCK;	 break;
		case KEYBOARDLIGHT::CapsLock:	keyLight = KEYBOARD_CAPS_LOCK;	 break;
		default: return;
	}
		
	if(m_hKeyboardDev == INVALID_HANDLE_VALUE)
		m_hKeyboardDev = OpenKeyboardDevice(NULL);

	if (m_hKeyboardDev == INVALID_HANDLE_VALUE)		// Was the device opened?
	{
		TRACE(_T("Unable to open the device. (error %d)\n"), GetLastError());
		return;
	}

	NT_ToggleKeyboardLight(m_hKeyboardDev, keyLight);	

	m_bKeyLightToggled = !m_bKeyLightToggled;
}


void CPopManDoc::Timer_ToggleKeyboardLight(HWND hwnd, UINT uMSG, UINT idEvent, DWORD dwTime)
{
	if(m_pDoc == NULL || m_pDoc->m_nTimerIDKeyboard == 0)
		return;

	m_pDoc->ToggleKeyboardLight();
}

void CPopManDoc::ResetKeyboardLight() 
{
	if(m_nTimerIDKeyboard != 0) 
	{
		::KillTimer(0, m_nTimerIDKeyboard);
		m_nTimerIDKeyboard = 0;

		if(m_bKeyLightToggled)
			ToggleKeyboardLight();

		CloseKeyboardDevice(m_hKeyboardDev);
		m_hKeyboardDev = INVALID_HANDLE_VALUE;
	}
}


void CPopManDoc::OnEmptyTrash() 
{
	if(CountMarkedMails() < 1) return;

	if(m_bShowEmptyTrashWarning) {
		CString message = i18n("You are about to irrevocably delete all marked messages.");
		CMessageBoxDialog dlg(AfxGetMainWnd()->GetActiveWindow(), message, "", MB_ICONEXCLAMATION|MB_OKCANCEL|MB_DONT_DISPLAY_AGAIN);
		int res = dlg.DoModal();

		if(dlg.GetDontDisplayAgain() == TRUE)
			m_bShowEmptyTrashWarning = FALSE;

		if(res == IDCANCEL)
			return;
	}


	CMails MailList;	

	POSITION pos = m_Accounts.GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc = m_Accounts.GetNext(pos);
		POSITION posM = pAcc->m_Mails.GetHeadPosition();
		while(posM != NULL) 
		{
			CMail* pMail = pAcc->m_Mails.GetNext(posM);
			if(pMail->IsMarkedForDelete())
				MailList.AddTail(pMail);
		}
	}

	if(!MailList.IsEmpty())
		DeleteMails(&MailList);	
}

void CPopManDoc::OnUpdateEmptyTrash(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(IsEmptyTrashEnabled());
}

BOOL CPopManDoc::IsEmptyTrashEnabled()
{
	return (CountMarkedMails() > 0 && !ConnectionsPending());
}

int CPopManDoc::CountMarkedMails()
{
	return CMail::getNumberOfMarkedMessages();
}


void CPopManDoc::ShowMarkMessagInfo()
{
	if(!m_bShowMarkDeleteInfo) return;

	CString s1 = i18n("You have just marked a message for deletion. To unmark it, press the Ins keyboard button. To delete all marked messages, press F12 or click the appropriate toolbar button.");
	CString s2 = i18n("You can also delete selected messages immediately by pressing Shift+Del.");

    CString s3 = StrFormat(i18n("If you want to restore the traditional behaviour (delete all selected messages immediately), uncheck \"{1}\" in the options dialog."), "s", i18n("Delete command marks message for deletion")); 
	CString message = s1 + "\n\n" + s2 + "\n\n" + s3;
	
	CMessageBoxDialog dlg(AfxGetMainWnd()->GetActiveWindow(), message, "", MB_ICONINFORMATION|MB_OK|MB_DONT_DISPLAY_AGAIN);
	dlg.DoModal();

	if(dlg.GetDontDisplayAgain() == TRUE)
		m_bShowMarkDeleteInfo = FALSE;

}


void CPopManDoc::OnRules() 
{
	CString stFile = m_RuleManager.GetFileName();

	CFileStatus status;
	if(!CFile::GetStatus(stFile, status)) 
	{
		try {
			CFile file(stFile, CFile::modeCreate | CFile::modeNoTruncate | CFile::modeWrite);
			file.Close();

			CString stFileExamples = CPopManApp::GetAppPath();
			if(stFileExamples.Right(1) != _T("\\")) stFileExamples += _T('\\');
			stFileExamples += szRulesExamplesFile;
			
			::CopyFile(stFileExamples, stFile, FALSE);		
		}
		catch(CFileException* e) {
			e->ReportError();
			e->Delete();
			return;
		}
	}

	::ShellExecute(NULL, _T("open"), stFile, NULL, NULL, SW_SHOWNORMAL);
}

static CRulesManager::StringType mapType(CDlgRulesItem::ItemType type)
{
    switch(type)
    {
    case CDlgRulesItem::ItemType::typeString:
        return CRulesManager::StringType::stringAny;

    case CDlgRulesItem::ItemType::typeWord:
        return CRulesManager::StringType::stringWord;

    case CDlgRulesItem::ItemType::typeRegEx:
        return CRulesManager::StringType::stringRegEx;

    default:
        return CRulesManager::StringType::stringWord;
    }
}

void AddUnique(CStringArray& array, const CString& newItem)
{
    for(int i = 0; i < array.GetSize(); ++i) 
    {
        if(array[i].CompareNoCase(newItem) == 0)
            return;
    }
    array.Add(newItem);
}

bool CPopManDoc::AddItemToRulesList(const CString& item, const CMail* pMail)
{
    CDlgRulesItem dlg;

    m_RuleManager.GetDefinedListNames(dlg.m_AvailableLists);
    dlg.m_stNewItem = item;
    dlg.m_nDestinationList = FindInStrArray(dlg.m_AvailableLists, m_stLastSelectedRulesDestList);
    
    if(!item.IsEmpty())
        dlg.m_NewItemSuggestions.Add(item);

    if(IsMailValid(pMail))
    {
        AddUnique(dlg.m_NewItemSuggestions, pMail->GetSubject());
        AddUnique(dlg.m_NewItemSuggestions, pMail->GetEntireFrom());
        AddUnique(dlg.m_NewItemSuggestions, pMail->GetFrom());
        AddUnique(dlg.m_NewItemSuggestions, pMail->GetReplyTo());
        AddUnique(dlg.m_NewItemSuggestions, pMail->GetTo());
    }

    if(IDOK != dlg.DoModal())
        return false;
    
    int listIdx = dlg.m_nDestinationList;
    if(listIdx >= 0 && listIdx < dlg.m_AvailableLists.GetSize())
        m_stLastSelectedRulesDestList = dlg.m_AvailableLists[listIdx];
    else
        m_stLastSelectedRulesDestList.Empty();

    CString err;

    if(m_stLastSelectedRulesDestList.IsEmpty() || !m_RuleManager.AddItemToList(m_stLastSelectedRulesDestList, dlg.m_stNewItem, mapType(dlg.m_ItemType), err)) 
    {
        CString stMsg = i18n("Failed to add item to Rules list!");
        if(err.GetLength() > 0) 
            stMsg += _T("\n\n") + err;

        AfxMessageBox(stMsg, MB_OK | MB_ICONERROR);
        return false;
    }
    return true;   
}

void CPopManDoc::OnMuteMode() 
{
	m_bMuteMode = !m_bMuteMode;
}

void CPopManDoc::OnUpdateMuteMode(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck(m_bMuteMode);
}
