// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// PopMan.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "PopMan.h"

#include "MainFrm.h"
#include "PopManDoc.h"
#include "PopManView.h"
#include <locale.h>
#include "StaticLink.h"
#include "DlgAccImport.h"
#include "DlgPass.h"
#include "RegKey.h"
#include "StrFunctions.h"
#include "htmlhelp.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern const UINT UWM_ARE_YOU_ME = ::RegisterWindowMessage(_T("PopMan-RegMsg-3D27B5E3-841D-4dc2-B0ED-5FAEA6DE1990"));

BOOL	CPopManApp::m_bLocalStorage = FALSE;
BOOL    CPopManApp::m_bCheckForUpdates = TRUE;
CString CPopManApp::m_stTheme;

CPopManApp::CMapIcon	CPopManApp::m_MapResourceIcon;
CPopManApp::CMapCursor	CPopManApp::m_MapResourceCursor;


//  i18nComment("PopMan Start")


/////////////////////////////////////////////////////////////////////////////
// CPopManApp

BEGIN_MESSAGE_MAP(CPopManApp, CWinApp)
	//{{AFX_MSG_MAP(CPopManApp)
	ON_COMMAND(ID_APP_ABOUT, OnAppAbout)
	ON_COMMAND(ID_APP_HELP, OnAppHelp)
	ON_UPDATE_COMMAND_UI(ID_APP_ABOUT, OnUpdateAppAbout)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_FILE_NEW, CWinApp::OnFileNew)
	ON_COMMAND(ID_FILE_OPEN, CWinApp::OnFileOpen)
END_MESSAGE_MAP()


CPopManApp theApp;
CSettings  Settings;


CPopManApp::CPopManApp()
{
	m_bPasswordProtect = FALSE;
}

BOOL CPopManApp::InitInstance()
{    
	CString stIniFile = GetAppPath();
	if(stIniFile.Right(1) != _T("\\")) stIniFile += _T('\\');
	stIniFile += szPopManIniFile;

	CFileStatus Status;
	m_bLocalStorage = CFile::GetStatus(stIniFile, Status);
	if(m_bLocalStorage)
	{
		Settings.InitIniFile(stIniFile);

		if(Status.m_size == 0)
		{
			CRegistryKey RegKey;
			CSettingsKey SettingsKey;
			if(RegKey.Open(HKEY_CURRENT_USER, szMainRegKey) && GetSettingsRootKey(SettingsKey))
				ImportRegistrySettings(RegKey, SettingsKey);

			ImportDataFile(szMailCacheFile);
			ImportDataFile(szBlackListFile);
            ImportDataFile(szWhiteListFile);            
            ImportDataFile(szLogFile);
            ImportDataFile(szRuleLogFile);
            ImportDataFile(szRulesFile);

            // In case a custom named rules file is used:
            CSettingsKey Key;
            if(OpenSettingsKey(Key, szSettingsKey))
	        {
                CString stRulesName;
                Key.QueryValue(szRulesFileNameSetting, stRulesName);
                if(stRulesName.IsEmpty())
                    stRulesName = szRulesFile;
                
                ImportDataFile(stRulesName);
            }

		}
	}
	else
	{
		Settings.InitRegistry(szMainRegKey);
	}
			
	
	CSettingsKey Key;
	if(OpenSettingsKey(Key, szSettingsKey))
	{
		Key.QueryValue(szPasswordEnabled, m_bPasswordProtect);
		Key.QueryValue(szPasswordValue, m_stPassword);
		Key.QueryValue(szThemeValue, m_stTheme);
		m_stPassword = Crypt(m_stPassword, FALSE);

		CString stFile;
		if(Key.QueryValue(szTranslationFile, stFile))
		{
			if(!stFile.IsEmpty())
			{
				CString stLangPath = GetAppLanguagePath();
				if(stLangPath.Right(1) != _T("\\")) stLangPath += _T('\\');
				stLangPath += stFile;
				LoadLanguageFile(stLangPath);
			}
		}
		else
			LoadUserDefaultLang();

		Key.Close();
	}
	else
		LoadUserDefaultLang();

	
	// Sicherstellen, dass nur eine Instanz zur selben Zeit luft:
	::CreateMutex(NULL, FALSE, _T("PopMan-3D27B5E3-841D-4dc2-B0ED-5FAEA6DE1990"));
	BOOL bFound = (::GetLastError() == ERROR_ALREADY_EXISTS || ::GetLastError() == ERROR_ACCESS_DENIED);

	if(bFound)
	{
		AfxMessageBox(i18n("Please start one instance only."));
		
		HWND hOther = NULL;
        ::EnumWindows(Searcher, (LPARAM)&hOther);

		if(hOther != NULL)
		{
			if(::IsIconic(hOther) || !::IsWindowVisible(hOther))
				::ShowWindow(hOther, SW_RESTORE);

			::SetForegroundWindow(hOther);
		}
		return FALSE; // Beenden!
	}



	if(m_bPasswordProtect)
	{
		CDlgAppPass DlgPass;
		DlgPass.m_stPass = m_stPassword;
		if(IDOK != DlgPass.DoModal())
			return FALSE;
	}
	


		
	if (!AfxSocketInit())
	{
		AfxMessageBox(i18n("Windows socket init failed."));
		return FALSE;
	}
	
	setlocale(LC_ALL, "");

	// Standardinitialisierung
	
	CSingleDocTemplate* pDocTemplate;
	pDocTemplate = new CSingleDocTemplate(
		IDR_MAINFRAME,
		RUNTIME_CLASS(CPopManDoc),
		RUNTIME_CLASS(CMainFrame),       
		RUNTIME_CLASS(CPopManView));
	AddDocTemplate(pDocTemplate);


	if(_tcscmp(m_lpCmdLine, _T("-minimize")) == 0)
		m_nCmdShow = SW_SHOWMINNOACTIVE;
	
	
	
	CCommandLineInfo cmdInfo;
	ParseCommandLine(cmdInfo);
	

	if (!ProcessShellCommand(cmdInfo))
		return FALSE;

	if(m_pMainWnd->IsWindowVisible())
		m_pMainWnd->ShowWindow(SW_SHOW);

	m_pMainWnd->UpdateWindow();
	

	CPopManDoc* pDoc = NULL;

	CFrameWnd* pFrame = dynamic_cast<CFrameWnd*>(m_pMainWnd);
	if(pFrame)
	{
		pDoc = dynamic_cast<CPopManDoc*>(pFrame->GetActiveDocument());
	}


	if(_tcscmp(m_lpCmdLine, _T("-updatesuccess")) == 0)
	{
		CString stMsg;
		CString stVersion;
		stVersion.Format(_T("%d.%d"), cstAppVersion.GetMajor(), cstAppVersion.GetMinor());

		if(cstAppVersion.IsBeta())
		{
			stMsg = StrFormat(i18n("{1} Beta version has been updated successfully!"), _T("s"), (LPCTSTR)stVersion);
		} 
		else
		{
			if(cstAppVersion.GetSub() > 0)
				stVersion = cstAppVersion.GetString();

			stMsg = StrFormat(i18n("You have successfully updated to version {1} !"), _T("s"), (LPCTSTR)stVersion);
		}
	
		AfxMessageBox(stMsg, MB_OK | MB_ICONINFORMATION);
	}
	


	if(!OpenSettingsKey(Key, szAccountsKey))
	{	
		CDlgAccImport Import(pDoc, TRUE);
		if(Import.LoadAccounts(TRUE) > 0)
			Import.DoModal();
	}

    
    if(pDoc) pDoc->OnStartUp();

	return TRUE;
}



int CPopManApp::ExitInstance() 
{
	SaveSettings();
	Settings.Close();
	FreeI18nBuffer();
	DestroyResources(m_MapResourceIcon, m_MapResourceCursor);
	
	return CWinApp::ExitInstance();
}

void CPopManApp::SaveSettings() 
{
	CSettingsKey Key;
	if(OpenSettingsKey(Key, szSettingsKey))
	{
		CString stFile = GetCurrentLangFile();
		int nIdx = stFile.ReverseFind(_T('\\'));
		stFile = stFile.Mid(nIdx+1);
		Key.SetValue(szTranslationFile, stFile);
		Key.SetValue(szPasswordEnabled, m_bPasswordProtect);
		Key.SetValue(szPasswordValue, Crypt(m_stPassword));
		Key.SetValue(szThemeValue, m_stTheme);
	}
}



BOOL CALLBACK CPopManApp::Searcher(HWND hWnd, LPARAM lParam)
{
	DWORD result;
	LRESULT ok = ::SendMessageTimeout(hWnd, UWM_ARE_YOU_ME, 0, 0, 
										SMTO_BLOCK | SMTO_ABORTIFHUNG,
										200,
										&result);
	if(ok == 0)
		return TRUE; 

	if(result == UWM_ARE_YOU_ME)
	{ 
		HWND* target = (HWND*)lParam;
		*target = hWnd;
		return FALSE; // gefunden!
	} 
    return TRUE; // die Suche geht weiter
} 



BOOL CPopManApp::LoadUserDefaultLang()
{
	UINT nLangID = PRIMARYLANGID(::GetUserDefaultLangID());
	if(nLangID == LANG_ENGLISH)
	{
		LoadLanguageFile(_T(""));
		return TRUE;
	}

	CLanguages Languages;
	if(!GetInstalledLanguages(Languages))
		return FALSE;


	for(int n = 0; n < Languages.GetSize(); n++)
	{
		if(Languages[n].nLangID == nLangID)
		{
			if(!Languages[n].stFile.IsEmpty())
			{
				CString stLangPath = GetAppLanguagePath();
				if(stLangPath.Right(1) != _T("\\")) stLangPath += _T('\\');
				stLangPath += Languages[n].stFile;
				return LoadLanguageFile(stLangPath);
			}
		}
	}

	LoadLanguageFile(_T(""));
	return TRUE;	
}

CString CPopManApp::GetAppPath()
{
	TCHAR szPath[MAX_PATH + 1];
	::GetModuleFileName(0, szPath, sizeof(szPath)/sizeof(TCHAR));

	TCHAR* pSlash = _tcsrchr(szPath, _T('\\'));
	if(pSlash != NULL)
	{
		pSlash[0] = _T('\0');
		return szPath;
	}
	
	return _T("");
}

CString CPopManApp::GetAppDataPath()
{
	CString stPath;

	if(m_bLocalStorage)
	{
		stPath = GetAppPath();
		if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
		stPath += _T("Data");
	}
	else
	{
		stPath = GetGlobalAppDataPath();
	}

	DWORD dwAtt = GetFileAttributes(stPath);
	if(dwAtt == 0xFFFFFFFF || (dwAtt & FILE_ATTRIBUTE_DIRECTORY) == 0)
		::CreateDirectory(stPath, NULL);

	return stPath;
}

CString CPopManApp::GetGlobalAppDataPath()
{
	LPITEMIDLIST pidl;

	if(NOERROR != ::SHGetSpecialFolderLocation(NULL, CSIDL_APPDATA, &pidl))
		return GetAppPath();

	TCHAR szPath[MAX_PATH+1];
	if(!::SHGetPathFromIDList(pidl, szPath))
		return GetAppPath();

	LPMALLOC pMalloc;
	if(NOERROR == ::SHGetMalloc(&pMalloc))
	{
		pMalloc->Free(pidl);
		pMalloc->Release();
	}

	CString stPath(szPath);
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += _T("PopMan");

	return stPath;
}

CString CPopManApp::GetAppTempPath()
{
	CString stPath = GetAppDataPath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += _T("temp");
	
	DWORD dwAtt = GetFileAttributes(stPath);
	if(dwAtt == 0xFFFFFFFF || (dwAtt & FILE_ATTRIBUTE_DIRECTORY) == 0)
		::CreateDirectory(stPath, NULL);

	return stPath;
}

CString CPopManApp::GetAppLanguagePath()
{
	CString stPath = GetAppPath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += _T("Languages");
	return stPath;
}

CString CPopManApp::GetAppPluginsPath()
{
	CString stPath = GetAppPath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += _T("Plugins");
	return stPath;
}

BOOL CPopManApp::GetInstalledLanguages(CLanguages& Languages)
{
	Languages.RemoveAll();
	LANGUAGE Item;
	Item.stAuthor = _T("Christian Huebner");
	Item.stEmail = szPopManEmail;
	Item.stComment = _T("PopMan default language");
	Item.stName = _T("English <default>");
	Item.stUpdated = szPopManUpdated;
	Item.nLangID = LANG_ENGLISH;
	Languages.Add(Item);

	WIN32_FIND_DATA FileData;
	HANDLE hHandle = NULL;
	CStringArray Files;

	CString stPath = GetAppLanguagePath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');

	CString stSearch = stPath;
	stSearch += _T("*.lng");

	
	hHandle = FindFirstFile(stSearch, &FileData);
	if(hHandle == INVALID_HANDLE_VALUE)
		return TRUE;

	if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0)
		Files.Add(FileData.cFileName);

	while(FindNextFile(hHandle, &FileData) != 0)
		if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0)
			Files.Add(FileData.cFileName);

	FindClose(hHandle);
	
	
	for(int n = 0; n < Files.GetSize(); n++)
	{
		CString stFile = Files[n];

		CString stExt = stFile.Right(4);
		stExt.MakeLower();
		if(stExt != _T(".lng")) continue;
		
		CFile File(stPath + stFile, CFile::modeRead);

        unsigned int nFileLen = static_cast<unsigned int>(File.GetLength());
		if(nFileLen > 4096) nFileLen = 4096;
		char* pData = new char[nFileLen + 1];
		File.Read(pData, nFileLen);
		pData[nFileLen] = (char)0;
		File.Close();
		
		USES_CONVERSION;
		CString stData = A2T(pData);

		LANGUAGE Item;
		Item.stFile = stFile;
		if(ExtractLanguageInfo(stData, Item))
			Languages.Add(Item);
		
		delete [] pData;
	}

	return TRUE;
}


BOOL CPopManApp::ExtractLanguageInfo(const CString& stData, LANGUAGE &Item)
{
	Item.stName = GetItem(stData, _T("\r\nLanguage="));
	if(Item.stName.IsEmpty()) return FALSE;

	Item.stAuthor = GetItem(stData, _T("\r\nAuthor="));
	if(Item.stAuthor.IsEmpty()) return FALSE;
	
	Item.stComment = GetItem(stData, _T("\r\nComment="));
	if(Item.stComment.IsEmpty()) return FALSE;

	Item.stUpdated = GetItem(stData, _T("\r\nDate="));
	if(Item.stUpdated.IsEmpty()) return FALSE;

	Item.stEmail = GetItem(stData, _T("\r\nEmail="));
	Item.nLangID = _ttoi(GetItem(stData, _T("\r\nLangID=")));
		
	Item.stName.TrimLeft();
	Item.stAuthor.TrimLeft();
	Item.stComment.TrimLeft();
	Item.stUpdated.TrimLeft();
	Item.stEmail.TrimLeft();

	return TRUE;
}

CString CPopManApp::GetItem(const CString& Src, LPCTSTR Search)
{
	int LenS = _tcslen(Search);
	int iStart = Src.Find(Search);
	
	int iEnd = iStart + LenS;

	if(iStart == -1)
		return _T("");

	iEnd = Src.Find(_T("\r\n"), iEnd);
	if(iEnd == -1)
		iEnd = Src.GetLength();
	
	return Src.Mid(iStart + LenS, iEnd - iStart - LenS); 
}

void CPopManApp::BroadcastChangedGUI()
{
	::EnumThreadWindows(::GetCurrentThreadId(), UpdateGUI, 0);
}

BOOL CALLBACK CPopManApp::UpdateGUI(HWND hWnd, LPARAM lParam)
{
	::SendMessage(hWnd, WM_UPDATE_GUI, 0, 0);	
	return TRUE; 
}

CString CPopManApp::GetAppThemesPath()
{
	CString stPath = GetAppPath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += _T("Themes");
	return stPath;
}

void CPopManApp::GetAvailableThemes(CStringArray& themes)
{
	themes.RemoveAll();

	WIN32_FIND_DATA FileData;
	HANDLE hHandle = NULL;

	CString stSearch = GetAppThemesPath();
	if(stSearch.Right(1) != _T("\\")) stSearch += _T('\\');
	stSearch += _T("*");

	
	hHandle = FindFirstFile(stSearch, &FileData);
	if(hHandle == INVALID_HANDLE_VALUE)
		return;

	if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY)
		if(FileData.cFileName[0] != _T('.'))
			themes.Add(FileData.cFileName);

	while(FindNextFile(hHandle, &FileData) != 0)
		if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY)
			if(FileData.cFileName[0] != _T('.'))
				themes.Add(FileData.cFileName);

	FindClose(hHandle);
}


bool CPopManApp::SetTheme(LPCTSTR szTheme)
{
//	TRACE("CPopManApp::SetTheme: %s \n", szTheme);

	CMapIcon mapIcons;
	CMapCursor mapCursors;

	POSITION pos = m_MapResourceIcon.GetStartPosition();
	while(pos) {
		UINT key = 0;
		ICONSET IconSet;
		m_MapResourceIcon.GetNextAssoc(pos, key, IconSet);
		mapIcons[key] = IconSet;
	}
	m_MapResourceIcon.RemoveAll();
	
	pos = m_MapResourceCursor.GetStartPosition();
	while(pos) {
		UINT key = 0;
		HCURSOR hCursor = NULL;
		m_MapResourceCursor.GetNextAssoc(pos, key, hCursor);
		mapCursors[key] = hCursor;
	}
	m_MapResourceCursor.RemoveAll();


	m_stTheme = szTheme;
	BroadcastChangedGUI();


	DestroyResources(mapIcons, mapCursors);
	return true;
}

void CPopManApp::DestroyResources(CMapIcon& mapIcons, CMapCursor& mapCursors)
{
	POSITION pos = mapIcons.GetStartPosition();
	while(pos) {
		UINT key = 0;
		ICONSET IconSet;
		mapIcons.GetNextAssoc(pos, key, IconSet);
		if(IconSet.hDefault) VERIFY(::DestroyIcon(IconSet.hDefault));
		if(IconSet.hSmall)	 VERIFY(::DestroyIcon(IconSet.hSmall));
		if(IconSet.hBig)	 VERIFY(::DestroyIcon(IconSet.hBig));
	}
	
	pos = mapCursors.GetStartPosition();
	while(pos) {
		UINT key = 0;
		HCURSOR hCursor = NULL;
		mapCursors.GetNextAssoc(pos, key, hCursor);
		VERIFY(::DestroyCursor(hCursor));
	}
}


bool CPopManApp::GetCurrentThemePath(CString& stPath)
{
	if(m_stTheme.IsEmpty())
		return false;

	stPath = GetAppThemesPath();
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += m_stTheme;

	DWORD dwAtt = GetFileAttributes(stPath);
	return (dwAtt != 0xFFFFFFFF && (dwAtt & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY);
}

HICON CPopManApp::GetIcon(UINT Icon, RESSIZE resSize)
{
	HICON hIcon = NULL;
	ICONSET IconSet;

	if(m_MapResourceIcon.Lookup(Icon, IconSet)) {
		hIcon = (resSize == RESSIZE::Default ? IconSet.hDefault : 
						(resSize == RESSIZE::Small ? IconSet.hSmall : IconSet.hBig));
		if(hIcon)
			return hIcon;
	}

	int SizeX = 0, SizeY = 0;
	switch(resSize)
	{
	case RESSIZE::Default:
		SizeX = SizeY = 0;
		break;
	case RESSIZE::Small:
		SizeX = SizeY = 16;
		break;
	case RESSIZE::Big:
		SizeX = SizeY = 32;
		break;
	}
	
	CString stPath;
	if(GetCurrentThemePath(stPath))
	{
		CString stFile;

		LPCTSTR szName = MapIconToName(Icon);
		if(szName == NULL)
			stFile.Format(_T("\\%d.ico"), Icon);
		else
			stFile.Format(_T("\\%s.ico"), szName);

		stPath += stFile;

		CFileStatus Status;
		if(CFile::GetStatus(stPath, Status))		
			hIcon = (HICON)::LoadImage(0, stPath, IMAGE_ICON, SizeX, SizeY, LR_LOADFROMFILE);
	}

	if(hIcon == NULL)
		hIcon = (HICON)::LoadImage(AfxGetResourceHandle(), MAKEINTRESOURCE(Icon), IMAGE_ICON, SizeX, SizeY, LR_DEFAULTCOLOR);
   	
	if(hIcon)
	{
		switch(resSize)
		{
		case RESSIZE::Default:
			m_MapResourceIcon[Icon].hDefault = hIcon;
			break;
		case RESSIZE::Small:
			m_MapResourceIcon[Icon].hSmall = hIcon;
			break;
		case RESSIZE::Big:
			m_MapResourceIcon[Icon].hBig = hIcon;
			break;
		}
	}

	return hIcon;
}


CPopManApp::RESNAMEMAP CPopManApp::mapIconName[] =
{
	{ IDR_MAINFRAME,		_T("MainWindow")		},
	{ IDR_MESSAGE,			_T("MessageWindow")		},
	{ IDI_TRAY,				_T("Tray_Mail")			},
	{ IDI_TRAY_ACTION,		_T("Tray_Action")		},
	{ IDI_TRAY_DISABLED,	_T("Tray_Disabled")		},
	{ IDI_TRAY_EMPTY,		_T("Tray_Empty")		},
	{ IDI_TRAY_ERROR,		_T("Tray_Error")		},
	{ IDI_TRAY_EXCLAMATION,	_T("Tray_Exclamation")	},
	{ IDI_TRAY_NEWMAIL,		_T("Tray_NewMail")		},
	{ IDI_TRAY_NUM,			_T("Tray_NumBkg")		},
	{ IDI_TAB_MAIL,			_T("Tab_Mail")			},
	{ IDI_TAB_UNREAD_MAIL,	_T("Tab_UnreadMail")	},
	{ IDI_VIRUS,			_T("Attachment_Virus")	},
	{ IDI_ATTACHMENT,		_T("Attachment")		},
	{ IDI_AUTO_UPDATE,		_T("Dlg_UpdateFiles")	},
	{ IDI_DOWNLOAD,			_T("Dlg_DownloadMail")	},
	{ IDI_DIALUP,			_T("Dlg_DialUp")		},
	{ IDI_LOGGING,			_T("Dlg_Logging")		},
	{ IDI_OPEN_FOLDER,		_T("Dlg_OpenFolder")	},
	{ IDI_PASSWORD,			_T("Dlg_Password")		},
	{ IDI_PREVIEW,			_T("Dlg_Preview")		},
	{ IDI_SORT_ASC,			_T("Sort_Ascending")	},
	{ IDI_SORT_DESC,		_T("Sort_Descending")	},
};


LPCTSTR CPopManApp::MapIconToName(UINT Icon)
{
	for(int i = 0; i < sizeof(mapIconName)/sizeof(mapIconName[0]); ++i)
		if(mapIconName[i].Res == Icon)
			return mapIconName[i].szName;
	return NULL;
}

HICON CPopManApp::GetCursor(UINT Cursor)
{
	HCURSOR hCursor = NULL;

	if(m_MapResourceCursor.Lookup(Cursor, hCursor))
		return hCursor;
	
	CString stPath;
	if(GetCurrentThemePath(stPath))
	{
		CString stFile;

		LPCTSTR szName = MapCursorToName(Cursor);
		if(szName == NULL)
			stFile.Format(_T("\\%d.cur"), Cursor);
		else
			stFile.Format(_T("\\%s.cur"), szName);

		stPath += stFile;

		CFileStatus Status;
		if(CFile::GetStatus(stPath, Status))		
			hCursor = (HCURSOR)::LoadImage(0, stPath, IMAGE_CURSOR, 0, 0, LR_LOADFROMFILE);
	}

	if(hCursor == NULL)
		hCursor = (HCURSOR)::LoadImage(AfxGetResourceHandle(), MAKEINTRESOURCE(Cursor), IMAGE_CURSOR, 0, 0, LR_DEFAULTCOLOR);
   

	if(hCursor)
		m_MapResourceCursor[Cursor] = hCursor;

	return hCursor;
}



CPopManApp::RESNAMEMAP CPopManApp::mapCursorName[] =
{
	{ IDC_DRAG_MOVE,		_T("Drag_Move")		},
	{ IDC_NO_DROP,			_T("No_Drop")		},
	{ IDC_HOVER_LINK,		_T("Hover_Link")	},
};


LPCTSTR CPopManApp::MapCursorToName(UINT Cursor)
{
	for(int i = 0; i < sizeof(mapCursorName)/sizeof(mapCursorName[0]); ++i)
		if(mapCursorName[i].Res == Cursor)
			return mapCursorName[i].szName;
	return NULL;
}


/////////////////////////////////////////////////////////////////////////////
// CAboutDlg-Dialog fr Info ber Anwendung

class CAboutDlg : public CDialog
{
public:
	CAboutDlg();

	//{{AFX_DATA(CAboutDlg)
	enum { IDD = IDD_ABOUTBOX };
	CStatic	m_staticIcon;
	//}}AFX_DATA

	//{{AFX_VIRTUAL(CAboutDlg)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    
	//}}AFX_VIRTUAL

protected:
	CStaticLink	 m_LinkWebsite;
	CStaticLink	 m_LinkEmail;
    CStaticLink	 m_LinkForum;

	//{{AFX_MSG(CAboutDlg)
	virtual BOOL OnInitDialog();
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
	//{{AFX_DATA_INIT(CAboutDlg)
	//}}AFX_DATA_INIT
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAboutDlg)
	DDX_Control(pDX, IDC_STATIC_ABOUT_ICON, m_staticIcon);
	DDX_Check(pDX, IDC_CHECK_UPDATE, CPopManApp::m_bCheckForUpdates);
	//}}AFX_DATA_MAP
}

BOOL CAboutDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();

	SetWindowText(i18n("About PopMan"));
	SetDlgItemText(IDOK, i18n("OK"));

    m_staticIcon.SetIcon(CPopManApp::GetIcon(IDR_MAINFRAME, CPopManApp::RESSIZE::Big));
	
	m_LinkWebsite.SubclassDlgItem(IDC_STATIC_WEBSITE_VAL, this);
	m_LinkEmail.SubclassDlgItem(IDC_STATIC_EMAIL_VAL, this);
    m_LinkForum.SubclassDlgItem(IDC_STATIC_FORUM_VAL, this);
	m_LinkEmail.m_stLink = _T("mailto:");
	m_LinkEmail.m_stLink += szPopManEmail;
	
	
	CString stVersion;
	if(cstAppVersion.IsBeta())
		stVersion.Format(_T("%d.%d Beta %d"), cstAppVersion.GetMajor(), cstAppVersion.GetMinor(), cstAppVersion.GetSub());
	else
		stVersion = cstAppVersion.GetString();

	stVersion = StrFormat(i18n("PopMan, Version {1}"), _T("s"), stVersion);
	SetDlgItemText(IDC_STATIC_VERSION, stVersion);

	SetDlgItemText(IDC_STATIC_WEBSITE, i18n("Website:"));
	SetDlgItemText(IDC_STATIC_EMAIL, i18n("Email:"));
    SetDlgItemText(IDC_STATIC_FORUM, i18n("Forum:"));

	SetDlgItemText(IDC_STATIC_WEBSITE_VAL, szPopManWebsite);
	SetDlgItemText(IDC_STATIC_EMAIL_VAL, szPopManEmail);
    SetDlgItemText(IDC_STATIC_FORUM_VAL, szPopManForum);

	SetDlgItemText(IDC_CHECK_UPDATE, i18n("Check for updates"));

	return TRUE;  
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
	//{{AFX_MSG_MAP(CAboutDlg)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()




void CPopManApp::OnAppAbout()
{
	CAboutDlg aboutDlg;
	aboutDlg.DoModal();
}


void CPopManApp::OnAppHelp() 
{
	TCHAR szHelpFileName[] = _T("PopMan.chm");
	TCHAR szPathName[MAX_PATH] = _T("");

	::GetModuleFileName(NULL, szPathName, MAX_PATH);
	LPTSTR pszFileName = _tcsrchr(szPathName, _T('\\')) + 1;
	*pszFileName = _T('\0');
	
	if(_tcslen(szPathName) + _tcslen(szHelpFileName) < MAX_PATH)
	{
		_tcscat(szPathName, _T("PopMan.chm"));
		//HINSTANCE hInstance = ShellExecute(0, _T("open"), szPathName, 0, 0, SW_SHOWNORMAL);
        ::HtmlHelp(::GetDesktopWindow(), szPathName, HH_DISPLAY_TOPIC, 0);
	}
}

void CPopManApp::OnUpdateAppAbout(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(AfxGetMainWnd()->IsWindowEnabled());	
}




BOOL GetSettingsRootKey(CSettingsKey& Key)
{
	return Settings.GetRootKey(Key);
}

BOOL OpenSettingsKey(CSettingsKey& Key, LPCTSTR szKeyName)
{
	return Settings.OpenSubKey(Key, szKeyName);
}

BOOL CreateSettingsKey(CSettingsKey& Key, LPCTSTR szKeyName)
{
	return Settings.CreateSubKey(Key, szKeyName);
}

BOOL FlushSettings()
{
	return Settings.Flush();
}

BOOL CPopManApp::ImportRegistrySettings(CRegistryKey& KeyIn, CSettingsKey& KeyOut)
{
	int Idx = 0;
	CString stName;
	CRegistryKey SubKeyIn;
	CSettingsKey SubKeyOut;
	while(KeyIn.EnumKey(Idx++, stName))
	{
		if(!SubKeyIn.Open(KeyIn, stName)) continue;
		if(!KeyOut.CreateSubKey(stName, SubKeyOut)) continue;
		ImportRegistrySettings(SubKeyIn, SubKeyOut);
		SubKeyIn.Close();
		SubKeyOut.Close();
	}

	Idx = 0;
	while(KeyIn.EnumValue(Idx++, stName))
	{
		DWORD dwType = 0;
		if(!KeyIn.QueryValueType(stName, dwType)) continue;

		if(dwType == REG_DWORD)
		{
			DWORD dwVal = 0;
			if(KeyIn.QueryValue(stName, dwVal))
				KeyOut.SetValue(stName, dwVal);
		}
		else if(dwType == REG_SZ)
		{
			CString stVal;
			if(KeyIn.QueryValue(stName, stVal))
				KeyOut.SetValue(stName, stVal);
		}
	}
	return TRUE;
}

void CPopManApp::ImportDataFile(LPCTSTR szFile)
{
	CString stFrom = GetGlobalAppDataPath();
	if(stFrom.Right(1) != _T("\\")) stFrom += _T('\\');

	CString stTo = GetAppDataPath();
	if(stTo.Right(1) != _T("\\")) stTo += _T('\\');

	stFrom += szFile;
	stTo += szFile;

	CFileStatus Status;
	if(CFile::GetStatus(stFrom, Status))
		::CopyFile(stFrom, stTo, TRUE);
}
