// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// PluginAccount.cpp
//
////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PluginAccount.h"
#include "PopManDoc.h"
#include "StrFunctions.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


CWnd* CPluginAccount::pThreadWnd = NULL;
DWORD CPluginAccount::dwInstanceCount = 0; 


CPluginAccount::CPluginAccount(CPopManDoc* pDoc, CProtocol& Protocol) : CAccount(pDoc), m_Protocol(Protocol)
{
	m_nTimeOut = 60;
	m_bBusy = FALSE;
	m_bCanceled = FALSE;
	m_bTimedOut = FALSE;
	m_bStopThread = FALSE;
	m_bThreadSuccess = FALSE;

	if(pThreadWnd == NULL)
		pThreadWnd = new CThreadWnd();

	++dwInstanceCount;
}

CPluginAccount::~CPluginAccount()
{
	//TRACE("~CPluginAccount()\n");

	--dwInstanceCount;
	if(dwInstanceCount == 0) {
		delete pThreadWnd;
		pThreadWnd = NULL;
	}
}

BOOL CPluginAccount::BeginSession(UINT ErrLocation, void (*fnThread)(LPVOID))
{
	if(IsBusy())
	{
		ErrorOccured(erBusy, ErrLocation, FALSE);
		return FALSE;
	}
	m_LastError = erNoError;
	m_bCanceled = FALSE;
	m_bTimedOut = FALSE;
	m_bStopThread = FALSE;
	m_bThreadSuccess = FALSE;
	m_bBusy = TRUE;


	CString stPass = m_stPass;
	BOOL    bRemember = FALSE;

	if(stPass.IsEmpty())
		stPass = m_stPassVolatile;

	if(stPass.IsEmpty())
	{
		if(m_pDoc != NULL && !m_pDoc->GetAccountPass(this, stPass, bRemember))
		{
			m_bCanceled = TRUE;
			ErrorOccured(erCanceled);
			return FALSE;
		}
		m_stPassVolatile = stPass;
		if(bRemember) m_stPass = stPass;
	}

	m_Protocol.ResetStartTime();

	CProtocolPlugin::paramsWorkerThread.fnFunction = fnThread;
	CProtocolPlugin::paramsWorkerThread.lParam = this;
    CProtocolPlugin::paramsWorkerThread.Action = CProtocolPlugin::WorkerThreadParams::actionCallFunction;
	if(!SetEvent(CProtocolPlugin::paramsWorkerThread.hActionEvent)) {
        ErrorOccured(erConnectFailed, 1404);
		return FALSE;
	}
	return TRUE;
}


void CPluginAccount::ErrorOccured(ERRORS Error, UINT ErrLocation, BOOL bTerminate)
{
	if(IsWorkerThread()) 
	{
		if(bTerminate)
			m_Protocol.Disconnect();
		
		paramErrorOccured.Error = Error;
		paramErrorOccured.ErrLocation = ErrLocation;
		paramErrorOccured.bTerminate = bTerminate;
		CallFunction(procErrorOccured);
		return;
	}
	
	if(bTerminate)
		m_bStopThread = TRUE;

	m_nErrorLocation = 0;
	m_stServerErrMessage.Empty();


	if(IsCanceled())
		m_LastError = erCanceled;
	else if(IsTimedOut())
		m_LastError = erTimedOut;
	else
	{
		m_LastError = Error;
		m_nErrorLocation = ErrLocation;
	
		m_stServerErrMessage = m_Protocol.LastErrorMessage();
		m_stServerErrMessage.TrimLeft();
		m_stServerErrMessage.TrimRight();	
		m_stServerErrMessage.Replace(_T('\r'), _T(' '));
		m_stServerErrMessage.Replace(_T('\n'), _T(' '));
	}
	
	
	CString stErr;
	stErr.Format(_T("\r\n --> ErrorOccured: Error=%d; ErrLoc=%d; Terminate=%d"), Error, ErrLocation, bTerminate);
	m_pDoc->Log(stErr);


	StateChanged(stError);

	if(bTerminate) {
		m_bBusy = FALSE;
		StateChanged(stIdle);
	}
}

void CPluginAccount::StateChanged(STATES State) 
{
	if(IsWorkerThread()) {
		paramStateChanged.State = State;
		CallFunction(procStateChanged);
		return;
	}

	CAccount::StateChanged(State);
}


void CPluginAccount::Close()  { 
	if(IsBusy()) 
		Cancel(); 
	else {
		BeginSession(1900, wt_Disconnect);
		WaitForComplete();
	}
	StateChanged(stIdle);
}

void CPluginAccount::wt_Disconnect(LPVOID lParam)
{
	CPluginAccount* pAcc = (CPluginAccount*)lParam;
	pAcc->m_bThreadSuccess = TRUE;
	pAcc->m_Protocol.Disconnect();
	pAcc->m_bBusy = FALSE; 
}

#include "OAuth2.h"

BOOL CPluginAccount::ConnectAndLogin(int *pMailsCount)
{
	StateChanged(stConnecting);

	CString stPass = m_stPass;
	if(stPass.IsEmpty()) stPass = m_stPassVolatile;

	m_Protocol.SetDoeventsProc((DWORD)this, CallbackDoevents);

	m_Protocol.SetParameters(m_stParameters);

	const CString Prefix = OAuthPassPrefix;
	if (stPass.Left(Prefix.GetLength()) == Prefix) {
		CString refreshToken = stPass.Mid(Prefix.GetLength());
		if (refreshToken.GetLength() > 0) {
			
			const CTime Now = CTime::GetCurrentTime();
			TRACE(_T("Now = %s\n"), Now.Format("%m%d%Y %H:%M:%S"));
			if (refreshToken == m_RefreshToken && !m_AccessToken.IsEmpty() && m_AccessTokenExpiresAt > Now) {
				TRACE(_T("m_AccessToken still valid till %s\n"), m_AccessTokenExpiresAt.Format("%m%d%Y %H:%M:%S"));
				stPass = m_AccessToken;
			}
			else {

				TRACE(_T("m_AccessToken NOT valid\n"));

				int expires_in = 0;
				CString accessToken = Google_GetOAuth2Token(refreshToken, expires_in);
				if (!accessToken.IsEmpty()) {
					m_AccessToken = accessToken;
					m_RefreshToken = refreshToken;
					m_AccessTokenExpiresAt = Now + CTimeSpan(0, 0, 0, expires_in);
					TRACE(_T("Got new m_AccessToken valid until %s\n"), m_AccessTokenExpiresAt.Format("%m%d%Y %H:%M:%S"));
					stPass = m_AccessToken;
				}
			}
		}
	}

	if(!m_Protocol.Connect(m_stServer, m_nPort, m_stUser, stPass, m_nTimeOut))
	{
		ErrorOccured(erConnectFailed, 1401);
		return FALSE;
	}
	if(IsAborted()) return FALSE;

	StateChanged(stLoggedIn);


	int MailCount = m_Protocol.CheckMessages();
	if(MailCount < 0 || !m_Protocol.LastErrorMessage().IsEmpty())
	{
		ErrorOccured(erResponseInvalid, 1402);
		return FALSE;
	}
	if(IsAborted()) return FALSE;

	if(pMailsCount != NULL)
		*pMailsCount = MailCount;


	if(!m_Protocol.UIDLs(m_UIDLCache))
	{
		ErrorOccured(erResponseInvalid, 1403);
		return FALSE;
	}
	
	if(IsAborted()) return FALSE;

	return TRUE;
}



void CPluginAccount::OnNewMail(CMail* pMail, bool isNew, bool& bDelete)
{
	if(IsWorkerThread()) {
		paramOnNewMail.pMail = pMail;
		paramOnNewMail.bNew = isNew;
		paramOnNewMail.bDelete = &bDelete;
		CallFunction(procOnNewMail);
		return;
	}

	m_pDoc->OnNewMail(pMail, isNew, bDelete);
}


bool CPluginAccount::IsAborted()
{
	if(!m_bStopThread) return false;

	m_Protocol.Disconnect();
	
	if(m_bCanceled)
		ErrorOccured(erCanceled);
	
	return true;
}


BOOL CPluginAccount::GetMailSize(int Idx, int& Size)
{
	if(IsAborted()) return FALSE;

	Size = m_Protocol.MessageSize(Idx);
	if(!m_Protocol.LastErrorMessage().IsEmpty()) {
		ErrorOccured(erResponseInvalid, 1501);
		return FALSE;
	}
	if(IsAborted()) return FALSE;
	return TRUE;
}


BOOL CPluginAccount::RetrieveMail(int Idx, BOOL bComplete, CString &stSrc)
{
	if(IsAborted()) return FALSE;

	bool bRet = false;
	if(bComplete)
		bRet = m_Protocol.RetrieveRaw(Idx, stSrc);
	else
		bRet = m_Protocol.RetrieveTop(Idx, NumberOfLinesForIncompleteMail(), stSrc);

	
	if(!bRet) {
		
		//TRACE("RetrieveMail Error=%s \n", (LPCSTR)m_Protocol.LastErrorMessage());
		ErrorOccured(erResponseInvalid, 1502);
	}
	
	if(IsAborted()) return FALSE;

	return bRet;
}


BOOL CPluginAccount::DeleteMail(int Idx)
{
	if(IsAborted()) return FALSE;

	m_nCurrentMail++;
	StateChanged(stDeletingMails);

	if(!m_Protocol.Delete(Idx)) {
		ErrorOccured(erResponseInvalid, 1503);
		return FALSE;
	}
	if(IsAborted()) return FALSE;
	return TRUE;
}

BOOL CPluginAccount::MarkMailAsSeen(int Idx)
{
	if (IsAborted()) return FALSE;

	m_nCurrentMail++;
	StateChanged(stMarkAsSeen);

	if (!m_Protocol.MarkAsSeen(Idx)) {
		ErrorOccured(erResponseInvalid, 1504);
		return FALSE;
	}
	if (IsAborted()) return FALSE;
	return TRUE;
}


//////////////////////////////////////////////////////////////////////////

BOOL CPluginAccount::ListMails(BOOL bHardReload)
{
	ListMails_bHardReload = bHardReload;

	if(!BeginSession(1100, wt_ListMails)) return FALSE;

	BOOL bRes = WaitForComplete();
	if(bRes)
		m_LastCheckTime = COleDateTime::GetCurrentTime();

	return bRes;
}

void CPluginAccount::wt_ListMails(LPVOID lParam)
{
	CPluginAccount* pAcc = (CPluginAccount*)lParam;
	pAcc->m_bThreadSuccess = pAcc->OnListMails(pAcc->ListMails_bHardReload);
	pAcc->m_bBusy = FALSE; 
}

BOOL CPluginAccount::OnListMails(BOOL bHardReload)
{
	int nMailsOnServer = 0;
	if(!ConnectAndLogin(&nMailsOnServer))
		return FALSE;
	
	if(bHardReload)
		ClearMails();
	
	int		 n = 0;
	CString  stUIDL;
	int		 nNewMails = 0;

	if(bHardReload == FALSE)
	{
		// How many new mails on server?
		for(n = 1; n <= nMailsOnServer; n++)
		{
			GetMailUIDL(n, stUIDL);
			if(!IsMailInList(stUIDL))
    			nNewMails++;
		}

		// Remove all mails that are not on the server anymore
		CMails mList;

		POSITION pos = m_Mails.GetHeadPosition();
		while(pos != NULL)
		{
			POSITION posDel = pos;
			CMail* pMail = m_Mails.GetNext(pos);
			if(-1 == FindInStrArray(m_UIDLCache, pMail->GetUIDL())) {
				mList.AddTail(pMail);
				m_Mails.RemoveAt(posDel);
			}
		}


		paramDocRemoveMails = &mList;
		CallFunction(procDocRemoveMails);


		// delete mails:
		pos = mList.GetHeadPosition();
		while(pos != NULL)
		{
			CMail* pMail = mList.GetNext(pos);
			delete pMail;
		}
	}

	if(bHardReload)
		m_nMailsCount = nMailsOnServer;
	else 
    {
		m_nMailsCount = nNewMails;

        if(m_bTopMessages && nNewMails > m_nTopMessagesCount)
            m_nMailsCount = m_nTopMessagesCount;
    }

	m_nCurrentMail = 1;

		
	bool bCheckForDoubleMails =(!bHardReload && m_Mails.GetCount());

	CList<int, int> autoDeleteList;
	
    int mailCount = 1;

    for(n = nMailsOnServer; n >= 1; n--, mailCount++)
	//for(n = 1; n <= nMailsOnServer; n++)
	{
		if(IsAborted()) return FALSE;

        if(!bHardReload && m_bTopMessages && mailCount > m_nTopMessagesCount)
            break;

		BOOL	 bComplete = FALSE;
		CString  stSource;
		int		 nSize = 0;
			
		if(!GetMailUIDL(n, stUIDL))
			return FALSE;

		// if mail is already loaded, continue with the next
		if(bCheckForDoubleMails && IsMailInList(stUIDL))
			continue;

		if(!GetMailSize(n, nSize))
			return FALSE;

		// // wenn Size <= m_MaxLoadSize -> vollstndig runterladen
		//bComplete = !m_bLimitLoadSize || (nSize <= m_nMaxLoadSize);
		
        bComplete = ( (m_RetrieveMode == RETRIEVE_MODE::AllComplete) || 
            ((m_RetrieveMode == RETRIEVE_MODE::SmallComplete) && ((unsigned)nSize < (m_nMaxSizeOfSmallMails * 1024))) );
		
		StateChanged(stLoadingMails);
		if(!RetrieveMail(n, bComplete, stSource))
			return FALSE;

	

		//stSource = "";
		//CFile File;

		//if (!File.Open(_T("D:\\PopMan\\voicemail.eml"), CFile::shareDenyNone | CFile::modeRead | CFile::typeBinary))
		//	return false;

		//const unsigned int nFileSize = static_cast<unsigned int>(File.SeekToEnd());
		//File.Seek(0, CFile::begin);

		//char* pBuffer = new char[nFileSize + 1];  // stSource.GetBuffer(nFileSize + 1)

		//File.Read(pBuffer, nFileSize);
		//File.Close();

		//pBuffer[nFileSize] = 0;

		//for (unsigned int i = 0; i < nFileSize; ++i) {
		//	stSource.AppendChar(pBuffer[i]);
		//}
		//nSize = nFileSize;
		//bComplete = TRUE;




		CMail* pMail = new CMail(this);
		if( pMail->Init(stSource, nSize, stUIDL, bComplete) )
		{
			m_Mails.AddHead(pMail);
			m_nCurrentMail++;

			
			// check, if mail is read
			int nPos = m_MailCache.FindSignature(pMail->GetSignature());
			if(nPos == -1)
				pMail->SetRead(FALSE);
			else
				pMail->SetRead(m_MailCache[nPos].m_bRead);
				
			
			bool bDelete = false;
			OnNewMail(pMail, (nPos == -1), bDelete);
			if(bDelete) {
				autoDeleteList.AddHead(n);
				delete m_Mails.RemoveHead();
			}
		}
	}

	m_nMailsCount = autoDeleteList.GetCount();
	m_nCurrentMail = 0;

	POSITION pos = autoDeleteList.GetHeadPosition();
	while(pos != NULL) {
		int idx = autoDeleteList.GetNext(pos);
		if(!DeleteMail(idx))
			break;
//		TRACE("AutoDelete Idx=%d \n", idx);
	}

		
	m_nCurrentMail = 0;
	m_nMailsCount = 0;

	m_Protocol.Disconnect();

    if(bHardReload || !m_bTopMessages || m_MailCache.GetSize() > max(500, 2*nMailsOnServer))
	    UpdateMailCache();
	SyncMailListWithUIDLs();
	
	StateChanged(stIdle);
	return TRUE;
}



//////////////////////////////////////////////////////////////////////////


BOOL CPluginAccount::DeleteMails(CMails* pMailList)
{
	DeleteMails_pMailList = pMailList;

	if(!BeginSession(1200, wt_DeleteMails)) return FALSE;
	return WaitForComplete();
}

void CPluginAccount::wt_DeleteMails(LPVOID lParam)
{
	CPluginAccount* pAcc = (CPluginAccount*)lParam;
	pAcc->m_bThreadSuccess = pAcc->OnDeleteMails(pAcc->DeleteMails_pMailList);
	pAcc->m_bBusy = FALSE; 
}


BOOL CPluginAccount::OnDeleteMails(CMails *pMailList)
{
	m_nMailsCount = pMailList->GetCount();
	m_nCurrentMail = 0;

	int nMailsOnServer = 0;
	if(!ConnectAndLogin(&nMailsOnServer))
		return FALSE;

	if(nMailsOnServer > 0) 
	{
		POSITION pos = pMailList->GetHeadPosition();
		while(pos != NULL)
		{
			CMail* pMail = pMailList->GetNext(pos);

			int nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
			if(nIdx < 0 || nIdx >= nMailsOnServer) 
			{
			//	LogMailNotFound(pMail);
				continue;
			}

//			TRACE("Lsche Mail Nr. %d \n", nIdx+1);
			if(!DeleteMail(nIdx+1))
				return FALSE;
		}
	}


	m_Protocol.Disconnect();

	// die gelschten Mails aus der Auflistung entfernen:
	POSITION pos = pMailList->GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = pMailList->GetNext(pos);

		int nPos = m_MailCache.FindSignature(pMail->GetSignature());
		if(nPos > -1) m_MailCache.RemoveAt(nPos);

		POSITION posDel = m_Mails.Find(pMail);
		if(posDel != NULL)
		{
			m_Mails.RemoveAt(posDel);
			delete pMail;
		}
	}

	StateChanged(stIdle);
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////


BOOL CPluginAccount::MarkMailsAsSeen(CMails* pMailList)
{
	MarkMails_pMailList = pMailList;

	if (!BeginSession(1200, wt_MarkMails)) return FALSE;
	return WaitForComplete();
}

void CPluginAccount::wt_MarkMails(LPVOID lParam)
{
	CPluginAccount* pAcc = (CPluginAccount*)lParam;
	pAcc->m_bThreadSuccess = pAcc->OnMarkMails(pAcc->MarkMails_pMailList);
	pAcc->m_bBusy = FALSE;
}


BOOL CPluginAccount::OnMarkMails(CMails *pMailList)
{
	m_nMailsCount = pMailList->GetCount();
	m_nCurrentMail = 0;

	int nMailsOnServer = 0;
	if (!ConnectAndLogin(&nMailsOnServer))
		return FALSE;

	if (nMailsOnServer > 0)
	{
		POSITION pos = pMailList->GetHeadPosition();
		while (pos != NULL)
		{
			CMail* pMail = pMailList->GetNext(pos);

			int nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
			if (nIdx < 0 || nIdx >= nMailsOnServer)
			{
				//	LogMailNotFound(pMail);
				continue;
			}

			//			TRACE("Lsche Mail Nr. %d \n", nIdx+1);
			if (!MarkMailAsSeen(nIdx + 1))
				return FALSE;
		}
	}


	m_Protocol.Disconnect();

	// die gelschten Mails aus der Auflistung entfernen:
	POSITION pos = pMailList->GetHeadPosition();
	while (pos != NULL)
	{
		CMail* pMail = pMailList->GetNext(pos);

		int nPos = m_MailCache.FindSignature(pMail->GetSignature());
		if (nPos > -1) m_MailCache.RemoveAt(nPos);

		POSITION posDel = m_Mails.Find(pMail);
		if (posDel != NULL)
		{
			m_Mails.RemoveAt(posDel);
			delete pMail;
		}
	}

	StateChanged(stIdle);
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////


BOOL CPluginAccount::DownloadMail(CMail* pMail, BOOL bKeepConnectionAlive)
{
	DownloadMail_pMail = pMail;
	DownloadMail_bKeepConnectionAlive = bKeepConnectionAlive;

	if(!BeginSession(1300, wt_DownloadMail)) return FALSE;
	return WaitForComplete();
}

void CPluginAccount::wt_DownloadMail(LPVOID lParam)
{
	CPluginAccount* pAcc = (CPluginAccount*)lParam;
	pAcc->m_bThreadSuccess = pAcc->OnDownloadMail(pAcc->DownloadMail_pMail, pAcc->DownloadMail_bKeepConnectionAlive);
	pAcc->m_bBusy = FALSE; 
}


BOOL CPluginAccount::OnDownloadMail(CMail *pMail, BOOL bKeepConnectionAlive)
{
	if(!m_Protocol.Connected())
		if(!ConnectAndLogin(NULL))
			return FALSE;

	const int nIndex = 1 + FindInStrArray(m_UIDLCache, pMail->GetUIDL());

	if(nIndex == 0)
	{
		ErrorOccured(erMailMissing, 1302);
		return FALSE;
	}

	StateChanged(stLoadingMail);
	CString stSource;
	if(!RetrieveMail(nIndex, TRUE, stSource))
		return FALSE;
	
	pMail->CompleteSource(stSource);

	if(!bKeepConnectionAlive)
	{
		m_Protocol.Disconnect();
		StateChanged(stIdle);
	}

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////

/*
BOOL CPluginAccount::MailboxChanged()
{
	if(!BeginSession(1400, wt_MailboxChanged)) return FALSE;
	return WaitForComplete();
}

void CPluginAccount::wt_MailboxChanged(LPVOID lParam)
{
	CPluginAccount* pAcc = (CPluginAccount*)lParam;
	pAcc->m_bThreadSuccess = pAcc->OnMailboxChanged();
	pAcc->m_bBusy = FALSE; 
}

BOOL CPluginAccount::OnMailboxChanged()
{
	int nMailsOnServer = 0;
	if(!ConnectAndLogin(&nMailsOnServer))
		return FALSE;

	return nMailsOnServer != m_Mails.GetCount();
}

*/

//////////////////////////////////////////////////////////////////////////

BOOL CPluginAccount::IsMailInList(const CString& stUIDL)
{
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		if(pMail->GetUIDL() == stUIDL)
			return TRUE;
	}	
	return FALSE;
}


BOOL CPluginAccount::WaitForComplete()
{
	bool bKilled = false;

	while(IsBusy())
	{
		if(IsCanceled()) {
			m_bStopThread = TRUE;  // Thread abbrechen?
			if(!bKilled)
				m_Protocol.Kill();
			bKilled = true;
			// ErrorOccured(erCanceled);
			// return FALSE;
		}

		COleDateTime now = COleDateTime::GetCurrentTime();
		COleDateTimeSpan diff = now - m_Protocol.GetStartTime();
		
		// give 5 seconds more, because it's better when the plugin timeouts itself
		if(diff.GetSeconds() > m_nTimeOut + 5) {
			m_bTimedOut = TRUE;
			m_bStopThread = TRUE; // Thread abbrechen?
			m_Protocol.Kill();
			// ErrorOccured(erTimedOut);
			// return FALSE;
		}

		Sleep(5);  // avoid 100% CPU time

		MSG msg;
		while(::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE)) 
		{ 
			AfxGetApp()->OnIdle(0);
			if(!AfxGetApp()->PumpMessage()) 
				return FALSE; 
		} 	
	}

	return m_bThreadSuccess;
}



//////////////////////////////////////////////////////////////////////////


BEGIN_MESSAGE_MAP(CPluginAccount::CThreadWnd, CWnd)
	ON_MESSAGE(WM_CHANGE_THREAD, CPluginAccount::CThreadWnd::OnChangeThread)
END_MESSAGE_MAP()



LRESULT CPluginAccount::CThreadWnd::OnChangeThread(WPARAM wParam, LPARAM lParam)
{
	CPluginAccount * pAcc = (CPluginAccount *)lParam; 
	if(pAcc == NULL) return 0;


	switch(wParam) {
	case procErrorOccured:

		pAcc->ErrorOccured(pAcc->paramErrorOccured.Error, pAcc->paramErrorOccured.ErrLocation, pAcc->paramErrorOccured.bTerminate);
		break;

	case procStateChanged:
		pAcc->StateChanged(pAcc->paramStateChanged.State);
		break;

	case procDocRemoveMails:

		pAcc->m_pDoc->RemoveMails(pAcc->paramDocRemoveMails);
		break;

	case procOnNewMail:
		
		pAcc->OnNewMail(pAcc->paramOnNewMail.pMail, pAcc->paramOnNewMail.bNew, *(pAcc->paramOnNewMail.bDelete));
		break;

	}

    return 0;
}

bool CPluginAccount::IsWorkerThread()
{
	return ::GetCurrentThreadId() == CProtocolPlugin::paramsWorkerThread.ThreadID;
}

LRESULT CPluginAccount::CallFunction(ThreadedProcs procID)
{
	return pThreadWnd->SendMessage(WM_CHANGE_THREAD, procID, (LPARAM)this);
}


void CPluginAccount::CallbackDoevents(DWORD dwContext)
{
	CPluginAccount * pAcc = (CPluginAccount *)dwContext; 
	if(pAcc == NULL) return;

	if(pAcc->m_bStopThread)
		pAcc->m_Protocol.Disconnect();
}



BOOL CPluginAccount::ReadSettings(const CSettingsKey& settings)
{	
	if(!CAccount::ReadSettings(settings))
		return FALSE;

	CString stParams;
	if(settings.QueryValue(szAccParametersValue, stParams))
		SetParameters(stParams);

	return TRUE;
}


void CPluginAccount::SaveSettings(CSettingsKey& settings) const
{
	CAccount::SaveSettings(settings);

	settings.SetValue(szAccParametersValue, m_stParameters);
}