// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Plugin.h
//
////////////////////////////////////////////////////////////////////////////////


#if !defined(AFX_PLUGIN_H_)
#define AFX_PLUGIN_H_

#if _MSC_VER > 1000
#pragma once
#endif


#include <afxtempl.h>
#include <afxmt.h>


class CPlugin : public CObject  
{
public:

	CPlugin() : m_hPlugin(NULL), m_nVersion(0), m_fnInit(NULL), m_fnFreePChar(NULL), m_fnUnload(NULL) { }

	virtual ~CPlugin()
	{
		Unload();
	}

	enum PluginType {
		piNotify		=  0, 
		piProtocol		=  1, 
		piRuleAction	=  2,
	};

	virtual BOOL Load(LPCTSTR szPluginFile);
	virtual BOOL Unload();

	void Plugin_Init() {
		if(m_fnInit != NULL) m_fnInit();
	}

	void Plugin_Unload() {
		if(m_fnUnload != NULL) m_fnUnload();
	}

	void FreePChar(LPCSTR* pStr) {
		if(m_fnFreePChar != NULL && pStr != NULL && *pStr != NULL)
			m_fnFreePChar(pStr);
	}

	int GetVersion()  const {  return m_nVersion; }

	PluginType GetType() const {  return m_Type;  }
	
	const CString& GetName() const { return m_stName;  }

	const CString& GetDLLName() const { return m_stDLLName;  }


protected:
	HINSTANCE	m_hPlugin;
	int			m_nVersion;
	CString		m_stName;
	CString		m_stDLLName;
	PluginType  m_Type;

	struct ShortString
	{
		unsigned char len;
		char          str[255];
	}; 

	typedef void (WINAPI *FNInit)();
	FNInit m_fnInit;

	typedef void (WINAPI *FNFreePChar)(LPCSTR*);
	FNFreePChar m_fnFreePChar;

	typedef void (WINAPI *FNUnload)();
	FNUnload m_fnUnload;

	static CString ShortStringToCString(const ShortString& shortStr) 
	{
		CString cStr;
		LPTSTR pStr = cStr.GetBuffer(shortStr.len + 1);
	#ifdef _UNICODE
		MultiByteToWideChar(CP_ACP, 0, shortStr.str, shortStr.len, pStr, shortStr.len);
	#else
		strncpy(pStr, shortStr.str, shortStr.len);
	#endif
		cStr.ReleaseBuffer(shortStr.len);
		return cStr;
	}


};




class CProtocol;

class CProtocols : public CTypedPtrList<CPtrList, CProtocol*>
{
public:

    CProtocol* GetProtocolByName(const CString& name);
};


class CProtocolPlugin : public CPlugin 
{
	friend class CProtocol;
	friend class CPluginAccount;

protected:
	CProtocolPlugin() :
		m_fnProtocols(NULL),
		m_fnConnect(NULL),
		m_fnDisconnect(NULL),
		m_fnConnected(NULL),
		m_fnCheckMessages(NULL),
		m_fnRetrieveHeader(NULL),
		m_fnRetrieveRaw(NULL),
		m_fnRetriveTop(NULL),
		m_fnRetrieveMsgSize(NULL),
		m_fnUIDL(NULL),
		m_fnDelete(NULL),
		m_fnLastErrorMsg(NULL),
		m_fnSetDoeventsProc(NULL),
		m_fnKill(NULL),
		m_fnGetDefaultParameters(NULL),
		m_fnSetParameters(NULL),
		m_fnValidateParameters(NULL)
	{
		++dwInstanceCount;
	}

	virtual BOOL Load(LPCTSTR szPluginFile);
	virtual BOOL Unload();

public:

	static CProtocolPlugin* LoadProtocolPlugin(LPCTSTR szPlugin);
	
	virtual ~CProtocolPlugin();

	const CProtocols&  Protocols();

protected:
	CProtocols m_Protocols;

	typedef void (WINAPI *FNProtocols)(ShortString*);
	typedef void (WINAPI *FNConnect)(LPCSTR Server, int Port, LPCSTR Protocol, LPCSTR UserName, LPCSTR Password, int Timeout);
	typedef void (WINAPI *FNDisconnect)();
	typedef bool (WINAPI *FNConnected)();
	typedef int  (WINAPI *FNCheckMessages)();
	typedef bool (WINAPI *FNRetrieveHeader)(int MsgNum, LPCSTR* pHeader);
	typedef bool (WINAPI *FNRetrieveRaw)(int MsgNum, LPCSTR* pRawMsg);
	typedef bool (WINAPI *FNRetrieveTop)(int MsgNum, int LineCount, LPCSTR* pDest);
	typedef int  (WINAPI *FNRetrieveMsgSize)(int MsgNum);
	typedef bool (WINAPI *FNUIDL)(LPCSTR* pUIDL, int MsgNum);
	typedef bool (WINAPI *FNDelete)(int MsgNum);
	typedef bool (WINAPI *FNMarkAsSeen)(int MsgNum);
	typedef LPCSTR (WINAPI *FNLastErrorMsg)();
	typedef void (WINAPI *FNDoeventsProc)(DWORD Context, DWORD ProcAddress);
	typedef void (WINAPI *FNKill)();
	typedef LPCSTR (WINAPI *FNGetDefaultParameters)();
	typedef bool (WINAPI *FNSetParameters)(LPCSTR params);
	typedef bool (WINAPI *FNValidateParameters)(LPCSTR params);

	FNProtocols			m_fnProtocols;
	FNConnect			m_fnConnect;
	FNDisconnect		m_fnDisconnect;
	FNConnected			m_fnConnected;
	FNCheckMessages		m_fnCheckMessages;
	FNRetrieveHeader	m_fnRetrieveHeader;
	FNRetrieveRaw		m_fnRetrieveRaw;
	FNRetrieveTop		m_fnRetriveTop;
	FNRetrieveMsgSize	m_fnRetrieveMsgSize;
	FNUIDL				m_fnUIDL;
	FNDelete			m_fnDelete;
	FNMarkAsSeen		m_fnMarkAsSeen;
	FNLastErrorMsg		m_fnLastErrorMsg;
	FNDoeventsProc		m_fnSetDoeventsProc;
	FNKill				m_fnKill;
	FNGetDefaultParameters m_fnGetDefaultParameters;
	FNSetParameters		m_fnSetParameters;
	FNValidateParameters m_fnValidateParameters;

protected:
	static struct WorkerThreadParams 
	{
		UINT ThreadID;
		HANDLE hActionEvent;
		HANDLE hLoadCompleteEvent;
		enum {  actionTerminate, 
				actionLoadPlugin,
				actionCallFunction, 
		} Action;

		CProtocolPlugin* pLoadPlugin;
		LPCTSTR szLoadParam;
		BOOL	bLoadResult;
		void (*fnFunction)(LPVOID lParam);
		LPVOID lParam;
	} paramsWorkerThread;

	static UINT WorkerThread(LPVOID lParam);

	static DWORD dwInstanceCount;
	static CWinThread* pWorkerThread;

};


class CProtocol : public CObject {

	friend class CProtocolPlugin;

public:
	CProtocol(CProtocolPlugin* pPlugin, const CString& Name, int Port, bool needsServerPort) 
		: m_pPlugin(pPlugin), m_stName(Name), m_nDefaultPort(Port), m_bNeedsServerAndPort(needsServerPort)
	{
		ResetStartTime();
	}
public:
	virtual ~CProtocol() { m_pPlugin = NULL; }

	const CString& GetName()   const	{ return m_stName; }
	int   GetDefaultPort()	   const	{ return m_nDefaultPort; }
	bool  NeedsServerAndPort() const	{ return m_bNeedsServerAndPort; }

	bool Connect(LPCTSTR Server, int Port, LPCTSTR UserName, LPCTSTR Password, int Timeout);
	void Disconnect();
	bool Connected();
	int  CheckMessages();
	bool RetrieveHeader(int MsgNum, CString& stHeader);
	bool RetrieveRaw(int MsgNum, CString& stRawMsg);
	bool RetrieveTop(int MsgNum, int LineCount, CString& stTop);
	int  MessageSize(int MsgNum);
	bool UIDL(CString& stUIDL, int MsgNum);
	bool UIDLs(CStringArray& arrayStr);
	bool Delete(int MsgNum);
	bool MarkAsSeen(int MsgNum);
	CString LastErrorMessage();

	void SetDoeventsProc(DWORD dwContext, void (WINAPI *fnDoevents)(DWORD));
	void Kill();

	bool SupportsParameters() const { 
        return m_pPlugin==NULL || m_pPlugin->m_fnSetParameters != NULL; 
    };

	CString GetDefaultParameters() const;
	bool SetParameters(LPCTSTR szParams);
	bool ValidateParameters(LPCTSTR szParams) const;

	const CProtocolPlugin* GetPlugin() const { return m_pPlugin; }

	COleDateTime GetStartTime() {
		m_CriticalSec.Lock();
		COleDateTime start = m_StartTime;
		m_CriticalSec.Unlock();
		return start;
	}

	void ResetStartTime() {
		m_CriticalSec.Lock();
		m_StartTime = COleDateTime::GetCurrentTime();
		m_CriticalSec.Unlock();
	}

protected:
	COleDateTime  m_StartTime;
	CCriticalSection m_CriticalSec;

	CProtocolPlugin* m_pPlugin;
	int			m_nDefaultPort;
	CString		m_stName;
	bool		m_bNeedsServerAndPort;

	static void LPCSTRToCString(LPCSTR pStr, CString& stString);

};


#endif // !defined(AFX_PLUGIN_H_)
