// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Plugin.cpp
//
////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Plugin.h"
#include "StrFunctions.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif



BOOL CPlugin::Load(LPCTSTR szPluginFile)
{
	m_hPlugin = LoadLibrary(szPluginFile);
	if(m_hPlugin == NULL) return FALSE;


	typedef int (WINAPI *FNInterfaceVersion)();
	FNInterfaceVersion fnVersion = (FNInterfaceVersion)::GetProcAddress(m_hPlugin, "InterfaceVersion");
	if(fnVersion == NULL) {
		Unload();
		return FALSE;
	}
	m_nVersion = fnVersion();


	
//	typedef ShortString (WINAPI *FNPluginName)();
	typedef void (WINAPI *FNPluginName)(ShortString*);
	FNPluginName fnName = (FNPluginName)::GetProcAddress(m_hPlugin, "PluginName");
	if(fnName == NULL) {
		Unload();
		return FALSE;
	}
	ShortString name;
	fnName(&name);

	m_stName = ShortStringToCString(name);

	typedef int (WINAPI *FNPluginType)();
	FNPluginType fnType = (FNPluginType)::GetProcAddress(m_hPlugin, "PluginType");
	if(fnType == NULL) {
		Unload();
		return FALSE;
	}
	m_Type = (PluginType)(unsigned char)fnType();


	
	m_fnInit = (FNInit)::GetProcAddress(m_hPlugin, "Init");
	m_fnFreePChar = (FNFreePChar)::GetProcAddress(m_hPlugin, "FreePChar");
	m_fnUnload = (FNUnload)::GetProcAddress(m_hPlugin, "Unload");


	if(m_fnInit != NULL)
		m_fnInit();


	m_stDLLName = szPluginFile;
	int idx = m_stDLLName.ReverseFind(_T('\\'));
	if(idx > -1) 
		m_stDLLName = m_stDLLName.Mid(idx+1);
	

	return TRUE;
}

BOOL CPlugin::Unload() 
{
	if(m_fnUnload != NULL)
		m_fnUnload();

	m_nVersion = 0;
	m_stName.Empty();
	m_stDLLName.Empty();

	m_fnInit = NULL;
	m_fnFreePChar = NULL;
	m_fnUnload = NULL;
	
	if(m_hPlugin == NULL)
		return TRUE;

	BOOL ret = FreeLibrary(m_hPlugin);

	m_hPlugin = NULL;

	return ret;
}



///////////////////////////////////////////////////////////////////////////

CProtocolPlugin::WorkerThreadParams CProtocolPlugin::paramsWorkerThread = {0};
DWORD CProtocolPlugin::dwInstanceCount = 0; 
CWinThread* CProtocolPlugin::pWorkerThread = NULL;


CProtocolPlugin* CProtocolPlugin::LoadProtocolPlugin(LPCTSTR szPlugin)
{
	if(dwInstanceCount == 0) 
	{
		paramsWorkerThread.hActionEvent = CreateEvent(0, FALSE, FALSE, NULL);
		if(paramsWorkerThread.hActionEvent == NULL) return NULL;

		paramsWorkerThread.hLoadCompleteEvent = CreateEvent(0, FALSE, FALSE, NULL);
		if(paramsWorkerThread.hLoadCompleteEvent == NULL) return NULL;

		pWorkerThread = AfxBeginThread(WorkerThread, &paramsWorkerThread);
		if(pWorkerThread == NULL) return NULL;

		paramsWorkerThread.ThreadID = pWorkerThread->m_nThreadID;
	}

	CProtocolPlugin* pPlugin = new CProtocolPlugin();
	paramsWorkerThread.Action = WorkerThreadParams::actionLoadPlugin;
	paramsWorkerThread.bLoadResult = FALSE;
	paramsWorkerThread.szLoadParam = szPlugin;
	paramsWorkerThread.pLoadPlugin = pPlugin;

	SetEvent(paramsWorkerThread.hActionEvent);
	if(WAIT_OBJECT_0 != WaitForSingleObject(paramsWorkerThread.hLoadCompleteEvent, INFINITE))
	{
		delete pPlugin;
		return NULL;
	}

	if(paramsWorkerThread.bLoadResult == TRUE)
		return pPlugin;
	
	delete pPlugin;
	return NULL;
}


CProtocolPlugin::~CProtocolPlugin()
{
	Unload();

	--dwInstanceCount;
	if(dwInstanceCount == 0) {

		paramsWorkerThread.Action = WorkerThreadParams::actionTerminate;
		SetEvent(paramsWorkerThread.hActionEvent);
		Sleep(5);

	}
}



BOOL CProtocolPlugin::Load(LPCTSTR szPluginFile)
{
	if(CPlugin::Load(szPluginFile) == FALSE)
		return FALSE;

	if(m_Type != PluginType::piProtocol) {
		Unload();
		return FALSE;
	}

	m_fnProtocols		= (FNProtocols)::GetProcAddress(m_hPlugin, "Protocols");
	m_fnConnect			= (FNConnect)::GetProcAddress(m_hPlugin, "Connect");
	m_fnDisconnect		= (FNDisconnect)::GetProcAddress(m_hPlugin, "Disconnect");
	m_fnConnected		= (FNConnected)::GetProcAddress(m_hPlugin, "Connected");
	m_fnCheckMessages	= (FNCheckMessages)::GetProcAddress(m_hPlugin, "CheckMessages");
	m_fnRetrieveHeader	= (FNRetrieveHeader)::GetProcAddress(m_hPlugin, "RetrieveHeader");
	m_fnRetrieveRaw		= (FNRetrieveRaw)::GetProcAddress(m_hPlugin, "RetrieveRaw");
	m_fnRetriveTop		= (FNRetrieveTop)::GetProcAddress(m_hPlugin, "RetrieveTop");
	m_fnRetrieveMsgSize = (FNRetrieveMsgSize)::GetProcAddress(m_hPlugin, "RetrieveMsgSize");
	m_fnUIDL			= (FNUIDL)::GetProcAddress(m_hPlugin, "UIDL");
	m_fnDelete			= (FNDelete)::GetProcAddress(m_hPlugin, "Delete");
	m_fnMarkAsSeen      = (FNMarkAsSeen)::GetProcAddress(m_hPlugin, "MarkAsSeen");	
	m_fnLastErrorMsg	= (FNLastErrorMsg)::GetProcAddress(m_hPlugin, "LastErrorMsg");
	m_fnSetDoeventsProc	= (FNDoeventsProc)::GetProcAddress(m_hPlugin, "SetDoeventsProc");
	m_fnKill			= (FNKill)::GetProcAddress(m_hPlugin, "Kill");
	m_fnGetDefaultParameters = (FNGetDefaultParameters)::GetProcAddress(m_hPlugin, "GetDefaultParameters");
	m_fnSetParameters	= (FNSetParameters)::GetProcAddress(m_hPlugin, "SetParameters");
	m_fnValidateParameters = (FNValidateParameters)::GetProcAddress(m_hPlugin, "ValidateParameters");
	return TRUE;
}



BOOL CProtocolPlugin::Unload()
{
	m_fnProtocols		=  NULL;
	m_fnConnect			=  NULL;
	m_fnDisconnect		=  NULL;
	m_fnConnected		=  NULL;
	m_fnCheckMessages	=  NULL;
	m_fnRetrieveHeader	=  NULL;
	m_fnRetrieveRaw		=  NULL;
	m_fnRetriveTop		=  NULL;
	m_fnRetrieveMsgSize =  NULL;
	m_fnUIDL			=  NULL;
	m_fnDelete			=  NULL;
	m_fnMarkAsSeen      =  NULL;
	m_fnLastErrorMsg	=  NULL;
	m_fnSetDoeventsProc =  NULL;
	m_fnKill			=  NULL;
	m_fnGetDefaultParameters =  NULL;
	m_fnSetParameters	=  NULL;
	m_fnValidateParameters = NULL;


		
	POSITION pos = m_Protocols.GetHeadPosition();
	while(pos != NULL)
	{
		CProtocol* pProto = m_Protocols.GetNext(pos);
		delete pProto;
	}
	m_Protocols.RemoveAll();

	return CPlugin::Unload();
}

const CProtocols& CProtocolPlugin::Protocols() 
{
	if(m_fnProtocols == NULL || m_Protocols.GetCount() > 0)
		return m_Protocols;

	ShortString proto;
	m_fnProtocols(&proto);
	CString stProtocols = ShortStringToCString(proto);

	CString item;

	while(!stProtocols.IsEmpty()) {
		item = Parse(stProtocols, _T(","));
		CString name = Parse(item, _T(":"));
		int port = 0;
		bool bHasPort = false;
		if(item.GetLength() > 0) {
			port = _ttoi(item);
			bHasPort = true;
		}

		CProtocol *pProto = new CProtocol(this, name, port, bHasPort);
		m_Protocols.AddTail(pProto);

	}

	return m_Protocols;
}


UINT CProtocolPlugin::WorkerThread(LPVOID lParam)
{
	volatile WorkerThreadParams* params = (WorkerThreadParams*)lParam;

	//TRACE("WorkerThread: %X \n", ::GetCurrentThreadId()); 

	while(true) 
	{
		HANDLE Event = params->hActionEvent;
		DWORD res = MsgWaitForMultipleObjects(1, &Event, FALSE, INFINITE, QS_ALLINPUT);

		if (res == (WAIT_OBJECT_0 + 1))
		{
			MSG msg; 
			while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) 
			{ 
				if (msg.message == WM_QUIT)  
					return 0; 
				TranslateMessage(&msg);
				DispatchMessage(&msg); 
			}
			continue;
		}

		switch(params->Action)
		{
		case WorkerThreadParams::actionTerminate:
			CloseHandle(params->hActionEvent);
			params->hActionEvent = NULL; 
			CloseHandle(params->hLoadCompleteEvent);
			params->hLoadCompleteEvent = NULL;
			return 0;
		case WorkerThreadParams::actionLoadPlugin:
			params->bLoadResult = params->pLoadPlugin->Load(params->szLoadParam);
			SetEvent(params->hLoadCompleteEvent);
			break;
		case WorkerThreadParams::actionCallFunction:
			params->fnFunction(params->lParam);
			break;
		}
	}
}


////////////////////////////////////////////////////////////////////////


bool CProtocol::Connect(LPCTSTR Server, int Port, LPCTSTR UserName, LPCTSTR Password, int Timeout)
{
	if(m_pPlugin == NULL || m_pPlugin->m_fnConnect == NULL)
		return false;

	bool isInRetry = false;
retry:

	try {

        USES_CONVERSION;
		const char* aName     = T2A((LPTSTR)(LPCTSTR)m_stName);
        const char* aServer   = T2A((LPTSTR)(LPCTSTR)Server);
        const char* aUserName = T2A((LPTSTR)(LPCTSTR)UserName);
        const char* aPassword = T2A((LPTSTR)(LPCTSTR)Password);

		m_pPlugin->m_fnConnect(aServer, Port, aName, aUserName, aPassword, Timeout*1000);
	}
	catch(...)
	{
		return false;
	}

	auto f = m_pPlugin->m_fnLastErrorMsg;
	if (!isInRetry && f != NULL) {
		LPCSTR err = f();
		if (err != NULL && strcmp(err,"Could not load SSL library.") == 0) {
			isInRetry = true;
			goto retry;
		}
	}

	return (m_pPlugin->m_fnLastErrorMsg == NULL || m_pPlugin->m_fnLastErrorMsg() == NULL);
}

void CProtocol::Disconnect() 
{
	ResetStartTime();
	if(m_pPlugin != NULL && m_pPlugin->m_fnDisconnect != NULL)
		m_pPlugin->m_fnDisconnect();
}

bool CProtocol::Connected() 
{
	ResetStartTime();
	if(m_pPlugin != NULL && m_pPlugin->m_fnConnected != NULL)
		return m_pPlugin->m_fnConnected();

	return false;
}

int CProtocol::CheckMessages() 
{
	ResetStartTime();
	if(m_pPlugin != NULL && m_pPlugin->m_fnCheckMessages != NULL)
		return m_pPlugin->m_fnCheckMessages();

	return 0;
}

bool CProtocol::RetrieveHeader(int MsgNum, CString& stHeader) 
{
	ResetStartTime();
	if(m_pPlugin == NULL || m_pPlugin->m_fnRetrieveHeader == NULL)
		return false;

	LPCSTR pHeader = NULL;
	bool ret = m_pPlugin->m_fnRetrieveHeader(MsgNum, &pHeader);
	if(!ret) return false;

	LPCSTRToCString(pHeader, stHeader);
	
	m_pPlugin->FreePChar(&pHeader);

	return true;
}

bool CProtocol::RetrieveRaw(int MsgNum, CString& stRawMsg) 
{
	ResetStartTime();

	if(m_pPlugin == NULL || m_pPlugin->m_fnRetrieveRaw == NULL)
		return false;

	LPCSTR pRaw = NULL;
	bool ret = m_pPlugin->m_fnRetrieveRaw(MsgNum, &pRaw);
	if(!ret) return false;

	LPCSTRToCString(pRaw, stRawMsg);
	
	m_pPlugin->FreePChar(&pRaw);

	return true;
}

bool CProtocol::RetrieveTop(int MsgNum, int LineCount, CString& stTop) 
{
	ResetStartTime();

	if(m_pPlugin == NULL || m_pPlugin->m_fnRetriveTop == NULL)
		return false;

	LPCSTR pTop = NULL;
	bool ret = m_pPlugin->m_fnRetriveTop(MsgNum, LineCount, &pTop);
	if(!ret) return false;

	LPCSTRToCString(pTop, stTop);
	
	m_pPlugin->FreePChar(&pTop);

	return true;
}

int CProtocol::MessageSize(int MsgNum) 
{
	ResetStartTime();

	if(m_pPlugin != NULL && m_pPlugin->m_fnRetrieveMsgSize != NULL)
		return m_pPlugin->m_fnRetrieveMsgSize(MsgNum);

	return 0;
}

bool CProtocol::UIDL(CString& stUIDL, int MsgNum) 
{
	ResetStartTime();

	if(m_pPlugin == NULL || m_pPlugin->m_fnUIDL == NULL)
		return false;

	LPCSTR pUIDL = NULL;
	bool ret = m_pPlugin->m_fnUIDL(&pUIDL, MsgNum);
	if(!ret) return false;

	LPCSTRToCString(pUIDL, stUIDL);
	
	m_pPlugin->FreePChar(&pUIDL);

	return true;
}

bool CProtocol::UIDLs(CStringArray& arrayStr)
{
	ResetStartTime();

	CString stData;
	if(!UIDL(stData, -1))
		return false;

	arrayStr.RemoveAll();
	
	// accept line separator "\n" and "\r\n" 

	while(!stData.IsEmpty())	// while end not reached
	{
		Parse(stData, _T(" ")); // remove index
		int strIdx = arrayStr.Add(Parse(stData, _T("\n")));
		CString& item = arrayStr[strIdx];
		int charIdx = item.GetLength()-1;
		if(charIdx > -1)
			if(item[charIdx] == _T('\r'))
				item.Delete(charIdx);	// remove trailing '\r'
	}

	return true;
}


bool CProtocol::Delete(int MsgNum) 
{
	ResetStartTime();

	if(m_pPlugin == NULL || m_pPlugin->m_fnDelete == NULL)
		return false;

	return m_pPlugin->m_fnDelete(MsgNum);
}

bool CProtocol::MarkAsSeen(int MsgNum)
{
	ResetStartTime();

	if (m_pPlugin == NULL || m_pPlugin->m_fnMarkAsSeen == NULL)
		return false;

	return m_pPlugin->m_fnMarkAsSeen(MsgNum);
}

CString CProtocol::LastErrorMessage() 
{
	ResetStartTime();

    if(m_pPlugin == NULL)
        return i18n("Protocol plugin not initialized correctly!");

	if(m_pPlugin->m_fnLastErrorMsg == NULL)
		return _T("");

	LPCSTR pError = m_pPlugin->m_fnLastErrorMsg();

	CString stError;
	LPCSTRToCString(pError, stError);

	return stError;
}

void CProtocol::SetDoeventsProc(DWORD dwContext, void (WINAPI *fnDoevents)(DWORD))
{
	if(m_pPlugin == NULL || m_pPlugin->m_fnSetDoeventsProc == NULL)
		return;

	m_pPlugin->m_fnSetDoeventsProc(dwContext, (DWORD)fnDoevents);
}

// Note: this function is usually called from within the PopMan main (gui) thread
void CProtocol::Kill() 
{
	if(m_pPlugin == NULL || m_pPlugin->m_fnKill == NULL)
		return;

	CWinThread* pThread = CProtocolPlugin::pWorkerThread;
	if(!pThread) return;

	bool bSuspendResume = (::GetCurrentThreadId() != pThread->m_nThreadID);

	// Suspend the worker thread, to ensure that it will not interrupt the kill call
	if(bSuspendResume)
		pThread->SuspendThread(); 

	try {
		m_pPlugin->m_fnKill();
	} catch (...) {}

	if(bSuspendResume) {
		pThread->ResumeThread();
		Sleep(50); 
	}
}

CString CProtocol::GetDefaultParameters() const
{
	if(m_pPlugin == NULL || m_pPlugin->m_fnGetDefaultParameters == NULL)
		return _T("");

	LPCSTR pParams= m_pPlugin->m_fnGetDefaultParameters();

	CString stParams;
	LPCSTRToCString(pParams, stParams);

	return stParams;
}

bool CProtocol::SetParameters(LPCTSTR szParams) 
{
	USES_CONVERSION;

	ResetStartTime();

	if(m_pPlugin == NULL || m_pPlugin->m_fnSetParameters == NULL)
		return false;
	
	return m_pPlugin->m_fnSetParameters(T2A((LPTSTR)szParams));
}

bool CProtocol::ValidateParameters(LPCTSTR szParams) const
{
	USES_CONVERSION;

    if(m_pPlugin == NULL)
        return true;

	if(m_pPlugin->m_fnValidateParameters == NULL)
		return false;
	
	return m_pPlugin->m_fnValidateParameters(T2A((LPTSTR)szParams));
}

void CProtocol::LPCSTRToCString(LPCSTR pStr, CString& stString) 
{
	if(pStr == NULL) {
		stString.Empty();
		return;
	}

#ifdef _UNICODE
	const int len = strlen(pStr);
	LPWSTR pDest = stString.GetBuffer(len+1);
	for (int i = 0; i < len; ++i) {
		const unsigned char c = *pStr++;
		*pDest++ = c;
	}
	stString.ReleaseBuffer(len);
#else
	stString = pStr;
#endif
}

CProtocol* CProtocols::GetProtocolByName(const CString& name)
{
    POSITION pos = GetHeadPosition();
	while(pos != NULL)
	{
		CProtocol* prot = GetNext(pos);
		if(prot != NULL && prot->GetName() == name)
        {
            return prot;
        }		    
	}
    return NULL;
}