// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Account.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "POP3Account.h"
#include "PopManDoc.h"
#include "StrFunctions.h"

#include <afxcoll.h>


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


CPOP3Account::CPOP3Account(CPopManDoc* pDoc, BOOL useTLS) : CAccount(pDoc)
{
	m_bUseTLS = useTLS;
	m_nPort = 110;
	m_bTopSupported = TRUE;
	m_bTopVerified = FALSE;
	m_bTopReChecked = FALSE;
	m_bStrictHeaderCheck = TRUE;
	SetTimeOut(30);
	m_Socket.SetUseTLS(useTLS ? true : false);
	m_Socket.SetEventSink(new CSyncSocketEventSink<CAccount>(this, &CAccount::OnReceivedData, &CAccount::OnSentData));
}

CPOP3Account::~CPOP3Account()
{
}


/*
BOOL CPOP3Account::MailboxChanged()
{
	CString stOldStat(m_stStatString);

	ConnectAndLogin(NULL, FALSE);
	m_Socket.Close();
	StateChanged(stIdle);
	
	return (stOldStat != m_stStatString);
}
*/


BOOL CPOP3Account::ListMails(BOOL bHardReload)
{
	m_LastError = erNoError;

	m_pDoc->Log(_T("\r\n\r\n *** ListMails ***"));

	if(m_Socket.IsBlocking())
	{
		ErrorOccured(erBusy, 101, FALSE);
		return FALSE;
	}
	int nMailsOnServer = 0;

	if(!ConnectAndLogin(&nMailsOnServer))
		return FALSE;

	if(bHardReload)
		ClearMails();

	int		 n = 0;
	CString  stUIDL;
	int      nSize;
	int		 nNewMails = 0;	

	if(bHardReload == FALSE)
	{
		// How many new mails on server?
		for(n = 1; n <= nMailsOnServer; n++)
		{
			GetMailSize(n, nSize);
			GetMailUIDL(n, stUIDL);
				
			if(IsMailInList(stUIDL, nSize))
				continue;

			nNewMails++;
		}

		// Remove all mails that are not on the server anymore
		CMails mList;

		POSITION pos = m_Mails.GetHeadPosition();
		while(pos != NULL)
		{
			POSITION posDel = pos;
			CMail* pMail = m_Mails.GetNext(pos);
			
			if(-1 == FindInStrArray(m_UIDLCache, pMail->GetUIDL())) {
				mList.AddTail(pMail);
				m_Mails.RemoveAt(posDel);
			}
		}	

		
		m_pDoc->RemoveMails(&mList);

		
		pos = mList.GetHeadPosition();
		while(pos != NULL)
		{
			CMail* pMail = mList.GetNext(pos);
			delete pMail;
		}

	}


	if(bHardReload)
		m_nMailsCount = nMailsOnServer;
    else 
    {
		m_nMailsCount = nNewMails;

        if(m_bTopMessages && nNewMails > m_nTopMessagesCount)
            m_nMailsCount = m_nTopMessagesCount;
    }


	m_nCurrentMail = 1;

	
	BOOL bCheckForDoubleMails = FALSE;
	
	if(!bHardReload && m_Mails.GetCount())
		bCheckForDoubleMails = TRUE;

	CList<int, int> autoDeleteList;
	
    int mailCount = 1;
	// wenn nMailsOnServer == 0, wird folgende Schleife nicht ausgefhrt:
	//for(n = 1; n <= nMailsOnServer; n++)
    for(n = nMailsOnServer; n >= 1; n--, mailCount++)
	{

        if(!bHardReload && m_bTopMessages && mailCount > m_nTopMessagesCount)
            break;


		BOOL	 bComplete = FALSE;
		CString  stSource;
			
		if(!GetMailSize(n, nSize))
			return FALSE;
		
		if(!GetMailUIDL(n, stUIDL))
			return FALSE;

		// wenn Mail schon geladen, dann mit der nchsten weitermachen
		if(bCheckForDoubleMails)
			if(IsMailInList(stUIDL, nSize))
				continue;

		// // wenn Size <= m_MaxLoadSize -> vollstndig runterladen
		// bComplete = !m_bLimitLoadSize || (nSize <= m_nMaxLoadSize);

        bComplete = ( (m_RetrieveMode == RETRIEVE_MODE::AllComplete) || 
            ((m_RetrieveMode == RETRIEVE_MODE::SmallComplete) && ((unsigned)nSize < (m_nMaxSizeOfSmallMails * 1024))) );
		
        //TRACE("bComplete: %d \n", bComplete);
		
		StateChanged(stLoadingMails);
		if(!RetrieveMail(n, bComplete, stSource))
			return FALSE;

		if(!IsMailInList(stSource))
		{
			CMail* pMail = new CMail(this);
			if( pMail->Init(stSource, nSize, stUIDL, bComplete) )
			{
				m_Mails.AddHead(pMail);
				m_nCurrentMail++;

				
				// prfen, ob Mail schon gelesen:
				int nPos = m_MailCache.FindSignature(pMail->GetSignature());
				if(nPos == -1)
					pMail->SetRead(FALSE);
				else
					pMail->SetRead(m_MailCache[nPos].m_bRead);
									
				
				if(m_pDoc != NULL) {
					bool bDelete = false;
					m_pDoc->OnNewMail(pMail, (nPos == -1), bDelete);
					if(bDelete == true) {
						autoDeleteList.AddHead(n);
						delete m_Mails.RemoveHead();
					}
				}
			}
		}

		//if(!m_Socket.IsConnected() && nMailsOnServer > n)
        if(!m_Socket.IsConnected() && n > 1)
		{
			int nMailsCount = 0;
			if(!ConnectAndLogin(&nMailsCount))
				return FALSE;
			
			if(nMailsCount != nMailsOnServer)
			{
				nMailsOnServer = nMailsCount;
				n = nMailsOnServer + 1;
				bCheckForDoubleMails = TRUE;
			}
		}
	}

	if(m_Socket.IsConnected()) {

		m_nMailsCount = autoDeleteList.GetCount();
		m_nCurrentMail = 0;

		POSITION pos = autoDeleteList.GetHeadPosition();
		while(pos != NULL) {
			int idx = autoDeleteList.GetNext(pos);
			DeleteMail(idx);
			//TRACE("AutoDelete Idx=%d \n", idx);
		}
	}


	
	// TOPSupported test:
	if(!m_bTopSupported && !m_bTopReChecked && m_Mails.GetCount() > 0 && m_Socket.IsConnected()) {
		int minSize = 100000;
		int minIdx = 1;
		for(n = 0; n < m_SizeCache.GetSize(); n++)
		{
			int size = _ttoi(m_SizeCache[n]);
			if(size < minSize) { 
				minIdx = n+1;
				minSize = size;
			}
		}
		if(minSize < 11000) {
			CString stHead;
			if(GetMailHeader(minIdx, stHead)) 
				m_bTopSupported = m_bTopVerified = TRUE;
			else
				m_bTopReChecked = TRUE;
		}
	}

	
	m_nCurrentMail = 0;
	m_nMailsCount = 0;

	SendAndReceive(_T("QUIT"), _T(""), _T("\r\n"));

	m_Socket.Close();
	

    if(bHardReload || !m_bTopMessages || m_MailCache.GetSize() > max(500, 2*nMailsOnServer))
	    UpdateMailCache();

	SyncMailListWithUIDLs();
	
	m_LastCheckTime = COleDateTime::GetCurrentTime();

	StateChanged(stIdle);
	return TRUE;
}



BOOL CPOP3Account::DeleteMails(CMails* pMailList)
{
	m_LastError = erNoError;

	m_pDoc->Log(_T("\r\n\r\n *** DeleteMails ***"));
	
	m_nMailsCount = pMailList->GetCount();
	m_nCurrentMail = 0;

	if(m_Socket.IsBlocking())
	{
		ErrorOccured(erBusy, 201, FALSE);
		return FALSE;
	}
	
	int nMailsOnServer = 0;
	if(!ConnectAndLogin(&nMailsOnServer))
		return FALSE;
	
	

	if(nMailsOnServer > 0)
	{
		TRACE1("nMailsCount: %d\n", nMailsOnServer);

		if(m_bUseUIDLForDelete) 
		{
			POSITION pos = pMailList->GetHeadPosition();
			while(pos != NULL)
			{
				CMail* pMail = pMailList->GetNext(pos);

				int nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
				if(nIdx < 0 || nIdx >= nMailsOnServer) 
					continue;
		
			//	TRACE("Lsche Mail Nr. %d \n", nIdx+1);
				if(!DeleteMail(nIdx+1))
					return FALSE;
			}
		}
		else
		{
			if(!DoSaveDelete(nMailsOnServer, pMailList))
				return FALSE;
		}

		TRACE0("Mails deleted...\n");
	}


	// alle zu lschenden Mails sind markiert, jetzt "QUIT" senden, um sie zu lschen:
	SendAndReceive(_T("QUIT"), _T(""), _T("\r\n"));

	// die gelschten Mails aus der Auflistung entfernen:
	POSITION pos = pMailList->GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = pMailList->GetNext(pos);

		int nPos = m_MailCache.FindSignature(pMail->GetSignature());
		if(nPos > -1) m_MailCache.RemoveAt(nPos);

		POSITION posDel = m_Mails.Find(pMail);
		if(posDel != NULL)
		{
			m_Mails.RemoveAt(posDel);
			delete pMail;
		}
	}

	m_Socket.Close();
	StateChanged(stIdle);

	return TRUE;
}



BOOL CPOP3Account::DoSaveDelete(int nMailsCount, CMails* pMailList) 
{
	CStringArray TopArray;

	if(m_bTopSupported)	 // prfen, ob TOP-Befehl wirklich untersttzt wird:
	{
		TopArray.SetSize(nMailsCount);
		
		POSITION pos = pMailList->GetHeadPosition();
		ASSERT(pos != NULL);
		CMail* pMail = pMailList->GetNext(pos);
		ASSERT(pMail != NULL);
		int nIdx = m_Mails.GetIdxByMail(pMail);
		
		if(nIdx >= nMailsCount)
		{
			nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
			if(nIdx < 0) nIdx = 0;
			 
			//ErrorOccured(erMailMissing, 211);
			//return FALSE;
		}

		if(!GetMailHeader(nIdx+1, TopArray[nIdx]) && !IsCanceled())
			m_bTopSupported = FALSE;

		if(IsCanceled())
		{
			ErrorOccured(erCanceled);
			return FALSE;
		}

		m_bTopVerified = TRUE;
	}



	if(!m_Socket.IsConnected())	// wenn der Top-Befehl nicht untersttzt wird,
	{					// musste mglicherweise die Verbindung getrennt werden
		
		TRACE0("!m_Socket.IsConnected()\n");

		if(!ConnectAndLogin(&nMailsCount))
			return FALSE;
		
		if(nMailsCount < 1)
		{
			// ErrorOccured(erMailMissing, 203);
			// return FALSE;
			return TRUE;
		}
	}

	
	if(m_bTopSupported)
	{
		POSITION pos = pMailList->GetHeadPosition();
		while(pos != NULL)
		{
			CMail* pMail = pMailList->GetNext(pos);
			ASSERT(pMail != NULL);

			int nIdx = m_Mails.GetIdxByMail(pMail);

			if(nIdx >= nMailsCount)
			{
				nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
				if(nIdx < 0) 
				{
					LogMailNotFound(pMail);
					continue;
				}
				// ErrorOccured(erMailMissing, 214);
				// return FALSE;
			}

			if(TopArray[nIdx].IsEmpty())
				GetMailHeader(nIdx+1, TopArray[nIdx]);
			
			if(IsCanceled())
			{
				ErrorOccured(erCanceled);
				return FALSE;
			}

			if(CMail::CompareHeaders(pMail->GetHeader(), TopArray[nIdx], m_bStrictHeaderCheck))
			{
				if(!DeleteMail(nIdx+1))
					return FALSE;
				
				TRACE1("Mail %d deleted\n", nIdx+1);
				
				continue;
			}

			
			// try to find the correct index using UIDL:

			nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
			if(nIdx < 0) 
			{
				LogMailNotFound(pMail);
				continue;
			}

			if(TopArray[nIdx].IsEmpty())
				GetMailHeader(nIdx+1, TopArray[nIdx]);
			
			if(IsCanceled())
			{
				ErrorOccured(erCanceled);
				return FALSE;
			}

			if(CMail::CompareHeaders(pMail->GetHeader(), TopArray[nIdx], m_bStrictHeaderCheck))
			{
				if(!DeleteMail(nIdx+1))
					return FALSE;
				
				TRACE1("Mail %d deleted\n", nIdx+1);
				
				continue;
			}


			// nicht gefunden -> Suche von vorn nach hinten:
			BOOL bDeleted = FALSE;
			for(int n = 0; n < nMailsCount; n++)
			{
				if(TopArray[n].IsEmpty())
					GetMailHeader(n+1, TopArray[n]);
				
				if(IsCanceled())
				{
					ErrorOccured(erCanceled);
					return FALSE;
				}

				if(CMail::CompareHeaders(pMail->GetHeader(), TopArray[n], m_bStrictHeaderCheck))
				{
					if(!DeleteMail(n+1))
						return FALSE;
					bDeleted = TRUE;
					TRACE1("Mail %d deleted\n", n+1);
					break;
				}

			} // for

			if(!bDeleted)	// disable StrictHeaderChecking?
			{
				for(int n = 0; n < nMailsCount; n++)
				{
					if(CMail::CompareHeaders(pMail->GetHeader(), TopArray[n], FALSE))
					{
						m_bStrictHeaderCheck = FALSE;

						if(!DeleteMail(n+1))
							return FALSE;
						
						bDeleted = TRUE;
						break;
					}
				}

				nIdx = FindInStrArray(m_UIDLCache, pMail->GetUIDL());
				if(nIdx >= 0 && CMail::HeadersMayEqual(pMail->GetHeader(), TopArray[nIdx]))
				{
					if(!DeleteMail(nIdx+1))
						return FALSE;
								
					TRACE1("Mail %d deleted\n", nIdx+1);
					continue;
				}


				if(!bDeleted)
				{
					LogMailNotFound(pMail);
					ErrorOccured(erMailMissing, 212);
					return FALSE;
				}
			}

		} // while

	}
	else
	{	//(!m_bTopSupported) -> alternative Methode ber UIDL / Size:
		
		TRACE0("UIDL...\n");

		CStringArray ServerUIDLs;
		if(!GetServerUIDLs(ServerUIDLs))
			return FALSE;

		CStringArray CurrentUIDLs;
		GetCurrentUIDLs(CurrentUIDLs);
		
		// UIDL-Integrittsprfung:

		if(!VerifyIntegrity(CurrentUIDLs, ServerUIDLs))
		{
			// UIDL funktioniert nicht - anderen Weg ber Size versuchen:

			CStringArray ServerSizes;
			GetServerSizes(ServerSizes);

			CStringArray CurrentSizes;
			GetCurrentSizes(CurrentSizes);
			
			if(!VerifyIntegrity(CurrentSizes, ServerSizes))
			{
				ErrorOccured(erMailLocFailure, 204); 
				return FALSE;
			}

			POSITION pos = pMailList->GetHeadPosition();
			while(pos != NULL)
			{
				CMail* pMail = pMailList->GetNext(pos);

				int nIndex = 0;
				int nSizeCount = 0;

				CString stSize;
				stSize.Format(_T("%d"), pMail->GetSize());

				for(int n = 0; n < ServerSizes.GetSize(); n++)
				{
					if(ServerSizes[n] == stSize)
						if(nSizeCount++ == 0)
							nIndex = n + 1;
				}
			
				if(nSizeCount != 1)	// die Gre darf nur einmal vorhanden sein
				{
					ErrorOccured(erMailLocFailure, 205); 
					return FALSE;
				}

				if(!DeleteMail(nIndex))
					return FALSE;
			
			}


		
		}
		else
		{	// UIDL-Check ok ...
		
			BOOL bLooseCheck = (nMailsCount == m_Mails.GetCount());

			POSITION pos = pMailList->GetHeadPosition();
			while(pos != NULL)
			{
				CMail* pMail = pMailList->GetNext(pos);

				int nIndex = 1 + FindInStrArray(ServerUIDLs, pMail->GetUIDL());

				if(nIndex == 0)
				{
					ErrorOccured(erMailMissing, 206);
					return FALSE;
				}
				

				// wenn UIDL mehrfach vorkommt :
				if(FindInStrArray(ServerUIDLs, pMail->GetUIDL(), nIndex) > -1)
				{
					BOOL bFound = FALSE;
					
					if(pMail->GetSize() < 60000)
					{
						int Idx = 0; 
									
						while(!bFound && (Idx = (1 + FindInStrArray(ServerUIDLs, pMail->GetUIDL(), Idx))) > 0)
						{
							CString stHeader;
							GetMailHeader(Idx, stHeader);
							if(CMail::CompareHeaders(pMail->GetHeader(), stHeader, m_bStrictHeaderCheck))
							{
								bFound = TRUE;	
								nIndex = Idx;
							}
						}
					
					}
					if(!bFound)
					{
						ErrorOccured(erMailLocFailure, 207); 
						return FALSE;
					}
				}

				int nSize = 0;
				if(!GetMailSize(nIndex, nSize))
					return FALSE;

				int nDiffSize = pMail->GetSize() - (UINT)nSize;

				if(bLooseCheck)
				{	// Anzahl der Mails hat sich nicht verndert -> tolerante berprfung:
					if(abs(nDiffSize) > 18)
					{
						CString stLog;
						stLog.Format(_T("\r\n Subject: \"%s\"; Size: %d; UIDL: %s; nDiffSize: %d; Idx: %d \r\n"), 
												pMail->GetSubject(), pMail->GetSize(), pMail->GetUIDL(), nDiffSize, nIndex);
						m_pDoc->Log(stLog);
						ErrorOccured(erMailLocFailure, 208); 
						return FALSE;
					}
				}
				else
				{  // Anzahl der Mails hat sich verndert -> strenge berprfung:

						if(nDiffSize != 0)
						{
							BOOL bFailure = TRUE;
							if(nDiffSize >= 0 && nDiffSize <= 18)
							{
								// wahrscheinlich wurde der Header der Mail vom Server um den Eintrag "Status: RO" erweitert 
								// wenn die Mail klein genug ist > Header vergleichen, um sicher zu gehn:
								if(nSize <= 100000)
								{
									CString stHeader;
									GetMailHeader(nIndex, stHeader);
									if(CMail::CompareHeaders(pMail->GetHeader(), stHeader, m_bStrictHeaderCheck))
										bFailure = FALSE;
								}
							}
							if(bFailure)
							{
								ErrorOccured(erMailLocFailure, 209); 
								return FALSE;
							}
						}
						else
						{  // (nDiffSize == 0)  -> Einmaligkeit der Mailgre besttigen:
							
							CString stSize;
							stSize.Format(_T("%d"), pMail->GetSize());

							CStringArray SizeArray;
							GetServerSizes(SizeArray);
							int nSizeCount = 0;

							for(int n = 0; n < SizeArray.GetSize(); n++)
							{
								if(SizeArray[n] == stSize)
									nSizeCount++;
							}
						
							if(nSizeCount != 1)
							{
								ErrorOccured(erMailLocFailure, 210); 
								return FALSE;
							}
						}
				}
				
				if(!DeleteMail(nIndex))
					return FALSE;

				TRACE1("Mail %d deleted\n", nIndex);
			} // while
	
		} // UIDL-Check

	} // m_bTopSupported

	return TRUE;
}

BOOL CPOP3Account::DownloadMail(CMail* pMail, BOOL bKeepConnectionAlive)
{
	m_LastError = erNoError;
	
	m_pDoc->Log(_T("\r\n\r\n *** DownloadMail ***"));

	if(m_Socket.IsBlocking())
	{
		ErrorOccured(erBusy, 301, FALSE);
		return FALSE;
	}
	
	int nMailsCount = -1;

	if(!m_Socket.IsConnected())
	{
		if(!ConnectAndLogin(&nMailsCount))
			return FALSE;
	}

	CStringArray ServerUIDLs;
	if(!GetServerUIDLs(ServerUIDLs))
		return FALSE;
	

	int nIndex = 1 + FindInStrArray(ServerUIDLs, pMail->GetUIDL());

	if(nIndex == 0)
	{
		ErrorOccured(erMailMissing, 302);
		return FALSE;
	}
	
	
	int nSize = 0;
	if(!GetMailSize(nIndex, nSize))
		return FALSE;

	int nDiffSize = pMail->GetSize() - (UINT)nSize;
	if(abs(nDiffSize) > 30)
	{
		ErrorOccured(erMailLocFailure, 303); 
		return FALSE;
	}


	if(FindInStrArray(ServerUIDLs, pMail->GetUIDL(), nIndex) > -1)
	{
		// UIDL kommt mehrfach vor!

		nIndex = m_Mails.GetIdxByMail(pMail);
		
		if(nMailsCount == -1)
			if(!SendStat(&nMailsCount))
				return FALSE;	

		BOOL bFound = FALSE;		
		for(int n = nIndex+1; n <= nMailsCount; n++)
		{
			CString stHeader;
			GetMailHeader(n, stHeader, FALSE);

			if(CMail::CompareHeaders(pMail->GetHeader(), stHeader, m_bStrictHeaderCheck))
			{
				nIndex = n;
				bFound = TRUE;
				break;
			}
		}
		if(!bFound)
		{
			ErrorOccured(erMailLocFailure, 304); 
			return FALSE;
		}
	}


	StateChanged(stLoadingMail);
	CString stSource;
	if(!RetrieveMail(nIndex, TRUE, stSource))
		return FALSE;
	
	pMail->CompleteSource(stSource);

	if(!bKeepConnectionAlive)
	{
		SendAndReceive(_T("QUIT"), _T(""), _T("\r\n"));
		m_Socket.Close();
		StateChanged(stIdle);
	}

	return TRUE;
}



void CPOP3Account::ErrorOccured(ERRORS Error, UINT ErrLocation, BOOL bTerminate)
{
	if(bTerminate)
		m_Socket.Close();

	m_nErrorLocation = 0;
	m_stServerErrMessage.Empty();

	if(m_Socket.IsCanceled())
		m_LastError = erCanceled;
	else if(IsTimedOut())
		m_LastError = erTimedOut;
	else
	{
		m_LastError = Error;
		m_nErrorLocation = ErrLocation;
		CString stBuf = m_Socket.GetLastReceived().Left(4);
		stBuf.MakeUpper();
		if(stBuf == "-ERR")
		{
			m_stServerErrMessage = m_Socket.GetLastReceived().Mid(5);
			m_stServerErrMessage.TrimLeft();
			m_stServerErrMessage.TrimRight();
		}
	}
	
	
	CString stErr;
	stErr.Format(_T("\r\n --> ErrorOccured: Error=%d; ErrLoc=%d; Terminate=%d"), Error, ErrLocation, bTerminate);
	m_pDoc->Log(stErr);

	//TRACE(stErr);

	StateChanged(stError);

	if(bTerminate)
		StateChanged(stIdle);
}


#include "OAuth2.h"

BOOL CPOP3Account::ConnectAndLogin(int* pMailsCount, BOOL bUpdateCaches)
{
	m_LastError = erNoError;

	CString stPass = m_stPass;
	BOOL    bRemember = FALSE;

	if(stPass.IsEmpty())
		stPass = m_stPassVolatile;

	if(stPass.IsEmpty())
	{
		if(m_pDoc) 
		{
			if(!m_pDoc->GetAccountPass(this, stPass, bRemember))
			{
				ErrorOccured(erCanceled);
				return FALSE;
			}
		}
	}

	bool useAUTH = false;

	const CString Prefix = OAuthPassPrefix;
	if (stPass.Left(Prefix.GetLength()) == Prefix) {
		CString refreshToken = stPass.Mid(Prefix.GetLength());
		if (refreshToken.GetLength() > 0) {
			useAUTH = true;
			const CTime Now = CTime::GetCurrentTime();
			TRACE(_T("Now = %s\n"), Now.Format("%m%d%Y %H:%M:%S"));
			if (refreshToken == m_RefreshToken && !m_AccessToken.IsEmpty() && m_AccessTokenExpiresAt > Now) {
				TRACE(_T("m_AccessToken still valid till %s\n"), m_AccessTokenExpiresAt.Format("%m%d%Y %H:%M:%S"));
				stPass = m_AccessToken;
			}
			else {

				TRACE(_T("m_AccessToken NOT valid\n"));

				int expires_in = 0;
				CString accessToken = Google_GetOAuth2Token(refreshToken, expires_in);
				if (!accessToken.IsEmpty()) {
					m_AccessToken = accessToken;
					m_RefreshToken = refreshToken;
					m_AccessTokenExpiresAt = Now + CTimeSpan(0, 0, 0, expires_in);
					TRACE(_T("Got new m_AccessToken valid until %s\n"), m_AccessTokenExpiresAt.Format("%m%d%Y %H:%M:%S"));
					stPass = m_AccessToken;
				}
			}
		}
	}

	if(!m_Socket.IsConnected())
	{
		CString stLog;
		stLog = _T("\r\n#################################\r\nServer: ");
		stLog += m_stServer;
		stLog += _T("\r\nPort: ");
		CString stPort;
		stPort.Format(_T("%d"), m_nPort);
		stLog += stPort;
		stLog += _T("\r\n#################################\r\n");
		m_pDoc->Log(stLog);


		StateChanged(stConnecting);


		if(!m_Socket.Create() || !m_Socket.Connect(m_stServer, m_nPort))
		{
			ErrorOccured(erConnectFailed, 401);
			return FALSE;
		}
		StateChanged(stConnected);
	}
	
	if(!WaitForResponse(_T("+OK"), _T("\r\n"))) 
	{
		ErrorOccured(erAuthFailed, 402);
		return FALSE;
	}
	
	StateChanged(stLogin);	

	if (!useAUTH) {

		if (!SendAndReceive(_T("USER ") + m_stUser, _T("+OK"), _T("\r\n")))
		{
			ErrorOccured(erAuthFailed, 403);
			return FALSE;
		}

		if (!SendAndReceive(_T("PASS ") + stPass, _T("+OK"), _T("\r\n")))
		{
			ErrorOccured(erAuthFailed, 404);
			return FALSE;
		}
	}
	else {

		CString cr;
		cr += _T("user=");
		cr += m_stUser;
		cr += _T('\01');
		cr += _T("auth=Bearer ");		
		cr += stPass;
		cr += _T('\01');
		cr += _T('\01');

		CString base64 = Base64Encode(cr);

		if (!SendAndReceive(_T("AUTH XOAUTH2 ") + base64, _T("+OK"), _T("\r\n")))
		{
			ErrorOccured(erAuthFailed, 404);
			return FALSE;
		}
	}

	if(bRemember)
		m_stPass = stPass;
	else
		m_stPassVolatile = stPass;


	StateChanged(stLoggedIn);
	
	int nMailsCount = 0;
	if(!SendStat(&nMailsCount))
		return FALSE;	

	if(pMailsCount)
		*pMailsCount = nMailsCount;

	if(nMailsCount == 0)
	{
		m_UIDLCache.RemoveAll();
		m_SizeCache.RemoveAll();
		return TRUE;
	}

	if(bUpdateCaches)
	{
		// Cache aktualisieren...
		if(!GetServerSizes(m_SizeCache, TRUE))
			return FALSE;

		if(!GetServerUIDLs(m_UIDLCache, TRUE))
		{
			if(IsCanceled())
			{
				ErrorOccured(erCanceled);
				return FALSE;
			}

			// UIDL wird nicht untersttzt -> Sizes als UIDL verwenden!
			m_UIDLCache.Copy(m_SizeCache);
		}

		if((m_UIDLCache.GetSize() != nMailsCount) || (m_SizeCache.GetSize() != nMailsCount))
		{
			TRACE(_T("m_UIDLCache.GetSize(): %d\n"), m_UIDLCache.GetSize());
			TRACE(_T("m_SizeCache.GetSize(): %d\n"), m_SizeCache.GetSize());
			TRACE(_T("nMailsCount: %d\n"), nMailsCount);
			ErrorOccured(erResponseInvalid, 405);
			return FALSE;
		}
	}

	return TRUE;
}

BOOL CPOP3Account::IsMailInList(const CString& stUIDL, int nSize)
{
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		if(pMail->GetUIDL() == stUIDL)
		{
			int nDiffSize = pMail->GetSize() - (UINT)nSize;
			if(nDiffSize >= 0 && nDiffSize <= 18)
				return TRUE;
		}
	}	
	return FALSE;
}

BOOL CPOP3Account::IsMailInList(const CString& stSource)
{
	int i = stSource.Find(_T("\r\n\r\n"));
	CString stHeader = stSource.Left(i);

	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		if(CMail::CompareHeaders(pMail->GetHeader(), stHeader, m_bStrictHeaderCheck))
			return TRUE;
	}	
	return FALSE;
}


BOOL CPOP3Account::WaitForResponse(LPCTSTR pszExpectedPrefix, LPCTSTR pszTerminator)
{
	return SendAndReceive(_T(""), pszExpectedPrefix, pszTerminator);
}

BOOL CPOP3Account::SendAndReceive(CString stSend, LPCTSTR pszExpectedPrefix, LPCTSTR pszTerminator, ABORT_CONDITION AbortCondition)
{
	if(m_Socket.IsBlocking())
	{
		::WSASetLastError(WSAEINPROGRESS);
		return FALSE;
	}

	CString& receivedStr = m_Socket.GetLastReceived();		

	receivedStr = _T("");
	
	if(!stSend.IsEmpty())
	{
		if(stSend.Right(2) != _T("\r\n"))
			stSend += _T("\r\n");
	
		if(!m_Socket.SendStr(stSend))
			return FALSE;
	}

    int lenOfPrefix     = _tcslen(pszExpectedPrefix);
    int lenOfTerminator = _tcslen(pszTerminator);
    BOOL bFindPrefix = (lenOfPrefix > 0); // if there is no prefix, we won't have to find it

    CString stBuf;

	while(1) {	
		
		if(!m_Socket.ReceiveStr(stBuf, 8192))
		{
			m_pDoc->Log(_T("\r\n SendAndReceive::ReceiveStr failed"));
			return FALSE;
		}
		
		// sometimes there are 0-Bytes included in the message body:
		stBuf.Replace(_T('\0'), _T(' '));

        if(receivedStr.GetAllocLength() < receivedStr.GetLength() + stBuf.GetLength()) {
            
           // TRACE("REALLOC on RECEIVE %d ", receivedStr.GetAllocLength());

            int newBuffLen;
            int minLen = receivedStr.GetLength() + 2*stBuf.GetLength();
            
            if(minLen < 4*1024*1024)
                newBuffLen = 2*minLen;
            else
                newBuffLen = minLen + 2*1024*1024;


            receivedStr.GetBuffer(newBuffLen);

           // TRACE("new Buff: %d %s\n", receivedStr.GetAllocLength(), (LPCTSTR)this->m_stName);
        }

		receivedStr += stBuf;
		
        if(bFindPrefix) {            
        
            if(receivedStr.GetLength() >= lenOfPrefix) {
                if(receivedStr.Find(pszExpectedPrefix) != 0)
    			    return FALSE; // prefix not found!
                else {
                    //receivedStr = receivedStr.Mid(lenOfPrefix);
                    receivedStr.Delete(0, lenOfPrefix);
                    bFindPrefix = FALSE;
                    //TRACE("Kaputt: %d \n", receivedStr.GetAllocLength());
                }
            }
        }
		

		switch(AbortCondition)
		{

		case abMsgTextComplete:
			
			if(CMail::IsMessageTextComplete(receivedStr) || receivedStr.GetLength() > 60000)
			{
				TRACE1("Laden abgebrochen nach: %d\n", receivedStr.GetLength());
				Close();
				m_pDoc->Log(_T("\r\n SendAndReceive:abMsgTextComplete"));
				return TRUE;	
			}
			break;


		case abEnough:

			if(receivedStr.GetLength() > 24000)
			{
				Close();
				m_pDoc->Log(_T("\r\n SendAndReceive:abEnough"));
				return TRUE;	
			}
			break;

        default: 
            break;
		}

        if(lenOfTerminator > 0) {

            int startFind = receivedStr.GetLength() - (5 * lenOfTerminator);
            if(startFind < 0) startFind = 0;

            if(receivedStr.Find(pszTerminator, startFind) != -1)
                break;
        }

	}
	
	
	
	return TRUE;
}




BOOL CPOP3Account::SendStat(int *pMailsCount)
{
	if(!m_Socket.IsConnected())
	{
		ErrorOccured(erNotConnected, 501);
		return FALSE;
	}

	if(!SendAndReceive(_T("STAT") , _T("+OK "), _T("\r\n")))
	{
		ErrorOccured(erResponseInvalid, 502);
		return FALSE;
	}
	
//	m_stStatString = m_Socket.m_stLastReceived;
	
	m_Socket.GetLastReceived().TrimLeft();
	int nMailsCount = _ttoi(Parse(m_Socket.GetLastReceived(), _T(" ")));
	
	if(nMailsCount < 0)
	{
		ErrorOccured(erResponseInvalid, 503);
		return FALSE;
	}

	if(pMailsCount)
		*pMailsCount = nMailsCount;

	return TRUE;
}


BOOL CPOP3Account::GetServerUIDLs(CStringArray& UIDLArray, BOOL bForceUpdate)
{
	if(bForceUpdate)
		return GetCmdListing(UIDLArray, _T("UIDL"), FALSE);

	UIDLArray.Copy(m_UIDLCache);
	return TRUE;
}

BOOL CPOP3Account::GetServerSizes(CStringArray& SizeArray, BOOL bForceUpdate)
{
	if(bForceUpdate)
		return GetCmdListing(SizeArray, _T("LIST"), TRUE);
	
	SizeArray.Copy(m_SizeCache);
	return TRUE;
}


BOOL CPOP3Account::GetCmdListing(CStringArray& ListingArray, const CString& stCmd, BOOL bErrorHandling)
{
	if(!m_Socket.IsConnected())
	{
		ErrorOccured(erNotConnected, 601);
		return FALSE;
	}
	
	if(!SendAndReceive(stCmd, _T("+OK"), _T("\r\n.\r\n")))
	{
		m_pDoc->Log(_T("\r\n GetCmdListing::SendAndReceive failed!"));
		if(bErrorHandling) 
			ErrorOccured(erResponseInvalid, 602);

		return FALSE;
	}
	
	ListingArray.RemoveAll();

	CString& St = m_Socket.GetLastReceived();
	
	Parse(St, _T("\r\n")); // 1. Zeile verwerfen
	
	while(St[0] != _T('.'))	// solange Ende nicht erreicht
	{
		Parse(St, _T(" ")); // Index verwerfen
		ListingArray.Add(Parse(St, _T("\r\n")));
	}

	return TRUE;
}


BOOL CPOP3Account::GetMailSize(int nMailNo, int& nSize)
{
	ASSERT(nMailNo > 0);

	nSize = _ttoi(m_SizeCache[nMailNo-1]);
	return (nSize > 0);
}

BOOL CPOP3Account::RetrieveMail(int nMailNo, BOOL bComplete, CString& stSource)
{
	ASSERT(nMailNo > 0);

	if(!m_Socket.IsConnected())
	{
		ErrorOccured(erNotConnected, 901);
		return FALSE;
	}

	if(bComplete)
	{
		CString stSend;
		stSend.Format(_T("RETR %d"), nMailNo);
		if(!SendAndReceive(stSend, _T("+OK"), _T("\r\n.\r\n"), abNone))
		{
			ErrorOccured(erResponseInvalid, 902);
			return FALSE;
		}
	}
	else
	{
		// nicht vollstndig laden:

		if(m_bTopSupported && !m_bTopVerified)	 // prfen, ob TOP-Befehl wirklich untersttzt wird, falls noch nicht geschehen:
		{
			CString stHead;
			if(!GetMailHeader(nMailNo, stHead) && !IsCanceled())
				m_bTopSupported = FALSE;
			
			if(IsCanceled())
			{
				ErrorOccured(erCanceled);
				return FALSE;
			}

			m_bTopVerified = TRUE;
		}

		if(!m_Socket.IsConnected())	// wenn der Top-Befehl nicht untersttzt wird,
		{					// musste mglicherweise die Verbindung getrennt werden
			
			if(!ConnectAndLogin())
				return FALSE;
		}

		
//		if(m_bTopSupported)
//		{	

		CString stSend;
		stSend.Format(_T("TOP %d %d"), nMailNo, NumberOfLinesForIncompleteMail());
		if(!SendAndReceive(stSend, _T("+OK"), _T("\r\n.\r\n"), abNone))
		{
			ErrorOccured(erResponseInvalid, 903);
			return FALSE;
		}

/*		}
		else
		{
			CString stSend;
			stSend.Format(_T("RETR %d"), nMailNo);
			if(!SendAndReceive(stSend, _T("+OK"), _T("\r\n.\r\n"), abMsgTextComplete))
			{
				ErrorOccured(erResponseInvalid, 904);
				return FALSE;
			}
		}
*/
	}


	CString& stData = m_Socket.GetLastReceived();
	Parse(stData, _T("\r\n"));

	//if(bComplete || m_bTopSupported)
		stSource = stData.Left(stData.ReverseFind(_T('.')));
	//else
	//	stSource = stData;
	
	stSource.Replace(_T("\r\n.."), _T("\r\n."));
    
    if(m_Socket.GetLastReceived().GetAllocLength() > 200*1024) {
        m_Socket.GetLastReceived().Empty(); // free unused memory
    }

	//stSource = "";
	//CFile File;

	//if (!File.Open("D:\Test.eml", CFile::shareDenyNone | CFile::modeRead | CFile::typeBinary))
	//	return false;

	//const unsigned int nFileSize = static_cast<unsigned int>(File.SeekToEnd());
	//File.Seek(0, CFile::begin);

	//char* pBuffer = stSource.GetBuffer(nFileSize + 1); //new char[nFileSize];

	//File.Read(pBuffer, nFileSize);
	//File.Close();

	//pBuffer[nFileSize] = 0;
	//stSource.ReleaseBuffer();

	return !stSource.IsEmpty();
}



BOOL CPOP3Account::DeleteMail(int nMailNo)
{
	ASSERT(nMailNo > 0);
	
	if(!m_Socket.IsConnected())
	{
		ErrorOccured(erNotConnected, 1001);
		return FALSE;
	}
	
	m_nCurrentMail++;
	StateChanged(stDeletingMails);

	CString stSend;
	stSend.Format(_T("DELE %d"), nMailNo);
	if(!SendAndReceive(stSend, _T("+OK"), _T("\r\n")))
	{
		ErrorOccured(erResponseInvalid, 1002);
		return FALSE;
	}
	return TRUE;
}



BOOL CPOP3Account::GetMailHeader(int nMailNo, CString& stHeader, BOOL bAbort)
{
	ASSERT(nMailNo > 0);
	
	if(!m_Socket.IsConnected())
	{
		ErrorOccured(erNotConnected, 1101);
		return FALSE;
	}

	CString stSend;
	stSend.Format(_T("TOP %d 0"), nMailNo);
	if(!SendAndReceive(stSend, _T("+OK"), _T("\r\n.\r\n"), bAbort ? abEnough : abNone))
	{
		ErrorOccured(erResponseInvalid, 1102);
		return FALSE;
	}

	CString& stData = m_Socket.GetLastReceived();
	Parse(stData, _T("\r\n")); // 1. Zeile verwerfen

	BOOL bNoHeadSep = (stData.Find(_T("\r\n\r\n")) == -1);
	stHeader = ConvertLineBreaks(Parse(stData, _T("\r\n\r\n")));
	
	stData.TrimLeft(_T(" \t\r\n"));
	stData.TrimRight(_T(" \t\r\n"));

	return (bNoHeadSep || stData == _T("."));
}



BOOL CPOP3Account::ReadSettings(const CSettingsKey& settings)
{	
	if(!CAccount::ReadSettings(settings))
		return FALSE;

	BOOL bSupported = TRUE;
	if(settings.QueryValue(szAccTopSupportedValue, bSupported))
		SetTopSupported(bSupported);

	BOOL bStrict = TRUE;
	if(settings.QueryValue(szAccStrictHeaderCheckValue, bStrict))
		SetStrictHeaderCheck(bStrict);

	return TRUE;
}


void CPOP3Account::SaveSettings(CSettingsKey& settings) const
{
	CAccount::SaveSettings(settings);

	settings.SetValue(szAccTopSupportedValue, IsTopSupported());
	settings.SetValue(szAccStrictHeaderCheckValue, IsStrictHeaderCheck());
}