// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// NewTip.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "NewTip.h"
#include "PopMan.h"	// we need this for "CPopManApp::GetCursor(IDC_HOVER_LINK)"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


CString CNewTip::m_stClassName;

#define ID_TIMER_DESTROY 1
#define ID_TIMER_HOVER   2

#define cstButtonMargin 5


CNewTip::CNewTip()
{
	m_headFont = NULL;
	m_Font = NULL;
	m_fnCallback = NULL;
	dwDownValue = 0;
	m_bCallingBack = false;
}

CNewTip::~CNewTip()
{
}


void CNewTip::Show(int nSeconds)
{
	DestroyWindow();

	if (m_stClassName.IsEmpty())
		m_stClassName = AfxRegisterWndClass(CS_SAVEBITS | CS_HREDRAW | CS_VREDRAW);

	for(int n = 0; n < m_Buttons.GetSize(); n++)
		m_Buttons[n].m_pParent = this;
		
	int nWidth = 0;
	int nHeight = 0;
	
	CalculateSize(nWidth, nHeight);

	int nScreenWidth  = ::GetSystemMetrics(SM_CXSCREEN);
	int nScreenHeight = ::GetSystemMetrics(SM_CYSCREEN);
	
	RECT rectWorkingArea;
	if(!::SystemParametersInfo(SPI_GETWORKAREA, NULL, &rectWorkingArea, 0))
	{
		rectWorkingArea.left = 0;
		rectWorkingArea.top  = 0;
		rectWorkingArea.right  = nScreenWidth;
		rectWorkingArea.bottom = nScreenHeight;
	}

	BOOL bRigthAlign  = TRUE;
	BOOL bBottomAlign = TRUE;

	HWND hwndSysTray = ::FindWindow(_T("Shell_TrayWnd"), NULL);
	HWND hwndNotifyWnd = ::FindWindowEx(hwndSysTray, 0, _T("TrayNotifyWnd"), 0);
	if(hwndNotifyWnd)
	{
		RECT rect;
		::GetWindowRect(hwndNotifyWnd, &rect);
		int ptX = rect.left + (rect.right - rect.left) / 2;
		int ptY = rect.top  + (rect.bottom - rect.top) / 2;
		
		bRigthAlign  = ((ptX > nScreenWidth / 2) ? TRUE : FALSE);
		bBottomAlign = ((ptY > nScreenHeight / 2) ? TRUE : FALSE);
	}

	int x = 0;
	int y = 0;

	if(bRigthAlign)
		x = rectWorkingArea.right - nWidth;
	else
		x = rectWorkingArea.left;

	if(bBottomAlign)
		y = rectWorkingArea.bottom - nHeight;
	else
		y = rectWorkingArea.top;

	CreateEx(WS_EX_TOOLWINDOW | WS_EX_TOPMOST, m_stClassName, NULL, WS_POPUP | WS_BORDER, x, y, nWidth, nHeight, 0, 0);
	ShowWindow(SW_SHOWNOACTIVATE);
	UpdateWindow();

	SetTimer(ID_TIMER_DESTROY, nSeconds * 1000, NULL);
	SetTimer(ID_TIMER_HOVER, 500, NULL);
}


BEGIN_MESSAGE_MAP(CNewTip, CWnd)
	//{{AFX_MSG_MAP(CNewTip)
	ON_WM_PAINT()
	ON_WM_TIMER()
	ON_WM_SETCURSOR()
	ON_WM_LBUTTONUP()
	ON_WM_LBUTTONDOWN()
	ON_WM_MOUSEMOVE()
	ON_WM_DESTROY()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


void CNewTip::CalculateSize(int& nWidth, int& nHeight)
{
	HDC screenDC = ::GetDC(0);
	HDC dc = ::CreateCompatibleDC(screenDC);
	::ReleaseDC(0, screenDC);

	HFONT oldFont = (HFONT)::SelectObject(dc, m_headFont);
	
	SIZE size;
	::GetTextExtentPoint32(dc, m_stHead, m_stHead.GetLength(), &size);
	m_nHeadHeight = int(size.cy * 1.5);
	int nHeadWidth = size.cx;

	
	::SelectObject(dc, m_Font);

	int nTabPos = LOWORD(::GetTabbedTextExtent(dc, _T("\t"), 1, 0, NULL));
	int nMaxWidth = 0;
	int n;
	for(n = 0; n < m_Lines.GetSize(); n++)
	{
		const CString& Line = m_Lines[n].stStr;

		int nLineWidth = LOWORD(::GetTabbedTextExtent(dc, Line, Line.GetLength(), 1, &nTabPos));
		if(nLineWidth > nMaxWidth)
			nMaxWidth = nLineWidth;
	}
	
	::GetTextExtentPoint32(dc, _T(" "), 1, &size);
	int nSpaceWidth = size.cx;
	m_nLineHeight = size.cy;



	int nMaxCaptionWidth = 0;
	for(n = 0; n < m_Buttons.GetSize(); n++)
	{
		const CString& Caption = m_Buttons[n].m_stCaption;

		int nCaptionWidth = LOWORD(::GetTabbedTextExtent(dc, Caption, Caption.GetLength(), 1, &nTabPos));
		if(nCaptionWidth > nMaxCaptionWidth)
			nMaxCaptionWidth = nCaptionWidth;
	}
	nMaxCaptionWidth += 4*nSpaceWidth;

	for(n = 0; n < m_Buttons.GetSize(); n++)
	{
		m_Buttons[n].SetSize(nMaxCaptionWidth, int(1.5 * m_nLineHeight));
	}
	
	nMaxCaptionWidth = m_Buttons.GetSize() * (nMaxCaptionWidth + cstButtonMargin);



	nWidth = nMaxWidth;
	if(nHeadWidth > nWidth) nWidth = nHeadWidth;
	if(nMaxCaptionWidth > nWidth)    nWidth = nMaxCaptionWidth;
	nWidth += 12;

	nHeight = m_nHeadHeight + (m_nLineHeight * m_Lines.GetSize()) + 8;
	nHeight += int(2.5 * m_nLineHeight);


	::SelectObject(dc, oldFont);
	::DeleteDC(dc);
}

void CNewTip::OnPaint() 
{
	CPaintDC dc(this); 
	
	COLORREF colorBkg = ::GetSysColor(COLOR_INFOBK);
	COLORREF colorText = ::GetSysColor(COLOR_INFOTEXT);

	CRect rcClient;
	GetClientRect(&rcClient);
	dc.FillSolidRect(&rcClient, colorBkg);

	dc.SetBkColor(colorBkg);
	dc.SetTextAlign(TA_LEFT | TA_TOP);
	dc.SetTextColor(colorText);

	int yOffset = 3;
	const int xOffset = 5;

	HFONT oldFont = (HFONT)::SelectObject(dc.m_hDC, m_headFont);
	dc.TextOut(xOffset, yOffset, m_stHead);
	yOffset += m_nHeadHeight;

	::SelectObject(dc.m_hDC, m_Font);

	CSize Size	= dc.GetTabbedTextExtent(_T("\t"), 1, 0, NULL);
	int nTabPos = Size.cx;

	for(int n = 0; n < m_Lines.GetSize(); n++)
	{
		dc.SetTextColor(m_Lines[n].nColor);
		CSize size = dc.TabbedTextOut(xOffset, yOffset, m_Lines[n].stStr, 1, &nTabPos, xOffset);

		m_Lines[n].rcPos.left = xOffset;
		m_Lines[n].rcPos.top = yOffset;
		m_Lines[n].rcPos.bottom = yOffset + m_nLineHeight;
		m_Lines[n].rcPos.right = xOffset + size.cx;

		yOffset += m_nLineHeight;
	}

	for(int i = 0; i < m_Buttons.GetSize(); i++)
	{
		int nX = xOffset;
		if(i > 0)
			nX = m_Buttons[i-1].m_X + m_Buttons[i-1].m_Width + cstButtonMargin;
		
		m_Buttons[i].SetPos(nX , yOffset + m_nLineHeight - 1);
		m_Buttons[i].Draw(dc, colorText);
	}

	::SelectObject(dc.m_hDC, oldFont);
}

void CNewTip::OnTimer(UINT nIDEvent) 
{
	CWnd::OnTimer(nIDEvent);

	if(nIDEvent == ID_TIMER_DESTROY)
	{
		if(!m_bCallingBack)
			DestroyWindow();
	}
	else if(nIDEvent == ID_TIMER_HOVER)
	{
		CPoint point;
		GetCursorPos(&point);
		ScreenToClient(&point);

		for(int i = 0; i < m_Buttons.GetSize(); i++)
			m_Buttons[i].OnMouseMove(point);
	}
}

BOOL CNewTip::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
	POINT point;
	::GetCursorPos(&point);
	ScreenToClient(&point);

	if(GetItemValueFromPos(point) > 0)		
//		::SetCursor(::LoadCursor(::GetModuleHandle(NULL), MAKEINTRESOURCE(IDC_HOVER_LINK)));
		::SetCursor(CPopManApp::GetCursor(IDC_HOVER_LINK));
	else
		::SetCursor(::LoadCursor(NULL, IDC_ARROW));
		
	return CWnd::OnSetCursor(pWnd, nHitTest, message);
}

DWORD CNewTip::GetItemValueFromPos(const POINT& Point)
{
	for(int n = 0; n < m_Lines.GetSize(); n++)
	{
		const CRect& rc = m_Lines[n].rcPos;
		if(rc.PtInRect(Point))
			return m_Lines[n].dwValue;
	}
	return 0;
}


void CNewTipButton::Draw(CDC& dc, COLORREF colorText)
{
	CPen   Pen(PS_SOLID, 1, RGB(0, 0, 0));

	COLORREF colorBkg = ::GetSysColor(COLOR_INFOBK);
	
	if(m_State == STATE::Pressed)
		colorBkg = RGB(245, 245, 245);

	CBrush Brush(colorBkg);

	CPen*   poldPen   = dc.SelectObject(&Pen);
	CBrush* poldBrush = dc.SelectObject(&Brush);
	
	CRect rect = GetRect();
	dc.Rectangle(rect);
	
	if(m_State == STATE::Hover || m_State == STATE::Pressed)
	{
		rect.left += 1;
		rect.top += 1;
		rect.right -= 1;
		rect.bottom -= 1;
		dc.Rectangle(rect);
	}

	UINT uFormat =  DT_SINGLELINE | DT_NOPREFIX  | DT_VCENTER |
					DT_END_ELLIPSIS | DT_CENTER;

	dc.SetBkColor(colorBkg);
	dc.SetTextColor(colorText);
	dc.DrawText(m_stCaption, GetRect(), uFormat);

	dc.SelectObject(poldBrush);
	dc.SelectObject(poldPen);
}



void CNewTipButton::OnMouseDown(CPoint point)
{
	if(GetRect().PtInRect(point))
	{
		m_bPressed = TRUE;
		m_State = STATE::Pressed;
		m_pParent->RedrawWindow(GetRect());
	}
}

void CNewTipButton::OnMouseMove(CPoint point)
{
	if(GetRect().PtInRect(point))
	{
		if(m_bPressed)
		{
			if(m_State != STATE::Pressed)
			{
				m_State = STATE::Pressed;
				m_pParent->RedrawWindow(GetRect());
			}
		}
		else
		{
			if(m_State != STATE::Hover)
			{
				m_State = STATE::Hover;
				m_pParent->RedrawWindow(GetRect());
			}
		}
	}
	else
	{
		if(m_State != STATE::Normal)
		{
			m_State = STATE::Normal;
			m_pParent->RedrawWindow(GetRect());
		}
	}
}

BOOL CNewTipButton::OnMouseUp(CPoint point)
{
	if(GetRect().PtInRect(point) && m_bPressed)
	{
		m_bPressed = FALSE;
		m_State = STATE::Hover;
		m_pParent->RedrawWindow(GetRect());
		return TRUE;
	}
	m_bPressed = FALSE;

	return FALSE;
}



void CNewTip::OnLButtonDown(UINT nFlags, CPoint point) 
{
	CWnd::OnLButtonDown(nFlags, point);

	dwDownValue = GetItemValueFromPos(point);
	
	SetCapture();

	for(int i = 0; i < m_Buttons.GetSize(); i++)
		m_Buttons[i].OnMouseDown(point);
}

void CNewTip::OnMouseMove(UINT nFlags, CPoint point) 
{
	CWnd::OnMouseMove(nFlags, point);

	for(int i = 0; i < m_Buttons.GetSize(); i++)
		m_Buttons[i].OnMouseMove(point);
}

void CNewTip::OnLButtonUp(UINT nFlags, CPoint point) 
{
	CWnd::OnLButtonUp(nFlags, point);
	ReleaseCapture();

	if(m_fnCallback != NULL)
	{
		DWORD dwVal = GetItemValueFromPos(point);
		if(dwVal > 0)
		{
			if(dwVal != dwDownValue) return;
			DoCallBack(NEWTIPEVENT::ItemClicked, dwVal);
            DestroyWindow();
            return;
		}
		

		for(int i = 0; i < m_Buttons.GetSize(); i++)
		{
			if(m_Buttons[i].OnMouseUp(point)) 
            {
				DoCallBack(NEWTIPEVENT::ButtonClicked, m_Buttons[i].m_dwID);
                DestroyWindow();
                return;
            }
		}
	}	
}

void CNewTip::DoCallBack(int event, DWORD val)
{
	m_bCallingBack = true;
	m_fnCallback(event, val);
	m_bCallingBack = false;
}

void CNewTip::OnDestroy() 
{
	KillTimer(ID_TIMER_DESTROY);
	KillTimer(ID_TIMER_HOVER);

	CWnd::OnDestroy();
}

