// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// MultiLineTip.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "MultiLineTip.h"
#include "StrFunctions.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


CString CMultiLineTip::m_stClassName;


CMultiLineTip::CMultiLineTip()
{
	CreateFonts();

    m_fnMouseMove = NULL;
    m_pMouseMoveReceiver = NULL;
    m_pClient = NULL;
    m_bAutoHide = false;
}

CMultiLineTip::~CMultiLineTip()
{
}


void CMultiLineTip::Show(const CWnd* pClient, const CString& stHead, const CString& stTip, bool bBelowCursor, bool bAutoHide, int boldLines[], int countBoldLines)
{
	DestroyWindow();

    m_pClient = pClient;
    m_bAutoHide = bAutoHide;

	m_stHead = stHead;
	m_Lines.RemoveAll();

	int nIdx = 0;
	while(nIdx > -1)
	{
		LINE Line;
		Line.stRight = ParseConst(stTip, _T("\r\n"), nIdx);

		if(Line.stRight.Find(_T('\t')) > -1)
			Line.stLeft = Parse(Line.stRight, _T("\t"));

		m_Lines.Add(Line);
    }

    for(int i = 0; boldLines && i<countBoldLines; ++i) 
    {
        int boldIdx = boldLines[i];
        if(boldIdx >= 0 && boldIdx < m_Lines.GetSize())
            m_Lines[boldIdx].isBold = true;
    }

	if (m_stClassName.IsEmpty())
		m_stClassName = AfxRegisterWndClass(CS_SAVEBITS | CS_HREDRAW | CS_VREDRAW);

	int nWidth = 0;
	int nHeight = 0;
	
	CalculateSize(nWidth, nHeight);

	POINT point;
	::GetCursorPos(&point);
	
	m_ptLT.x = point.x - 10;
	m_ptLT.y = point.y - 10;


	int x = point.x + 1;
	int y = point.y - (bBelowCursor ? -20 : nHeight);

	int nScreenWidth = ::GetSystemMetrics(SM_CXSCREEN);
	if(x + nWidth > nScreenWidth - 1)
		x = nScreenWidth - nWidth - 1;

	if(y < 1)
		y = 1;

	CreateEx(WS_EX_TOOLWINDOW | WS_EX_TOPMOST, m_stClassName, NULL, WS_POPUP | WS_BORDER, x, y, nWidth, nHeight, 0, 0);
	ShowWindow(SW_SHOWNOACTIVATE);
	UpdateWindow();

	if(bAutoHide || m_pClient) SetTimer(1, 200, NULL);
} 


void CMultiLineTip::OnDestroy() 
{
	CWnd::OnDestroy();
	KillTimer(1);
}


BEGIN_MESSAGE_MAP(CMultiLineTip, CWnd)
	//{{AFX_MSG_MAP(CMultiLineTip)
	ON_WM_TIMER()
	ON_WM_PAINT()
	ON_WM_DESTROY()
	ON_WM_SETTINGCHANGE()
	ON_WM_MOUSEMOVE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


void CMultiLineTip::OnTimer(UINT nIDEvent) 
{
	CWnd::OnTimer(nIDEvent);

	POINT point;
	::GetCursorPos(&point);

    if(m_bAutoHide) 
    {

	    if( point.x < m_ptLT.x || 
		    point.x > m_ptLT.x + 20 ||
		    point.y < m_ptLT.y ||
		    point.y > m_ptLT.y + 20
	      )
	    {
		    KillTimer(nIDEvent);
		    DestroyWindow();
            return;
	    }
    }

    if(m_pClient)
    {
        CRect rect;
        m_pClient->GetWindowRect(&rect);
        if(!rect.PtInRect(point))
        {
            KillTimer(nIDEvent);
		    DestroyWindow();
            return;
        }
    }
}


void CMultiLineTip::OnPaint() 
{
	CPaintDC dc(this);

	COLORREF colorBkg = ::GetSysColor(COLOR_INFOBK);
	COLORREF colorText = ::GetSysColor(COLOR_INFOTEXT);

	CRect rcClient;
	GetClientRect(&rcClient);
	dc.FillSolidRect(&rcClient, colorBkg);

	dc.SetBkColor(colorBkg);
	dc.SetTextAlign(TA_LEFT | TA_TOP);
	dc.SetTextColor(colorText);

	int yOffset = 2;
	const int xOffset = 4;

	CFont* poldFont = dc.SelectObject(&m_boldFont);
	dc.TextOut(xOffset, yOffset, m_stHead);
	yOffset += m_nHeadHeight;

	CSize Size	 = dc.GetTabbedTextExtent(_T("\t"), 1, 0, NULL);
	int nTabPos = Size.cx;

    bool isBoldFontSelected = true;

	for(int n = 0; n < m_Lines.GetSize(); n++)
	{
        if(m_Lines[n].isBold && !isBoldFontSelected) {
            dc.SelectObject(&m_boldFont);
            isBoldFontSelected = true;
        }
        else if(!m_Lines[n].isBold && isBoldFontSelected) {
            dc.SelectObject(&m_Font);
            isBoldFontSelected = false;
        }

        const CString& strLeft  = m_Lines[n].stLeft;
        const CString& strRight = m_Lines[n].stRight;

		dc.TextOut(xOffset, yOffset, strLeft);
		dc.TabbedTextOut(xOffset + m_Lines[n].nOffset, yOffset, strRight, 1, &nTabPos, xOffset + m_Lines[n].nOffset);

        if(strLeft.IsEmpty() && strRight.IsEmpty())
            yOffset += m_nEmptyLineHeight;
        else
		    yOffset += m_nLineHeight;
	}

	dc.SelectObject(poldFont);
}


void CMultiLineTip::CalculateSize(int& nWidth, int& nHeight)
{
    SIZE size;
	HDC screenDC = ::GetDC(0);
	HDC dc = ::CreateCompatibleDC(screenDC);
	::ReleaseDC(0, screenDC);

	HFONT oldFont = (HFONT)::SelectObject(dc, m_boldFont.m_hObject);    	    
	::GetTextExtentPoint32(dc, m_stHead, m_stHead.GetLength(), &size);
	m_nHeadHeight = int(size.cy * 1.4);
	int nHeadWidth = size.cx;
    
    if(m_stHead.IsEmpty())
    {
        nHeadWidth = 0;
        m_nHeadHeight = 0;
    }
	
	::SelectObject(dc, m_Font.m_hObject);

	int nTabPos = LOWORD(::GetTabbedTextExtent(dc, _T("\t"), 1, 0, NULL));

	int n = 0;
	
	int nAllMax = 0;
	int nLeftMax = 0;
	int nRightMax = 0;


    int countEmptyLines = 0;
    	
	for(n = 0; n < m_Lines.GetSize(); n++)
	{
		const CString& Left = m_Lines[n].stLeft;
		const CString& Right = m_Lines[n].stRight;

        if(Left.IsEmpty() && Right.IsEmpty())
            ++countEmptyLines;

		::GetTextExtentPoint32(dc, Left, Left.GetLength(), &size);
		if(size.cx > nLeftMax)
			nLeftMax = size.cx;

		size.cx = LOWORD(::GetTabbedTextExtent(dc, Right, Right.GetLength(), 1, &nTabPos));
		if(Left.IsEmpty())
		{
			if(size.cx > nAllMax)
				nAllMax = size.cx;
		}
		else
		{
			if(size.cx > nRightMax)
				nRightMax = size.cx;
		}
	}
	
	::GetTextExtentPoint32(dc, _T(" "), 1, &size);
	int nSpaceWidth = size.cx * 2;
	m_nLineHeight = size.cy;
    m_nEmptyLineHeight = m_nLineHeight / 2;

	for(n = 0; n < m_Lines.GetSize(); n++)
	{
		if(!m_Lines[n].stLeft.IsEmpty())
			m_Lines[n].nOffset = nLeftMax + nSpaceWidth;
		else
			m_Lines[n].nOffset = 0;
	}

	
	nWidth = nLeftMax + nSpaceWidth + nRightMax; 
	if(nAllMax > nWidth) nWidth = nAllMax;
	if(nHeadWidth > nWidth) nWidth = nHeadWidth;
	nWidth += 10;

	nHeight = 6 + m_nHeadHeight + 
             (m_nLineHeight      * (m_Lines.GetSize()-countEmptyLines)) +
             (m_nEmptyLineHeight * countEmptyLines);

	::SelectObject(dc, oldFont);
	::DeleteDC(dc);
}

void CMultiLineTip::OnSettingChange(UINT uFlags, LPCTSTR lpszSection) 
{
	CWnd::OnSettingChange(uFlags, lpszSection);
	CreateFonts();
}

void CMultiLineTip::CreateFonts()
{
	m_Font.DeleteObject();
	m_boldFont.DeleteObject();

	HWND hwndTooltip = ::FindWindow(_T("tooltips_class32"), NULL);

	HFONT Font = (HFONT)::SendMessage(hwndTooltip, WM_GETFONT, 0, 0);
	if(Font)
	{
		LOGFONT lf;
		::GetObject(Font, sizeof(lf), &lf);
		m_Font.CreateFontIndirect(&lf);
	}
	else
	{ 
		m_Font.CreatePointFont(80, _T("MS Sans Serif"));
	}


	LOGFONT lf;
	::GetObject(m_Font, sizeof(lf), &lf);
	lf.lfWeight = FW_BOLD;
	m_boldFont.CreateFontIndirect(&lf);
}


void CMultiLineTip::OnMouseMove(UINT nFlags, CPoint point) 
{
	CWnd::OnMouseMove(nFlags, point);

    if(m_pMouseMoveReceiver && m_fnMouseMove)
        m_fnMouseMove(m_pMouseMoveReceiver);
}
