// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// MessageView.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "PopMan.h"
#include "MessageView.h"
#include <afxdlgs.h>
#include "RegKey.h"
#include "Mail.h"
#include "PopManDoc.h"
#include "MessageFrm.h"
#include "DlgSource.h"
#include "StrFunctions.h"
#include "FileDlgEx.h"
#include "PrintMail.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


const TCHAR szFaceNameValue[] = _T("FaceName");
const TCHAR szPointsValue[]   = _T("Points");
const TCHAR szBoldValue[]     = _T("Bold");
const TCHAR szItalicValue[]   = _T("Italic");
const TCHAR szForeColorValue[]= _T("ForeColor");


static UINT WM_FINDREPLACE = ::RegisterWindowMessage(FINDMSGSTRING);

CPtrList	CMessageView::m_List;


// uncomment the next define if you want a PopMan without attachment support 
// (attachments can't be opend or saved):

//#define NO_ATTACHMENTS 


/////////////////////////////////////////////////////////////////////////////
// CMessageView

IMPLEMENT_DYNCREATE(CMessageView, CFormView)

CMessageView::CMessageView()
	: CFormView(CMessageView::IDD)
{
	m_pFindDlg = NULL;
	m_bMatchCase = FALSE;
	m_bMatchWholeWord = FALSE;
	m_bSearchDown = TRUE;
	m_bSmallHead = FALSE;
	m_pMail = NULL;
	m_hIconVirus = NULL;
	m_List.AddTail(this);
	m_bResetTimerID = 0;
	m_bDelayReset = true;
	//{{AFX_DATA_INIT(CMessageView)
	//}}AFX_DATA_INIT
}

CMessageView::~CMessageView()
{
	POSITION pos = m_List.Find(this);
	if(pos)
		m_List.RemoveAt(pos);
}

void CMessageView::DoDataExchange(CDataExchange* pDX)
{
	CFormView::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CMessageView)
	DDX_Control(pDX, IDC_MESSAGE_TO, m_editTo);
	DDX_Control(pDX, IDC_MESSAGE_FROM, m_editFrom);
	DDX_Control(pDX, IDC_MESSAGE_DATE, m_editDate);
	DDX_Control(pDX, IDC_MESSAGE_SUBJECT, m_editSubject);
	DDX_Control(pDX, IDC_LIST_ATTACHMENTS, m_ListAttachments);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CMessageView, CFormView)
	//{{AFX_MSG_MAP(CMessageView)
	ON_WM_SIZE()
	ON_COMMAND(ID_EDIT_SELECT_ALL, OnSelectAll)
	ON_COMMAND(ID_EDIT_COPY, OnCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateCopy)
	ON_COMMAND(ID_EXTRAS_FONT, OnFont)
	ON_WM_DESTROY()
	ON_COMMAND(ID_EXTRAS_FONT_COLOR, OnFontColor)
	ON_COMMAND(ID_FILE_PRINT, OnPrintMessage)
	ON_COMMAND(ID_EDIT_FIND, OnFind)
	ON_COMMAND(ID_EDIT_FIND_NEXT, OnFindNext)
	ON_WM_CREATE()
	ON_COMMAND(ID_FILE_SAVE_MAIL, OnSaveMail)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE_MAIL, OnUpdateSaveMail)
	ON_COMMAND(ID_FILE_VIEW, OnOpenMail)
	ON_UPDATE_COMMAND_UI(ID_FILE_VIEW, OnUpdateOpenMail)
	ON_COMMAND(ID_FILE_MAIL_SOURCE, OnMailSource)
	ON_UPDATE_COMMAND_UI(ID_FILE_MAIL_SOURCE, OnUpdateMailSource)
	ON_COMMAND(ID_FILE_VIEW_EXTERNALLY, OnViewExternally)
	ON_UPDATE_COMMAND_UI(ID_FILE_VIEW_EXTERNALLY, OnUpdateViewExternally)
	ON_COMMAND(ID_FILE_REPLY_TO, OnReplyTo)
	ON_UPDATE_COMMAND_UI(ID_FILE_REPLY_TO, OnUpdateReplyTo)
	ON_WM_PAINT()
	ON_NOTIFY(NM_RCLICK, IDC_LIST_ATTACHMENTS, OnRClickAttachments)
	ON_UPDATE_COMMAND_UI(ID_ATTACHMENT_OPEN, OnUpdateAttachmentOpen)
	ON_UPDATE_COMMAND_UI(ID_ATTACHMENT_SAVE, OnUpdateAttachmentSave)
	ON_UPDATE_COMMAND_UI(ID_ATTACHMENT_SAVE_ALL, OnUpdateAttachmentSaveAll)
	ON_COMMAND(ID_ATTACHMENT_OPEN, OnAttachmentOpen)
	ON_COMMAND(ID_ATTACHMENT_SAVE, OnAttachmentSave)
	ON_COMMAND(ID_ATTACHMENT_SAVE_ALL, OnAttachmentSaveAll)
	ON_NOTIFY(NM_DBLCLK, IDC_LIST_ATTACHMENTS, OnDblclkListAttachments)
	ON_COMMAND(ID_FILE_DELETE_MAILS, OnDeleteMail)
	ON_UPDATE_COMMAND_UI(ID_FILE_DELETE_MAILS, OnUpdateDeleteMail)
	ON_COMMAND(ID_GOTO_NEXT_TAB, OnGotoNextTab)
	ON_COMMAND(ID_GOTO_PREV_TAB, OnGotoPrevTab)
	ON_WM_TIMER()
	ON_COMMAND(ID_FILE_DELETE_UNMARK, OnUnmarkDelete)
	ON_COMMAND(ID_RULES_ADD_LISTITEM, OnRulesAddListitem)
	ON_UPDATE_COMMAND_UI(ID_RULES_ADD_LISTITEM, OnUpdateRulesAddListitem)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_FILE_DELETE_MAILS_SHIFT, OnDeleteShift)
	ON_REGISTERED_MESSAGE(WM_FINDREPLACE, OnFindDlgMessage)
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// Diagnose CMessageView

#ifdef _DEBUG
void CMessageView::AssertValid() const
{
	CFormView::AssertValid();
}

void CMessageView::Dump(CDumpContext& dc) const
{
	CFormView::Dump(dc);
}
#endif //_DEBUG


int CMessageView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CFormView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	m_TextBox.Create(this, CRect(0,0,1,1));
	
	return 0;
}


void CMessageView::OnInitialUpdate() 
{
	CFormView::OnInitialUpdate();
	
	// den eingesunkenen Rahmen beseitigen:		
	ModifyStyleEx(WS_EX_CLIENTEDGE, 0, SWP_FRAMECHANGED);

	LoadSettings();
	
	CFont* pFont = GetDlgItem(IDC_MESSAGE_STATIC_FROM)->GetFont();
	LOGFONT lf;
	pFont->GetLogFont(&lf);
	lf.lfWeight = FW_BOLD;
	m_FontStatic.CreateFontIndirect(&lf);

	GetDlgItem(IDC_MESSAGE_STATIC_FROM)->SetFont(&m_FontStatic);
	GetDlgItem(IDC_MESSAGE_STATIC_TO)->SetFont(&m_FontStatic);
	GetDlgItem(IDC_MESSAGE_STATIC_DATE)->SetFont(&m_FontStatic);
	GetDlgItem(IDC_MESSAGE_STATIC_SUBJECT)->SetFont(&m_FontStatic);
	GetDlgItem(IDC_MESSAGE_STATIC_ATTACHMENT)->SetFont(&m_FontStatic);
	

	if(m_bSmallHead)
	{
		GetDlgItem(IDC_MESSAGE_STATIC_TO)->ShowWindow(SW_HIDE);
		GetDlgItem(IDC_MESSAGE_TO)->ShowWindow(SW_HIDE);

		GetDlgItem(IDC_MESSAGE_STATIC_DATE)->ShowWindow(SW_HIDE);
		GetDlgItem(IDC_MESSAGE_DATE)->ShowWindow(SW_HIDE);

		GetDlgItem(IDC_MESSAGE_STATIC_ATTACHMENTS)->ShowWindow(SW_HIDE);
		GetDlgItem(IDC_LIST_ATTACHMENTS)->ShowWindow(SW_HIDE);
	}

	// Scrollleisten unterbinden:
	SetScrollSizes(MM_TEXT, CSize(1, 1));

	m_ListAttachments.SetBkColor(GetSysColor(COLOR_BTNFACE));
	m_ListAttachments.SetTextBkColor(GetSysColor(COLOR_BTNFACE));
	
	
	InitGUIText();

	::SetFocus(m_TextBox.m_hWnd);
}


void CMessageView::ArrangeHeadControls()
{
	const int cstLeftLabel = 6;
	const int cstWidthLabel = GetMaxLabelWidth();  //55;
	const int cstLeftText = cstLeftLabel + cstWidthLabel + 5;
	const int cstHeight = 18;
	const int cstDist = m_bSmallHead ? 3 : 3;
	
	int nWidthText = 0;
	int nTop = 5;

	RECT rect;
	GetClientRect(&rect);
	nWidthText = rect.right - cstLeftText - 2;

	
	MoveControl(IDC_MESSAGE_STATIC_FROM, cstLeftLabel, nTop, cstWidthLabel, cstHeight);
	MoveControl(IDC_MESSAGE_FROM, cstLeftText, nTop, nWidthText, cstHeight);
	nTop += cstHeight + cstDist;

	if(m_bSmallHead == FALSE)
	{
		MoveControl(IDC_MESSAGE_STATIC_DATE, cstLeftLabel, nTop, cstWidthLabel, cstHeight);
		MoveControl(IDC_MESSAGE_DATE, cstLeftText, nTop, nWidthText, cstHeight);
		nTop += cstHeight + cstDist;
	
		MoveControl(IDC_MESSAGE_STATIC_TO, cstLeftLabel, nTop, cstWidthLabel, cstHeight);
		MoveControl(IDC_MESSAGE_TO, cstLeftText, nTop, nWidthText, cstHeight);
		nTop += cstHeight + cstDist;
	}

	MoveControl(IDC_MESSAGE_STATIC_SUBJECT, cstLeftLabel, nTop, cstWidthLabel, cstHeight);
	MoveControl(IDC_MESSAGE_SUBJECT, cstLeftText, nTop, nWidthText, cstHeight);
	nTop += cstHeight + cstDist;

	MoveControl(IDC_MESSAGE_STATIC_ATTACHMENTS, cstLeftLabel, nTop+1, cstWidthLabel, cstHeight);
	MoveControl(IDC_LIST_ATTACHMENTS, cstLeftText, nTop+0, nWidthText, GetSystemMetrics(SM_CYSMICON)+6);
	if(m_ListAttachments.m_hWnd)
		m_ListAttachments.ModifyStyle(WS_HSCROLL, 0, SWP_FRAMECHANGED);
}


int CMessageView::GetMaxLabelWidth()
{
	CString stFrom;
	CString stTo;
	CString stSubject;
	CString stDate;
	CString stAttach;
	
	GetDlgItemText(IDC_MESSAGE_STATIC_FROM,			stFrom);
	GetDlgItemText(IDC_MESSAGE_STATIC_TO,			stTo);
	GetDlgItemText(IDC_MESSAGE_STATIC_SUBJECT,		stSubject);
	GetDlgItemText(IDC_MESSAGE_STATIC_DATE,			stDate);
	GetDlgItemText(IDC_MESSAGE_STATIC_ATTACHMENT,	stAttach);

	HDC screenDC = ::GetDC(0);
	HDC hDC = ::CreateCompatibleDC(screenDC);
	::ReleaseDC(0, screenDC);

	HFONT oldFont = (HFONT)::SelectObject(hDC, m_FontStatic.m_hObject);

	int nMaxWidth = 0;
	int nItemWidth = 0;
	
	nItemWidth = LOWORD(::GetTabbedTextExtent(hDC, stFrom, stFrom.GetLength(), 0, NULL)); 
	if(nItemWidth > nMaxWidth) nMaxWidth = nItemWidth;
	
	nItemWidth = LOWORD(::GetTabbedTextExtent(hDC, stSubject, stSubject.GetLength(), 0, NULL)); 
	if(nItemWidth > nMaxWidth) nMaxWidth = nItemWidth;

	if(!m_bSmallHead)
	{
		nItemWidth = LOWORD(::GetTabbedTextExtent(hDC, stTo, stTo.GetLength(), 0, NULL)); 
		if(nItemWidth > nMaxWidth) nMaxWidth = nItemWidth;

		nItemWidth = LOWORD(::GetTabbedTextExtent(hDC, stDate, stDate.GetLength(), 0, NULL)); 
		if(nItemWidth > nMaxWidth) nMaxWidth = nItemWidth;
	}

	BOOL bAttachment = (m_pMail && m_pMail->m_Attachments.GetCount() > 0);
	if(bAttachment)
	{
		nItemWidth = LOWORD(::GetTabbedTextExtent(hDC, stAttach, stAttach.GetLength(), 0, NULL)); 
		if(nItemWidth > nMaxWidth) nMaxWidth = nItemWidth;
	}
	
	nMaxWidth += LOWORD(::GetTabbedTextExtent(hDC, _T(" "), 1, 0, NULL)); 
	
	
	::SelectObject(hDC, oldFont);
	::DeleteDC(hDC);

	return nMaxWidth;
}

void CMessageView::OnDestroy() 
{
	SaveSettings();
	CFormView::OnDestroy();
}

void CMessageView::OnSize(UINT nType, int cx, int cy) 
{
	CFormView::OnSize(nType, cx, cy);

	ArrangeHeadControls();
	ArrangeTextBox(cx, cy);
	ArrangeAttachments();
}

void CMessageView::ArrangeTextBox(int cx, int cy)
{
	if(m_TextBox.GetSafeHwnd())
	{
		CWnd* pWnd = NULL;
		int yOffset = 85;

		BOOL bAttachment = (m_pMail && m_pMail->m_Attachments.GetCount() > 0);
		
		
		pWnd = GetDlgItem(IDC_MESSAGE_STATIC_ATTACHMENTS);
		if(pWnd)
			pWnd->ShowWindow(bAttachment ? SW_SHOW : SW_HIDE);

		pWnd = GetDlgItem(IDC_LIST_ATTACHMENTS);
		if(pWnd)
			pWnd->ShowWindow(bAttachment ? SW_SHOW : SW_HIDE);


		pWnd = GetDlgItem(bAttachment ? IDC_LIST_ATTACHMENTS : IDC_MESSAGE_SUBJECT);
		if(pWnd)
		{
			RECT rect;
			pWnd->GetWindowRect(&rect);	

			POINT point = { rect.right, rect.bottom };
			ScreenToClient(&point);
			rect.bottom = point.y;

			yOffset = rect.bottom + 6;
		}
		if(m_bSmallHead)
			m_TextBox.MoveWindow(-2, yOffset, cx+4, cy - yOffset + 2);
		else
			m_TextBox.MoveWindow(0, yOffset, cx, cy - yOffset);
	}
}

void CMessageView::OnSelectAll() 
{
	m_TextBox.SelectAll();
}


void CMessageView::OnCopy() 
{
	HWND hEditBox = ::GetFocus();
	if(hEditBox)
		::SendMessage(hEditBox, WM_COPY, 0, 0);
}

void CMessageView::OnUpdateCopy(CCmdUI* pCmdUI) 
{
	DWORD dwStart = 0, dwEnd = 0;

	HWND hEditBox = ::GetFocus();
	if(hEditBox)
		::SendMessage(hEditBox, EM_GETSEL, (WPARAM)&dwStart, (LPARAM)&dwEnd);
	
	pCmdUI->Enable(dwStart != dwEnd);
}

void CMessageView::OnFont() 
{
	LOGFONT lgFont;
	m_Font.GetLogFont(&lgFont);
	
	CFontDialog fDlg(&lgFont, CF_SCREENFONTS);
	if(IDOK != fDlg.DoModal())
		return;
	
	
	m_Font.DeleteObject();
	
	fDlg.GetCurrentFont(&lgFont);
	m_Font.CreateFontIndirect(&lgFont);
	m_TextBox.SetFont(&m_Font);	
	UpdateAllViews();
}


void CMessageView::OnFontColor() 
{
	CColorDialog cDlg(m_TextBox.m_TextColor);
	if(IDOK != cDlg.DoModal())
		return;

	m_TextBox.m_TextColor = cDlg.GetColor();
	m_TextBox.Invalidate();
	m_TextBox.UpdateWindow();	
	UpdateAllViews();
}


void CMessageView::LoadSettings()
{
	CString  stFaceName(_T("Courier New"));
	int  	 nPoints = 10;
	BOOL	 bBold   = FALSE;
	BOOL	 bItalic = FALSE;
	COLORREF clForeColor = m_TextBox.m_TextColor;

	CSettingsKey Key;
	if(OpenSettingsKey(Key, szMsgWindowKey))
	{
		Key.QueryValue(szFaceNameValue, stFaceName);
		Key.QueryValue(szPointsValue,   nPoints);
		Key.QueryValue(szBoldValue,     bBold);
		Key.QueryValue(szItalicValue,   bItalic);
		Key.QueryValue(szForeColorValue, clForeColor);
	}

	m_Font.CreateFont(HeightFromPoints(nPoints), 0, 0, 0, bBold ? FW_BOLD : FW_NORMAL,
					 (BYTE)bItalic, (BYTE)FALSE, 0, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, 
					 CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, stFaceName);
	

	m_TextBox.SetFont(&m_Font);
	m_TextBox.m_TextColor = clForeColor;
}


void CMessageView::SaveSettings()
{
	LOGFONT lf;
	m_Font.GetLogFont(&lf);
		
	CString stFaceName(lf.lfFaceName);
	int  	nPoints = PointsFromHeight(lf.lfHeight);
	BOOL	bBold   = (lf.lfWeight > FW_NORMAL);
	BOOL	bItalic = lf.lfItalic;
	
	CSettingsKey Key;
	if(CreateSettingsKey(Key, szMsgWindowKey))
	{
		Key.SetValue(szFaceNameValue, stFaceName);
		Key.SetValue(szPointsValue,   nPoints);
		Key.SetValue(szBoldValue,     bBold);
		Key.SetValue(szItalicValue,   bItalic);
		Key.SetValue(szForeColorValue,m_TextBox.m_TextColor);
	}
}


inline int CMessageView::PointsFromHeight(int iHeight)
{
	HDC hDC = ::GetDC(NULL);
	int iPoints = -MulDiv(iHeight, 72, ::GetDeviceCaps(hDC, LOGPIXELSY));
	::ReleaseDC(NULL, hDC);
	return iPoints;
}

inline int CMessageView::HeightFromPoints(int iPoints)
{
	HDC hDC = ::GetDC(NULL);
	int iHeight = -MulDiv(iPoints, ::GetDeviceCaps(hDC, LOGPIXELSY), 72);
	::ReleaseDC(NULL, hDC);
	return iHeight;
}


void CMessageView::OnPrintMessage() 
{
	OnPrintMail(m_pMail, TRUE, m_TextBox.GetFont(), this);
}

void CMessageView::OnFind() 
{
	if(m_pFindDlg)
	{
		m_pFindDlg->SetFocus();
		return;
	}

	int nStart, nEnd;
	m_TextBox.GetSel(nStart, nEnd);
	if(nStart != nEnd)
	{
		m_stFind = m_TextBox.GetText();
		m_stFind = m_stFind.Mid(nStart, nEnd-nStart);
		int nCR = m_stFind.Find(_T('\r'));
		if(nCR != -1)
			m_stFind = m_stFind.Left(nCR);
	}

	ASSERT(m_pFindDlg == NULL);
	m_pFindDlg = new CFindReplaceDialog();
	DWORD dwFlags = (m_bSearchDown ? FR_DOWN : 0) | (m_bMatchWholeWord ? FR_WHOLEWORD : 0) | (m_bMatchCase ? FR_MATCHCASE : 0);
	m_pFindDlg->Create(TRUE, m_stFind, NULL, dwFlags, this);
}

void CMessageView::OnFindNext() 
{
	if(m_stFind.IsEmpty())
		OnFind();
	else
		FindText(m_stFind, m_bMatchCase, m_bMatchWholeWord, m_bSearchDown);
}


LRESULT CMessageView::OnFindDlgMessage(WPARAM wParam, LPARAM lParam)
{
	ASSERT(m_pFindDlg != NULL);

	if (m_pFindDlg->IsTerminating())
	{
		m_pFindDlg = NULL;
		return 0;
	}

	if(m_pFindDlg->FindNext())
	{
		m_stFind = m_pFindDlg->GetFindString();
		m_bMatchCase = m_pFindDlg->MatchCase();
		m_bMatchWholeWord = m_pFindDlg->MatchWholeWord();
		m_bSearchDown = m_pFindDlg->SearchDown();

		FindText(m_stFind, m_bMatchCase, m_bMatchWholeWord, m_bSearchDown);
	}

	return 0;
}


void CMessageView::FindText(CString stFind, BOOL bMatchCase, BOOL bMatchWholeWord, BOOL bSearchDown)
{
	CString stText;
	stText = m_TextBox.GetText();

	int nStart, nEnd;
	m_TextBox.GetSel(nStart, nEnd);
	int iStartPos = (bSearchDown ? nEnd : nStart - 1);

	int posFound = AdvancedSearch(bSearchDown, stText, stFind, iStartPos, bMatchCase, bMatchWholeWord);
	
	if(posFound > -1)
	{	// gefunden:
		int nStart = posFound;
		int nEnd = nStart + stFind.GetLength();
		m_TextBox.SetSel(nStart, nEnd);
	}
	else
	{	// nicht gefunden:
		CString stPrompt = StrFormat(i18n("\"{1}\" not found."), _T("s"), (LPCTSTR)stFind);
		MessageBox(stPrompt, NULL, MB_OK | MB_ICONINFORMATION);
		if(m_pFindDlg != NULL)
			m_pFindDlg->SetActiveWindow();
	}
}


void CMessageView::OnTimer(UINT nIDEvent) 
{
	if(m_bResetTimerID != 0) 
	{
		KillTimer(m_bResetTimerID);
		m_bResetTimerID = 0;
		m_bDelayReset = false;
		ShowMail(NULL);
		m_bDelayReset = true;
	}
	

	CFormView::OnTimer(nIDEvent);
}

void CMessageView::ShowMail(CMail *pMail)
{

	if(m_bDelayReset)
	{
		if(pMail == NULL)
		{
			if(m_bResetTimerID != 0) return;
			// init Reset timer:
			m_bResetTimerID = SetTimer(1, 50, NULL);
			return;
		}
		else
		{	// abort Reset:
			if(m_bResetTimerID != 0)
				KillTimer(m_bResetTimerID);
			m_bResetTimerID = 0;
		}
	}

	if(pMail != NULL)
	{
		try 
		{
			pMail = dynamic_cast<CMail*>((CObject*)pMail);
			if(pMail == 0)
				return;
		} 
		catch (...)
		{
			return;
		}
	}

	m_pMail = pMail;
	
	CRect rect;
	GetClientRect(&rect);
	ArrangeTextBox(rect.Width(), rect.Height());
	ArrangeHeadControls();


	if(pMail == NULL)
	{
		SetDlgItemText(IDC_MESSAGE_FROM, _T(""));
		SetDlgItemText(IDC_MESSAGE_TO, _T(""));
		SetDlgItemText(IDC_MESSAGE_SUBJECT, _T(""));
		SetDlgItemText(IDC_MESSAGE_DATE, _T(""));
		m_TextBox.SetText(_T(""));
		if(m_ListAttachments.m_hWnd)
			m_ListAttachments.DeleteAllItems();
		return;
	}

	SetDlgItemText(IDC_MESSAGE_FROM, pMail->GetFromComplete());
	SetDlgItemText(IDC_MESSAGE_TO, pMail->GetTo());
	SetDlgItemText(IDC_MESSAGE_SUBJECT, pMail->GetSubject());

	
	COleDateTime DateTime(pMail->GetDate());
	//CString stDate =  DateTime.Format(_T("%#c"));
	CString stDate = DateTime.Format();
	SetDlgItemText(IDC_MESSAGE_DATE, stDate);

	bool bNotify = !pMail->IsRead();
	m_TextBox.SetText(pMail->GetExtractedMsg(), &pMail->GetHtmlLinks());
	pMail->SetRead();

	if(bNotify)
		AfxGetMainWnd()->SendMessageToDescendants(WM_MAIL_READ, (WPARAM)pMail, 0);
	

	ListAttachments();

}

void CMessageView::ListAttachments()
{

	m_ListAttachments.DeleteAllItems();

	if(m_pMail)
	{
		LV_ITEM lvItem;
		lvItem.mask = LVIF_TEXT | LVIF_IMAGE;

		m_ImageList.DeleteImageList();
		m_ImageList.Create(GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON), ILC_COLOR24, m_pMail->m_Attachments.GetCount()+1, 1); 
		m_ListAttachments.SetImageList(&m_ImageList, LVSIL_SMALL);

		POSITION pos = m_pMail->m_Attachments.GetHeadPosition();
		while(pos != NULL)
		{
			CAttachment* pAttachment = m_pMail->m_Attachments.GetNext(pos);
			HICON hIcon = NULL;
			if(pAttachment->IsExecuteable()) 
				hIcon = m_hIconVirus;
			else
				hIcon = m_ExtIcon.GetIconFromExtension(pAttachment->GetExtension());

			m_ImageList.Add(hIcon);
			lvItem.iItem = m_ListAttachments.GetItemCount();
			lvItem.iImage = lvItem.iItem;
			lvItem.iSubItem = 0;
			CString stName = pAttachment->GetFileName();
			stName += _T(" (") + (pAttachment->IsComplete() ? pAttachment->GetSizeFormated() : _T("??")) + _T(')');
			lvItem.pszText = (LPTSTR)(LPCTSTR)stName;
			m_ListAttachments.InsertItem(&lvItem);
			m_ListAttachments.SetItemData(lvItem.iItem, (DWORD)pAttachment);
		}

		ArrangeAttachments();
	}
}



void CMessageView::ArrangeAttachments()
{
	if(m_ListAttachments.m_hWnd == 0 || m_ListAttachments.GetItemCount() == 0)
		return;

	m_ListAttachments.Arrange(LVA_ALIGNTOP);
		
	bool bScrollBars = false;
	CRect prevRect, curRect;
	for(int i = 1; i < m_ListAttachments.GetItemCount(); ++i)
	{
		m_ListAttachments.GetItemRect(i-1, &prevRect, LVIR_BOUNDS);
		m_ListAttachments.GetItemRect(i, &curRect, LVIR_BOUNDS);

		bool bSameLine = abs(prevRect.top - curRect.top) < 8;
		if(prevRect.right > curRect.left && bSameLine)
			m_ListAttachments.SetItemPosition(i, CPoint(prevRect.right + 5, curRect.top));
		if(!bSameLine)
			bScrollBars = true;
	}

	DWORD Add = bScrollBars ? WS_VSCROLL : LVS_NOSCROLL;
	DWORD Remove = bScrollBars ? LVS_NOSCROLL : WS_VSCROLL;
	m_ListAttachments.ModifyStyle(WS_HSCROLL | Remove, Add, SWP_FRAMECHANGED);
}


void CMessageView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{	
	// no call to base function (would cause flicker!)
}

void CMessageView::UpdateDisplay(CFont *pFont, COLORREF TextColor)
{
	m_TextBox.m_TextColor = TextColor;

	if(pFont)
	{
		m_Font.DeleteObject();
		LOGFONT lf;
		pFont->GetLogFont(&lf);
		m_Font.CreateFontIndirect(&lf);
		m_TextBox.SetFont(&m_Font);
	}
}

void CMessageView::UpdateAllViews()
{
	POSITION pos = m_List.GetHeadPosition();
	while(pos)
	{
		CMessageView* pMsg = (CMessageView*)m_List.GetNext(pos);	

		if(pMsg == this)
			continue;

		pMsg->UpdateDisplay(&m_Font, m_TextBox.m_TextColor);
	}
	SaveSettings();
}

void CMessageView::OnSaveMail() 
{
	CPopManDoc* pDoc = GetDocument();
	CMails MailList;
	MailList.AddTail(m_pMail);
	if(pDoc)
		pDoc->SaveMail(&MailList);
}

void CMessageView::OnUpdateSaveMail(CCmdUI* pCmdUI) 
{
	CPopManDoc* pDoc = GetDocument();
	pCmdUI->Enable(pDoc != NULL && pDoc->IsMailValid(m_pMail) && !pDoc->ConnectionsPending());	
}

void CMessageView::OnOpenMail() 
{
	CMessageFrm* pMessage = new CMessageFrm(GetDocument(), m_pMail);
}

void CMessageView::OnUpdateOpenMail(CCmdUI* pCmdUI) 
{
	CPopManDoc* pDoc = GetDocument();
	pCmdUI->Enable(pDoc != NULL && pDoc->IsMailValid(m_pMail));	
}

void CMessageView::OnMailSource() 
{
	CDlgSource* pDlgSrc = new CDlgSource(m_pMail->GetSubject(), m_pMail->GetSource(), this->GetParentFrame());
}

void CMessageView::OnUpdateMailSource(CCmdUI* pCmdUI) 
{
	CPopManDoc* pDoc = GetDocument();
	pCmdUI->Enable(pDoc != NULL && pDoc->IsMailValid(m_pMail));	
}

void CMessageView::OnViewExternally()
{
	CPopManDoc* pDoc = GetDocument();
	pDoc->ViewMailExternally(m_pMail);
}

void CMessageView::OnUpdateViewExternally(CCmdUI* pCmdUI)
{
	CPopManDoc* pDoc = GetDocument();
	pCmdUI->Enable(pDoc != NULL && pDoc->IsMailValid(m_pMail));
}

void CMessageView::OnReplyTo() 
{
	CPopManDoc* pDoc = GetDocument();
	if(pDoc)
		pDoc->ReplyToMail(m_pMail);
}

void CMessageView::OnUpdateReplyTo(CCmdUI* pCmdUI) 
{
	CPopManDoc* pDoc = GetDocument();
	pCmdUI->Enable(pDoc != NULL && pDoc->IsMailValid(m_pMail));		
}

void CMessageView::OnPaint() 
{
	CPaintDC dc(this); // device context for painting

	if(m_bSmallHead)
	{
		RECT rect;
		GetClientRect(&rect);
		
		CPen Pen1(PS_SOLID, 1, GetSysColor(COLOR_3DHIGHLIGHT));
		CPen* pOldPen = dc.SelectObject(&Pen1);
		dc.MoveTo(rect.right, 0);
		dc.LineTo(0, 0);
		dc.LineTo(0, rect.bottom);
		
		dc.SelectObject(pOldPen);
	}
}


void CMessageView::MoveControl(int ID, int Left, int Top, int Width, int Height)
{
	CWnd * pWnd = GetDlgItem(ID);
	if(pWnd == NULL || pWnd->GetSafeHwnd() == NULL)
		return;

	RECT rect;

	rect.left = Left;
	rect.top = Top;
	rect.bottom = Top + Height;
	rect.right = Left + Width;
	pWnd->MoveWindow(&rect);
}



static void MenuTextCmd(CMenu* pMenu, int Cmd, LPCTSTR szText)
{
	pMenu->ModifyMenu(Cmd, MF_BYCOMMAND|MF_STRING, Cmd, szText);
}

void CMessageView::OnRClickAttachments(NMHDR* pNMHDR, LRESULT* pResult) 
{

#ifndef NO_ATTACHMENTS
	
	CMenu Menu;
	if(!Menu.LoadMenu(IDR_ATTACHMENT))
		return;

	CMenu* pSubMenu = Menu.GetSubMenu(0);
	if(pSubMenu == NULL)
		return;
	
	pSubMenu->SetDefaultItem(0, TRUE);
	CPoint mouse;
	GetCursorPos(&mouse);

	MenuTextCmd(pSubMenu, ID_ATTACHMENT_OPEN,		i18n("&Open"));
	MenuTextCmd(pSubMenu, ID_ATTACHMENT_SAVE,		i18n("&Save As..."));
	MenuTextCmd(pSubMenu, ID_ATTACHMENT_SAVE_ALL,	i18n("Save &All..."));
		
	pSubMenu->TrackPopupMenu(TPM_RIGHTBUTTON|TPM_LEFTALIGN , mouse.x, mouse.y, GetParentFrame(), NULL);

#endif

	*pResult = 0;
}

void CMessageView::OnUpdateAttachmentOpen(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_ListAttachments.GetSelectedCount() == 1);	
}

void CMessageView::OnUpdateAttachmentSave(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_ListAttachments.GetSelectedCount() >= 1);	
}

void CMessageView::OnUpdateAttachmentSaveAll(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_ListAttachments.GetItemCount() >= 0);	
}

void CMessageView::OnAttachmentOpen() 
{
#ifndef NO_ATTACHMENTS

	POSITION pos = m_ListAttachments.GetFirstSelectedItemPosition();
	if(pos == NULL) return;

	int nItem = m_ListAttachments.GetNextSelectedItem(pos);
	
	CAttachment* pAttachment = (CAttachment*)m_ListAttachments.GetItemData(nItem);

	if(!IsAttachmentValid(pAttachment))
	{
		AfxMessageBox(i18n("This attachment can not be accessed anymore."));
		return;
	}
		
	if(pAttachment->IsExecuteable()) {
		CString stMsg = StrFormat(i18n("\"{1}\" is very likely to contain a virus, worm or any other sort of dangerous code!"), _T("s"), (LPCTSTR)pAttachment->GetFileName());
		stMsg += _T("\r\n");
		stMsg += i18n("Opening this file is therefore denied.");

		AfxMessageBox(stMsg, MB_OK | MB_ICONEXCLAMATION);
		return;
	}


	if(!pAttachment->IsComplete())
	{
		if(!CompleteAttachments())
			return;
	}


	CString stFileName = CPopManApp::GetAppTempPath();
	if(stFileName.Right(1) != _T("\\")) stFileName += _T('\\');
	stFileName += pAttachment->GetFileName();

	if (SaveAttachment(stFileName, pAttachment)) {		
		::ShellExecute(NULL, NULL, stFileName, NULL, NULL, SW_SHOWNORMAL);
	}	

#endif
}

BOOL CMessageView::IsAttachmentValid(const CAttachment* pAttachment)
{
	if(pAttachment != NULL)
	{
		try 
		{
			pAttachment = dynamic_cast<CAttachment*>((CObject*)pAttachment);
		} 
		catch (...)
		{
			pAttachment = NULL;
		}
	}
	return (pAttachment != NULL);
}


void CMessageView::OnAttachmentSave() 
{
#ifndef NO_ATTACHMENTS

	if(m_ListAttachments.GetSelectedCount() == 1)
	{
		POSITION pos = m_ListAttachments.GetFirstSelectedItemPosition();
		if(pos == NULL) return;

		int nItem = m_ListAttachments.GetNextSelectedItem(pos);
		CAttachment* pAttachment = (CAttachment*)m_ListAttachments.GetItemData(nItem);

		if(!IsAttachmentValid(pAttachment))
		{
			AfxMessageBox(i18n("This attachment can not be accessed anymore."));
			return;
		}
		
		
		if(pAttachment->IsExecuteable()) {
			CString stMsg = StrFormat(i18n("\"{1}\" is very likely to contain a virus, worm or any other sort of dangerous code!"), _T("s"), (LPCTSTR)pAttachment->GetFileName());
			stMsg += _T("\r\n");
			stMsg += i18n("Do you really want to save this file to your hard disk?");

			if(IDNO == AfxMessageBox(stMsg, MB_YESNO | MB_DEFBUTTON2 | MB_ICONEXCLAMATION))
				return;
		}

		if(!pAttachment->IsComplete())
		{
			if(!CompleteAttachments())
				return;
		}

		CString stFileName = pAttachment->GetFileName();
		CFileDlgEx Dlg;
		Dlg.AddFilter(i18n("Attachment"), _T("*.*"));
		if(!Dlg.GetSaveFileName(stFileName))
			return;
		
		SaveAttachment(stFileName, pAttachment);
		
	
	} 
	else
	{
		CAttachments mList;
		
		POSITION pos = m_ListAttachments.GetFirstSelectedItemPosition();
		while(pos != NULL)
		{
			int nItem = m_ListAttachments.GetNextSelectedItem(pos);
			CAttachment* pAttachment = (CAttachment*)m_ListAttachments.GetItemData(nItem);
			if(!IsAttachmentValid(pAttachment))
			{
				AfxMessageBox(i18n("This attachment can not be accessed anymore."));
				return;
			}
			mList.AddTail(pAttachment);
		}

		SaveAttachmentList(mList);
	}

#endif
}

BOOL CMessageView::SaveAttachmentList(CAttachments& List)
{
#ifndef NO_ATTACHMENTS

	BOOL bExec = FALSE;
	BOOL bIncomplete = FALSE;

	POSITION pos = List.GetHeadPosition();
	while(pos != NULL)
	{
		CAttachment* pAttachment = List.GetNext(pos);
		if(pAttachment->IsExecuteable())
			bExec = TRUE;
		
		if(!pAttachment->IsComplete())
			bIncomplete = TRUE;
	}

	if(bExec) 
	{
		CString stMsg;
		stMsg =  i18n("These attachments are very likely to contain a virus, worm or any other sort of dangerous code!");
		stMsg += _T("\r\n");
		stMsg += i18n("Do you really want to save these files to your hard disk?");

		if(IDNO == AfxMessageBox(stMsg, MB_YESNO | MB_DEFBUTTON2 | MB_ICONEXCLAMATION))
			return FALSE;
	}

	if(bIncomplete)
	{
		if(!CompleteAttachments())
			return FALSE;
	}

	
	CString stFolder;
	if(!CFileDlgEx::BrowseForFolder(stFolder, i18n("Please specify where to save the attachments.")))
		return FALSE;

	if(stFolder.Right(1) != _T("\\")) stFolder += _T('\\');

	CString stFileName;
	pos = List.GetHeadPosition();
	while(pos != NULL)
	{
		CAttachment* pAttachment = List.GetNext(pos);
		stFileName = stFolder + pAttachment->GetFileName();
		CFileStatus FileStatus;
		if(TRUE == CFile::GetStatus(stFileName, FileStatus))
		{
			CString stMsg = StrFormat(i18n("The file \"{1}\" does already exist!\r\nOverwrite it?"), _T("s"), (LPCTSTR)stFileName);
			if(IDYES != AfxMessageBox(stMsg, MB_YESNO | MB_DEFBUTTON2 | MB_ICONEXCLAMATION))
				continue;
		}
		
		if(!SaveAttachment(stFileName, pAttachment))
			return FALSE;
	}
	return TRUE;

#else
	return FALSE;
#endif
}

void CMessageView::OnAttachmentSaveAll() 
{
#ifndef NO_ATTACHMENTS

	if(m_ListAttachments.GetItemCount() == 1)
	{
		m_ListAttachments.SetItemState(0, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
		OnAttachmentSave();
		return;
	}


	CAttachments mList;
	
	for(int n = 0; n < m_ListAttachments.GetItemCount(); n++)
	{
		CAttachment* pAttachment = (CAttachment*)m_ListAttachments.GetItemData(n);
		if(!IsAttachmentValid(pAttachment))
		{
			AfxMessageBox(i18n("This attachment can not be accessed anymore."));
			return;
		}
		mList.AddTail(pAttachment);
	}

	SaveAttachmentList(mList);

#endif
}

void CMessageView::OnDblclkListAttachments(NMHDR* pNMHDR, LRESULT* pResult) 
{
	*pResult = 0;
	OnAttachmentOpen();
}



BOOL CMessageView::SaveAttachment(LPCTSTR szFileName, const CAttachment* pAttachment)
{
#ifndef NO_ATTACHMENTS
	try 
	{
		CFile File(szFileName, CFile::modeCreate | CFile::modeWrite);

		const CString& stData = pAttachment->GetContent();

		#ifndef _UNICODE
	
			File.Write(stData, stData.GetLength());

		#else

			int nLen = stData.GetLength();
			unsigned char* szBuf = new unsigned char[nLen];
            for(int i = 0; i < nLen; ++i)
                szBuf[i] = static_cast<unsigned char>(stData[i]);

			//WideCharToMultiByte(CP_ACP, 0, (LPCWSTR)stData, nLen, szBuf, nLen, NULL, NULL);
			
			File.Write(szBuf, nLen);
		
			delete[] szBuf;

		#endif
		
		File.Close();

	}
	catch(CFileException* e)
	{
		e->ReportError();
		e->Delete();
		return FALSE;
	}

	return TRUE;
#else
	return FALSE;
#endif
}

BOOL CMessageView::CompleteAttachments()
{
#ifndef NO_ATTACHMENTS

	CPopManDoc* pDoc = GetDocument();
	if(!pDoc->IsMailValid(m_pMail)) {
		AfxMessageBox(i18n("This attachment can not be accessed anymore."));
		return FALSE;
	}

	BOOL bConnectionCanceled = FALSE;
	if(!pDoc->DownloadMail(m_pMail, bConnectionCanceled, FALSE))
	{
		if(!bConnectionCanceled && !m_pMail->GetAccount()->IsCanceled())
			AfxMessageBox(i18n("Downloading email failed.\r\nPress Ctrl+F5 to update the email listing and try again."));

		return FALSE;
	}

	ListAttachments();

	return TRUE;

#else
	return FALSE;
#endif

}


void CMessageView::OnDeleteShift() 
{
	DeleteMail(true);
}

void CMessageView::OnDeleteMail() 
{
	bool bShift = (::GetKeyState(VK_SHIFT) & 0x8000) != 0;
	DeleteMail(bShift);
}

void CMessageView::OnUpdateDeleteMail(CCmdUI* pCmdUI) 
{
	CPopManDoc* pDoc = GetDocument();
	if(pDoc == NULL) return;

	if(m_pMail == NULL || pDoc->IsMailValid(m_pMail) == FALSE) {
		pCmdUI->Enable(FALSE);
		return;
	}


	if(pDoc->m_bMarkDelete)
		pCmdUI->Enable(!m_pMail->IsMarkedForDelete());
	else
		pCmdUI->Enable(!pDoc->ConnectionsPending());
}



BOOL CMessageView::DeleteMail(bool bShift)
{
	CWaitCursor Wait;

	CPopManDoc* pDoc = GetDocument();
	if(pDoc == NULL) return FALSE;
	if(!pDoc->IsMailValid(m_pMail))	return FALSE;


	if(GetDocument()->m_bMarkDelete && !bShift) 
	{
		m_pMail->SetMarkedForDelete();
		GetDocument()->ShowMarkMessagInfo();
		AfxGetMainWnd()->RedrawWindow();
		return TRUE;
	}
	else
	{
		if(pDoc->ConnectionsPending()) return FALSE;

		CMails MailList;
		MailList.AddTail(m_pMail);

		if(pDoc->m_bConfirmDeleteMsg)
		{
			int ret = NULL;

			CString stMsg, stPrompt;
			stMsg = i18n("Are you sure you want to delete this email?");
			stMsg += _T("\r\n\r\n");
			stMsg += _T("\"%s\"");
			stPrompt.Format(stMsg, (LPCTSTR)m_pMail->GetSubject());
			ret = AfxMessageBox(stPrompt, MB_YESNO);
			
			if(ret == IDNO)
				return FALSE;
		}
		
		if(pDoc->DeleteMails(&MailList))
			return TRUE;

		AfxMessageBox(i18n("Deleting email failed!"), MB_ICONERROR);
		return FALSE;
	}
}



void CMessageView::OnUnmarkDelete() 
{
	CPopManDoc* pDoc = GetDocument();
	if(pDoc == NULL) return;
	if(!pDoc->IsMailValid(m_pMail))	return;

	m_pMail->SetMarkedForDelete(FALSE);
	AfxGetMainWnd()->RedrawWindow();	
}


void CMessageView::InitGUIText()
{
	m_hIconVirus = CPopManApp::GetIcon(IDI_VIRUS);  // ::LoadIcon(::GetModuleHandle(0), MAKEINTRESOURCE(IDI_VIRUS));

	SetDlgItemText(IDC_MESSAGE_STATIC_FROM,			i18n("From:"));
	SetDlgItemText(IDC_MESSAGE_STATIC_DATE,			i18n("Date:"));
	SetDlgItemText(IDC_MESSAGE_STATIC_TO,			i18n("To:"));
	SetDlgItemText(IDC_MESSAGE_STATIC_SUBJECT,		i18n("Subject:"));
	SetDlgItemText(IDC_MESSAGE_STATIC_ATTACHMENTS,	i18n("Attach:"));

	ArrangeHeadControls();

	ListAttachments();

	m_TextBox.SetHoverLinkCursor(CPopManApp::GetCursor(IDC_HOVER_LINK));
}



void CMessageView::OnGotoNextTab() 
{

	AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_GOTO_NEXT_PANE, 0);
	AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_GOTO_NEXT_TAB, 0);
	
}

void CMessageView::OnGotoPrevTab() 
{
	AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_GOTO_NEXT_PANE, 0);
	AfxGetMainWnd()->SendMessage(WM_COMMAND, ID_GOTO_PREV_TAB, 0);
}

bool CMessageView::GetText(const CEdit& edit, CString& text)
{
    if(edit.GetSafeHwnd() != GetFocus()->GetSafeHwnd())
        return false;

    edit.GetWindowText(text);

    int start = 0, end = 0;
    edit.GetSel(start, end);

    if(start != end) {
        text = text.Mid(start, end-start);
    }

    return true;
}

void CMessageView::OnRulesAddListitem() 
{
    CPopManDoc* pDoc = GetDocument();
    if(!pDoc) return;

    CString text;

    if(
        !GetText(m_editSubject, text) &&
        !GetText(m_editFrom,    text) &&
        !GetText(m_editTo,      text) &&
        !GetText(m_editDate,    text)    
      )
        text = m_TextBox.GetSelectedText();

    int idxLB = text.FindOneOf(_T("\r\n"));
    if(idxLB >= 0)
        text = text.Mid(0,idxLB);

    text.TrimLeft();
    text.TrimRight();

    pDoc->AddItemToRulesList(text, m_pMail);
}

void CMessageView::OnUpdateRulesAddListitem(CCmdUI* pCmdUI) 
{
    CPopManDoc* pDoc = GetDocument();
	pCmdUI->Enable(pDoc && pDoc->IsAddItemToRulesListEnabled());
}
