// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// MainFrm.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "PopMan.h"

#include "MainFrm.h"
#include "PopManDoc.h"
#include "RegKey.h"
#include "DlgSettings.h"
#include "MessageView.h"
#include "PopManView.h"
#include "StrFunctions.h"
#include "DlgAccReport.h"
#include <mmsystem.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


const TCHAR szWndStatusBarValue[]  = _T("StatusBar");
const TCHAR szWndToolBarValue[]    = _T("ToolBar");
const TCHAR szPreviewValue[]       = _T("Preview");
const TCHAR szPreviewLayoutValue[] = _T("PreviewLayout");
const TCHAR szPaneSizeValue[]	   = _T("PaneSize");
const TCHAR szXHideValue[]		   = _T("XHide");
const TCHAR szMinHideValue[]	   = _T("MinHide");



extern const UINT UWM_ARE_YOU_ME;

const int TimerID_ShutDown = 1;
const int TimerID_ShowActivate = 2;
const int TimerID_RepositionTrashButton = 3;


//  i18nComment("Main Window")


/////////////////////////////////////////////////////////////////////////////
// CMainFrame

IMPLEMENT_DYNCREATE(CMainFrame, CFrameWnd)

BEGIN_MESSAGE_MAP(CMainFrame, CFrameWnd)
	//{{AFX_MSG_MAP(CMainFrame)
	ON_WM_CREATE()
	ON_WM_CLOSE()
	ON_WM_TIMER()
	ON_WM_SIZE()
	ON_COMMAND(ID_VIEW_STATUS_BAR, OnViewStatusBar)
	ON_COMMAND(ID_TRAY_SHOW, OnShowWindow)
	ON_WM_SYSCOMMAND()
	ON_UPDATE_COMMAND_UI(ID_TRAY_SHOW, OnUpdateShowWindow)
	ON_COMMAND(ID_EXTRAS_SETTINGS, OnSettings)
	ON_COMMAND(ID_GOTO_NEXT_PANE, OnGotoNextPane)
	ON_UPDATE_COMMAND_UI(ID_EXTRAS_SETTINGS, OnUpdateSettings)
	ON_COMMAND(ID_VIEW_MULTITAB, OnViewMultitab)
	ON_UPDATE_COMMAND_UI(ID_VIEW_MULTITAB, OnUpdateViewMultitab)
	ON_WM_MENUSELECT()
	ON_WM_INITMENUPOPUP()
	ON_WM_PAINT()
	ON_WM_INITMENU()
	ON_COMMAND(ID_APP_ESCAPE, OnAppEscape)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_GOTO_PREV_PANE, OnGotoNextPane)
	ON_REGISTERED_MESSAGE(UWM_ARE_YOU_ME, OnAreYouMe)
	ON_WM_QUERYENDSESSION()
	ON_NOTIFY_EX_RANGE(TTN_NEEDTEXTW, 0, 0xFFFF, OnToolTipText)
	ON_NOTIFY_EX_RANGE(TTN_NEEDTEXTA, 0, 0xFFFF, OnToolTipText)
	ON_MESSAGE(WM_UPDATE_GUI, OnUpdateGUI)
	ON_COMMAND(ID_ACCOUNTS_0, OnListAccount)
	ON_COMMAND(ID_TAB_CHECK_MAIL, OnCheckTab)
	ON_COMMAND(ID_SELECT_THEME, OnSelectTheme)
	ON_MESSAGE(WM_POWERBROADCAST, OnPowerBroadcast)
	ON_MESSAGE(WM_MAIL_DEL_MARKED, OnMailMarkedForDelete)
END_MESSAGE_MAP()




static UINT indicators[] =
{
	ID_SEPARATOR,
	ID_ERROR           // Statusleistenanzeige
};

/////////////////////////////////////////////////////////////////////////////
// CMainFrame Diagnose

#ifdef _DEBUG
void CMainFrame::AssertValid() const
{
	CFrameWnd::AssertValid();
}

void CMainFrame::Dump(CDumpContext& dc) const
{
	CFrameWnd::Dump(dc);
}

#endif //_DEBUG

CMainFrame::CMainFrame() 
{
	TRACE(_T(" CMainFrame::CMainFrame\n"));
	m_pMessageView = NULL;
	m_bPreview = TRUE;
	m_PreviewLayout = CPropPage_Layout::PREVIEW_LAYOUT::layoutBelow;
	m_pSplitter = NULL;
	m_nPaneSize = 200;
	m_bXHide = TRUE;
	m_bMinHide = FALSE;
	m_nSelectedMenuIdx = 0;
	disabledImageList = NULL;
	normalImageList = NULL;
//	m_bReloadToolbar = false;
    m_bIsHiddenMaximized = FALSE;
}

CMainFrame::~CMainFrame()
{
	if(m_pSplitter)
		delete m_pSplitter;

	TRACE(_T("CMainFrame::~CMainFrame()\n"));
}

LRESULT CMainFrame::OnAreYouMe(WPARAM wParam, LPARAM)
{
	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if (pDoc == NULL)
		return UWM_ARE_YOU_ME;

	CString stDummy;
    switch(wParam)
	{
		case 1: return pDoc->GetMailCount(CPopManDoc::NUMMAILSTRAY::Unread, stDummy);  break;
		case 2: return pDoc->GetMailCount(CPopManDoc::NUMMAILSTRAY::All, stDummy);  break;
		case 3:
		{
			LRESULT PopManStatus = 0;
	        if(!pDoc->m_bNotification) PopManStatus |= 1;
	        if(pDoc->ConnectionsPending()) PopManStatus |= 2;
			if(pDoc->GetErrorStatus()) PopManStatus |= 4;
	        return PopManStatus;
			break;
		}
		case 4:
		{
			COleDateTimeSpan MinutesSinceCheck;
			if(pDoc->GetLastCheckTime().GetStatus() == COleDateTime::DateTimeStatus::valid) 
				MinutesSinceCheck = COleDateTime::GetCurrentTime() - pDoc->GetLastCheckTime();
            else
				MinutesSinceCheck.SetDateTimeSpan(101, 0, 0, 0);
			return static_cast<LRESULT>(MinutesSinceCheck.GetTotalMinutes());
			break;
		}
		default: return UWM_ARE_YOU_ME;
	}
}

LRESULT CMainFrame::OnMailMarkedForDelete(WPARAM wParam, LPARAM lParam)
{
	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if(!pDoc) return TRUE;

	int count = pDoc->CountMarkedMails();
	m_ButtonTrash.ShowWindow(count > 0 ? SW_SHOW : SW_HIDE);

	CString stCap;
	if(count > 0) stCap.Format(_T(" (%d)"), count);
	stCap = i18n("Empty Trash") + stCap;
	
	m_ButtonTrash.SetWindowText(stCap);
	

	CPopManView* pView = GetMainView();
	if(pView != NULL) 
		pView->OnMailMarkedChange();

	return TRUE;
}

LRESULT CMainFrame::OnPowerBroadcast(WPARAM wParam, LPARAM lParam)
{
	if(wParam == PBT_APMRESUMESUSPEND || wParam == PBT_APMRESUMESTANDBY || wParam == PBT_APMRESUMECRITICAL)
    {
		m_bRepositionTrashButton = true; // necessary to fix a Windows XP toolbar button problem
    }
   
	return TRUE;
}

void CMainFrame::OnPaint() // we need to trigger the toolbar reload after the desktop has become visible again
{
	CPaintDC dc(this);

	if(m_bRepositionTrashButton) {
		m_bRepositionTrashButton = false;
		SetTimer(TimerID_RepositionTrashButton, 500, NULL); // we need a short delay to make the toolbar reload effective
	}

}

BOOL CMainFrame::PreCreateWindow(CREATESTRUCT& cs) 
{
	CWinApp* pApp = AfxGetApp();
	CRect rect;
	BOOL  bMaximized;
		
	cs.lpszClass = AfxRegisterWndClass(
      CS_DBLCLKS,		// if you need double-clicks
      NULL,				// no cursor (use default)
      NULL,				// no background brush
	  AfxGetApp()->LoadIcon(IDR_MAINFRAME)); // app icon

	cs.style &= ~FWS_ADDTOTITLE; 

	if(!LoadWindowSettings(bMaximized, rect))
	{
		bMaximized = FALSE;
		rect.left = 40;
		rect.top = 40;
		rect.bottom = 620;
		rect.right = 900;
	}
	
    m_bIsHiddenMaximized = bMaximized;

	switch(pApp->m_nCmdShow)
	{
		case SW_RESTORE:
		case SW_SHOW:
		case SW_SHOWDEFAULT:
		case SW_SHOWNORMAL:
		case SW_SHOWMAXIMIZED:
			if (bMaximized)
				pApp->m_nCmdShow = SW_SHOWMAXIMIZED;
			break;

		case SW_MINIMIZE:
		case SW_SHOWMINNOACTIVE:
		{
			pApp->m_nCmdShow = SW_HIDE;
			break;
		}
		
	}
	
	if (rect.Width() > 0 && rect.Height() > 0)
	{
		RECT rcIS, rcScreen = {0, 0, ::GetSystemMetrics(SM_CXSCREEN), ::GetSystemMetrics(SM_CYSCREEN)};
		if(::IntersectRect(&rcIS, &rcScreen, &rect))
		{
			cs.x = rect.left;
			cs.y = rect.top;
			cs.cx = rect.Width();
			cs.cy = rect.Height();
		}
	}
	
	return CFrameWnd::PreCreateWindow(cs);
}

int GuessImgWidth(int Width, int imgHeight)
{
	if(Width % imgHeight == 0) return imgHeight;
	if(Width % (imgHeight+1) == 0) return imgHeight+1;
	if(Width % (imgHeight-1) == 0) return imgHeight-1;
	if(Width % (imgHeight+2) == 0) return imgHeight+2;
	if(Width % (imgHeight-2) == 0) return imgHeight-2;
	int i = 3;
	while(Width % (imgHeight+i) != 0)
		++i;
	return i;
}

int CMainFrame::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CFrameWnd::OnCreate(lpCreateStruct) == -1)
		return -1;

	SetIcon(CPopManApp::GetIcon(IDR_MAINFRAME, CPopManApp::RESSIZE::Big), TRUE);
	SetIcon(CPopManApp::GetIcon(IDR_MAINFRAME, CPopManApp::RESSIZE::Small), FALSE);

	HDC hDC = ::GetDC(NULL);
	int lfHeight = -MulDiv(8, GetDeviceCaps(hDC, LOGPIXELSY), 72);
	::ReleaseDC(NULL, hDC);

	m_Font.CreateFont(lfHeight, 0, 0, 0, FW_BOLD, FALSE, FALSE, 0, DEFAULT_CHARSET,
	                 OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, 
					 DEFAULT_PITCH, NULL);

	if (!m_wndToolBar.CreateEx(this, TBSTYLE_FLAT, WS_CHILD | WS_VISIBLE | CBRS_TOP
		| CBRS_TOOLTIPS | CBRS_FLYBY | CBRS_SIZE_FIXED))
	{
		TRACE0("Symbolleiste konnte nicht erstellt werden\n");
		return -1;      
	}
	
	if(!m_wndToolBar.LoadToolBar(IDR_TOOLBAR))
	{
		TRACE0("Symbolleiste konnte nicht erstellt werden\n");
		return -1;      
	}
	
	m_ButtonTrash.Create(i18n("Empty Trash"), BS_PUSHBUTTON | WS_CHILD, CRect(0,0,0,0), &m_wndToolBar, ID_FILE_EMPTY_TRASH);    

	LoadToolbarBitmap(false);
		
	if (!m_wndStatusBar.Create(this) ||
		!m_wndStatusBar.SetIndicators(indicators, sizeof(indicators)/sizeof(UINT)))
	{
		TRACE0("Statusleiste konnte nicht erstellt werden\n");
		return -1;      
	}

	m_wndStatusBar.SetPaneStyle(0, SBPS_NORMAL | SBPS_STRETCH);
	m_wndStatusBar.SetFont(&m_Font);
	
	m_wndStatusBar.SetPaneTextColor(1, RGB(218, 0, 0));
	m_wndStatusBar.m_fnOnPaneDblClick = OnStatusBarDblClick;
	

	

	BOOL bStatusBar = TRUE, bToolBar = TRUE;
	LoadBarSettings(bStatusBar, bToolBar);

	if(!bStatusBar)
	{
		CControlBar* pBar = &m_wndStatusBar;
		ShowControlBar(pBar, FALSE, FALSE);
	}

	if(!bToolBar)
	{
		CControlBar* pBar = &m_wndToolBar;
		ShowControlBar(pBar, FALSE, FALSE);
	}

	InitGUIText();

		
	CString stTitle = _T("PopMan");
	//stTitle.Format(_T("PopMan %d.%d.%d"), cstAppVersion.GetMajor(), cstAppVersion.GetMinor(), cstAppVersion.GetSub());
	
	if(cstAppVersion.IsBeta())
		stTitle += _T(" Beta");

	SetWindowText(stTitle);

	return 0;
}



void CMainFrame::LoadToolbarBitmap(bool bSwitch)
{
	SIZE sizeButton = {0}, sizeImage = {0};
	int imageCount = 0;
	bool bUseCustomToolbar = false;
	HBITMAP hBitmap = NULL, hDisabledBitmap = NULL;

	static HIMAGELIST hNormalList = NULL; 

	CString stThemePath;
		
	if(CPopManApp::GetCurrentThemePath(stThemePath))
	{
		if(stThemePath.Right(1) != _T("\\")) stThemePath += _T('\\');
		CString stBitmap = stThemePath + _T("toolbar.bmp");
	
		CFileStatus state;
		if(CFile::GetStatus(stBitmap, state))
		{
			hBitmap = (HBITMAP)::LoadImage(0, stBitmap, IMAGE_BITMAP, 0, 0, LR_LOADTRANSPARENT | LR_LOADMAP3DCOLORS | LR_LOADFROMFILE);
			if(hBitmap) 
			{
				BITMAP BitMap;
				if(0 != ::GetObject(hBitmap, sizeof(BITMAP), &BitMap)) 
				{
					bUseCustomToolbar = true;
					sizeImage.cy = BitMap.bmHeight;
					sizeImage.cx = GuessImgWidth(BitMap.bmWidth, BitMap.bmHeight);
					imageCount = BitMap.bmWidth / sizeImage.cx;
				}
			}

			stBitmap = stThemePath + _T("toolbarDisabled.bmp");
			if(CFile::GetStatus(stBitmap, state))
				hDisabledBitmap = (HBITMAP)::LoadImage(0, stBitmap, IMAGE_BITMAP, 0, 0, LR_LOADTRANSPARENT | LR_LOADMAP3DCOLORS | LR_LOADFROMFILE);
		}
	}


	if(!bUseCustomToolbar)
	{
		sizeImage.cx = 16;
		sizeImage.cy = 15;
    }
    else 
    {
		CBitmap normalBitmap;
		normalBitmap.Attach(hBitmap);

		normalImageList = new CImageList;
		normalImageList->Create(sizeImage.cx, sizeImage.cy, ILC_COLOR32|ILC_MASK, imageCount, 1);
		normalImageList->Add(&normalBitmap, ::GetSysColor(COLOR_3DFACE));

		if(hDisabledBitmap) {
			CBitmap disabledBitmap;
			disabledBitmap.Attach(hDisabledBitmap);

			disabledImageList = new CImageList;
			disabledImageList->Create(sizeImage.cx, sizeImage.cy, ILC_COLOR32|ILC_MASK, imageCount, 1);
			disabledImageList->Add(&disabledBitmap, ::GetSysColor(COLOR_3DFACE));
		}
		else
			disabledImageList = NULL;
	}

    sizeButton.cx =  sizeImage.cx + 7;
	sizeButton.cy =  sizeImage.cy + 6;

	const LONG height = sizeButton.cy;

	if(!bUseCustomToolbar && hNormalList != NULL) {
		delete m_wndToolBar.GetToolBarCtrl().SetImageList(CImageList::FromHandle(hNormalList));
		delete m_wndToolBar.GetToolBarCtrl().SetDisabledImageList(NULL);
		hNormalList = NULL;
		normalImageList = NULL;
		disabledImageList = NULL;
	}

	if(bSwitch)
		m_wndToolBar.LoadToolBar(IDR_TOOLBAR);


	if(bUseCustomToolbar) {

		CImageList* pList = m_wndToolBar.GetToolBarCtrl().SetImageList(normalImageList);
		if(pList) {
			if(hNormalList == NULL)
				hNormalList = pList->m_hImageList;
			else
				delete pList;
		}
		else
			hNormalList = NULL;


		delete m_wndToolBar.GetToolBarCtrl().SetDisabledImageList(disabledImageList);
	}
	

	m_wndToolBar.GetToolBarCtrl().SetExtendedStyle(TBSTYLE_EX_DRAWDDARROWS);
	TBBUTTONINFO tbi;
	tbi.dwMask = TBIF_STYLE;
	tbi.cbSize = sizeof(TBBUTTONINFO);
	m_wndToolBar.GetToolBarCtrl().GetButtonInfo(ID_FILE_LIST_MAILS, &tbi);
	tbi.fsStyle |= TBSTYLE_DROPDOWN;
	m_wndToolBar.GetToolBarCtrl().SetButtonInfo(ID_FILE_LIST_MAILS, &tbi);
	
    // **SNH fudge - for some reason it needs text even tho it's not displayed else paint gets messed up
    m_wndToolBar.SetButtonText(1, _T("U"));

	m_wndToolBar.SetSizes(sizeButton, sizeImage);
	if(bSwitch && m_wndToolBar.m_hWnd && m_wndToolBar.IsWindowVisible())
		ShowControlBar(&m_wndToolBar, TRUE, FALSE);

	m_FontTrash.Detach();
	m_FontTrash.CreatePointFont(height < 28 ? 70 : 90, _T("Tahoma"));
	m_ButtonTrash.SetFont(&m_FontTrash);

    RecalcLayout();  // **SNH helps paint problem ?

	SetTrashButtonPos(); // move and size ButtonTrash, but retain width (width will be set in InitGuiText)
}

void CMainFrame::SetTrashButtonPos()
{
    int count = m_wndToolBar.GetCount();
	CRect rect;
	m_wndToolBar.GetItemRect(count-1, rect);
	int left = rect.right + rect.Width();
	int height = min(rect.Height()-2, 28);
	int top = (rect.Height() - height)/2;
	
	m_ButtonTrash.GetClientRect(rect);
	
	rect.SetRect(left, top, left+rect.Width(), top + height);

	m_ButtonTrash.MoveWindow(rect);
}

void CMainFrame::OnClose() 
{
	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	
	pDoc->CancelConnections();
	SetTimer(TimerID_ShutDown, 100, NULL);
                                 
	SaveWindowSettings();
	SaveBarSettings();
}
  

void CMainFrame::OnTimer(UINT nIDEvent) 
{
	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
    if(pDoc == NULL)
		return;

	switch(nIDEvent)
	{
	case TimerID_ShutDown:

		if(!pDoc->ConnectionsPending())
		{
			KillTimer(TimerID_ShutDown);
			delete normalImageList;
			delete disabledImageList;
			CFrameWnd::OnClose();
		}
		break;

	case TimerID_ShowActivate:

		MessageBeep(MB_ICONEXCLAMATION);
		SetForegroundWindow();
		SetWindowPos(&wndTopMost, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_SHOWWINDOW);
		SetForegroundWindow();
			
		break;

    case TimerID_RepositionTrashButton:
		
		KillTimer(TimerID_RepositionTrashButton);
		SetTrashButtonPos();

		break;

	}
}


void CMainFrame::OnSize(UINT nType, int cx, int cy) 
{
	CFrameWnd::OnSize(nType, cx, cy);

	RecalcStatusBar();
}


void CMainFrame::SaveWindowSettings()
{
	CSettingsKey Key;
	if(CreateSettingsKey(Key, szMainWindowKey))
	{
        BOOL maximized = (IsZoomed() || ((!IsWindowVisible() || IsIconic()) && m_bIsHiddenMaximized));

		Key.SetValue(szWndMaximizedValue, maximized);

		if(!IsZoomed() && !IsIconic() && IsWindowVisible())
		{
			RECT rect;
			GetWindowRect(&rect);

			Key.SetValue(szWndLeftValue,   rect.left);
			Key.SetValue(szWndTopValue,    rect.top);
			Key.SetValue(szWndWidthValue,  rect.right - rect.left);
			Key.SetValue(szWndHeightValue, rect.bottom - rect.top);
			Key.SetValue(szWndStatusBarValue, m_wndStatusBar.IsWindowVisible());
		}

		Key.SetValue(szPreviewValue, m_bPreview);
		Key.SetValue(szPreviewLayoutValue, m_PreviewLayout);
		if(m_bPreview && m_pSplitter != NULL)
		{
			int nSize = 0;
			int nMin = 0;

			if(m_PreviewLayout == CPropPage_Layout::PREVIEW_LAYOUT::layoutBelow)
				m_pSplitter->GetRowInfo(0, nSize, nMin);
			else
				m_pSplitter->GetColumnInfo(0, nSize, nMin);

			Key.SetValue(szPaneSizeValue, nSize); 
		}

		Key.SetValue(szXHideValue, m_bXHide);
		Key.SetValue(szMinHideValue, m_bMinHide);
	}
}

BOOL CMainFrame::LoadWindowSettings(BOOL& bMaximized, CRect& FrameRect)
{
	CSettingsKey Key;
	if(!OpenSettingsKey(Key, szMainWindowKey))
		return FALSE;

	if(!Key.QueryValue(szWndMaximizedValue, bMaximized))
		return FALSE;

	if(!Key.QueryValue(szWndLeftValue, FrameRect.left))
		return FALSE;

	if(!Key.QueryValue(szWndTopValue, FrameRect.top))
		return FALSE;
	
	DWORD dwValue = 0;

	if(!Key.QueryValue(szWndWidthValue, dwValue))
		return FALSE;
	FrameRect.right = FrameRect.left + dwValue;
		
	if(!Key.QueryValue(szWndHeightValue, dwValue))
		return FALSE;
	FrameRect.bottom = FrameRect.top + dwValue;

	Key.QueryValue(szPreviewValue, m_bPreview);

	int Val = 0;
	if(Key.QueryValue(szPreviewLayoutValue, Val))
		m_PreviewLayout = (CPropPage_Layout::PREVIEW_LAYOUT)Val;

	Key.QueryValue(szPaneSizeValue, m_nPaneSize);
	Key.QueryValue(szXHideValue, m_bXHide);
	Key.QueryValue(szMinHideValue, m_bMinHide);

	return TRUE;
}

BOOL CMainFrame::LoadBarSettings(BOOL& bStatusBar, BOOL& bToolBar)
{
	CSettingsKey Key;
	if(!OpenSettingsKey(Key, szMainWindowKey))
		return FALSE;
	
	if(!Key.QueryValue(szWndStatusBarValue, bStatusBar))
		return FALSE;

	if(!Key.QueryValue(szWndToolBarValue, bToolBar))
		return FALSE;
	
	return TRUE;
}

void CMainFrame::SaveBarSettings()
{
	CSettingsKey Key;
	if(CreateSettingsKey(Key, szMainWindowKey))
	{
		Key.SetValue(szWndStatusBarValue, (m_wndStatusBar.GetStyle() & WS_VISIBLE) != 0 ? TRUE : FALSE);
		Key.SetValue(szWndToolBarValue, (m_wndToolBar.GetStyle() & WS_VISIBLE) != 0 ? TRUE : FALSE);
	}
}

void CMainFrame::PostNcDestroy() 
{
	// Close all message windows:
	::EnumThreadWindows(::GetCurrentThreadId(), CloseWindow, 0);	
	
	CFrameWnd::PostNcDestroy();
}

BOOL CALLBACK CMainFrame::CloseWindow(HWND hWnd, LPARAM lParam)
{
	// Don't destroy ToolTip windows, because this may cause serious
	// problems on Win98 !
	TCHAR classname[8] = _T("");
	::GetClassName(hWnd, classname, sizeof(classname)/sizeof(TCHAR));
	if(_tcsncicmp(classname, _T("ToolTip"), 7) != 0)
		::DestroyWindow(hWnd);

	return TRUE;
}


void CMainFrame::OnViewStatusBar() 
{
	CControlBar* pBar = &m_wndStatusBar;
	ShowControlBar(pBar, (pBar->GetStyle() & WS_VISIBLE) == 0, FALSE);

	RecalcStatusBar();
}


void CMainFrame::RecalcStatusBar()
{
	CStatusBarCtrl& StatusCtrl = m_wndStatusBar.GetStatusBarCtrl();
	if(StatusCtrl.m_hWnd == NULL)
		return;

	RECT rect;
	int  iStatusBarWidth;

	StatusCtrl.GetWindowRect(&rect);
	iStatusBarWidth = rect.right - rect.left;

	UINT nID, nStyle;
	int cxWidth;

	m_wndStatusBar.GetPaneInfo(0, nID, nStyle, cxWidth);
	m_wndStatusBar.SetPaneInfo(0, nID, nStyle, int(0.6 * iStatusBarWidth));

	m_wndStatusBar.GetPaneInfo(1, nID, nStyle, cxWidth);
	m_wndStatusBar.SetPaneInfo(1, nID, nStyle, int(0.4 * iStatusBarWidth));
}

void CMainFrame::OnShowWindow() 
{
	RestoreFromTray();
}

void CMainFrame::OnSysCommand(UINT nID, LPARAM lParam) 
{
	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if(pDoc != NULL)
	{
		if((nID & 0xFFF0) == SC_MINIMIZE)
		{
            m_bIsHiddenMaximized = IsZoomed();
			SaveWindowSettings();
			pDoc->OnMainWndMinimize();
		}
	}

	if((nID & 0xFFF0) == SC_MINIMIZE && m_bMinHide)
	{
        // CFrameWnd::OnSysCommand(nID, lParam);
		HideToTray();
		return;
	}

	if((nID & 0xFFF0) == SC_CLOSE && m_bXHide)
	{
		HideToTray();
		return;
	}

	CFrameWnd::OnSysCommand(nID, lParam);
}

void CMainFrame::RestoreFromTray() {

	SetForegroundWindow();

    if(m_bIsHiddenMaximized)
        ShowWindow(SW_SHOWMAXIMIZED);
    else
        ShowWindow(SW_RESTORE);

	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if(pDoc == NULL)  
		return;

	pDoc->OnMainWndRestore();
}

void CMainFrame::HideToTray()
{
	SaveWindowSettings();

    m_bIsHiddenMaximized = IsZoomed();

	ShowWindow(SW_MINIMIZE);
	ShowWindow(SW_HIDE);

	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if(pDoc == NULL)
		return;

	pDoc->OnMainWndHide();
}

void CMainFrame::OnUpdateShowWindow(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!IsWindowVisible() || IsIconic());	
}

void CMainFrame::OnSettings() 
{
	CPropSheet_Settings Settings(i18n("Options"));
	Settings.m_psh.dwFlags |= PSH_NOAPPLYNOW;

	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if (pDoc == NULL)
		return;
	
	CPropPage_General PageGeneral(i18n("General"));
	PageGeneral.m_bUseDialUp = pDoc->m_bUseDUN;
	PageGeneral.m_bAutoStart = pDoc->AutoStartEnabled();
	PageGeneral.m_bCheckMailOnStartup = pDoc->m_bCheckMailOnStartup;
	PageGeneral.m_nStartCheckDelay = pDoc->m_nStartCheckDelay;
	PageGeneral.m_bXHide = m_bXHide;
	PageGeneral.m_bMinHide = m_bMinHide;
	PageGeneral.m_bConfirmDeleteMain = pDoc->m_bConfirmDeleteMain;
	PageGeneral.m_bConfirmDeleteMsg = pDoc->m_bConfirmDeleteMsg;
	PageGeneral.m_bMarkDelete = pDoc->m_bMarkDelete;
	PageGeneral.m_bPasswordProtect = ((CPopManApp*)AfxGetApp())->m_bPasswordProtect;
	PageGeneral.m_stPassword = ((CPopManApp*)AfxGetApp())->m_stPassword;

	Settings.AddPage(&PageGeneral);

	CPropPage_AutoCheck PageAutoCheck(i18n("AutoCheck"));
	PageAutoCheck.m_nUpdateInterval = pDoc->GetUpdateInterval();
	PageAutoCheck.m_bAutoCheckMinimized = pDoc->m_bAutoCheckMinimized;
	PageAutoCheck.m_bSuspendAutoCheck = pDoc->m_bSuspendAutoCheck;
	PageAutoCheck.m_nSuspendFrom = pDoc->m_nSuspendFrom;
	PageAutoCheck.m_nSuspendTo = pDoc->m_nSuspendTo;
	PageAutoCheck.m_pAccounts = &pDoc->m_Accounts;
	PageAutoCheck.m_bEnableCustomAutoCheck = pDoc->m_bCustomAutoCheck;
	Settings.AddPage(&PageAutoCheck);

	CPropPage_Notification PageNotify(i18n("Notification"));
	PageNotify.m_bSoundNotification = pDoc->m_bSoundNotify;
	PageNotify.m_stSoundFile = pDoc->m_stSoundFile;
	PageNotify.m_bShowBallonOption = pDoc->m_TrayIcon.BallonsEnabled();
	PageNotify.m_nNotify = (CPropPage_Notification::NOTIFYSTATES)pDoc->m_nNotify;
	PageNotify.m_bShowMainWnd = pDoc->m_bShowMainWnd;
	PageNotify.m_bShowNewMailTip = pDoc->m_bShowNewMailTip;
	PageNotify.m_bDetailedNewTip = pDoc->m_bDetailedNewTip;
	PageNotify.m_bBallonTip = pDoc->m_bBallonTip;
	PageNotify.m_bNotifyProtectedOnly = pDoc->m_bNotifyProtectedOnly;
	PageNotify.m_KeyboardLight = (CPropPage_Notification::KEYBOARDLIGHT)pDoc->getKeyboardLight();
	Settings.AddPage(&PageNotify);

	CPropPage_EmailClient PageEmailClient(i18n("Email Client"));
	PageEmailClient.m_stEmailClient = pDoc->m_stEmailClient;
	PageEmailClient.m_nReplyOption = (CPropPage_EmailClient::REPLYOPTIONS)pDoc->m_nReplyOption;
	PageEmailClient.m_stReplyParam = pDoc->m_stReplyParam;
	PageEmailClient.m_bHTMLReply = pDoc->m_bHTMLReply;
	PageEmailClient.m_bEnforceAddress = pDoc->m_bEnforceAddress;
	PageEmailClient.m_ClientLaunchAction = (CPropPage_EmailClient::CLIENTLAUNCHACTION)pDoc->m_ClientLaunchAction;

	Settings.AddPage(&PageEmailClient);

	CPropPage_Layout PageLayout(i18n("View"));
	PageLayout.m_bPreview = m_bPreview;
	PageLayout.m_PreviewLayout = m_PreviewLayout;
	PageLayout.m_bMarkUnread = pDoc->m_bMarkUnread;
	Settings.AddPage(&PageLayout);

	CPropPage_TrayIcon PageTrayIcon(i18n("Tray Icon"));
	PageTrayIcon.m_bNumMailsInTray = pDoc->m_bNumMailsInTray;
	PageTrayIcon.m_bRotateMailNum = pDoc->m_bRotateMailNum;
	PageTrayIcon.m_numMails = (CPropPage_TrayIcon::NUMMAILSTRAY)pDoc->m_numMails;
	PageTrayIcon.m_nRotationInterval = pDoc->m_nRotationInterval;
	PageTrayIcon.m_bHighColorTray = pDoc->m_bHighColorTray;
	PageTrayIcon.m_bHighColorTrayEnabled = pDoc->m_TrayIcon.HighColorTrayEnabled();
	PageTrayIcon.m_bAdvancedTrayInfo = pDoc->m_bAdvancedTrayInfo;
	Settings.AddPage(&PageTrayIcon);

	CPropPage_Language PageLanguage(i18n("Language"));
	Settings.AddPage(&PageLanguage);

	CString stLangFile = GetCurrentLangFile();

	CPropPage_MouseAction PageMouseAction(i18n("Mouse Actions"));
	PageMouseAction.m_actionLeftClick = (CPropPage_MouseAction::TRAYACTIONS)pDoc->m_actionLeftClick;
	PageMouseAction.m_actionRightClick = (CPropPage_MouseAction::TRAYACTIONS)pDoc->m_actionRightClick;
	PageMouseAction.m_actionMiddleClick = (CPropPage_MouseAction::TRAYACTIONS)pDoc->m_actionMiddleClick;
	PageMouseAction.m_actionDoubleClick = (CPropPage_MouseAction::TRAYACTIONS)pDoc->m_actionDoubleClick;
	PageMouseAction.m_bDoubleClickDelay = pDoc->m_TrayIcon.m_bDoubleClickDelay;
	Settings.AddPage(&PageMouseAction);

	CPropPage_Logging PageLogging(i18n("Logging"));
	PageLogging.m_bLoggingEnabled = pDoc->IsLogEnabled();
    PageLogging.m_bLoggingRules = pDoc->IsRulesLogEnabled();
    PageLogging.m_nMaxLogSizeRules = pDoc->GetMaxRulesLogSize() / 1024;
	PageLogging.m_bCompactLogging = pDoc->m_bCompactLogging;
	PageLogging.m_nMaxLogSize = pDoc->GetMaxLogSize() / 1024;
	Settings.AddPage(&PageLogging);
	
	if (Settings.DoModal() == IDOK)
	{
		pDoc->m_bUseDUN = PageGeneral.m_bUseDialUp;
		pDoc->SetAutoStart(PageGeneral.m_bAutoStart);
		pDoc->m_bCheckMailOnStartup = PageGeneral.m_bCheckMailOnStartup;
		pDoc->m_nStartCheckDelay = PageGeneral.m_nStartCheckDelay;
		m_bXHide = PageGeneral.m_bXHide;
		m_bMinHide = PageGeneral.m_bMinHide;
		pDoc->m_bConfirmDeleteMain = PageGeneral.m_bConfirmDeleteMain;
		pDoc->m_bConfirmDeleteMsg = PageGeneral.m_bConfirmDeleteMsg;
		pDoc->m_bMarkDelete = PageGeneral.m_bMarkDelete;
		
		((CPopManApp*)AfxGetApp())->m_bPasswordProtect = PageGeneral.m_bPasswordProtect;
		((CPopManApp*)AfxGetApp())->m_stPassword = PageGeneral.m_stPassword;

		pDoc->SetUpdateInterval(PageAutoCheck.m_nUpdateInterval);
		pDoc->m_bAutoCheckMinimized = PageAutoCheck.m_bAutoCheckMinimized;
		pDoc->m_bSuspendAutoCheck = PageAutoCheck.m_bSuspendAutoCheck;
		pDoc->m_nSuspendFrom = PageAutoCheck.m_nSuspendFrom;
		pDoc->m_nSuspendTo = PageAutoCheck.m_nSuspendTo;
		pDoc->m_bCustomAutoCheck = PageAutoCheck.m_bEnableCustomAutoCheck;

		pDoc->m_bSoundNotify = PageNotify.m_bSoundNotification;
		pDoc->m_stSoundFile = PageNotify.m_stSoundFile;	
		pDoc->m_nNotify = (CPopManDoc::NOTIFYSTATES)PageNotify.m_nNotify;
		pDoc->m_bShowMainWnd = PageNotify.m_bShowMainWnd;
		pDoc->m_bShowNewMailTip = PageNotify.m_bShowNewMailTip;
		pDoc->m_bDetailedNewTip = PageNotify.m_bDetailedNewTip;
		pDoc->m_bBallonTip = PageNotify.m_bBallonTip;
		pDoc->m_bNotifyProtectedOnly = PageNotify.m_bNotifyProtectedOnly;
		pDoc->setKeyboardLight((CPopManDoc::KEYBOARDLIGHT)PageNotify.m_KeyboardLight);

		pDoc->m_stEmailClient = PageEmailClient.m_stEmailClient;
		pDoc->m_nReplyOption = (CPopManDoc::REPLYOPTIONS)PageEmailClient.m_nReplyOption;
		pDoc->m_stReplyParam = PageEmailClient.m_stReplyParam;
		pDoc->m_bHTMLReply = PageEmailClient.m_bHTMLReply;
		pDoc->m_bEnforceAddress = PageEmailClient.m_bEnforceAddress;
		pDoc->m_ClientLaunchAction = (CPopManDoc::CLIENTLAUNCHACTION)PageEmailClient.m_ClientLaunchAction;

		BOOL boldPreview = m_bPreview;
		CPropPage_Layout::PREVIEW_LAYOUT oldPreviewLayout = m_PreviewLayout;

		m_bPreview = PageLayout.m_bPreview;
		m_PreviewLayout = PageLayout.m_PreviewLayout;
		pDoc->m_bMarkUnread = PageLayout.m_bMarkUnread;
		pDoc->RepaintAllViews();
	
		pDoc->m_actionLeftClick = (CPopManDoc::TRAYACTIONS)PageMouseAction.m_actionLeftClick;
		pDoc->m_actionRightClick = (CPopManDoc::TRAYACTIONS)PageMouseAction.m_actionRightClick;
		pDoc->m_actionMiddleClick = (CPopManDoc::TRAYACTIONS)PageMouseAction.m_actionMiddleClick;
		pDoc->m_actionDoubleClick = (CPopManDoc::TRAYACTIONS)PageMouseAction.m_actionDoubleClick;
		pDoc->m_TrayIcon.m_bDoubleClickDelay = PageMouseAction.m_bDoubleClickDelay;
		pDoc->m_bNumMailsInTray = PageTrayIcon.m_bNumMailsInTray;
		pDoc->m_bRotateMailNum = PageTrayIcon.m_bRotateMailNum;
		pDoc->m_numMails = (CPopManDoc::NUMMAILSTRAY)PageTrayIcon.m_numMails;
		pDoc->m_nRotationInterval = PageTrayIcon.m_nRotationInterval;
		pDoc->m_bHighColorTray = PageTrayIcon.m_bHighColorTray;
		pDoc->m_bAdvancedTrayInfo = PageTrayIcon.m_bAdvancedTrayInfo;
		if(pDoc->m_bHighColorTray && !pDoc->m_TrayIcon.HighColorTrayEnabled())
			pDoc->m_TrayIcon.EnableHighColorTray();

		pDoc->EnableLog(PageLogging.m_bLoggingEnabled != FALSE);
		pDoc->m_bCompactLogging = PageLogging.m_bCompactLogging;
		pDoc->SetMaxLogSize(1024*PageLogging.m_nMaxLogSize);
        pDoc->EnableRulesLog(PageLogging.m_bLoggingRules != FALSE);
        pDoc->SetMaxRulesLogSize(1024*PageLogging.m_nMaxLogSizeRules);
	
		pDoc->UpdateTrayIcon();

		CString stNewLangFile = GetCurrentLangFile();
		if(stLangFile != stNewLangFile)
			CPopManApp::BroadcastChangedGUI();
		

		if(m_bPreview == boldPreview && m_PreviewLayout == oldPreviewLayout)
			return;

		CView* pView = NULL;

		if(boldPreview == FALSE)
			pView = GetActiveView();
		else
			pView = (CView*)m_pSplitter->GetPane(0, 0);	
				
		ASSERT(pView != NULL);

		CDocument* pDoc = pView->GetDocument();

		CCreateContext Context;
		Context.m_pNewViewClass = RUNTIME_CLASS(CPopManView);
		Context.m_pCurrentDoc = pDoc;
		Context.m_pNewDocTemplate = pDoc->GetDocTemplate();
		Context.m_pLastView = NULL;
		Context.m_pCurrentFrame = NULL;

		BOOL bAutoDelete = pDoc->m_bAutoDelete;
		pDoc->m_bAutoDelete = FALSE;
		pDoc->RemoveView(pView);

		if(boldPreview)
		{
			m_pSplitter->DestroyWindow();
			delete m_pSplitter;
			m_pSplitter = NULL;
			m_pMessageView = NULL;
		}
		else
		{
			pView->DestroyWindow();
		}

		CWnd* pWnd = CreateSplitterWnd(TRUE, &Context, m_bPreview, (m_PreviewLayout == CPropPage_Layout::PREVIEW_LAYOUT::layoutBelow));
		pDoc->m_bAutoDelete = bAutoDelete;

		try 
		{
			CPopManView* pView = dynamic_cast<CPopManView*>(pWnd);
			if(pView)
				pView->Refresh();
		} 
		catch (...)
		{
		}
	}
	else
	{
		CString stNewLangFile = GetCurrentLangFile();
		if(stLangFile != stNewLangFile)
			LoadLanguageFile(stLangFile);
	}

	
}

LRESULT CMainFrame::OnUpdateGUI(WPARAM, LPARAM)
{
	SetIcon(CPopManApp::GetIcon(IDR_MAINFRAME, CPopManApp::RESSIZE::Big), TRUE);
	SetIcon(CPopManApp::GetIcon(IDR_MAINFRAME, CPopManApp::RESSIZE::Small), FALSE);

	InitGUIText();
	SendMessageToDescendants(WM_COMMAND, ID_APP_UPDATE_GUI, 0);

	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if(pDoc != NULL) pDoc->OnUpdateGUI();

	LoadToolbarBitmap(true);
	InitGUIText();
	return 1;
}

BOOL CMainFrame::OnCreateClient(LPCREATESTRUCT lpcs, CCreateContext* pContext) 
{
	CreateSplitterWnd(FALSE, pContext, m_bPreview, (m_PreviewLayout == CPropPage_Layout::PREVIEW_LAYOUT::layoutBelow));

	if(m_bPreview && m_pSplitter)
	{
		if(m_PreviewLayout == CPropPage_Layout::PREVIEW_LAYOUT::layoutBelow)
			m_pSplitter->SetRowInfo(0, m_nPaneSize, 0);
		else
			m_pSplitter->SetColumnInfo(0, m_nPaneSize, 0);
	}	

	return TRUE;
}

CWnd* CMainFrame::CreateSplitterWnd(BOOL bInitialUpdate, CCreateContext *pContext, BOOL bPreview, BOOL bBelow)
{
	CView* pMainView = NULL;

	if(bPreview == FALSE)
	{
		pMainView = (CView*)CreateView(pContext);
	}
	else
	{
		m_pSplitter = new CSplitterWnd;
		m_pSplitter->CreateStatic(this, bBelow ? 2 : 1, bBelow ? 1 : 2);

		m_pSplitter->CreateView(0, 0, RUNTIME_CLASS(CPopManView), CSize(100, 100), pContext);
		m_pSplitter->CreateView(bBelow ? 1 : 0, bBelow ? 0 : 1, RUNTIME_CLASS(CMessageView), CSize(1, 1), pContext);
		m_pMessageView = (CMessageView*)m_pSplitter->GetPane(bBelow ? 1 : 0, bBelow ? 0 : 1);
		m_pMessageView->m_bSmallHead = TRUE;

		CRect rect;
		GetWindowRect(&rect);

		if(bBelow)
			m_pSplitter->SetRowInfo(0, rect.Height() / 2, 0);
		else
			m_pSplitter->SetColumnInfo(0, rect.Width() / 2, 0);

		pMainView = (CView*)m_pSplitter->GetPane(0, 0);
		
	}

	if(bInitialUpdate)
	{
		//pContext->m_pNewDocTemplate->InitialUpdateFrame(this, pContext->m_pCurrentDoc);
		pMainView->OnInitialUpdate();
		if(m_pMessageView != NULL)
			m_pMessageView->OnInitialUpdate();
		SetActiveView(pMainView);
		RecalcLayout();
	}

	return pMainView;
}


void CMainFrame::OnGotoNextPane() 
{
	if(m_pSplitter == NULL)
		return;
	
	BOOL bBelow = (m_PreviewLayout == CPropPage_Layout::PREVIEW_LAYOUT::layoutBelow);
	CView* pPopManView = (CView*)m_pSplitter->GetPane(0, 0);
	CView* pMsgView = (CView*)m_pSplitter->GetPane(bBelow ? 1 : 0, bBelow ? 0 : 1);
	CView* pActiveView = GetActiveView();

	if(pActiveView == pPopManView)
		SetActiveView(pMsgView);
	else
		SetActiveView(pPopManView);
}

void CMainFrame::ActivateFrame(int nCmdShow) 
{
	CFrameWnd::ActivateFrame(nCmdShow);
}


BOOL CMainFrame::OnQueryEndSession()
{
	SaveWindowSettings();
	SaveBarSettings();

	CPopManApp* pApp = (CPopManApp*)AfxGetApp();
	if(pApp != NULL) pApp->SaveSettings();
	CPopManView* pView = GetMainView();
	if(pView != NULL)
		pView->SaveSettings();

	return TRUE;
}


void CMainFrame::OnUpdateSettings(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(AfxGetMainWnd()->IsWindowEnabled());
}

static void MenuTextPos(CMenu* pMenu, int Pos, LPCTSTR szText)
{
	pMenu->ModifyMenu(Pos, MF_BYPOSITION|MF_STRING, Pos, szText);
}

static void MenuTextCmd(CMenu* pMenu, int Cmd, LPCTSTR szText)
{
	pMenu->ModifyMenu(Cmd, MF_BYCOMMAND|MF_STRING, Cmd, szText);
}

UINT GetThemeMenuIdx()
{
	return 3;
}

void CMainFrame::InitGUIText()
{
	CMenu* pMenu = GetMenu();
	if(pMenu == NULL) return;

	MenuTextPos(pMenu, 0, i18n("&File"));
		MenuTextCmd(pMenu, ID_FILE_ACCOUNTS,		i18n("&Accounts..."));
		MenuTextCmd(pMenu, ID_FILE_LIST_MAILS,		i18n("&Update\tF5"));
		MenuTextCmd(pMenu, ID_FILE_CANCEL,			i18n("&Cancel\tEsc"));
		MenuTextCmd(pMenu, ID_FILE_VIEW,			i18n("&Open...\tCtrl+O"));
		MenuTextCmd(pMenu, ID_FILE_MAIL_SOURCE,		CString(i18n("Sou&rce...")) + _T("\tF10"));
		MenuTextCmd(pMenu, ID_FILE_VIEW_EXTERNALLY, CString(i18n("View E&xternally...")) + _T("\tF11"));
		MenuTextCmd(pMenu, ID_FILE_SAVE_MAIL,		i18n("&Save As...\tCtrl+S"));
		MenuTextCmd(pMenu, ID_FILE_REPLY_TO,		i18n("Repl&y To Sender\tCtrl+R"));
		MenuTextCmd(pMenu, ID_FILE_BLOCK_SENDER,	i18n("&Block Sender"));
		MenuTextCmd(pMenu, ID_FILE_DELETE_MAILS,	i18n("&Delete\tDel"));
		MenuTextCmd(pMenu, ID_FILE_EMPTY_TRASH,		i18n("Delete &Marked Emails\tF12"));
		MenuTextCmd(pMenu, ID_APP_EXIT,				i18n("&Exit"));
	
	MenuTextPos(pMenu, 1, i18n("&View"));
		MenuTextCmd(pMenu, ID_VIEW_TOOLBAR,			i18n("&Toolbar"));
		MenuTextCmd(pMenu, ID_VIEW_STATUS_BAR,		i18n("&Statusbar"));
		MenuTextCmd(pMenu, ID_VIEW_HIDE_MARKED,		i18n("&Hide Marked Messages"));
        MenuTextCmd(pMenu, ID_VIEW_TRASHTAB,		i18n("Move Marked Messages to Tr&ash Tab"));
		MenuTextCmd(pMenu, ID_VIEW_MULTITAB,		i18n("Tab &View\tCtrl+M"));

	MenuTextPos(pMenu, 2, i18n("&Tools"));
		MenuTextCmd(pMenu, ID_EXTRAS_FONT,			i18n("&Font..."));
		MenuTextCmd(pMenu, ID_EXTRAS_FONT_COLOR,	i18n("Font &Color..."));
		
		CMenu* pMenuTools = pMenu->GetSubMenu(2);
		MenuTextPos(pMenuTools, GetThemeMenuIdx(),	i18n("&Themes"));

		MenuTextCmd(pMenu, ID_APP_EMAIL_CLIENT,			i18n("Launch &Email Client\tCtrl+E"));
		MenuTextCmd(pMenu, ID_EXTRAS_IMPORT_ACCOUNTS,	i18n("&Import Accounts..."));
		MenuTextCmd(pMenu, ID_EXTRAS_WHITELIST,			i18n("&White List..."));
        MenuTextCmd(pMenu, ID_EXTRAS_BLACKLIST,			i18n("&Black List..."));
		MenuTextCmd(pMenu, ID_EXTRAS_RULES,				i18n("&Rules..."));
        MenuTextCmd(pMenu, ID_RULES_ADD_LISTITEM,		i18n("Add Item To Rules &List...\tF9"));
        
		MenuTextCmd(pMenu, ID_EXTRAS_SETTINGS,			i18n("&Options"));

	MenuTextPos(pMenu, 3, i18n("&Help"));
		MenuTextCmd(pMenu, ID_APP_HELP,				i18n("&Help Topics...\tF1"));
		MenuTextCmd(pMenu, ID_HELP_CHECK_UPDATE,	i18n("Check For &Updates"));
		MenuTextCmd(pMenu, ID_APP_ABOUT,			i18n("&About PopMan..."));
	
	DrawMenuBar();

	OnMailMarkedForDelete(0, 0); // update text of "empty trash" button


	CString stText(i18n("Empty Trash"));
	stText += _T(" (000) ");

	CClientDC dc(&m_ButtonTrash);
	CSize size(100, 100);
	CFont * pOriginalFont = dc.SelectObject(&m_FontTrash);
	if(pOriginalFont) {
		size = dc.GetTextExtent(stText);
		dc.SelectObject(pOriginalFont);
	}
	//TRACE("InitGuiText: size.cx = %d \n", size.cx);

	CRect rect;
	m_ButtonTrash.GetWindowRect(rect);
	m_wndToolBar.ScreenToClient(rect);
	rect.right = rect.left + long(size.cx*1.2);
	
	m_ButtonTrash.MoveWindow(rect);
}


void CMainFrame::OnInitMenuPopup(CMenu* pPopupMenu, UINT nIndex, BOOL bSysMenu) 
{
	CFrameWnd::OnInitMenuPopup(pPopupMenu, nIndex, bSysMenu);

	CMenu* pMenu = GetMenu();
	if(pMenu == NULL) return;
	CMenu* pMenuTools = pMenu->GetSubMenu(2);
	CMenu* pPopup = pMenuTools->GetSubMenu(GetThemeMenuIdx());
	if(pPopup && pPopup->m_hMenu == pPopupMenu->m_hMenu) 
	{
		while(pPopupMenu->GetMenuItemCount() > 0)
			pPopupMenu->DeleteMenu(0, MF_BYPOSITION);

		pPopupMenu->AppendMenu(MF_STRING, ID_SELECT_THEME, i18n("Default"));

		CStringArray Themes;
		CPopManApp::GetAvailableThemes(Themes);
		
		CString currTheme = CPopManApp::GetCurrentTheme();

		int i = 0;
		int checkPos = 0;
		for(; i < Themes.GetSize(); ++i)
		{
			const CString& Theme = Themes[i];
			pPopupMenu->AppendMenu(MF_STRING, ID_SELECT_THEME, Theme);
			if(currTheme == Theme)
				checkPos = i+1;
		}

		pPopupMenu->CheckMenuRadioItem(0, i, checkPos, MF_BYPOSITION); 

	}
}

void CMainFrame::GetMessageString(UINT nID, CString& rMessage) const
{
	rMessage = GetCommnadString(nID);
	rMessage = Parse(rMessage, _T("\r\n"));
}

BOOL CMainFrame::OnToolTipText(UINT, NMHDR* pNMHDR, LRESULT* pResult)
{

#ifndef _countof
#define _countof(array) (sizeof(array)/sizeof(array[0]))
#endif

	ASSERT(pNMHDR->code == TTN_NEEDTEXTA || pNMHDR->code == TTN_NEEDTEXTW);

	// need to handle both ANSI and UNICODE versions of the message
	TOOLTIPTEXTA* pTTTA = (TOOLTIPTEXTA*)pNMHDR;
	TOOLTIPTEXTW* pTTTW = (TOOLTIPTEXTW*)pNMHDR;

	CString strTipText;
	UINT nID = pNMHDR->idFrom;
	if (pNMHDR->code == TTN_NEEDTEXTA && (pTTTA->uFlags & TTF_IDISHWND) ||
		pNMHDR->code == TTN_NEEDTEXTW && (pTTTW->uFlags & TTF_IDISHWND))
	{
		// idFrom is actually the HWND of the tool
		nID = ((UINT)(WORD)::GetDlgCtrlID((HWND)nID));
	}

	if (nID != 0) // will be zero on a separator
	{
		strTipText = GetCommnadString(nID);
		Parse(strTipText, _T("\r\n"));

		if(!strTipText.IsEmpty())
		{
		
		#ifndef _UNICODE
			if (pNMHDR->code == TTN_NEEDTEXTA)
				lstrcpyn(pTTTA->szText, strTipText, _countof(pTTTA->szText));
			else
				_mbstowcsz(pTTTW->szText, strTipText, _countof(pTTTW->szText));
		#else
			if (pNMHDR->code == TTN_NEEDTEXTA)
				_wcstombsz(pTTTA->szText, strTipText, _countof(pTTTA->szText));
			else
				lstrcpyn(pTTTW->szText, strTipText, _countof(pTTTW->szText));
		#endif
			*pResult = 0;

			// bring the tooltip window above other popup windows
			//::SetWindowPos(pNMHDR->hwndFrom, HWND_TOP, 0, 0, 0, 0, SWP_NOACTIVATE|SWP_NOSIZE|SWP_NOMOVE|SWP_NOOWNERZORDER);

			return TRUE;    // message was handled
		}
	}
	return CMainFrame::OnToolTipText(nID, pNMHDR, pResult);
}

CString CMainFrame::GetCommnadString(UINT nID) const
{
	CString stMessage;

	switch(nID)
	{
	case ID_FILE_ACCOUNTS:
		stMessage = i18n("Edit email accounts.\r\nAccounts");
		break;
	case ID_FILE_LIST_MAILS:
		stMessage = i18n("Update email listing.\r\nUpdate");
		break;
	case ID_FILE_CANCEL:
		stMessage = i18n("Cancel current operation.\r\nCancel");
		break;
	case ID_FILE_VIEW:
		stMessage = i18n("Show email content.\r\nShow");
		break;
	case ID_FILE_MAIL_SOURCE:
		stMessage = i18n("Show source code of this email.\r\nSource code");
		break;
	case ID_FILE_VIEW_EXTERNALLY:
		stMessage = i18n("Save this email to a temporary file with extension eml and open it.\r\nView externally");
		break;
	case ID_FILE_SAVE_MAIL:
		stMessage = i18n("Save selected email to file.\r\nSave");
		break;
	case ID_FILE_REPLY_TO:
		stMessage = i18n("Create a reply message to the sender.\r\nReply");
		break;
	case ID_FILE_DELETE_MAILS:
		stMessage = i18n("Delete selected emails.\r\nDelete");
		break;
	case ID_FILE_EMPTY_TRASH:
		stMessage = i18n("Delete marked emails.\r\nDelete marked emails");
		break;
	case ID_FILE_DELETE_UNMARK:
		stMessage = i18n("Unmark (undelete) selected emails.\r\nUnmark");
		break;
	case ID_FILE_BLOCK_SENDER:
		stMessage = i18n("Add sender to Black List.\r\nBlock Sender");
		break;
    case ID_FILE_TRUST_SENDER:
		stMessage = i18n("Add sender to White List.\r\nTrust Sender");
		break;
	case ID_APP_EXIT:
		stMessage = i18n("Quit the application.\r\nExit");
		break;
	case ID_VIEW_TOOLBAR:
		stMessage = i18n("Show or hide the toolbar.\r\nToggle ToolBar");
		break;
	case ID_VIEW_STATUS_BAR:
		stMessage = i18n("Show or hide the status bar.\r\nToggle StatusBar");
		break;
	case ID_VIEW_MULTITAB:
		stMessage = i18n("Enable\\Disable the Tab view.\r\nToggle Tab view");
		break;
	case ID_VIEW_HIDE_MARKED:
		stMessage = i18n("Hide messages that are marked for delete.\r\nHide marked messages");
		break;
    case ID_VIEW_TRASHTAB:
        stMessage = i18n("Move marked messages to a separate \"trash\" tab.\r\nEnable trash tab");
		break;
	case ID_EXTRAS_FONT:
		stMessage = i18n("Change font.\r\nFont");
		break;
	case ID_EXTRAS_FONT_COLOR:
		stMessage = i18n("Change font color.\r\nFont Color");
		break;
	case ID_APP_EMAIL_CLIENT:
		stMessage = i18n("Launch email client.\r\nEmail Client");
		break;
	case ID_EXTRAS_IMPORT_ACCOUNTS:
		stMessage = i18n("Import email accounts from Outlook Express.\r\nImport Accounts");
		break;
	case ID_EXTRAS_RULES:
		stMessage = i18n("Edit the rules.\r\nRules");
		break;
    case ID_RULES_ADD_LISTITEM:
		stMessage = i18n("Add item to Rules list.\r\nAdd To List");
		break;
    case ID_EXTRAS_WHITELIST:
		stMessage = i18n("Edit list of trusted senders.\r\nWhiteList");
		break;
	case ID_EXTRAS_BLACKLIST:
		stMessage = i18n("Edit list of blocked senders.\r\nBlackList");
		break;
	case ID_EXTRAS_SETTINGS:
		stMessage = i18n("Change program settings.\r\nSettings");
		break;
	case ID_APP_HELP:
		stMessage = i18n("Show Help Topics.\r\nHelp");
		break;
	case ID_HELP_CHECK_UPDATE:
		stMessage = i18n("Check if a new version is available.\r\nCheck for updates");
		break;
	case ID_APP_ABOUT:
		stMessage = i18n("Display program information, version number and copyright.\r\nAbout");
		break;
	case ID_TRAY_SHOW:
		stMessage = i18n("Show the main window.\r\nShow");
		break;
	case ID_APP_AUTOCHECK:
		stMessage = i18n("Enable AutoChecking.\r\nEnable");
		break;
    case ID_APP_MUTE_MODE:
		stMessage = i18n("Suspend New Mail Notification Sound.\r\nMute");
		break;      
	case ID_ATTACHMENT_OPEN:
		stMessage = i18n("Open the selected attachment.\r\nOpen");
		break;
	case ID_ATTACHMENT_SAVE:
		stMessage = i18n("Save the selected attachments.\r\nSave");
		break;
	case ID_ATTACHMENT_SAVE_ALL:
		stMessage = i18n("Save all attachments.\r\nSave All");
		break;
	case AFX_IDS_IDLEMESSAGE:
		{
			CFrameWnd* pFrame = (CFrameWnd*)this;
			const CPopManDoc* pDoc = (CPopManDoc*)pFrame->GetActiveDocument();
			if(pDoc == NULL) break;
			stMessage = pDoc->GetCurrentStateText();
			break;
		}
	case AFX_IDS_SCSIZE:
		stMessage = i18n("Change the window size.");
		break;
	case AFX_IDS_SCMOVE:
		stMessage = i18n("Change the window position.");
		break;
	case AFX_IDS_SCMINIMIZE:
		stMessage = i18n("Reduce the window to an icon.");
		break;
	case AFX_IDS_SCMAXIMIZE:
		stMessage = i18n("Enlarge the window to full size.");
		break;
	case AFX_IDS_SCCLOSE:
		stMessage = i18n("Close the active window.");
		break;
	case AFX_IDS_SCRESTORE:
		stMessage = i18n("Restore the window to normal size.");
		break;
	case ID_PROTECTMESSAGE:
		stMessage = i18n("Toggle the current [PROTECTED] state of the selected messages.\r\nToggle [PROTECTED]");
		break;
	case ID_CONTEXT_MARKASSEENONSERVER:
		stMessage = i18n("Mark selected email as seen (read) on server.\r\nMark As Seen on Server");
		break;
	default:
		stMessage.Empty();
		break;
	}

	return stMessage;
}


/*
void CMainFrame::SendMessageToDescendants(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	for (HWND hWndChild = ::GetTopWindow(hWnd); hWndChild != NULL; hWndChild = ::GetNextWindow(hWndChild, GW_HWNDNEXT))
	{
		::SendMessage(hWndChild, message, wParam, lParam);
		if (::GetTopWindow(hWndChild) != NULL)
			SendMessageToDescendants(hWndChild, message, wParam, lParam);
	}
}
*/

BOOL CMainFrame::OnNotify(WPARAM wParam, LPARAM lParam, LRESULT* pResult) 
{
	LPNMHDR		lpnm	= (LPNMHDR)lParam;
	LPNMTOOLBAR lpnmTB	= (LPNMTOOLBAR)lParam;

	if(lpnm->code == TBN_DROPDOWN)
	{
   		RECT rect;
		::SendMessage(lpnmTB->hdr.hwndFrom, TB_GETRECT, (WPARAM)lpnmTB->iItem, (LPARAM)&rect);
		::MapWindowPoints(lpnmTB->hdr.hwndFrom, HWND_DESKTOP, (LPPOINT)&rect, 2);   
		
		TPMPARAMS tpm;
		tpm.cbSize = sizeof(TPMPARAMS);
		tpm.rcExclude.top    = rect.top;
		tpm.rcExclude.left   = rect.left;
		tpm.rcExclude.bottom = rect.bottom;
		tpm.rcExclude.right  = rect.right;

		CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
		if (pDoc == NULL) return FALSE;

		CMenu mnContext;
		mnContext.CreatePopupMenu();

		mnContext.AppendMenu(MF_STRING, ID_FILE_LIST_MAILS, i18n("&Update\tF5"));
		mnContext.SetDefaultItem(ID_FILE_LIST_MAILS);
		if(pDoc->m_Accounts.GetCount() > 0)
			mnContext.AppendMenu(MF_SEPARATOR);

		
		POSITION pos = pDoc->m_Accounts.GetHeadPosition();
		while(pos != NULL)
		{
			CAccount* pAcc = pDoc->m_Accounts.GetNext(pos);
			CString stName = pAcc->m_stName;
			stName.Replace(_T("&"), _T("&&"));
			mnContext.AppendMenu(MF_STRING, ID_ACCOUNTS_0, stName);
		}
	
		::TrackPopupMenuEx(mnContext.m_hMenu, TPM_LEFTALIGN|TPM_LEFTBUTTON|TPM_VERTICAL, rect.left, rect.bottom, GetSafeHwnd(), &tpm);

		return FALSE; 
	}

	return CFrameWnd::OnNotify(wParam, lParam, pResult);
}


void CMainFrame::OnMenuSelect(UINT nItemID, UINT nFlags, HMENU hSysMenu) 
{
	CFrameWnd::OnMenuSelect(nItemID, nFlags, hSysMenu);

	MENUITEMINFO info;
	info.cbSize = sizeof(info);
	info.fMask = MIIM_STATE;

	int count = ::GetMenuItemCount(hSysMenu);
	for(int i=0; i < count; ++i) 
	{
		if(GetMenuItemInfo(hSysMenu, i, TRUE, &info)) {
			if(info.fState & MFS_HILITE) {
				m_nSelectedMenuIdx = i; 
				break;
			}
		}
	}

  // GetMainView()->SendMessage(WM_MENUSELECT, MAKELONG(nItemID, nFlags), (UINT)hSysMenu);

}
void CMainFrame::OnListAccount()
{
	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if (pDoc == NULL) return;

	POSITION pos = pDoc->m_Accounts.FindIndex(m_nSelectedMenuIdx - 2); // The first item is "Update", the second is the separator!
	if(pos == NULL) return;
	CAccount* pAcc = pDoc->m_Accounts.GetNext(pos);

	CAccounts Accs;
	Accs.AddTail(pAcc);

    BOOL hardReload = (GetKeyState(VK_CONTROL) & 8000) != 0;

	pDoc->ListMailsFromAccs(Accs, TRUE, hardReload);
}


void CMainFrame::OnStatusBarDblClick(int nPane)
{
/*
	if(nPane != 1) return;

	CMainFrame* pFrame = (CMainFrame*)AfxGetMainWnd();
	if (pFrame == NULL) return;

	CString stErr = pFrame->m_wndStatusBar.GetPaneText(1);
	stErr.TrimLeft();
	if(stErr.IsEmpty()) return;

	CPopManDoc* pDoc = (CPopManDoc*)pFrame->GetActiveDocument();
	if (pDoc == NULL) return;


	CDlgAccReport DlgReport;
	DlgReport.m_pAccounts = &pDoc->m_Accounts;

	DlgReport.DoModal();
*/
}


CPopManView* CMainFrame::GetMainView()
{
	CWnd* pWnd = NULL;

	if(m_pSplitter == NULL)
		pWnd = GetActiveView();
	else
		pWnd = m_pSplitter->GetPane(0, 0);

	CPopManView* pPopManView = NULL;
	try 
	{
		pPopManView = dynamic_cast<CPopManView*>(pWnd);
	} 
	catch (...)
	{
		return NULL;
	}
	return pPopManView;
}

void CMainFrame::OnViewMultitab() 
{
	CPopManView* pView = GetMainView();
	if(pView == NULL) return;

	pView->SetMultiTabbing(!pView->GetMultiTabbing());

}

void CMainFrame::OnUpdateViewMultitab(CCmdUI* pCmdUI) 
{
	CPopManView* pView = GetMainView();
	if(pView == NULL) return;

	pCmdUI->SetCheck(pView->GetMultiTabbing());
}

void CMainFrame::OnCheckTab()
{
	CPopManView* pView = GetMainView();
	if(pView == NULL) return;

	pView->CheckCurrentTab();
}

void CMainFrame::OnSelectTheme()
{
	if(m_nSelectedMenuIdx <= 0) {
		CPopManApp::SetTheme(_T(""));
		return;
	}

	CStringArray Themes;
	CPopManApp::GetAvailableThemes(Themes);

	int Idx = m_nSelectedMenuIdx - 1;
	if(Themes.GetUpperBound() >= Idx)
		CPopManApp::SetTheme(Themes[Idx]);
}





void CMainFrame::OnInitMenu(CMenu* pMenu) 
{
	CFrameWnd::OnInitMenu(pMenu);
	
	if(!pMenu) return;

	if(GetMenu()->m_hMenu != pMenu->m_hMenu) // consider only the main menu
		return;

	CPopManDoc* pDoc = (CPopManDoc*)GetActiveDocument();
	if(!pDoc) return;


	CMenu* pMenuFile = pMenu->GetSubMenu(0); // File submenu
	CMenu* pMenuView = pMenu->GetSubMenu(1); // View submenu

    bool bMarkDelete = (pDoc->m_bMarkDelete || pDoc->CountMarkedMails() > 0);

    if(bMarkDelete) 
    {
        InsertMenu(pMenuFile, ID_FILE_EMPTY_TRASH, false, ID_FILE_DELETE_MAILS, true);
        InsertMenu(pMenuView, ID_VIEW_HIDE_MARKED, true,  ID_VIEW_MULTITAB, false);      
    }
    else
    {
        RemoveMenu(pMenuFile, ID_FILE_EMPTY_TRASH);
        RemoveMenu(pMenuView, ID_VIEW_HIDE_MARKED);             
    }

	CPopManView* pView = GetMainView();
	
    if(bMarkDelete && pView && pView->GetMultiTabbing())
        InsertMenu(pMenuView, ID_VIEW_TRASHTAB,    true,  ID_VIEW_MULTITAB, false);
    else
        RemoveMenu(pMenuView, ID_VIEW_TRASHTAB); 

}


void CMainFrame::RemoveMenu(CMenu* pMenu, unsigned int menuID)
{
    if(!pMenu) return;
	pMenu->RemoveMenu(menuID, MF_BYCOMMAND);

	// remove multiple separator menus
	bool prevIsSeparator = false;
	for(int i = 0; i < pMenu->GetMenuItemCount(); ++i) 
    {
		bool curIsSeparator = ((MF_SEPARATOR & pMenu->GetMenuState(i, MF_BYPOSITION)) != 0);
		if (curIsSeparator && prevIsSeparator)
			pMenu->RemoveMenu(i--, MF_BYPOSITION);
		prevIsSeparator = curIsSeparator;
	}
}

void CMainFrame::InsertMenu(CMenu* pMenu, unsigned int menuID, bool bBefore, unsigned int menuIDReference, bool bSeparate)
{
    if(!pMenu || pMenu->GetMenuState(menuID, MF_BYCOMMAND) != 0xFFFFFFFF) return;
	
	for(int i = 0; i < pMenu->GetMenuItemCount(); ++i)
    {
		if(pMenu->GetMenuItemID(i) == menuIDReference) 
        {
            int pos = bBefore ? i : i+1;
			
			pMenu->InsertMenu(pos, MF_BYPOSITION|MF_STRING, menuID, _T("")); // caption will be set by InitGuiText
            
            if(bSeparate) 
            {
                if(bBefore) ++pos;
                pMenu->InsertMenu(pos, MF_BYPOSITION|MF_SEPARATOR, 0, _T(""));
            }

			InitGUIText();
			return;
		}
	}	
}

void CMainFrame::OnAppEscape() 
{
    HideToTray();
}
