// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// MailList.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "MailList.h"
#include "PopManDoc.h"
#include "PopManView.h"
#include "RegKey.h"
#include "DlgSource.h"
#include "MessageFrm.h"
#include "MessageView.h"
#include "MainFrm.h"
#include "StrFunctions.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


//  i18nComment("List View")

const TCHAR szColWidthValue[]   = _T("Width");
const TCHAR szColIndexValue[]   = _T("Index");
const TCHAR szSortColumnValue[] = _T("SortColumn");
const TCHAR szSortAscValue[]    = _T("SortAsc");

const int TimerID_ItemChanged = 1;
const int TimerID_RulesTip    = 2;


CMailList::COLUMN CMailList::Columns[] = 
{	// ID                     Order Width  Caption CaptionMenu  UseCaption  RegKey         Alignment     Command-ID
	{  COLUMNS::colAttach,	  0,	20,    0,	   0,			false,	  _T("Attach"),   LVCFMT_LEFT,  ID_COLUMN_ATTACH    },
	{  COLUMNS::colFrom,      1,    130,   0,	   0,			true,	  _T("From"),     LVCFMT_LEFT,  ID_COLUMN_FROM      },
	{  COLUMNS::colSubject,   2,    280,   0,	   0,			true,	  _T("Subject"),  LVCFMT_LEFT,  ID_COLUMN_SUBJECT   },
	{  COLUMNS::colSize,      3,     65,   0,	   0,			true,	  _T("Size"),     LVCFMT_RIGHT, ID_COLUMN_SIZE      },
	{  COLUMNS::colTo,        4,    110,   0,	   0,			true,	  _T("To"),       LVCFMT_LEFT,  ID_COLUMN_TO        },
	{  COLUMNS::colReceived,  5,    100,   0,	   0,			true,	  _T("Date"),     LVCFMT_RIGHT, ID_COLUMN_DATE      },
	{  COLUMNS::colAccount,   6,    100,   0,	   0,			true,	  _T("Account"),  LVCFMT_LEFT,  ID_COLUMN_ACCOUNT   },
	{  COLUMNS::colSent,	  -1,   100,   0,	   0,			true,	  _T("Sent"),	  LVCFMT_RIGHT, ID_COLUMN_SENT		},
	{  COLUMNS::colPriority,  -1,    80,   0,	   0,			true,	  _T("Priority"), LVCFMT_LEFT,  ID_COLUMN_PRIORITY  },
	{  COLUMNS::colUserAgent, -1,   100,   0,	   0,			true,	  _T("UserAgent"),LVCFMT_LEFT,  ID_COLUMN_USERAGENT },
    {  COLUMNS::colRules,     -1,   100,   0,	   0,			true,	  _T("Rules"),    LVCFMT_LEFT,  ID_COLUMN_RULES     },
	{  COLUMNS::colSender,    -1,   100,   0,	   0,			true,	  _T("Sender"),   LVCFMT_LEFT,  ID_COLUMN_SENDER },
	{  COLUMNS::colFromComplete,-1, 120,   0,	   0,			true,	  _T("FromComplete"), LVCFMT_LEFT,  ID_COLUMN_FROM_COMPLETE },
};


BEGIN_MESSAGE_MAP(CMailList, CListCtrl)
	//{{AFX_MSG_MAP(CMailList)
	ON_NOTIFY_REFLECT(NM_CUSTOMDRAW, OnCustomDraw)
	ON_COMMAND(ID_FILE_DELETE_MAILS, OnDeleteMails)
	ON_UPDATE_COMMAND_UI(ID_FILE_DELETE_MAILS, OnUpdateDeleteMails)
	ON_COMMAND(ID_EDIT_SELECT_ALL, OnSelectAll)
	ON_COMMAND(ID_FILE_SAVE_MAIL, OnSaveMail)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE_MAIL, OnUpdateSaveMail)
	ON_NOTIFY_REFLECT(LVN_COLUMNCLICK, OnColumnClick)
	ON_COMMAND(ID_FILE_MAIL_SOURCE, OnMailSource)
	ON_UPDATE_COMMAND_UI(ID_FILE_MAIL_SOURCE, OnUpdateMailSource)
	ON_WM_SETCURSOR()
	ON_WM_CONTEXTMENU()
	ON_COMMAND(ID_FILE_VIEW, OnView)
	ON_UPDATE_COMMAND_UI(ID_FILE_VIEW, OnUpdateView)
	ON_NOTIFY_REFLECT(NM_DBLCLK, OnDblclk)
	ON_NOTIFY_REFLECT(NM_RETURN, OnReturn)
	ON_COMMAND(ID_FILE_REPLY_TO, OnReplyTo)
	ON_UPDATE_COMMAND_UI(ID_FILE_REPLY_TO, OnUpdateReplyTo)
	ON_WM_TIMER()
	ON_COMMAND(ID_FILE_BLOCK_SENDER, OnBlockSender)
	ON_UPDATE_COMMAND_UI(ID_FILE_BLOCK_SENDER, OnUpdateBlockSender)
	ON_NOTIFY_REFLECT(LVN_ITEMCHANGED, OnItemChanged)
	ON_WM_GETDLGCODE()
	ON_COMMAND(ID_FILE_DELETE_MAILS_SHIFT, OnDeleteMailsShift)
	ON_COMMAND(ID_FILE_DELETE_UNMARK, OnDeleteUnmark)
	ON_UPDATE_COMMAND_UI(ID_FILE_DELETE_UNMARK, OnUpdateDeleteUnmark)
	ON_COMMAND(ID_RULES_ADD_LISTITEM, OnRulesAddListitem)
	ON_UPDATE_COMMAND_UI(ID_RULES_ADD_LISTITEM, OnUpdateRulesAddListitem)
	ON_WM_WINDOWPOSCHANGING()
	ON_WM_MOUSEMOVE()
	ON_UPDATE_COMMAND_UI(ID_FILE_TRUST_SENDER, OnUpdateTrustSender)
	ON_COMMAND(ID_FILE_TRUST_SENDER, OnTrustSender)
	ON_COMMAND(ID_FILE_DELETE_MAILS_CTRL, OnDeleteMails)
	ON_COMMAND(ID_APP_UPDATE_GUI, OnUpdateGUI)
	ON_WM_DESTROY()
	ON_COMMAND(ID_COLUMN_FROM, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_SUBJECT, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_SIZE, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_TO, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_DATE, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_ACCOUNT, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_ATTACH, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_SENT, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_PRIORITY, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_USERAGENT, OnColumnSelect)
    ON_COMMAND(ID_COLUMN_RULES, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_SENDER, OnColumnSelect)
	ON_COMMAND(ID_COLUMN_FROM_COMPLETE, OnColumnSelect)
	ON_COMMAND(ID_PROTECTMESSAGE, OnProtectMessage)
	//}}AFX_MSG_MAP
    ON_COMMAND(ID_TAB_MARKALL_READ, OnMarkAllRead)
    ON_UPDATE_COMMAND_UI(ID_TAB_MARKALL_READ, OnUpdateMarkAllRead)
    ON_COMMAND(ID_TAB_MARKALL_UNREAD, OnMarkAllUnread)
    ON_UPDATE_COMMAND_UI(ID_TAB_MARKALL_UNREAD, OnUpdateMarkAllUnread)
	ON_COMMAND(ID_CONTEXT_MARKASSEENONSERVER, &CMailList::OnMarkAsSeenOnServer)
	ON_COMMAND(ID_FILE_VIEW_EXTERNALLY, OnViewMailExternally)
	ON_UPDATE_COMMAND_UI(ID_FILE_VIEW_EXTERNALLY, OnUpdateViewMailExternally)	
END_MESSAGE_MAP()

// we have to use PreTranslateMessage because for some reason ON_NOTIFY_REFLECT(NM_RETURN, OnReturn) does not work anymore
BOOL CMailList::PreTranslateMessage(MSG* pMsg)
{
	if (pMsg->message == WM_KEYDOWN && pMsg->wParam == VK_RETURN)
	{
		OnView();
		return TRUE;
	}
	return CListCtrl::PreTranslateMessage(pMsg);
}

CMailList::CMailList()
{ 
	m_pDoc = NULL;
	m_IDSortColumn = COLUMNS::colReceived;
	m_bSortAscending = FALSE;
	m_prevMail = NULL;
	m_pAccounts = NULL;
	m_bShowUnmarkedMail = TRUE;    
    m_bShowMarkedMail = TRUE;
    m_bDisableRedraw = false;
    m_pLastTooltipMail = NULL;
    m_bTimerRulesTip = false;
}

BOOL CMailList::PreCreateWindow(CREATESTRUCT& cs)
{
	cs.style |= LVS_REPORT | LVS_SHOWSELALWAYS;
	return CListCtrl::PreCreateWindow(cs);
}

void CMailList::UpdateColumnsText()
{
	const int nColCount = sizeof(Columns)/sizeof(COLUMN);

	LV_COLUMN column;
	column.mask = LVCF_TEXT;

	for(int n = 0; n < nColCount; n++)
	{
		if(Columns[n].nOrderIndex != -1 && Columns[n].bCaptionInHeader)	
		{
			column.pszText = (LPTSTR)Columns[n].szCaption;
			SetColumn(Columns[n].nOrderIndex, &column);
		}
	}
}

void CMailList::InitGUIText()
{
	m_hClip = CPopManApp::GetIcon(IDI_ATTACHMENT); //(HICON)::LoadImage(AfxGetResourceHandle(), MAKEINTRESOURCE(IDI_CLIP), IMAGE_ICON, 16, 16, LR_DEFAULTCOLOR);
	m_Header.SetSortIcons(CPopManApp::GetIcon(IDI_SORT_ASC), CPopManApp::GetIcon(IDI_SORT_DESC));

	const int nColCount = sizeof(Columns)/sizeof(COLUMN);

	for(int n = 0; n < nColCount; n++)
	{
		switch(Columns[n].colID)
		{
		case COLUMNS::colFrom:
			Columns[n].szCaption = i18n("From");
			break;
		case COLUMNS::colSubject:
			Columns[n].szCaption = i18n("Subject");
			break;
		case COLUMNS::colSize:
			Columns[n].szCaption = i18n("Size");
			break;
		case COLUMNS::colTo:
			Columns[n].szCaption = i18n("To");
			break;
		case COLUMNS::colReceived:
			Columns[n].szCaption = i18n("Received");
			break;
		case COLUMNS::colAccount:
			Columns[n].szCaption = i18n("Account");
			break;
		case COLUMNS::colAttach:
			Columns[n].szCaption = i18n("Attachment");
			break;
		case COLUMNS::colSent:
			Columns[n].szCaption = i18n("Sent");
			break;
		case COLUMNS::colPriority:
			Columns[n].szCaption = i18n("Priority");
			break;
		case COLUMNS::colUserAgent:
			Columns[n].szCaption = i18n("UserAgent");
			break;
        case COLUMNS::colRules:
			Columns[n].szCaption = i18n("Rules");
			break;
		case COLUMNS::colSender:
			Columns[n].szCaption = i18n("Sender");
			break;
		case COLUMNS::colFromComplete:
			Columns[n].szCaption = i18n("From");
			Columns[n].szCaptionMenu = i18n("From <Sender>");
			break;
		}
	}
}

void CMailList::OnInitialUpdate(const CSettingsKey& settings)
{
#ifndef LVS_EX_DOUBLEBUFFER
    const unsigned long LVS_EX_DOUBLEBUFFER = 0x00010000;
#endif

   	SetExtendedStyle(LVS_EX_FULLROWSELECT | LVS_EX_HEADERDRAGDROP | LVS_EX_DOUBLEBUFFER );
	DeleteAllItems();
	while(DeleteColumn(0));
	
	InitGUIText();

	settings.QueryValue(szSortColumnValue, m_IDSortColumn);
	settings.QueryValue(szSortAscValue, m_bSortAscending);

	const int nColCount = sizeof(Columns)/sizeof(COLUMN);
	int n;
	
	for(n = 0; n < nColCount; n++)
	{
		CString ColName = Columns[n].szKey;
		settings.QueryValue(ColName + _T("Wdt"), Columns[n].nWidth);
		settings.QueryValue(ColName + _T("Idx"), Columns[n].nOrderIndex);
	}


	
	// sort by nOrderIndex:
	for(int i = nColCount-1; i > 0; --i)
		for(int j = 0; j < i; ++j)
			if(Columns[j].nOrderIndex > Columns[j+1].nOrderIndex) 
			{
				COLUMN temp = Columns[j];
				Columns[j] = Columns[j+1];
				Columns[j+1] = temp;
			}


	int count = 0;
	for(n = 0; n < nColCount; n++)
	{
		if(Columns[n].nOrderIndex != -1)
		{
			Columns[n].nOrderIndex = count++;
			LPCTSTR szCaption = _T("");
			if (Columns[n].bCaptionInHeader)
				szCaption = Columns[n].szCaption;
			AddColumn(szCaption, max(Columns[n].nWidth,10), Columns[n].dwAlign);
		}
	}

	if(m_Header.m_hWnd == NULL)
		m_Header.SubclassWindow(::GetDlgItem(m_hWnd, 0));
	
	int attIdx = 0;
	if(IndexFromID(COLUMNS::colAttach, attIdx))
		m_Header.SetColumnIcon(attIdx, m_hClip);

	UpdateSortIndicator();	
}


int CMailList::AddColumn(LPCTSTR szCaption, int nWidth, int nAlign)
{
	int nIdx = GetHeaderCtrl()->GetItemCount();

	LV_COLUMN column;
	column.mask = LVCF_FMT | LVCF_SUBITEM | LVCF_TEXT | LVCF_WIDTH;
	column.iSubItem = nIdx;
	column.pszText = (LPTSTR)szCaption;
	column.fmt = nAlign;
	column.cx = nWidth;
	// insert new column at the end:
	
	InsertColumn(nIdx, &column);  
	return nIdx;
}

void CMailList::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
	switch(lHint) 
	{

		case CPopManDoc::NOTIFICATIONS::ntClear:
			 {
				DeleteAllItems();
				break;
			 } 

		case CPopManDoc::NOTIFICATIONS::ntNewMail:
			 {
				if(pHint == NULL)
					return;

				CMail& Mail = *(CMail*)pHint;

				if(m_pAccounts && m_pAccounts->Find(Mail.GetAccount()) != NULL)
                    if(ShowMail(Mail))
						AddRow(&Mail); 

				break;
			 } 

		case CPopManDoc::NOTIFICATIONS::ntMailsDeleted:
			{
				if(pHint == NULL)
					return;
			
				CMails* pMailList = (CMails*)pHint;
	
				POSITION pos = pMailList->GetHeadPosition();
				while(pos != NULL)
				{
					CMail* pMail = (CMail*)pMailList->GetNext(pos);
					for(int n = 0; n < GetItemCount(); n++)
					{
						if(pMail == (CMail*)GetItemData(n))
						{
							SetItemData(n, 0);
							break;
						}
					}
				}
				
				for (int n = GetItemCount() - 1; n >= 0; n--)
				{
					if (GetItemData(n) == 0)
					{
						DeleteItem(n);
					}
				}				

				break;			
			}
	}
}

void CMailList::AddRow(const CMail* pMail, BOOL bSorted)
{

	LV_ITEM lvItem;
	lvItem.mask = LVIF_TEXT | LVIF_IMAGE;
	
	// insert new row:
	lvItem.iItem = GetItemCount();
	lvItem.iSubItem = 0;
	lvItem.pszText = NULL;
	lvItem.iImage = 0;
	InsertItem(&lvItem);

	lvItem.mask = LVIF_TEXT;

    CString tmp;

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetFrom();
	if(IndexFromID(COLUMNS::colFrom, lvItem.iSubItem))
		SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetFromAddress();
	if (IndexFromID(COLUMNS::colSender, lvItem.iSubItem))
		SetItem(&lvItem);
	
	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetFromComplete();
	if (IndexFromID(COLUMNS::colFromComplete, lvItem.iSubItem))
		SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetSubject();
	if(IndexFromID(COLUMNS::colSubject, lvItem.iSubItem))
		SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetSizeFormated();
	if(IndexFromID(COLUMNS::colSize, lvItem.iSubItem))
		SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetTo();
	if(IndexFromID(COLUMNS::colTo, lvItem.iSubItem))
		SetItem(&lvItem);

	tmp = GetFormatedDate(pMail->GetReceived());
	lvItem.pszText = (LPTSTR)(LPCTSTR)tmp;
	if(IndexFromID(COLUMNS::colReceived, lvItem.iSubItem))
		SetItem(&lvItem);

	tmp = GetFormatedDate(pMail->GetDate());
	lvItem.pszText = (LPTSTR)(LPCTSTR)tmp;
	if(IndexFromID(COLUMNS::colSent, lvItem.iSubItem))
		SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetUserAgent();
	if(IndexFromID(COLUMNS::colUserAgent, lvItem.iSubItem))
		SetItem(&lvItem);

	switch(pMail->GetPriority()) {
	case CMail::MailPriority::High:	tmp = i18n("High"); break;
	case CMail::MailPriority::Low:	tmp = i18n("Low");  break;
	default: tmp = i18n("Normal");
	}
	lvItem.pszText = (LPTSTR)(LPCTSTR)tmp;
	if(IndexFromID(COLUMNS::colPriority, lvItem.iSubItem))
		SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetAccount()->m_stName;
	if(IndexFromID(COLUMNS::colAccount, lvItem.iSubItem))
		SetItem(&lvItem);

//    TRACE("AddRow: %s  \n", (LPCTSTR)pMail->GetSubject());
    if(IndexFromID(COLUMNS::colRules, lvItem.iSubItem)) {
        lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetRules();
		SetItem(&lvItem);
    }


	SetItemData(lvItem.iItem, (DWORD)pMail);

	if(bSorted && m_IDSortColumn != -1)
		SortItems(CompareFunc, (DWORD)this);
}


BOOL CMailList::IndexFromID(COLUMNS colID, int& nIndex)
{
	for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
	{
		if(Columns[n].colID == colID)
		{	
			if(Columns[n].nOrderIndex == -1)
				return FALSE;

			nIndex = Columns[n].nOrderIndex;
			return TRUE;
		}
	}
	return FALSE;
}

CMailList::COLUMNS CMailList::GetIDFromIndex(int nIndex)
{
	for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
	{
		if(Columns[n].nOrderIndex == nIndex)
			return Columns[n].colID;
	}
	return (COLUMNS)0;
}


void CMailList::OnMarkAsSeenOnServer()
{
	CWaitCursor Wait;

	POSITION pos = GetFirstSelectedItemPosition();
	if (pos == NULL)
		return;

	CMails MailList;
	CMail* pMail = NULL;

	int first = 0x7FFFFFFF; // biggest positive 4-byte integer
	int last = 0;
	while (pos)
	{
		int nItem = GetNextSelectedItem(pos);
		first = min(first, nItem);
		last = max(last, nItem);
		pMail = (CMail*)GetItemData(nItem);

		MailList.AddTail(pMail);
	}

	if (GetDocument()->ConnectionsPending()) return;

	/*if (GetDocument()->m_bConfirmDeleteMain)
	{
		int nCount = MailList.GetCount();
		int ret = NULL;

		if (nCount == 1)
			ret = AfxMessageBox(i18n("Are you sure you want to delete the selected email?"), MB_YESNO);
		else
		{
			CString stPrompt = StrFormat(i18n("Are you sure you want to delete the {1} selected emails?"), _T("d"), nCount);
			ret = AfxMessageBox(stPrompt, MB_YESNO);
		}

		if (ret == IDNO)
			return;
	}*/
	GetDocument()->MarkMailsAsSeen(&MailList);	
}

void CMailList::OnDeleteMails() 
{
	bool bShift = (::GetKeyState(VK_SHIFT) & 0x8000) != 0;
	DoDeleteMails(bShift);
}

void CMailList::OnDeleteMailsShift() // called by Delete+Shift
{
	DoDeleteMails(true);
}


void CMailList::DoDeleteMails(bool bShift)
{
	CWaitCursor Wait;
	
	POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

	CMails MailList;
	CMail* pMail = NULL;
	
	int first = 0x7FFFFFFF; // biggest positive 4-byte integer
	int last = 0;
	while(pos)
	{
		int nItem = GetNextSelectedItem(pos);
		first = min(first, nItem);
		last  = max(last, nItem);
		pMail = (CMail*)GetItemData(nItem);

		if(!pMail->GetAlwaysProtected() || (pMail->GetAlwaysProtected() && IsCTRLpressed() == TRUE))
			MailList.AddTail(pMail);
		else
			pMail = NULL;
	}

	if(GetDocument()->m_bMarkDelete && !bShift) 
	{
		POSITION pos = MailList.GetHeadPosition();
		while(pos != NULL)
		{
			pMail = MailList.GetNext(pos);
			pMail->SetMarkedForDelete(TRUE, TRUE);
		}

		RedrawItems(first, last);

        //TRACE("Before...\n");
        if(pMail) pMail->SetMarkedForDelete(TRUE, FALSE); // trigger event
        //TRACE("After...\n");

		GetDocument()->ShowMarkMessagInfo();

	}
	else
	{
		if(GetDocument()->ConnectionsPending()) return;

		if(GetDocument()->m_bConfirmDeleteMain)
		{
			int nCount = MailList.GetCount();
			int ret = NULL;

			if(nCount == 1)
				ret = AfxMessageBox(i18n("Are you sure you want to delete the selected email?"), MB_YESNO);
			else
			{
				CString stPrompt = StrFormat(i18n("Are you sure you want to delete the {1} selected emails?"), _T("d"), nCount);
				ret = AfxMessageBox(stPrompt, MB_YESNO);
			}
			
			if(ret == IDNO)
				return;
		}
		GetDocument()->DeleteMails(&MailList);
	}
}



void CMailList::OnUpdateDeleteMails(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsDeleteMailsEnabled());
}


void CMailList::OnDeleteUnmark() 
{
	POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

    CMails MailList;    

	int first = 0x7FFFFFFF; // biggest positive 4-byte integer
	int last = 0;
	while(pos)
	{
		int nItem = GetNextSelectedItem(pos);
		first = min(first, nItem);
		last  = max(last, nItem);
		CMail* pMail = (CMail*)GetItemData(nItem);
		MailList.AddTail(pMail);
	}

    CMail* pMail = NULL;
    pos = MailList.GetHeadPosition();
	while(pos != NULL)
	{
		pMail = MailList.GetNext(pos);
        pMail->SetMarkedForDelete(FALSE, TRUE);
	}
    pMail->SetMarkedForDelete(FALSE, FALSE);

    first = min(first, GetItemCount()-1);
    last  = min(last,  GetItemCount()-1);

    if(first > -1 && last > -1 && first <= last)
	    RedrawItems(first, last);
}

void CMailList::OnUpdateDeleteUnmark(CCmdUI* pCmdUI) 
{
	bool bEnabled = false;
	POSITION pos = GetFirstSelectedItemPosition();
	while(pos)
	{
		int nItem = GetNextSelectedItem(pos);
		const CMail* pMail = (CMail*)GetItemData(nItem);
		if(pMail->IsMarkedForDelete()) {
			bEnabled = true;
			break;
		}
	}
	
	pCmdUI->Enable(bEnabled);
}


void CMailList::OnSaveMail() 
{
	POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

	CMails MailList;
	CMail* pMail = NULL;
	
	while(pos)
	{
		int nItem = GetNextSelectedItem(pos);
		pMail = (CMail*)GetItemData(nItem);

		MailList.AddTail(pMail);
	}

	GetDocument()->SaveMail(&MailList);
}

void CMailList::OnUpdateSaveMail(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsSaveMailEnabled());	
}


void CMailList::OnViewMailExternally() 
{
	POSITION pos = GetFirstSelectedItemPosition();
	if (pos == NULL)
		return;
	
	int nItem = GetNextSelectedItem(pos);
	CMail * pMail = (CMail*)GetItemData(nItem);

	GetDocument()->ViewMailExternally(pMail);
}

void CMailList::OnUpdateViewMailExternally(CCmdUI* pCmdUI) 
{
	bool enabled = ((GetSelectedCount() == 1) && !GetDocument()->ConnectionsPending());
	pCmdUI->Enable(enabled);
}


void CMailList::OnSelectAll() 
{
	SetItemState(-1, LVIS_SELECTED, LVIS_SELECTED);
}


/*
BOOL CMailList::LoadColumnSettings(LPCTSTR szColumn, UINT& nWidth, int& nOrderIndex)
{	
	CSettingsKey Key;
	if(!OpenSettingsKey(Key, szColumnsKey))
		return FALSE;

	if(!Key.OpenSubKey(szColumn))
		return FALSE;


	if(!Key.QueryValue(szColWidthValue, nWidth))
		return FALSE;

	if(!Key.QueryValue(szColIndexValue, nOrderIndex))
		return FALSE;

	return TRUE;
}


void CMailList::SaveColumnSettings(LPCTSTR szColumn, UINT nWidth, int nOrderIndex)
{
	CSettingsKey Key;
	if(!CreateSettingsKey(Key, szColumnsKey))
		return;

	if(Key.CreateSubKey(szColumn))
	{
		Key.SetValue(szColWidthValue,   nWidth);
		Key.SetValue(szColIndexValue,   nOrderIndex);
	}
}
*/

/*
void CMailList::OnDestroy() 
{
	LV_COLUMN column;
	column.mask =  LVCF_ORDER;
	

  SaveSettings();
	
	for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
	{
		int nWidth, iOrder;

		if(Columns[n].nOrderIndex != -1)
		{
			GetColumn(Columns[n].nOrderIndex, &column);
			iOrder = column.iOrder;
			nWidth = GetColumnWidth(Columns[n].nOrderIndex);
		} 
		else
		{
			nWidth = Columns[n].nWidth;
			iOrder = -1;
		}
		
		SaveColumnSettings(Columns[n].szKey, 
						   nWidth, 
						   iOrder);
	}


	CListCtrl::OnDestroy();
}
*/


void CMailList::OnColumnClick(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_LISTVIEW* pNMListView = (NM_LISTVIEW*) pNMHDR;
	
	int idColumn = GetIDFromIndex(pNMListView->iSubItem);

	if (idColumn == m_IDSortColumn) 
		m_bSortAscending = !m_bSortAscending; 
	else
		m_bSortAscending = TRUE;

	m_IDSortColumn = idColumn;
	UpdateSortIndicator();

	SortItems(CompareFunc, (DWORD)this);

	*pResult = 0;
}

int CALLBACK CMailList::CompareFunc(LPARAM lParam1, LPARAM lParam2, LPARAM lParamSort)
{
	const CMail* pM1 = reinterpret_cast<CMail*>(lParam1);
	const CMail* pM2 = reinterpret_cast<CMail*>(lParam2);
	const CMailList* pView = reinterpret_cast<CMailList*>(lParamSort);
	
    int value = 0;

	switch(pView->m_IDSortColumn)
	{
	case COLUMNS::colSize:
		if(pM1->GetSize() < pM2->GetSize())
			value = -1;
        else if(pM1->GetSize() > pM2->GetSize())
			value = 1;

        break;	

	case COLUMNS::colReceived:
		if(pM1->GetReceived() < pM2->GetReceived())
			value = -1;
		else if(pM1->GetReceived() > pM2->GetReceived())
			value = 1;

        break;

	case COLUMNS::colSent:
		if(pM1->GetDate() < pM2->GetDate())
			value = -1;
		else if(pM1->GetDate() > pM2->GetDate())
			value = 1;
		
		break;
	
	case COLUMNS::colAttach:
		if(pM1->m_Attachments.GetCount() < pM2->m_Attachments.GetCount())
			value = -1;
		else if(pM1->m_Attachments.GetCount() > pM2->m_Attachments.GetCount())
			value = 1;
		
		break;

	case COLUMNS::colPriority:
	{
		int p1 = (int)pM1->GetPriority();
		int p2 = (int)pM2->GetPriority();

		if(p1 < p2)
			value = -1;
		else if(p1 > p2)
			value = 1;
		
		break;
	}

	case COLUMNS::colFrom:
    {
		const CString& s1 = pM1->GetFrom();
		const CString& s2 = pM2->GetFrom();
        value = s1.CompareNoCase(s2);
    
		break;
    }

	case COLUMNS::colSender:
	{
		const CString& s1 = pM1->GetFromAddress();
		const CString& s2 = pM2->GetFromAddress();
		value = s1.CompareNoCase(s2);

		break;
	}

	case COLUMNS::colFromComplete:
	{
		const CString& s1 = pM1->GetFromComplete();
		const CString& s2 = pM2->GetFromComplete();
		value = s1.CompareNoCase(s2);

		break;
	}
	
	case COLUMNS::colTo:
    {
		const CString& s1 = pM1->GetTo();
		const CString& s2 = pM2->GetTo();
        value = s1.CompareNoCase(s2);
		break;
    }
	
	case COLUMNS::colSubject:
    {
		const CString& s1 = pM1->GetSubject();
		const CString& s2 = pM2->GetSubject();
        value = s1.CompareNoCase(s2);
		break;
    }

	case COLUMNS::colUserAgent:
    {
		const CString& s1 = pM1->GetUserAgent();
		const CString& s2 = pM2->GetUserAgent();
        value = s1.CompareNoCase(s2);
		break;
    }

    case COLUMNS::colRules:
    {
		const CString& s1 = pM1->GetRules();
		const CString& s2 = pM2->GetRules();
        value = s1.CompareNoCase(s2);
		break;
    }

	case COLUMNS::colAccount:
    {
		const CString& s1 = pM1->GetAccount()->m_stName;
		const CString& s2 = pM2->GetAccount()->m_stName;
		value = s1.CompareNoCase(s2);

		if(value == 0)
		{
			if(pM1->GetReceived() < pM2->GetReceived())
				value = -1;
			else if(pM1->GetReceived() > pM2->GetReceived())
				value = 1;
		}

		break;
    }

    default:
        break;
	}

    return value * (pView->m_bSortAscending ? 1 : -1);
}



void CMailList::OnMailSource() 
{
    CMail* pMail = GetSelectedMail();
    if(!pMail) return;
	
	CDlgSource* pDlgSrc = new CDlgSource(pMail->GetSubject(), pMail->GetSource());
}

void CMailList::OnUpdateMailSource(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsOneMailSelected());
}


BOOL CMailList::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
	if(GetDocument()->ConnectionsPending())
		::SetCursor(::LoadCursor(NULL, IDC_APPSTARTING));
	else
		::SetCursor(::LoadCursor(NULL, IDC_ARROW));
			
	return TRUE;
}


static void MenuTextCmd(CMenu* pMenu, int Cmd, LPCTSTR szText)
{
	pMenu->ModifyMenu(Cmd, MF_BYCOMMAND|MF_STRING, Cmd, szText);
}

void CMailList::OnContextMenu(CWnd* pWnd, CPoint point) 
{	
	if(GetHeaderCtrl()->m_hWnd == pWnd->m_hWnd)
	{
		InitGUIText(); // necessary because the translation may have been changed (rendering the stored pointers invalid!)
	
		CMenu mnContext;
		mnContext.CreatePopupMenu();

		for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
		{
			LPCTSTR szCaption = Columns[n].szCaptionMenu == 0 ? Columns[n].szCaption : Columns[n].szCaptionMenu;
			mnContext.AppendMenu(MF_STRING, Columns[n].nCmdID, szCaption);
			mnContext.CheckMenuItem(Columns[n].nCmdID, (Columns[n].nOrderIndex != -1 ? MF_CHECKED : MF_UNCHECKED));
		}
		
		mnContext.TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, point.x, point.y, AfxGetMainWnd());

		return;
	}

	int iSel = GetSelectedCount();
	if(iSel != 0)
	{
		CMenu mnContext;
		mnContext.LoadMenu(IDR_CONTEXT_ITEM);	
		CMenu* pSubMenu = mnContext.GetSubMenu(0);

		MenuTextCmd(pSubMenu, ID_FILE_VIEW,			i18n("&Open...\tCtrl+O"));
		MenuTextCmd(pSubMenu, ID_FILE_MAIL_SOURCE,  CString(i18n("Sou&rce...")) + _T("\tF10"));
		MenuTextCmd(pSubMenu, ID_FILE_VIEW_EXTERNALLY, CString(i18n("View E&xternally...")) + _T("\tF11"));
		MenuTextCmd(pSubMenu, ID_FILE_SAVE_MAIL,	i18n("&Save As...\tCtrl+S"));
		MenuTextCmd(pSubMenu, ID_FILE_REPLY_TO,		i18n("Repl&y To Sender\tCtrl+R"));
		MenuTextCmd(pSubMenu, ID_FILE_TRUST_SENDER,	i18n("&Trust Sender"));
        MenuTextCmd(pSubMenu, ID_FILE_BLOCK_SENDER,	i18n("&Block Sender"));
		MenuTextCmd(pSubMenu, ID_FILE_DELETE_MAILS,	i18n("&Delete\tDel"));
		MenuTextCmd(pSubMenu, ID_FILE_DELETE_UNMARK,i18n("&Unmark\tIns"));
		MenuTextCmd(pSubMenu, ID_PROTECTMESSAGE,    i18n("To&ggle [PROTECTED] State\tCtrl+T"));
		MenuTextCmd(pSubMenu, ID_CONTEXT_MARKASSEENONSERVER, i18n("Mark As Seen On Server\tF8"));
		MenuTextCmd(pSubMenu, ID_EDIT_SELECT_ALL, i18n("Select &All\tCtrl+A"));

		if(!GetDocument()->m_bMarkDelete && GetDocument()->CountMarkedMails() < 1)
			pSubMenu->DeleteMenu(ID_FILE_DELETE_UNMARK, MF_BYCOMMAND);

		pSubMenu->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, point.x, point.y, AfxGetMainWnd());
	}
}

BOOL CMailList::IsSaveMailEnabled()
{
	return ((GetSelectedCount() > 0) && !GetDocument()->ConnectionsPending());
}

BOOL CMailList::IsDeleteMailsEnabled()
{
	if(GetDocument()->m_bMarkDelete) 
	{
		bool bEnabled = false;
		POSITION pos = GetFirstSelectedItemPosition();
		while(pos)
		{
			int nItem = GetNextSelectedItem(pos);
			CMail* pMail = (CMail*)GetItemData(nItem);
			if(!pMail->IsMarkedForDelete()) {
				bEnabled = true;
				break;
			}
		}
		
		return bEnabled;
	}
	else
		return ((GetSelectedCount() > 0) && !GetDocument()->ConnectionsPending());
}

BOOL CMailList::IsOneMailSelected()
{
	return (GetSelectedCount() == 1);
}

void CMailList::OnColumnSelect()
{
	COLUMN* pColumn = NULL;
	int nCmd = GetCurrentMessage()->wParam;

	for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
	{
		if(Columns[n].nCmdID == nCmd)
		{
			pColumn = &Columns[n];
			break;
		}
	}

	COLUMN& Col = *pColumn;

	if(Col.nOrderIndex == -1)
	{
		// make column visible:
		LPCTSTR szCaption = _T("");
		if (Col.bCaptionInHeader)
			szCaption = Col.szCaption;
		Col.nOrderIndex = AddColumn(szCaption, max(Col.nWidth, 10), Col.dwAlign);

		int attIdx = 0;
		if(IndexFromID(COLUMNS::colAttach, attIdx))
			m_Header.SetColumnIcon(attIdx, m_hClip);
	} 
	else
	{
		// remove column:
		DeleteColumn(Col.nOrderIndex);
		m_Header.RemoveColumnIcon(Col.nOrderIndex);
		for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
		{
			if(Columns[n].nOrderIndex > Col.nOrderIndex) {
				Columns[n].nOrderIndex--;
				if(Columns[n].colID == COLUMNS::colAttach) {
					m_Header.RemoveColumnIcon(Columns[n].nOrderIndex+1);	
					m_Header.SetColumnIcon(Columns[n].nOrderIndex, m_hClip);
				}
			}
		}

		Col.nOrderIndex = -1;
		UpdateSortIndicator();
	}
	Refresh();
}

void CMailList::Refresh(BOOL bRetainSelectedMail)
{
    CMail* pSelectedMail = (bRetainSelectedMail ? GetSelectedMail() : NULL);

    m_bDisableRedraw = true;

	DeleteAllItems();

	if(m_pAccounts) 
    {
	    CAccounts& Accounts = *m_pAccounts;

	    POSITION pos1 = Accounts.GetHeadPosition();
	    while(pos1 != NULL)
	    {
		    CAccount* pAcc = Accounts.GetNext(pos1);
		    
		    CMails& Mails = pAcc->m_Mails;
		    POSITION pos2 = Mails.GetHeadPosition();
		    while(pos2 != NULL)
		    {
			    const CMail* pMail = Mails.GetNext(pos2);
                if(ShowMail(*pMail))
				    AddRow(pMail, FALSE);
		    }
	    }
	    
	    if(m_IDSortColumn != -1)
		    SortItems(CompareFunc, (DWORD)this);

	    if(bRetainSelectedMail)
            SelectMail(pSelectedMail);
    }

    m_bDisableRedraw = false;
}


void CMailList::UpdateSortIndicator()
{
	int nIdx = 0;

	if(IndexFromID((COLUMNS)m_IDSortColumn, nIdx))
		m_Header.SetSortColumn(nIdx, m_bSortAscending);
	else
		m_IDSortColumn = -1;
}


void CMailList::SaveSettings(CSettingsKey& settings) const
{
	settings.SetValue(szSortColumnValue, m_IDSortColumn);
	settings.SetValue(szSortAscValue, m_bSortAscending);

	LV_COLUMN column;
	column.mask =  LVCF_ORDER;
	
	for(int n = 0; n < sizeof(Columns)/sizeof(COLUMN); n++)
	{
		int nWidth, iOrder;

		if(Columns[n].nOrderIndex != -1)
		{
			GetColumn(Columns[n].nOrderIndex, &column);
			iOrder = column.iOrder;
			nWidth = GetColumnWidth(Columns[n].nOrderIndex);
		} 
		else
		{
			nWidth = Columns[n].nWidth;
			iOrder = -1;
		}
		
		CString ColName = Columns[n].szKey;
		settings.SetValue(ColName + _T("Wdt"), nWidth);
		settings.SetValue(ColName + _T("Idx"), iOrder);
	}
}

/*
void CMailList::LoadSettings()
{
	TRACE(_T("CMailList::LoadSettings()\n"));

	CSettingsKey Key;
	
	if(OpenSettingsKey(Key, szSettingsKey))
	{
		Key.QueryValue(szSortColumnValue, m_IDSortColumn);
		Key.QueryValue(szSortAscValue, m_bSortAscending);
	}
	Key.Close();

	HFONT hFont = (HFONT)SendMessage(WM_GETFONT, 0, 0);      
	LOGFONT lf;
	::GetObject(hFont, sizeof(LOGFONT), &lf);
		
	CString  stFaceName(lf.lfFaceName);
	int  	 nPoints = PointsFromHeight(lf.lfHeight);
	BOOL	 bBold   = (lf.lfWeight > FW_NORMAL);
	BOOL	 bItalic = lf.lfItalic;
	COLORREF clForeColor = GetTextColor();

	if(OpenSettingsKey(Key, szMainWindowKey))
	{
		Key.QueryValue(szFaceNameValue, stFaceName);
		Key.QueryValue(szPointsValue,   nPoints);
		Key.QueryValue(szBoldValue,     bBold);
		Key.QueryValue(szItalicValue,   bItalic);
		Key.QueryValue(szForeColorValue, clForeColor);
	}

	m_Font.CreateFont(HeightFromPoints(nPoints), 0, 0, 0, bBold ? FW_BOLD : FW_NORMAL,
					 bItalic, FALSE, 0, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, 
					 CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, stFaceName);
	
	m_BoldFont.DeleteObject();
	LOGFONT lg;
	m_Font.GetLogFont(&lg);
	lg.lfWeight = FW_BOLD;
	m_BoldFont.CreateFontIndirect(&lg);

	SetFont(&m_Font);
	SetTextColor(clForeColor);
}
*/


/*
void CMailList::SaveSettings()
{
	CSettingsKey Key;

	if(CreateSettingsKey(Key, szSettingsKey))
	{
		Key.SetValue(szSortColumnValue, m_IDSortColumn);
		Key.SetValue(szSortAscValue, m_bSortAscending);
	}
	Key.Close();

	LOGFONT lf;
	m_Font.GetLogFont(&lf);
		
	CString stFaceName(lf.lfFaceName);
	int  	nPoints = PointsFromHeight(lf.lfHeight);
	BOOL	bBold   = (lf.lfWeight > FW_NORMAL);
	BOOL	bItalic = lf.lfItalic;
	
	
	if(CreateSettingsKey(Key, szMainWindowKey))
	{
		Key.SetValue(szFaceNameValue, stFaceName);
		Key.SetValue(szPointsValue,   nPoints);
		Key.SetValue(szBoldValue,     bBold);
		Key.SetValue(szItalicValue,   bItalic);
		Key.SetValue(szForeColorValue, GetTextColor());
	}
}

  */


void CMailList::OnView() 
{
    CMail* pMail = GetSelectedMail();
    if(!pMail) return;

/* DEBUG -->
	CAccounts& Accs = GetDocument()->m_Accounts;
	POSITION pos1 = Accs.GetHeadPosition();
	while(pos1 != NULL)
	{
		CAccount* pAcc = Accs.GetNext(pos1);
		
		CMails& Mails = pAcc->m_Mails;
		POSITION pos2 = Mails.GetHeadPosition();
		while(pos2 != NULL)
		{
			CMail* pMail = Mails.GetNext(pos2);
			TRACE(_T(" %X\t %s \n"), pMail, (LPCTSTR)pMail->GetSubject());
		}
	}
// <-- DEBUG */

	ASSERT(GetDocument()->IsMailValid(pMail));
	if(GetDocument()->IsMailValid(pMail))
		CMessageFrm* pMessage = new CMessageFrm(GetDocument(), pMail);
}

void CMailList::OnUpdateView(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsOneMailSelected());
}

void CMailList::OnDblclk(NMHDR* pNMHDR, LRESULT* pResult) 
{
	OnView();
	*pResult = 0;
}

void CMailList::OnReturn(NMHDR* pNMHDR, LRESULT* pResult) 
{
	OnView();
	*pResult = 0;
}



void CMailList::OnCustomDraw(NMHDR* pNMHDR, LRESULT* pResult)
{
	LPNMLVCUSTOMDRAW  lplvcd = (LPNMLVCUSTOMDRAW)pNMHDR;

	switch(lplvcd->nmcd.dwDrawStage)
	{
	case CDDS_PREPAINT:
		*pResult = CDRF_NOTIFYITEMDRAW;          // ask for item notifications.
		break;

	case CDDS_ITEMPREPAINT:
	{
		*pResult = CDRF_DODEFAULT | CDRF_NOTIFYPOSTPAINT;

		int nItem = lplvcd->nmcd.dwItemSpec;
		CMail* pMail = (CMail*)GetItemData(nItem);
		if (pMail != NULL) {

			if (!pMail->IsRead() && GetDocument()->m_bMarkUnread)
			{
				if (!pMail->IsMarkedForDelete())
					::SelectObject(lplvcd->nmcd.hdc, m_BoldFont.m_hObject);
				else
					::SelectObject(lplvcd->nmcd.hdc, m_BoldStrikeFont.m_hObject);
				*pResult = CDRF_NEWFONT | CDRF_NOTIFYPOSTPAINT;
			}
			else if (pMail->IsMarkedForDelete()) {
				::SelectObject(lplvcd->nmcd.hdc, m_StrikeFont.m_hObject);
				*pResult = CDRF_NEWFONT | CDRF_NOTIFYPOSTPAINT;
			}

			if (pMail->m_bCustomColor)
				lplvcd->clrText = pMail->m_CustomColor;
			else if (pMail->GetAccount()->m_bCustomColor)
				lplvcd->clrText = pMail->GetAccount()->m_CustomColor;
		}
		break;

	}
	case CDDS_ITEMPOSTPAINT:
	{
		*pResult = CDRF_DODEFAULT;
		int nItem = (int)lplvcd->nmcd.dwItemSpec;
		
		CMail* pMail = (CMail*)GetItemData(nItem);
		int nColIdx = -1;
		
		if (pMail != NULL && pMail->m_Attachments.GetCount() > 0 
			&& IndexFromID(COLUMNS::colAttach, nColIdx))
		{
			CRect rcItem, rcHead;

			GetItemRect(nItem, &rcItem, LVIR_BOUNDS);
			int offset = rcItem.left;
			GetHeaderCtrl()->GetItemRect(nColIdx, &rcHead);
	
			rcItem.left = rcHead.left + offset;
			rcItem.right = rcHead.right + offset;

			POINT pt;
			pt.x = rcItem.left + (rcItem.Width() - 16) / 2;
			pt.y = rcItem.top  + (rcItem.Height() - 16) / 2;
	
			::DrawIconEx(lplvcd->nmcd.hdc,  pt.x, pt.y, m_hClip, 16, 16, 0, 0, DI_NORMAL);

			*pResult = CDRF_SKIPDEFAULT;
		}

		break;
	}
	default:
		*pResult = CDRF_DODEFAULT;
	}
}

void CMailList::OnItemChanged(NMHDR* pNMHDR, LRESULT* pResult) 
{
	*pResult = 0;

	NM_LISTVIEW* pNMListView = (NM_LISTVIEW*)pNMHDR;
	
	if((pNMListView->uChanged & LVIF_STATE) == 0)
		return;

	if(pNMListView->uNewState & LVIS_FOCUSED)
	{
		CMail* pMail = (CMail*)GetItemData(pNMListView->iItem);

		if(GetFirstSelectedItemPosition() != NULL)
			SetMailInMessageView(pMail);
	}

	SetTimer(TimerID_ItemChanged, 50, NULL);
}

void CMailList::OnTimer(UINT nIDEvent) 
{
    if(nIDEvent == TimerID_ItemChanged)
    {
	    if(GetFirstSelectedItemPosition() == NULL)
		    SetMailInMessageView(NULL);

	    KillTimer(nIDEvent);
    }
    else if(nIDEvent == TimerID_RulesTip)
    {
        CMail* pMail = NULL;
        
        POINT point;
        ::GetCursorPos(&point);
        ScreenToClient(&point);

        if(IsOnNonEmptyRulesSubItem(point, pMail)) 
        {        
            ShowRulesTip(pMail);
            m_pLastTooltipMail = pMail;
        }
        
        KillTimer(nIDEvent);
        m_bTimerRulesTip = false;
    }

	CListCtrl::OnTimer(nIDEvent);
}

int CMailList::GetSelectedItem() const
{
	POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL) return -1;

    int nSelectedItem = GetNextSelectedItem(pos);

    const UINT flags = LVIS_SELECTED | LVIS_FOCUSED;

	while(pos) // if there are more selected items, look for a focused item
    {
        int nItem = GetNextSelectedItem(pos);
        if(flags == GetItemState(nItem, flags)) {
            nSelectedItem = nItem;
            break;
        }
	}

	return nSelectedItem;
}

CMail* CMailList::GetSelectedMail() const
{
    int selItem = GetSelectedItem();
    if(selItem < 0) return NULL;
	return (CMail*)GetItemData(selItem);
}

void CMailList::OnReplyTo() 
{
	if(!IsOneMailSelected())
		return;
	
	GetDocument()->ReplyToMail(GetSelectedMail());
}

void CMailList::OnUpdateReplyTo(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsOneMailSelected());	
}



void CMailList::OnUpdateGUI()
{
	InitGUIText();
	UpdateColumnsText();

	int attIdx = 0;
	if(IndexFromID(COLUMNS::colAttach, attIdx))
		m_Header.SetColumnIcon(attIdx, m_hClip);
	
	Invalidate();
	UpdateWindow();
}

void CMailList::OnBlockSender() 
{

	POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

	CMails MailList;
	CString stSenders;
	
	while(pos)
	{
		int nItem = GetNextSelectedItem(pos);
		CMail* pMail = (CMail*)GetItemData(nItem);
		stSenders += pMail->GetFromAddress() + _T("\r\n");
		MailList.AddTail(pMail);
	}

	CString stPrompt = i18n("This will add the following email addresses to the Black List:");
	stPrompt += _T("\r\n\r\n");
	stPrompt += stSenders;
	stPrompt += _T("\r\n");
	stPrompt += i18n("Note: All future emails from senders on the Black List will be deleted automatically!");
	
	if(IDCANCEL == AfxMessageBox(stPrompt, MB_OKCANCEL))
		return;
	
	GetDocument()->BlackListSenders(MailList);
}

void CMailList::OnUpdateBlockSender(CCmdUI* pCmdUI) 
{
    BOOL bEnable = FALSE;

    POSITION pos = GetFirstSelectedItemPosition();
	while(pos)
	{
		const CMail* pMail = (CMail*)GetItemData(GetNextSelectedItem(pos));
		if( !GetDocument()->m_BlackList.Contains(pMail->GetFromAddress()) &&
            !GetDocument()->m_WhiteList.Contains(pMail->GetFromAddress()) ) 
        {
            bEnable = TRUE;
            break;
        }
	}

	pCmdUI->Enable(bEnable);	
}


void CMailList::OnTrustSender() 
{
    POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

	CMails MailList;
	CString stSenders;
	
	while(pos)
	{
		int nItem = GetNextSelectedItem(pos);
		CMail* pMail = (CMail*)GetItemData(nItem);
		stSenders += pMail->GetFromAddress() + _T("\r\n");
		MailList.AddTail(pMail);
	}

	CString stPrompt = i18n("This will add the following email addresses to the White List:");
	stPrompt += _T("\r\n\r\n");
	stPrompt += stSenders;
	
	if(IDCANCEL == AfxMessageBox(stPrompt, MB_OKCANCEL))
		return;
	
	GetDocument()->WhiteListSenders(MailList);
}

void CMailList::OnUpdateTrustSender(CCmdUI* pCmdUI) 
{
    BOOL bEnable = FALSE;

    POSITION pos = GetFirstSelectedItemPosition();
	while(pos)
	{
		const CMail* pMail = (CMail*)GetItemData(GetNextSelectedItem(pos));
		if( !GetDocument()->m_WhiteList.Contains(pMail->GetFromAddress()) ) 
        {
            bEnable = TRUE;
            break;
        }
	}

	pCmdUI->Enable(bEnable);	
}


void CMailList::SetTopIndex(int nItem)
{
	if(GetItemCount() <= GetCountPerPage() || GetTopIndex() == nItem || nItem < 0)
		return;

	EnsureVisible(GetItemCount()-1, FALSE);
	EnsureVisible(nItem, FALSE);
}


void CMailList::SetMailInMessageView(CMail *pMail)
{
	CMainFrame* pFrame = (CMainFrame*)AfxGetMainWnd();
	CMessageView* pMsgView = NULL;
	if(pFrame)
		pMsgView = pFrame->GetMessageView();

	if(pMail == NULL || GetDocument()->IsMailValid(pMail))
	{
		if(pMsgView != NULL && pMail != m_prevMail) pMsgView->ShowMail(pMail);
		m_prevMail = pMail;
	}
}

BOOL CMailList::DeleteAllItems()
{
	BOOL res = CListCtrl::DeleteAllItems();
	SetMailInMessageView(NULL);
	return res;
}

CMail* CMailList::GetDisplayedMail() const
{
	return m_prevMail;
}

int CMailList::SelectMail(const CMail* pMail)
{
    if(!pMail) return -1;
	for(int i = 0; i < GetItemCount(); ++i)
	{
		if(pMail == (CMail*)GetItemData(i)) 
        {
			SelectItem(i);
			return i;
		}
	}
	return -1;
}

void CMailList::SelectItem(int idx)
{
    if(idx >= 0 && idx < GetItemCount())
    {
        if(GetFirstSelectedItemPosition())
	        SetItemState(-1, 0, LVIS_SELECTED | LVIS_FOCUSED);

        SetItemState(idx, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
        EnsureVisible(idx, FALSE);
    }
}

// necessary to make the Return work:
UINT CMailList::OnGetDlgCode()
{
	return CListCtrl::OnGetDlgCode() | DLGC_WANTMESSAGE;
}


void CMailList::SetFont(CFont* pFont, BOOL bRedraw)
{
	if(pFont)
	{
		m_BoldFont.DeleteObject();
		m_StrikeFont.DeleteObject();
		m_BoldStrikeFont.DeleteObject();

		LOGFONT lg;
		pFont->GetLogFont(&lg);
		LONG stdWeight = lg.lfWeight;

		lg.lfWeight = FW_BOLD;
		m_BoldFont.CreateFontIndirect(&lg);

		lg.lfStrikeOut = TRUE;
		m_BoldStrikeFont.CreateFontIndirect(&lg);
		
		lg.lfWeight = stdWeight;
		m_StrikeFont.CreateFontIndirect(&lg);
	}

	CListCtrl::SetFont(pFont, bRedraw);
}




static BOOL IsMarkedMail  (const CMail& Mail) { return  Mail.IsMarkedForDelete(); }
static BOOL IsUnmarkedMail(const CMail& Mail) { return !Mail.IsMarkedForDelete(); }

void CMailList::OnMailMarkedChange()
{
    if(!ShowMarkedMails())
	   RemoveMailsFromList(IsMarkedMail);

    if(!ShowUnmarkedMails())
	   RemoveMailsFromList(IsUnmarkedMail);

    if (
         (ShowUnmarkedMails() && (CountMailsInAccounts(IsUnmarkedMail) != CountMailsInList(IsUnmarkedMail)))
          ||
         (ShowMarkedMails()   && (CountMailsInAccounts(IsMarkedMail)   != CountMailsInList(IsMarkedMail)))
       )
        Refresh(TRUE);
}

int CMailList::RemoveMailsFromList(fnMailPredicate predicate)
{
    int count = 0;
    for(int n = GetItemCount()-1; n >= 0; --n)
	{
		const CMail& Mail = *(CMail*)GetItemData(n);
		if(predicate(Mail)) 
        {
			const UINT flags = LVIS_SELECTED | LVIS_FOCUSED;
			bool SelNext = (flags == GetItemState(n, flags));

			DeleteItem(n);
            ++count;

			if(SelNext && n < GetItemCount()) 
				SelectItem(n);
		}
	}
    return count;
}

int CMailList::CountMailsInList(fnMailPredicate predicate) const
{
    int counter = 0;
    for(int n = 0; n < GetItemCount(); n++)
	{
		const CMail& Mail = *((CMail*)GetItemData(n));
        if(predicate(Mail))
	        ++counter;
	}
    return counter;
}

int CMailList::CountMailsInAccounts(fnMailPredicate predicate) const
{
    return m_pAccounts->CountMails(predicate);
}

void CMailList::OnRulesAddListitem() 
{
    CPopManDoc* pDoc = GetDocument();
    if(!pDoc) return;

    pDoc->AddItemToRulesList(_T(""), GetSelectedMail());	
}

void CMailList::OnUpdateRulesAddListitem(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(GetDocument() && GetDocument()->IsAddItemToRulesListEnabled());
}

void CMailList::OnWindowPosChanging(WINDOWPOS FAR* lpwndpos) 
{
    if(m_bDisableRedraw)
        lpwndpos->flags |= SWP_NOREDRAW;

    CListCtrl::OnWindowPosChanging(lpwndpos);
}


void CMailList::OnMouseMove(UINT nFlags, CPoint point) 
{
	CListCtrl::OnMouseMove(nFlags, point);
    CMail* pMail = NULL;

    if(IsOnNonEmptyRulesSubItem(point, pMail)) 
    {
        if(m_pLastTooltipMail == NULL || m_RulesTip.m_hWnd == NULL)
        {    
            if(!m_bTimerRulesTip)
               m_bTimerRulesTip = (0 != SetTimer(TimerID_RulesTip, 500, NULL));
        }
        else
        {
            if(pMail != m_pLastTooltipMail)
            {
                ShowRulesTip(pMail);
                m_pLastTooltipMail = pMail;
            }
        }
    }
    else 
    {
        m_pLastTooltipMail = NULL;
        if(m_bTimerRulesTip) KillTimer(TimerID_RulesTip);
        m_bTimerRulesTip = false;
        m_RulesTip.DestroyWindow();
    }
}

bool CMailList::IsOnNonEmptyRulesSubItem(CPoint point, CMail*& pMail)
{
    LVHITTESTINFO info;
    info.pt = point;
 	
	SubItemHitTest(&info);

    if(info.iItem > -1 && COLUMNS::colRules == GetIDFromIndex(info.iSubItem))
    {
        pMail = (CMail*)GetItemData(info.iItem);
        
        if(pMail && !pMail->GetRules().IsEmpty())
            return true;
    }
    return false;
}

int CountLineBreaks(const CString& str)
{
    int res = 0;
    for(int i=0; i < str.GetLength(); ++i) {
        if(str[i] == _T('\n'))
            ++res;
    }
    return res;
}

void CMailList::ShowRulesTip(const CMail* pMail) 
{
    if(pMail->m_RuleInfos.GetCount() == 0) return;

    CString buffer;
    buffer.GetBuffer(255);

    int* boldLineIdxArray = new int[pMail->m_RuleInfos.GetCount()];

    int ruleCount = 0;

    POSITION pos = pMail->m_RuleInfos.GetHeadPosition();
    while(pos)
    {
        const CMail::RuleInfo& rule = pMail->m_RuleInfos.GetNext(pos);
        boldLineIdxArray[ruleCount++] = CountLineBreaks(buffer);
        buffer += rule.Name;
        buffer += _T("\r\n"); 

        buffer += rule.Info;
        if(pos) buffer += _T("\r\n\r\n");
    }
    
    m_RulesTip.Show(this, _T(""), buffer, true, false, boldLineIdxArray, pMail->m_RuleInfos.GetCount());
    m_RulesTip.SetMouseMoveReceiver(this, OnTipMouseMove);

    delete boldLineIdxArray;
}


void CMailList::OnTipMouseMove(CObject *pThis)
{
    if(pThis) {

        CMailList* pList = (CMailList*)pThis;
          
        POINT point;
        ::GetCursorPos(&point);
        pList->ScreenToClient(&point);

        pList->OnMouseMove(0, point);
    }
}

void CMailList::OnMarkAllRead()
{
    for(int n = 0; n < GetItemCount(); n++)
	{
		CMail* pMail = (CMail*)GetItemData(n);
        if(!pMail->IsRead()) {
            pMail->SetRead();
            AfxGetMainWnd()->SendMessageToDescendants(WM_MAIL_READ, (WPARAM)pMail, 0);
        }
	}
    Invalidate();
	UpdateWindow();
}
    
void CMailList::OnMarkAllUnread()
{
    for(int n = 0; n < GetItemCount(); n++)
	{
		CMail* pMail = (CMail*)GetItemData(n);
        if(pMail->IsRead()) {
            pMail->SetRead(FALSE);
            AfxGetMainWnd()->SendMessageToDescendants(WM_MAIL_READ, (WPARAM)pMail, 0);
        }
	}
    Invalidate();
	UpdateWindow();
}

void CMailList::OnUpdateMarkAllRead(CCmdUI* pCmdUI)
{
    BOOL bEnabled = FALSE;
    for(int n = 0; n < GetItemCount() && !bEnabled; n++)
	{
		CMail* pMail = (CMail*)GetItemData(n);
        bEnabled = !pMail->IsRead();
	}
    pCmdUI->Enable(bEnabled);
}
 
void CMailList::OnUpdateMarkAllUnread(CCmdUI* pCmdUI)
{
    BOOL bEnabled = FALSE;
    for(int n = 0; n < GetItemCount() && !bEnabled; n++)
	{
		CMail* pMail = (CMail*)GetItemData(n);
        bEnabled = pMail->IsRead();
	}
    pCmdUI->Enable(bEnabled);
}

void CMailList::OnProtectMessage() 
{
	POSITION pos = GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

	while(pos)
	{
		const int nItem = GetNextSelectedItem(pos);
		CMail * const pMail = (CMail*)GetItemData(nItem);
		pMail->SetAlwaysProtected(!pMail->GetAlwaysProtected());

		LV_ITEM lvItem = {0};
		lvItem.iItem = nItem;
		lvItem.mask = LVIF_TEXT;
		lvItem.pszText = (LPTSTR)(LPCTSTR)pMail->GetSubject();
		if(IndexFromID(COLUMNS::colSubject, lvItem.iSubItem))
			SetItem(&lvItem);

		if(pMail == m_prevMail) 
		{
			m_prevMail = NULL;
			SetMailInMessageView(pMail);
		}
	}
}
