// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Mail.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "Mail.h"
#include "StrFunctions.h"
#include "Account.h"
#include <locale.h>
#include "resource.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define SIZE(Array) ( sizeof(Array) / sizeof(Array[0]) )

int CMail::nMarkedMessages = 0;


BOOL CMail::Init(const CString& stSource, int nSize, const CString& stUIDL, BOOL bComplete)
{
	if(nSize < 1 || stUIDL.IsEmpty() || stSource.IsEmpty())
		return FALSE;

	m_bRead = FALSE;
	m_bMarkedForDelete = FALSE;
	m_bComplete = bComplete;
	m_nSize = nSize;
	m_stUIDL = stUIDL;
	m_stSource = ConvertLineBreaks(stSource);
	m_bAlwaysProtected = FALSE;


	float fSizeKB = (float)m_nSize/(float)1024.0;
	if (fSizeKB < 1.0) fSizeKB = 1.0;
	m_stSizeFormated.Format(_T("%.0f KB"), fSizeKB);
	
	int i = m_stSource.Find(_T("\r\n\r\n"));
	if(i > 0)
		m_stHeader = m_stSource.Left(i);
	else 
		m_stHeader = m_stSource;
	
    
	return ParseMailSource();
}


CMail::~CMail() 
{ 
	TRACE2("~CMail: %X, %s\n", (int)this, (LPCTSTR)m_stSubject); 
	
	ClearAttachments();

	if(m_bMarkedForDelete)
		--nMarkedMessages;
	if(nMarkedMessages < 0) nMarkedMessages = 0;

    for(size_t i = 0; i < m_HtmlLinks.size(); ++i)
        delete [] ((TCHAR*)m_HtmlLinks[i].szLink);
}


void CMail::ClearAttachments()
{
	POSITION pos = m_Attachments.GetHeadPosition();
	while(pos != NULL)
		delete m_Attachments.GetNext(pos);
	
	m_Attachments.RemoveAll();
}

void CMail::SetRead(BOOL bRead)
{ 
	m_bRead = bRead; 
	GetAccount()->UpdateMailCache(this);
}

void CMail::SetMarkedForDelete(BOOL bMark, BOOL bSurpressEvents) 
{
	if(bMark && !m_bMarkedForDelete)
		++nMarkedMessages;
	if(!bMark && m_bMarkedForDelete)
		--nMarkedMessages;

	if(nMarkedMessages < 0) nMarkedMessages = 0;


	m_bMarkedForDelete = bMark;
	
    if(!bSurpressEvents)
	    AfxGetMainWnd()->SendMessage(WM_MAIL_DEL_MARKED, 0, 0);
}

void CMail::SetSubject(const CString& stNewSubject)
{
	m_stSubject = stNewSubject;
}

void CMail::SetAlwaysProtected(BOOL bProtection) 
{ 
	if(m_bAlwaysProtected != bProtection) 
	{
		m_bAlwaysProtected = bProtection;

		CString protectedPrefix = i18n("[PROTECTED]");
		protectedPrefix += _T(" ");

		if(bProtection) 
		{
			SetSubject(protectedPrefix + GetSubject());
		}
		else 
		{
			CString strSubject = GetSubject();
			if(strSubject.Find(protectedPrefix) == 0) 
			{
				strSubject.Delete(0, protectedPrefix.GetLength());
				SetSubject(strSubject);
			}
		}		
	}	
}

BOOL CMail::ParseMailSource(BOOL bResetContent)
{
	if(m_stSource.IsEmpty())
		return FALSE;

	//CFile File(_T("D:\\PopMan\\Attachment_ct.txt"), CFile::modeRead);
	//unsigned int nFileLen = static_cast<unsigned int>(File.GetLength());
	//char *pData = new char[nFileLen + 1];
	//File.Read(pData, nFileLen);
	//pData[nFileLen] = 0;
	//File.Close();
	//m_stSource = CString(pData);
		
	CString temp;  

	// alle Zeichen linksseits des Semikolon verwerfen:
	temp = GetProperty(_T("Received"));
	if(temp.Find(_T(';')) != -1)
		Parse(temp, _T(";"));

	m_tmReceived = FormatDate(temp);


	m_tmDate= FormatDate(GetProperty(_T("Date")));


	if(m_tmDate.GetYear() < 1990)
		m_tmDate = m_tmReceived;

	if(m_tmReceived.GetYear() < 1990)
		m_tmReceived = m_tmDate;

	
	temp = Parse(GetProperty(_T("X-Priority")), _T(" "));
	TCHAR p = _T('3');
	if(temp.GetLength() == 1)
		p = temp[0];
	switch(p) {
	case _T('1'): case _T('2'): m_Priority = MailPriority::High; break;
	case _T('4'): case _T('5'): m_Priority = MailPriority::Low;  break;
	default: m_Priority = MailPriority::Normal;
	}

	if(m_Priority == MailPriority::Normal)
	{
		temp = GetProperty(_T("Importance"));
		if(FindNoCase(temp, _T("High")) > -1)
			m_Priority = MailPriority::High;
		else if(FindNoCase(temp, _T("Low")) > -1)
			m_Priority = MailPriority::Low;
	}

	m_stUserAgent = GetProperty(_T("User-Agent"));
	if(m_stUserAgent.IsEmpty())
		m_stUserAgent = GetProperty(_T("X-Mailer"));
	m_stUserAgent.TrimLeft();

	m_stFromAddress = ExtractMail(GetProperty(_T("From")));


	m_stReplyTo = ExtractMail(GetProperty(_T("Reply-To")));
	if(m_stReplyTo.IsEmpty())
		m_stReplyTo = m_stFromAddress;


	m_stFrom = ExtractNames(GetProperty(_T("From"), true));

	m_stFromComplete = m_stFrom;
	if (m_stFromComplete != m_stFromAddress) {
		m_stFromComplete += _T(" <");
		m_stFromComplete += m_stFromAddress;
		m_stFromComplete += _T(">");
	}

	m_stTo = ExtractNames(GetProperty(_T("To"), true));
	m_stSubject = GetProperty(_T("Subject"), true);
	m_stSubject.Replace((TCHAR)(unsigned)0x96, _T('-'));
	m_stSubject.Trim();
	ExtractMessageText();
	



	ExtractAttachments(bResetContent);	
	
	// m_stSignature = Crypt(m_stSubject +  m_tmReceived.Format(_T("%d%m%y%H%M%S")) +  m_stTo + m_stFrom); 
	m_stSignature = Crypt(m_stSubject +  m_tmReceived.Format(_T("%d%m%y%H%M%S")) + GetProperty(_T("Message-ID"))); 

	//auto tt = _T("=?utf-8?B?Ik1lZGl1bSAoNTAgZmVlZHMpIiDnmoTkuqTku5jkv6Hmga/v?=\r\n  =?utf-8?B?vIhNeUNvbWllcmNlIFNoYXJlLWl0IOiuouWNleWPtyA3NjgzMzg4OTPv?=\r\n =?utf-8?B?vIk=?=");
	//m_stSubject = ISODecode(tt);

	return TRUE;
}

CString CMail::ExtractNames(CString Src)
{
	CString Name, Result, temp(GetNextPart(Src));

	while(!temp.IsEmpty())
	{
		if(!Result.IsEmpty())
			Result += _T("; ");

		Name = GetQuotation(temp);
		if(Name.IsEmpty())
		{
			temp.TrimLeft(); temp.TrimRight();
			if(temp.Left(1) == _T('<') && temp.Right(1) == _T('>'))
				Name = temp.Mid(1, temp.GetLength() - 2);
			else
				Name = RemoveMail(temp);
		}
		Name.TrimLeft(); Name.TrimRight();

		if(Name.IsEmpty() || Name == _T("\"\""))
			Name = ExtractMail(temp);

		Result += Name;
		temp = GetNextPart(Src);
	}

	// Backslash entfernen bzw. 2 aufeineanderfolgende Backslash in einen umwandeln:
    for(int n = 0; n < Result.GetLength(); n++) {
		if(Result[n] == '\\')
			Result.Delete(n);
    }

    // remove wrapping single quote if found:
    if(Result.GetLength() >= 2) {
        if(Result[0] == _T('\'')) {
            if(Result.Find(_T('\''), 1) == Result.GetLength()-1) {
                Result.Delete(Result.GetLength()-1, 1);
                Result.Delete(0, 1);
                //Result = Result.Mid(1, Result.GetLength()-2);
            }
                
        }
    }

	return Result;
}

CString CMail::GetNextPart(CString& Src)
{
	BOOL bInQuote = FALSE;
	TCHAR cPrev = _T('\0');
	for(int n = 0; n < Src.GetLength(); n++)
	{
		if(Src[n] == _T('\"'))
			bInQuote = !bInQuote;

		if(Src[n] == _T(',') || Src[n] == _T(';'))
		{
			if(!bInQuote && cPrev != _T('\\'))
			{
				CString temp(Src.Left(n));
				//Src = Src.Mid(n + 1);
                Src.Delete(0, n+1);
				return temp;
			}
		}
		cPrev = Src[n];
	}
	
	CString temp(Src);
	Src.Empty();
	return temp;
}

CString CMail::GetQuotation(const CString& Src)
{
	int qStart = -1, qEnd = -1;
	TCHAR cPrev = _T('\0');

	for(int n = 0; n < Src.GetLength(); n++)
	{
		if(Src[n] == _T('"') && cPrev != _T('\\'))
		{
			if(qStart == -1)
				qStart = n+1;
			else {
				qEnd = n;
				break;
			}
		}
		cPrev = Src[n];
	}
	if(qStart == -1)
		return "";

	if(qEnd == -1)
		return Src.Mid(qStart);

	return Src.Mid(qStart, qEnd - qStart);
}

CString CMail::RemoveMail(const CString& Src)
{
	int qStart = -1, qEnd = -1;
	TCHAR cPrev = _T('\0');

	for(int n = 0; n < Src.GetLength(); n++)
	{
		if(Src[n] == _T('<') && cPrev != _T('\\'))
		{
			if(qStart == -1)
				qStart = n;
		}
		else if(Src[n] == _T('>') && cPrev != _T('\\'))
		{
			qEnd = n;
		}
		
		cPrev = Src[n];
	}

	if(qStart == -1 || qEnd == -1 || qEnd < qStart)
		return Src;
	
	CString temp(Src);
	temp.Delete(qStart, qEnd - qStart + 1);
	return temp;
}


COleDateTime CMail::FormatDate(CString Src)
{
	CString stMonth, stTime;
	UINT nDay, nMonth, nYear, nHours, nMinutes, nSeconds;
	int iMailOffset = 0, iLocalOffset = 0;
	
	if(Src.IsEmpty())
		return COleDateTime();

	Src.Replace(_T("\r\n"), _T(" "));

	// Wochentag verwerfen:
	if(Src.Find(_T(", ")) != -1)
		Parse(Src, _T(", "));

	Src.TrimLeft();

	nDay = _ttoi(Parse(Src, _T(" ")));
	
	stMonth = Parse(Src, _T(" "));
	
	nYear = _ttoi(Parse(Src, _T(" ")));
	
	stTime = Parse(Src, _T(" "));
	nHours = _ttoi(Parse(stTime, _T(":")));
	nMinutes = _ttoi(Parse(stTime, _T(":")));
	nSeconds = _ttoi(Parse(stTime, _T(":")));

	CString stMailOffset = Parse(Src, _T(" "));
	BOOL bNoMailOffset = stMailOffset.IsEmpty();
	if(!bNoMailOffset)
		iMailOffset = _ttoi(stMailOffset) / 100;
	
	stMonth.MakeUpper();
	if(stMonth == "JAN")
		nMonth = 1;
	else if(stMonth == _T("FEB"))
		nMonth = 2; 
	else if(stMonth == _T("MAR"))
		nMonth = 3; 
	else if(stMonth == _T("APR"))
		nMonth = 4;
	else if(stMonth == _T("MAY"))
		nMonth = 5; 
	else if(stMonth == _T("JUN"))
		nMonth = 6; 
	else if(stMonth == _T("JUL"))
		nMonth = 7;
	else if(stMonth == _T("AUG"))
		nMonth = 8; 
	else if(stMonth == _T("SEP"))
		nMonth = 9; 
	else if(stMonth == _T("OCT"))
		nMonth = 10;
	else if(stMonth == _T("NOV"))
		nMonth = 11; 
	else if(stMonth == _T("DEC"))
		nMonth = 12; 
	else
		nMonth = 1;
	
	TIME_ZONE_INFORMATION TZ_Info;
	BOOL bIsDayLight = (TIME_ZONE_ID_DAYLIGHT == ::GetTimeZoneInformation(&TZ_Info));
	iLocalOffset = -TZ_Info.Bias/60 + (bIsDayLight ? 1 : 0);

	COleDateTime      Time(nYear, nMonth, nDay, nHours, nMinutes, nSeconds);
	if(!bNoMailOffset)
	{
		COleDateTimeSpan  Offset(0, iLocalOffset - iMailOffset, 0, 0);
		Time += Offset;
	}
						
	return Time;		
}

// Gibt den String zurck, der von Search und dem ersten Zeilenumbruch,
// auf dem kein Tab bzw. Leerzeichen folgt, eingeschlossen wird.
// Wird nach Search kein Zeilenumbruch gefunden, dann wird der gesamte
// String nach Search zurckgegeben:
CString CMail::GetItem(const CString& Src, LPCTSTR Search, int nStart, int* pIdxFound)
{
	if (pIdxFound) *pIdxFound = -1;
	int LenS = _tcslen(Search);
	int iStart = FindNoCase(Src, Search, nStart);
	
	if (iStart == -1 && LenS > 2 && Search[0] == '\r' && Search[1] == '\n') {
		iStart = FindNoCase(Src, &Search[2], nStart);
		LenS -= 2;
		if (iStart > 0) {
			return _T("");
		}
	}

	if(iStart == -1)
		return _T("");
	else
		if(pIdxFound) *pIdxFound = iStart + LenS;

	int iEnd = iStart + LenS - 2;

	do {
		iEnd = Src.Find(_T("\r\n"), iEnd + 2);
		if(iEnd == -1)
		{
			iEnd = Src.GetLength();
			break;
		}
			
//	} while((Src.Mid(iEnd+2, 1) == _T('\t') || Src.Mid(iEnd+2, 1) == _T(' ')));
    } while(iEnd+2 < Src.GetLength() && (Src[iEnd+2] == _T('\t') || Src[iEnd+2] == _T(' ')) );
	

	return Src.Mid(iStart + LenS, iEnd - iStart - LenS); 
}

// Dekodiert Text, der mit Base64 bzw. quoted-printable kodiert ist:
CString CMail::ISODecode(const CString& Src)
{
	CString stRes;
    CString outerText;
	int Idx = 0;
    bool firstEncodedItem = true;

	bool isNormal = true;
	CString rawBytes;
	CString lastCharset;
	
	CString front;
	CString trail;

	while(1)
	{
		int oldIdx = Idx;
		Idx = Src.Find(_T("=?"), oldIdx);
		if(Idx == -1)
		{
			trail = Src.Mid(oldIdx);			
			stRes += trail;			
			break;
		}

        outerText = Src.Mid(oldIdx, Idx - oldIdx);
		
		if (firstEncodedItem) {
			front = outerText;
		}

        if(!firstEncodedItem) { // ignore whitespaces between encoded text fragments
            CString tmp = outerText;
            tmp.TrimLeft();
            if(tmp.IsEmpty())
                outerText = _T("");
        }

		if (!outerText.IsEmpty() && !firstEncodedItem) {
			isNormal = false;
		}
		stRes += outerText;

		int IdxQ = FindNoCase(Src, _T("?Q?"), Idx);
		int IdxB = FindNoCase(Src, _T("?B?"), Idx);

		if(IdxQ == -1 && IdxB == -1)
		{
			isNormal = false;
			stRes += Src.Mid(Idx);
			break;
		}

		int  Idx2 = 0;
		if(IdxQ >= 0 && IdxB >= 0)
			Idx2 = min(IdxQ, IdxB);
		else
			Idx2 = (IdxQ == -1 ? IdxB : IdxQ);
		
		BOOL bBase64 = (Idx2 == IdxB);
		CString stCharset = Src.Mid(Idx+2, Idx2 - Idx - 2);

		if (!lastCharset.IsEmpty() && lastCharset != stCharset) {
			isNormal = false;
		}
		lastCharset = stCharset;

		int IdxEnd = Src.Find(_T("?="), Idx2 + 3);
		if(IdxEnd == -1)
		{
			isNormal = false;
			stRes += Src.Mid(Idx);
			break;
		}

		CString stContent;
		if(bBase64)
			stContent = Base64Decode( Src.Mid(Idx2 + 3, IdxEnd - Idx2 - 3) );
		else
			stContent = QPDecode( Src.Mid(Idx2 + 3, IdxEnd - Idx2 - 3), TRUE );

		rawBytes += stContent;

		ConvertEncodedText(stContent, stCharset);		

		stRes += stContent;

		Idx = IdxEnd + 2;

        firstEncodedItem = false;
	}

	stRes.Replace(_T("\r\n"), _T(""));	// Zeilenumbrche entfernen
	stRes.Replace(_T('\t'), _T(' '));	// Tabs durch Leerzeichen ersetzen

	if (stRes.Find(_T("=?")) >= 0 && stRes != Src) {
		return ISODecode(stRes);
	}

	if (isNormal) {
		stRes = front;
		ConvertEncodedText(rawBytes, lastCharset);
		stRes += rawBytes;
		stRes += trail;		
		stRes.Replace(_T("\r\n"), _T(""));	// Zeilenumbrche entfernen
		stRes.Replace(_T('\t'), _T(' '));	// Tabs durch Leerzeichen ersetzen
	}
	
	return stRes;
}



void CMail::ExtractMessageText()
{
	CString stMsg;
	const CString& Src = m_stSource;
	BOOL bHTML = FALSE;

	m_stPlainText.Empty();
	m_stHtml.Empty();

	int idxBodyStart = Src.Find(_T("\r\n\r\n"));
	if(idxBodyStart == -1) return;
	idxBodyStart += 4;

	int Idx = 0;
	CString stHead = ParseConst(Src, _T("\r\n\r\n"), Idx);

	int iStart = 0;
	CString stContent = GetItem(stHead, _T("\r\nContent-Type:"), 0, &iStart);
	stContent.TrimLeft();

	if(stContent.IsEmpty())
	{
		// Die Mail enthlt nur einfachen, uncodierten Text
		m_stPlainText = Src.Mid(idxBodyStart);

		return;
	}

	if(FindNoCase(stContent, _T("text")) == 0 || FindNoCase(stContent, _T("HTML")) == 0)
	{
		bHTML = (FindNoCase(stContent, _T("HTML")) != -1);

		// Die Mail enthlt nur einfachen Text bzw. HTML (mglicherweise codiert)
	    //TRACE("Encoding: %d \n", (int)GetEncodingType(Src, 0));
		stMsg = DecodeString(Src.Mid(idxBodyStart), GetEncodingType(Src, 0));
		

		CString stStr = stContent;
		ParseNoCase(stStr, _T("charset"));
		stStr = ParseNoCase(stStr, _T("\n"));
		ConvertEncodedText(stMsg, stStr);

		if(bHTML)
			m_stHtml = stMsg;
		else
			m_stPlainText = stMsg;

		return;
	}

	BOOL bFoundMessage = FALSE;

	if (FindNoCase(stContent, _T("boundary")) == -1)
	{	// Decodierung fehlgeschlagen; den rohen Quelltext verwenden:
		if (!bFoundMessage)
			m_stPlainText = Src.Mid(idxBodyStart);
		return;
	}

	ParseNoCase(stContent, _T("boundary"));
	CString stBoundary = ExtractBoundary(stContent);

	if (stBoundary.IsEmpty())
	{	// Decodierung fehlgeschlagen; den rohen Quelltext verwenden:
		if (!bFoundMessage)
			m_stPlainText = Src.Mid(idxBodyStart);
		return;
	}

	while (iStart >= 0 && iStart < Src.GetLength())
	{
		iStart = Src.Find(_T("--") + stBoundary, iStart);
		if(iStart == -1)
		{	// Decodierung fehlgeschlagen; den rohen Quelltext verwenden:
			if(!bFoundMessage)
				m_stPlainText = Src.Mid(idxBodyStart);
			return;
		}

		iStart += stBoundary.GetLength() + 2;

evalcontent:

		int iSecStart = iStart;
		stContent = GetItem(Src, _T("\r\nContent-Type:"), iSecStart, &iStart);
		stContent.TrimLeft();

		if(FindNoCase(stContent, _T("text")) == 0)
		{
			bHTML = (FindNoCase(stContent, _T("HTML")) != -1);
		
			int nTextStart = Src.Find(_T("\r\n\r\n"), iStart);
			if(nTextStart == -1)
				return;
			nTextStart += 4;
			int nEnd = Src.Find(_T("--") + stBoundary, nTextStart);
			BOOL bCorrectEnd = (nEnd != -1);
			if(!bCorrectEnd)
				nEnd = Src.GetLength();

			int nTextLen = nEnd - nTextStart - (bCorrectEnd ? 2 : 0);
			
			stMsg = DecodeString(Src.Mid(nTextStart, nTextLen), GetEncodingType(Src, iSecStart));


			CString stStr = stContent;
			ParseNoCase(stStr, _T("charset"));
			stStr = ParseNoCase(stStr, _T("\n"));
			ConvertEncodedText(stMsg, stStr);


			if(bHTML)
				m_stHtml = stMsg;
			else
			{
				if(m_stPlainText.IsEmpty())
					m_stPlainText = stMsg;
				else
				{
					m_stPlainText += _T("\r\n\r\n------------------------------------------------------\r\n\r\n");
					m_stPlainText += stMsg;
				}
			}
			
			bFoundMessage = TRUE;

			if(!bHTML && bCorrectEnd)
			{
				iStart = nEnd;
				goto evalcontent;	// HTML-Abschnitt suchen
			}

			return;
		}
		else if (FindNoCase(stContent, _T("multipart")) == 0)
		{

			if (FindNoCase(stContent, _T("boundary")) == -1)
			{	// Decodierung fehlgeschlagen; den rohen Quelltext verwenden:
				if (!bFoundMessage)
					m_stPlainText = Src.Mid(idxBodyStart);
				return;
			}

			ParseNoCase(stContent, _T("boundary"));
			stBoundary = ExtractBoundary(stContent);

			if (stBoundary.IsEmpty())
			{	// Decodierung fehlgeschlagen; den rohen Quelltext verwenden:
				if (!bFoundMessage)
					m_stPlainText = Src.Mid(idxBodyStart);
				return;
			}
		}
	}
}



CMail::ENCODING_TYPE CMail::GetEncodingType(const CString& stSrc, int startPos)
{
	int pos = FindNoCase(stSrc, _T("\r\n\r\n"), startPos);
	if(pos == -1) pos = stSrc.GetLength();
	CString stContent = stSrc.Mid(startPos, pos-startPos);
	
	CString stEncoding = GetItem(stContent, _T("\r\nContent-Transfer-Encoding:"));
	stEncoding.TrimLeft();

    //TRACE("EncodingStr: %s \n", (LPCTSTR)stEncoding);

	if(FindNoCase(stEncoding, _T("quoted-printable")) == 0) 
		return ENCODING_TYPE::enQuotedPrintable;
	else if(FindNoCase(stEncoding, _T("base64")) == 0)
		return ENCODING_TYPE::enBase64;
	else
		return ENCODING_TYPE::enNone;
}


inline CString CMail::DecodeString(const CString& stMsgText, ENCODING_TYPE Encoding)
{
	switch(Encoding)
	{
	case ENCODING_TYPE::enBase64:
		return Base64Decode(stMsgText);
		
	case ENCODING_TYPE::enQuotedPrintable:
		return QPDecode(stMsgText);
	
	default:
		return stMsgText;
	}
}

// Gibt an, ob der bergebene String bereits eine vollstndige Nachricht enthlt
// Plaintext oder Html !!!
//           ----
BOOL CMail::IsMessageTextComplete(const CString& Src)
{
	int iStart = 0;
	CString stContent = GetItem(Src, _T("\r\nContent-Type:"), 0, &iStart);
	stContent.TrimLeft();

	if(stContent.IsEmpty())
		return FALSE;       // ob die Mail bereits vollstndig ist, kann nicht festgestellt werden!
							
	if(FindNoCase(stContent, _T("text")) == 0)
		return FALSE;       // ob die Mail bereits vollstndig ist, kann nicht festgestellt werden!
		
	while(1)
	{
		if(FindNoCase(stContent, _T("boundary")) == -1)
			return FALSE;
		
		Parse(stContent, _T("boundary"));
		CString stBoundary = ExtractBoundary(stContent);
		if(stBoundary.IsEmpty())
			return FALSE;

		iStart = Src.Find(_T("--") + stBoundary, iStart);
		if(iStart == -1)
			return FALSE;

		iStart += stBoundary.GetLength() + 2;

evalcontent: 

		int iSecStart = iStart;
		stContent = GetItem(Src, _T("\r\nContent-Type:"), iSecStart, &iStart);
		stContent.TrimLeft();

		if(FindNoCase(stContent, _T("text")) == 0)
		{
			int nTextStart = Src.Find(_T("\r\n\r\n"), iStart);
			if(nTextStart == -1) 
				return FALSE;

			nTextStart += 4;

			int nEnd = Src.Find(_T("--") + stBoundary, nTextStart);
			
			BOOL bCorrectEnd = (nEnd != -1);
			if(!bCorrectEnd)
				return FALSE;

			if(FindNoCase(stContent, _T("HTML")) != -1)
				return TRUE;

			// prfen, der Textabschnitt gltig ist:
			if(!bCorrectEnd)
				nEnd = Src.GetLength();

			int nTextLen = nEnd - nTextStart - (bCorrectEnd ? 2 : 0);
			CString stMsg = DecodeString(Src.Mid(nTextStart, nTextLen), GetEncodingType(Src, iSecStart));

			if(IsValidTextContent(stMsg))
				return TRUE;

			// nach HTML-Abschnitt weitersuchen...
			iStart = nEnd;
			goto evalcontent;
		}

	}
	return FALSE;
}

CString CMail::ExtractBoundary(CString stStr)
{
	if (stStr.GetLength() == 0)
		return _T("");

	const int idx = stStr.Find(_T(';'));
	if (idx > 0) {
		stStr = stStr.Left(idx);
	}

	CString stBoundary = GetQuotation(stStr);
	if(stBoundary.IsEmpty() && stStr[0] == _T('='))
	{
		int nCRLF = FindNoCase(stStr, _T("\r\n"));
		if(nCRLF == -1)
			nCRLF = stStr.GetLength();
		
		stBoundary = stStr.Mid(1, nCRLF - 1);
	}
	return stBoundary;
}


const CString& CMail::GetSignature() const
{
 	return m_stSignature;
}

CString CMail::ExtractMail(const CString &Src)
{
	LPCTSTR pszData = Src;
	int nOffset = 0;
	CString Mail;

	while(1)
	{
		LPCTSTR pAt = _tcschr(pszData + nOffset, _T('@'));
		if(pAt == NULL)
			break;
		
		nOffset = pAt - pszData + 1;

		LPCTSTR pR = pAt;
		int res = 0;

		while((res = CheckMailChar(*++pR)) == 1);
		if(res == 3)  // invalid char
			continue;

		LPCTSTR pL = pAt - 1;
		while(pL >= pszData && (res = CheckMailChar(*pL)) == 1)
			pL--;
		if(res == 3)  // invalid char
			continue;

		pL++;
	
		if((pL < pAt) && pR - pAt > 2)
		{
			Mail = pL;
			Mail = Mail.Left(pR - pL);
			break;
		}
	}

	return Mail;
}


inline int CMail::CheckMailChar(TCHAR s)
{
	unsigned int c = (unsigned int)s;

	if( c >= _T('a') && c <= _T('z')	||
		c >= _T('A') && c <= _T('Z')	||
		c >= _T('0') && c <= _T('9')	||
		c == _T('-')					||
		c == _T('_')					||
		c == _T('.')
	   )
		return 1; // valid char

	if(c <= _T(' ') || c == _T('<') || c == _T('>'))
		return 2;	// white space

	return 3;		// invalid char

}

bool IsEmpty(const CString& stText) 
{
	for(int n = 0; n < stText.GetLength(); n++)
	{
		if((unsigned)stText[n] > _T(' '))
		{
			return false;
		}
	}
	return true;
}

const CString& CMail::GetExtractedMsg() const
{
	if(!m_stExtractedMsg.IsEmpty())
		return m_stExtractedMsg;
	
	const bool bIsHtmlEmpty = IsEmpty(GetHtml());

	if(!bIsHtmlEmpty) 
	{
		m_stExtractedMsg = GetTextFromHTML(GetHtml(), &m_HtmlLinks);
		if(IsEmpty(m_stExtractedMsg))
			m_stExtractedMsg = GetPlainText();
	}
	else 
	{
		m_stExtractedMsg = GetPlainText();
	}

	/*
	if(bIsEmpty)
		m_stExtractedMsg = GetTextFromHTML(GetHtml(), &m_HtmlLinks);
	else
	{
		m_stExtractedMsg = GetPlainText();

		if(!IsValidTextContent(m_stExtractedMsg))
		{
			if(!GetHtml().IsEmpty())
				m_stExtractedMsg = GetTextFromHTML(GetHtml(), &m_HtmlLinks);
		}
	}
	*/
	
	
	m_stExtractedMsg = ConvertLineBreaks(m_stExtractedMsg);
	
	return m_stExtractedMsg;
}


BOOL CMail::IsValidTextContent(const CString& stText)
{
	if(stText.GetLength() > 400)
		return TRUE;

	if(	  FindNoCase(stText, _T("HTML formatted message")) > -1  ||
		  FindNoCase(stText, _T("does not support MIME encoding")) > -1 ||
		  FindNoCase(stText, _T("MIME-enabled Email Client")) > -1  ||
		  FindNoCase(stText, _T("does not support the display of HTML")) > -1  ||
		  FindNoCase(stText, _T("view this message in a different mail client")) > -1  ||
		  FindNoCase(stText, _T("This is an HTML message")) > -1  ||
		  FindNoCase(stText, _T("Bitte benutzen Sie einen HTML faehigen email-clienten!")) > -1  ||
		  FindNoCase(stText, _T("e-mail-Programm HTML-Nachrichten nicht direkt")) > -1  ||
		  stText.GetLength() == 1  )
	{
		return FALSE;
	}
	return TRUE;
}

CString CMail::GetQuottedMsg() const
{
	CString stMsg(_T("\r\n\r\n----- Original Message -----"));
	stMsg += _T("\r\nFrom: ") + GetProperty(_T("From"), true);
	stMsg += _T("\r\nTo: ") + GetProperty(_T("To"), true);
	setlocale(LC_ALL, "English");
	stMsg += _T("\r\nSent: ") + GetDate().Format(_T("%A, %B %d, %Y %X"));
	setlocale(LC_ALL, "");
	stMsg += _T("\r\nSubject: ") + GetProperty(_T("Subject"), true);
	stMsg += _T("\r\n\r\n");

	CString stBody = GetExtractedMsg();
	stBody.Replace(_T("\r\n"), _T("\r\n> "));
	if(stBody[0] != _T('>'))
		stBody = _T("> ") + stBody;

	stMsg += stBody;
	
	return stMsg;
}



void CMail::ExtractAttachments(BOOL bClearAttachments)
{
	if(bClearAttachments)
		ClearAttachments();

	const CString stItem = _T("\r\nContent-Type: ");
	int nSrcIdx = 0;
	int nAttachmentIdx = 0;

	while(-1 < (nSrcIdx = FindNoCase(m_stSource, stItem, nSrcIdx + 1)))
	{
		nSrcIdx += stItem.GetLength();
		CString stHead = GetPartHead(m_stSource, nSrcIdx);

		BOOL bAttachment = FALSE;
		CString stFileName;
		if(ParseContentDisposition(stHead, bAttachment, stFileName))
		{
			if(bAttachment && !stFileName.IsEmpty())
			{
				AddAttachment(stFileName, nSrcIdx, nAttachmentIdx++);
				continue;
			}
		}

		BOOL bImageOrText = FALSE;
		ParseContentType(stHead, bImageOrText, stFileName);
		if(stFileName.IsEmpty())
			continue;

		if(bImageOrText == FALSE || FindNoCase(stHead, _T("Content-ID:")) < 0)
		{
			AddAttachment(stFileName, nSrcIdx, nAttachmentIdx++);
		}
	}
}


void CMail::AddAttachment(const CString& stFileName, int nSrcIdx, int nAttachmentIdx)
{
	BOOL bComplete = FALSE;
	int IdxEnd = -1;
	CString stBoundary;
	CString stContent;
	ENCODING_TYPE Encoding = ENCODING_TYPE::enNone;
	int idxBoundEnd = 0;

	int Idx = AdvancedSearch(FALSE, m_stSource, _T("\r\n--"), nSrcIdx);
	if(Idx == -1) goto exit;

	idxBoundEnd = m_stSource.Find(_T("\r\n"), Idx+1);
	if(idxBoundEnd == -1) goto exit;
	stBoundary = m_stSource.Mid(Idx, idxBoundEnd - Idx);

	Idx += stBoundary.GetLength();

	while(Idx < m_stSource.GetLength()-1)
	{
		TCHAR C = m_stSource[Idx];
		if(C < _T(' '))
			Idx++;
		else
			break;
	}
	
	Encoding = GetEncodingType(m_stSource, Idx);

	Idx = m_stSource.Find(_T("\r\n\r\n"), Idx);
	if(Idx == -1) goto exit;
	Idx += 4;
	
	IdxEnd = m_stSource.Find(stBoundary, Idx);
	bComplete = (IdxEnd > -1);
	if(bComplete)
	{
		stContent = m_stSource.Mid(Idx, IdxEnd - Idx);
		if(Encoding == ENCODING_TYPE::enBase64) {
			stContent.TrimLeft();
			stContent.TrimRight();
		}
		stContent = DecodeString(stContent, Encoding);
	}

exit:

	if (stContent.IsEmpty() && this->IsComplete()) { // assume mail only contains one attachment (no text at all)
		Encoding = GetEncodingType(m_stSource, 0);
		stContent = m_stPlainText; 
		if (Encoding == ENCODING_TYPE::enBase64) {
			stContent.TrimLeft();
			stContent.TrimRight();
		}
		stContent = DecodeString(stContent, Encoding);
		bComplete = true;
		m_stPlainText.Empty();
	}

	if(nAttachmentIdx >= m_Attachments.GetCount())
		m_Attachments.AddTail(new CAttachment(stFileName, stContent, bComplete));
	else
	{
		 POSITION pos = m_Attachments.FindIndex(nAttachmentIdx);
		 CAttachment* pAtt = m_Attachments.GetNext(pos);
		 pAtt->Complete(stFileName, stContent);
	}

}


CString CMail::GetPartHead(const CString& stSource, int nTypeStart)
{
	int nHeadStart = 0;
	int nHeadEnd = 0;

	while(--nTypeStart > 2)
	{
		if(stSource[nTypeStart] == _T('\n'))
		{
			if(stSource.Mid(nTypeStart-3, 4) == _T("\r\n\r\n"))
			{
				nHeadStart = nTypeStart + 1;
				break;
			}
		}
	}

	nHeadEnd = stSource.Find(_T("\r\n\r\n"), nHeadStart);
	if(nHeadEnd < 0)
		return  stSource.Mid(nHeadStart);

	return stSource.Mid(nHeadStart, nHeadEnd - nHeadStart);
}

CString GetValue(CString& str) {
	str.TrimLeft();
	if (str.GetLength() == 0) return "";
	if (str[0] == '"') {
		const int iEnd = str.Find('"', 1);
		if (iEnd < 0) {
			const CString res = str.Mid(1);
			str.Empty();
			return res;
		}
		else {
			const CString res = str.Mid(1, iEnd - 1);
			str = str.Mid(iEnd + 1);
			return res;
		}
	}
	else {
		const int iEnd = str.FindOneOf(_T("; "));
		if (iEnd < 0) {
			const CString res = str;
			str.Empty();
			return res;
		}
		else {
			const CString res = str.Left(iEnd);
			str = str.Mid(iEnd); // keep space
			return res;
		}		
	}
}

CString ParseRFC2231(CString content, CString name) {

	const CString NameStar = name + _T('*');
	const int LenNameStar = NameStar.GetLength();
	int nIdxStart = FindNoCase(content, NameStar);
	if (nIdxStart < 0) return _T("");

	const int nextChar = nIdxStart + LenNameStar;
	if (nextChar >= content.GetLength()) return _T("");

	const TCHAR c = content[nextChar];

	if (c != '0' && c != '=') return _T("");

	if (c == '=') { // single part
		
		content.Delete(0, nextChar + 1);
		CString line = GetValue(content);

		//content.TrimLeft(_T("\" "));
		CString encoding = Parse(line, _T("'"));
		if (line.IsEmpty()) return _T("");
		Parse(line, _T("'")); // skip lang
		if (line.IsEmpty()) return _T("");
	    CString bytes = QPDecode2(line);
		ConvertEncodedText(bytes, encoding);
		return bytes;
	}
	else { // multipart

		CString encoding;
		CString bytes;
		CString result;

		content.Delete(0, nIdxStart);
		bool first = true;
		int lineNumber = 0;

		while (true) {

			content.TrimLeft(_T("; \t"));

			if (content.IsEmpty()) break;

			CString num;
			num.Format(_T("%d"), lineNumber);

			CString expected = NameStar + num;

			if (FindNoCase(content, expected) != 0) return _T("");

			content.Delete(0, expected.GetLength());

			if (content[0] == _T('*')) {
				content.Delete(0);
				if (content.IsEmpty() || content[0] != _T('=')) return _T("");
				content.Delete(0);
				CString line = GetValue(content);
				if (first) {
					encoding = Parse(line, _T("'"));
					if (line.IsEmpty()) return _T("");
					Parse(line, _T("'")); // skip lang
					if (line.IsEmpty()) return _T("");
					bytes = QPDecode2(line);
					first = false;
				}
				else {
					bytes += QPDecode2(line);
				}
			}
			else if (content[0] == _T('=')) {
				content.Delete(0);
				CString line = GetValue(content);
				if (line.IsEmpty()) return _T("");

				if (!bytes.IsEmpty()) {
					ConvertEncodedText(bytes, encoding);
					result += bytes;
					bytes.Empty();
				}

				result += line;
				first = true;
			}
			else {
				return _T("");
			}

			lineNumber++;
		} // end while

		if (!bytes.IsEmpty()) {
			ConvertEncodedText(bytes, encoding);
			result += bytes;
		}
		
		return result;
	}
}

BOOL CMail::ParseContentDisposition(CString stHead, BOOL &bAttachment, CString &stFileName)
{
	CString stContent = ISODecode(GetPropertyFromSrc(stHead, _T("Content-Disposition")));
	if (stContent.IsEmpty())
		return FALSE;

	bAttachment = (FindNoCase(stContent, _T("attachment")) == 0);

	TCHAR szFileName[] = _T("filename");

	int nIdxStart = FindNoCase(stContent, szFileName);
	if(nIdxStart < 0)
		stFileName.Empty();
	else
	{
		const int fileNameLen = SIZE(szFileName) - 1; // -1 = terminating 0 char
		stContent.Delete(0, nIdxStart + fileNameLen);

		if (stContent.GetLength() > 1 && stContent[0] == '*') {
			stContent = GetPropertyFromSrc(stHead, _T("Content-Disposition"));
			stFileName = ParseRFC2231(stContent, szFileName);
		}
		else {
			Parse(stContent, _T("="));
			stFileName = GetValue(stContent);	
		}
		stFileName.TrimRight();
	}

	return TRUE;
}

void CMail::ParseContentType(const CString& stHead, BOOL& bImageOrText, CString& stName)
{
	stName.Empty();

	CString stCT = GetPropertyFromSrc(stHead, _T("Content-Type"));
	stCT.TrimLeft();
	if(stCT.IsEmpty())
		return;
	
	CString stType = Parse(stCT, _T("/"));
	stType.MakeLower();
	bImageOrText = (stType == _T("image") || stType == _T("text"));
			
	LPCTSTR szName = _T("name");
	int nIdxStart = FindNoCase(stCT, szName);
	if(nIdxStart < 0)
		return;
	else
	{
		const int lenName = 4;
        stCT.Delete(0, nIdxStart + lenName);
		Parse(stCT, _T("="));
		stName = GetValue(stCT);
		stName = ISODecode(stName);
	}
}

CString CMail::GetProperty(LPCTSTR szProperty, bool bISODecoded) const
{
	const CString& property = GetPropertyFromSrc(m_stHeader, szProperty);
	return bISODecoded ? ISODecode(property) : property;
}

CString CMail::GetTextMail() const
{
	CString stFrom		= i18n("From");		stFrom += _T(": ");
	CString stTo		= i18n("To");		stTo += _T(": ");
	CString stSubject	= i18n("Subject");	stSubject += _T(": ");
	CString stDate		= i18n("Date");		stDate += _T(": ");

	CString stMsg;
	stMsg += stFrom + GetProperty(_T("From"), true) + _T("\r\n");
	stMsg += stTo + GetProperty(_T("To"), true) + _T("\r\n");
	stMsg += stSubject + GetProperty(_T("Subject"), true) + _T("\r\n");
	stMsg += stDate + GetDate().Format(/*_T("%#c")*/) + _T("\r\n");
	stMsg += _T("\r\n");

	stMsg += GetExtractedMsg();
	
	return stMsg;
}

CString CMail::GetPropertyFromSrc(const CString& stSrc, LPCTSTR szProperty)
{
	CString stRes;
	CString stFind(szProperty);
	stFind += _T(":");

	if(FindNoCase(stSrc, stFind) == 0)  
	{
		stRes = GetItem(stSrc, stFind); 
		stRes.Replace(_T("\r\n "), _T(" "));
		stRes.Replace(_T("\r\n\t"), _T(" "));
		stRes.TrimLeft();
		return stRes;
	}

	stFind = _T("\r\n") + stFind;
	stRes = GetItem(stSrc, stFind); 
	stRes.Replace(_T("\r\n "), _T(" "));
	stRes.Replace(_T("\r\n\t"), _T(" "));
	stRes.TrimLeft();
	return stRes;
}


BOOL CMail::CompareHeaders(const CString& stHeaderA, const CString& stHeaderB, BOOL bStrict /* = TRUE */)
{
	LPCTSTR Properties[] = 
	{
		_T("Received"),		// "Received" will not be checked if bStrict == FALSE!
		_T("Message-ID"),
		_T("From"),
		_T("To"),
		_T("Subject"),
		_T("Date"),
		_T("Return-Path"),
	};

	if(stHeaderA == stHeaderB)
		return TRUE;

	CString stItemA;
	CString stItemB;

	for(int n = (bStrict ? 0 : 1); n < SIZE(Properties); n++)
	{
		stItemA = GetPropertyFromSrc(stHeaderA, Properties[n]);
		stItemB = GetPropertyFromSrc(stHeaderB, Properties[n]);

		stItemA.TrimRight();
		stItemB.TrimRight();

		if(stItemA != stItemB)
			return FALSE;
	}

	return TRUE;
}


BOOL CMail::HeadersMayEqual(const CString& stHeaderA, const CString& stHeaderB)
{
	LPCTSTR Properties[] = 
	{
		_T("Received"),	
		_T("Message-ID"),
		_T("Subject"),
		_T("Date"),
	};

	CString stItemA;
	CString stItemB;

	for(int n = 0; n < SIZE(Properties); n++)
	{
		stItemA = GetPropertyFromSrc(stHeaderA, Properties[n]);
		stItemB = GetPropertyFromSrc(stHeaderB, Properties[n]);

		stItemA.TrimRight();
		stItemB.TrimRight();

		if(stItemA == stItemB)
			return TRUE;
	}

	return FALSE;
}


BOOL CMail::CompleteSource(const CString &stSource)
{
	m_stSource = ConvertLineBreaks(stSource);
	m_stExtractedMsg.Empty();
	m_bComplete = TRUE;

	return ParseMailSource(FALSE);
}

CString CMail::GetTextFromHTML(const CString &stHTML, std::vector<LinkInfo>* pLinks)
{
	CString stText;
	LPTSTR szText = stText.GetBuffer(stHTML.GetLength()+1);

	LPTSTR pW = szText;

	for(int i = 0; i < stHTML.GetLength()+1; ++i)
		*pW++ = ' ';

	// CString html = _T(" <a href='https://popman.de/test.htm'><img src='thesrc'>Test</a>");

	HTMLToText(stHTML, szText, TRUE, 0, 0, 0, pLinks);
	
	stText.ReleaseBuffer();
	return stText;
}

const CString& CMail::GetEntireFrom() const
{
    if(!m_stEntireFrom.IsEmpty())
        return m_stEntireFrom;

    m_stEntireFrom = GetProperty(_T("From"), true);
    return m_stEntireFrom;
}

const CString& CMail::GetRules() const
{
    if(!m_stRules.IsEmpty() || m_RuleInfos.GetCount() == 0)
        return m_stRules;

    m_stRules.GetBuffer(m_RuleInfos.GetCount()*16);

    POSITION pos = m_RuleInfos.GetHeadPosition();
    while(pos)
    {
        const RuleInfo& rule = m_RuleInfos.GetNext(pos);
        m_stRules += rule.Name;
        if(pos) m_stRules += _T(", ");
    }

    return m_stRules;
}
