/*
 * Copyright (C) 1999-2001, 2005, 2008, 2016 Free Software Foundation, Inc.
 * This file is part of the GNU LIBICONV Library.
 *
 * The GNU LIBICONV Library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The GNU LIBICONV Library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the GNU LIBICONV Library; see the file COPYING.LIB.
 * If not, see <https://www.gnu.org/licenses/>.
 */

/*
 * GBK
 */

/*
 * GBK, as described in Ken Lunde's book, is an extension of GB 2312-1980
 * (shifted by adding 0x8080 to the range 0xA1A1..0xFEFE, as used in EUC-CN).
 * It adds the following ranges:
 *
 * (part of GBK/1)  0xA2A1-0xA2AA  Small Roman numerals
 * GBK/3   0x{81-A0}{40-7E,80-FE}  6080 new characters, all in Unicode
 * GBK/4   0x{AA-FE}{40-7E,80-A0}  8160 new characters, 8080 in Unicode
 * GBK/5   0x{A8-A9}{40-7E,80-A0}  166 new characters, 153 in Unicode
 *
 * Furthermore, all four tables I have looked at
 *   - the CP936 table by Microsoft, found on ftp.unicode.org in 1999,
 *   - the GBK table by Sun, investigated on a Solaris 2.7 machine,
 *   - the GBK tables by CWEX, found in the Big5+ package,
 *   - the GB18030 standard (second printing),
 * agree in the following extensions. (Ken Lunde must have overlooked these
 * differences between GB2312 and GBK. Also, the CWEX tables have additional
 * differences.)
 *
 * 1. Some characters in the GB2312 range are defined differently:
 *
 *     code    GB2312                         GBK
 *    0xA1A4   0x30FB # KATAKANA MIDDLE DOT   0x00B7 # MIDDLE DOT
 *    0xA1AA   0x2015 # HORIZONTAL BAR        0x2014 # EM DASH
 *
 * 2. 19 characters added in the range 0xA6E0-0xA6F5.
 *
 * 3. 4 characters added in the range 0xA8BB-0xA8C0.
 *
 * CP936 as of 1999 was identical to GBK. However, since 1999, Microsoft has
 * added new mappings to CP936...
 */



 /* Our own notion of wide character, as UCS-4, according to ISO-10646-1. */
typedef unsigned int ucs4_t;

/* State used by a conversion. 0 denotes the initial state. */
typedef unsigned int state_t;

/* iconv_t is an opaque type. This is the real iconv_t type. */
typedef struct conv_struct * conv_t;

/*
* Data type for conversion multibyte -> unicode
*/
struct mbtowc_funcs {
	int(*xxx_mbtowc) (conv_t conv, ucs4_t *pwc, unsigned char const *s, size_t n);
	/*
	* int xxx_mbtowc (conv_t conv, ucs4_t *pwc, unsigned char const *s, size_t n)
	* converts the byte sequence starting at s to a wide character. Up to n bytes
	* are available at s. n is >= 1.
	* Result is number of bytes consumed (if a wide character was read),
	* or -1 if invalid, or -2 if n too small,
	* or RET_SHIFT_ILSEQ(number of bytes consumed) if invalid input after a shift
	* sequence was read,
	* or RET_TOOFEW(number of bytes consumed) if only a shift sequence was read.
	*/
	int(*xxx_flushwc) (conv_t conv, ucs4_t *pwc);
	/*
	* int xxx_flushwc (conv_t conv, ucs4_t *pwc)
	* returns to the initial state and stores the pending wide character, if any.
	* Result is 1 (if a wide character was read) or 0 if none was pending.
	*/
};

/* Return code if invalid input after a shift sequence of n bytes was read.
(xxx_mbtowc) */
#define RET_SHIFT_ILSEQ(n)  (-1-2*(n))
/* Return code if invalid. (xxx_mbtowc) */
#define RET_ILSEQ           RET_SHIFT_ILSEQ(0)
/* Return code if only a shift sequence of n bytes was read. (xxx_mbtowc) */
#define RET_TOOFEW(n)       (-2-2*(n))
/* Retrieve the n from the encoded RET_... value. */
#define DECODE_SHIFT_ILSEQ(r)  ((unsigned int)(RET_SHIFT_ILSEQ(0) - (r)) / 2)
#define DECODE_TOOFEW(r)       ((unsigned int)(RET_TOOFEW(0) - (r)) / 2)
/* Maximum value of n that may be used as argument to RET_SHIFT_ILSEQ or RET_TOOFEW. */
#define RET_COUNT_MAX       ((INT_MAX / 2) - 1)

/*
* Data type for conversion unicode -> multibyte
*/
struct wctomb_funcs {
	int(*xxx_wctomb) (conv_t conv, unsigned char *r, ucs4_t wc, size_t n);
	/*
	* int xxx_wctomb (conv_t conv, unsigned char *r, ucs4_t wc, size_t n)
	* converts the wide character wc to the character set xxx, and stores the
	* result beginning at r. Up to n bytes may be written at r. n is >= 1.
	* Result is number of bytes written, or -1 if invalid, or -2 if n too small.
	*/
	int(*xxx_reset) (conv_t conv, unsigned char *r, size_t n);
	/*
	* int xxx_reset (conv_t conv, unsigned char *r, size_t n)
	* stores a shift sequences returning to the initial state beginning at r.
	* Up to n bytes may be written at r. n is >= 0.
	* Result is number of bytes written, or -2 if n too small.
	*/
};

/* Return code if invalid. (xxx_wctomb) */
#define RET_ILUNI      -1
/* Return code if output buffer is too small. (xxx_wctomb, xxx_reset) */
#define RET_TOOSMALL   -2

typedef struct {
	unsigned short indx; /* index into big table */
	unsigned short used; /* bitmask of used entries */
} Summary16;

/*
* Include all the converters.
*/

#include "gbkext1.h"
#include "gbkext2.h"
//#include "gbkext_inv.h"
#include "cp936ext.h"
#include "gb2312.h"


static int
ascii_mbtowc(conv_t conv, ucs4_t *pwc, const unsigned char *s, size_t n)
{
	unsigned char c = *s;
	if (c < 0x80) {
		*pwc = (ucs4_t)c;
		return 1;
	}
	return RET_ILSEQ;
}



static int
gbk_mbtowc (conv_t conv, ucs4_t *pwc, const unsigned char *s, size_t n)
{
  unsigned char c = *s;

  if (c >= 0x81 && c < 0xff) {
    if (n < 2)
      return RET_TOOFEW(0);
    if (c >= 0xa1 && c <= 0xf7) {
      unsigned char c2 = s[1];
      if (c == 0xa1) {
        if (c2 == 0xa4) {
          *pwc = 0x00b7;
          return 2;
        }
        if (c2 == 0xaa) {
          *pwc = 0x2014;
          return 2;
        }
      }
      if (c2 >= 0xa1 && c2 < 0xff) {
        unsigned char buf[2];
        int ret;
        buf[0] = c-0x80; buf[1] = c2-0x80;
        ret = gb2312_mbtowc(conv,pwc,buf,2);
        if (ret != RET_ILSEQ)
          return ret;
        buf[0] = c; buf[1] = c2;
        ret = cp936ext_mbtowc(conv,pwc,buf,2);
        if (ret != RET_ILSEQ)
          return ret;
      }
    }
    if (c >= 0x81 && c <= 0xa0)
      return gbkext1_mbtowc(conv,pwc,s,2);
    if (c >= 0xa8 && c <= 0xfe)
      return gbkext2_mbtowc(conv,pwc,s,2);
    if (c == 0xa2) {
      unsigned char c2 = s[1];
      if (c2 >= 0xa1 && c2 <= 0xaa) {
        *pwc = 0x2170+(c2-0xa1);
        return 2;
      }
    }
  }
  return RET_ILSEQ;
}

static int
ces_gbk_mbtowc(conv_t conv, ucs4_t *pwc, const unsigned char *s, size_t n)
{
	unsigned char c = *s;

	/* Code set 0 (ASCII or GB 1988-89) */
	if (c < 0x80)
		return ascii_mbtowc(conv, pwc, s, n);
	/* Code set 1 (GBK) */
	if (c >= 0x81 && c < 0xff) {
		if (n < 2)
			return RET_TOOFEW(0);
		return gbk_mbtowc(conv, pwc, s, 2);
	}
	return RET_ILSEQ;
}