// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// DlgSettings.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "popman.h"
#include "DlgSettings.h"
#include "FileDlgEx.h"
#include "StrFunctions.h"
#include "RegKey.h"
#include "DialogHelper.h"
#include "Account.h"
#include <mmsystem.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//  i18nComment("Options Dialog")



//////////////////////////////////////////////////////////////////////////////////////
// CPropSheet_Settings
//////////////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNAMIC(CPropSheet_Settings, CPropertySheet)

CPropSheet_Settings::CPropSheet_Settings(LPCTSTR pszCaption, CWnd* pParentWnd, UINT iSelectPage)
	:CPropertySheet(pszCaption, pParentWnd, iSelectPage)
{
	m_szTitle = NULL;
	if(pszCaption)
	{
		if(pszCaption == GetLastOutputStr())
			m_szTitle = GetLastInputStr();
	}
}

CPropSheet_Settings::~CPropSheet_Settings()
{
}


BEGIN_MESSAGE_MAP(CPropSheet_Settings, CPropertySheet)
	//{{AFX_MSG_MAP(CPropSheet_Settings)
	ON_WM_CREATE()
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, OnUpdateGUI)
END_MESSAGE_MAP()


int CPropSheet_Settings::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	EnableStackedTabs(FALSE);

	if (CPropertySheet::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	return 0;
}


BOOL CPropSheet_Settings::OnInitDialog() 
{
	BOOL bResult = CPropertySheet::OnInitDialog();
	
	OnUpdateGUI();
	
	return bResult;
}

void CPropSheet_Settings::OnUpdateGUI()
{
	SetTitle(i18nStr(m_szTitle));
	SetDlgItemText(IDOK, i18n("OK"));
	SetDlgItemText(IDCANCEL, i18n("Cancel"));

	CTabCtrl* pTab = GetTabControl();
	if(pTab == NULL)
		return;

	TCITEM Item;
	Item.mask = TCIF_TEXT;

	for(int n = 0; n < GetPageCount(); n++)
	{
		CPropertyPage* pPage = GetPage(n);
		Item.pszText = (LPTSTR)i18nStr(pPage->m_psp.pszTitle);
		pTab->SetItem(n, &Item);
	}
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_Notification 
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CPropPage_Notification, CPropertyPage)

CPropPage_Notification::KeyMap CPropPage_Notification::m_KeyMap[] =
{
	{_T("ScrollLock"), KEYBOARDLIGHT::ScrollLock},
	{_T("NumLock"),    KEYBOARDLIGHT::NumLock},
	{_T("CapsLock"),   KEYBOARDLIGHT::CapsLock}
};

CPropPage_Notification::CPropPage_Notification(LPCTSTR szTitle) : CPropertyPage(CPropPage_Notification::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_Notification)
	m_stSoundFile = _T("");
	m_bSoundNotification = FALSE;
	m_bNotifyProtectedOnly = FALSE;
	m_bBallonTip = TRUE;
	//}}AFX_DATA_INIT
	m_bShowBallonOption = FALSE;
	m_nNotify = NOTIFYSTATES::AutoChecked;
	m_bShowNewMailTip = TRUE;
	m_bDetailedNewTip = TRUE;
}

CPropPage_Notification::~CPropPage_Notification()
{
}

void CPropPage_Notification::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_Notification)
	DDX_Control(pDX, IDC_COMBO_KEYBOARD, m_cboKeyboard);
	DDX_Control(pDX, IDC_CHECK_NOTIFY_KEYBOARD, m_chkKeyboard);
	DDX_Control(pDX, IDC_CHECK_NOTIFY_BALLONTIP, m_chkBallon);
	DDX_Control(pDX, IDC_EDIT_SOUND_FILE, m_editSoundFile);
	DDX_Control(pDX, IDC_CHECK_NOTIFY_SOUND, m_chkSound);
	DDX_Text(pDX, IDC_EDIT_SOUND_FILE, m_stSoundFile);
	DDX_Check(pDX, IDC_CHECK_NOTIFY_SOUND, m_bSoundNotification);
	DDX_Check(pDX, IDC_PROTECTEDONLYTIPS_CHECK, m_bNotifyProtectedOnly);
	DDX_Check(pDX, IDC_CHECK_NOTIFY_BALLONTIP, m_bBallonTip);
	//}}AFX_DATA_MAP
	DDX_Check(pDX, IDC_CHECK_NOTIFY_SHOW_MAIN, m_bShowMainWnd);
	DDX_Control(pDX, IDC_CHECK_NOTIFY_NEWMAILTIP, m_chkNewMailTip);
	DDX_Check(pDX, IDC_CHECK_NOTIFY_NEWMAILTIP, m_bShowNewMailTip);
	DDX_Check(pDX, IDC_CHECK_NOTIFY_DETAILED_NEWTIP, m_bDetailedNewTip);

	if (pDX->m_bSaveAndValidate && (m_stSoundFile.GetLength() > 0) && (m_chkSound.GetCheck() == 1))
	{
		CFileStatus FileStatus;
		if(0 == CFile::GetStatus(m_stSoundFile, FileStatus))
		{
			AfxMessageBox(StrFormat(i18n("The file \"{1}\" doesn't exist!"), _T("s"), (LPCTSTR)m_stSoundFile));
			pDX->PrepareEditCtrl(IDC_EDIT_SOUND_FILE);
			pDX->Fail();
		}
	}	

	if(pDX->m_bSaveAndValidate)
	{
		if(((CButton*)GetDlgItem(IDC_RADIO_NOTIFY_MINIMIZED))->GetCheck() == 1)
			m_nNotify = NOTIFYSTATES::Minimized;
		else if(((CButton*)GetDlgItem(IDC_RADIO_NOTIFY_ALWAYS))->GetCheck() == 1)
			m_nNotify = NOTIFYSTATES::Always;
		else 
			m_nNotify = NOTIFYSTATES::AutoChecked;

		if(m_chkKeyboard.GetCheck() != 1)
			m_KeyboardLight = KEYBOARDLIGHT::None;
		else
		{
			CString stName;
			m_cboKeyboard.GetWindowText(stName);

			for(int n = 0; n < sizeof(m_KeyMap)/sizeof(m_KeyMap[0]); n++)
			{
				if(m_KeyMap[n].Caption == stName) {
					m_KeyboardLight = m_KeyMap[n].Value;
					break;
				}
			}
		}
	}
	else
	{
		((CButton*)GetDlgItem(IDC_RADIO_NOTIFY_ALWAYS))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_NOTIFY_AUTOCHECK))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_NOTIFY_MINIMIZED))->SetCheck(0);

		UINT nIDRadio = 0;
		if(m_nNotify == NOTIFYSTATES::Minimized)
			nIDRadio = IDC_RADIO_NOTIFY_MINIMIZED;
		else if(m_nNotify == NOTIFYSTATES::Always)
			nIDRadio = IDC_RADIO_NOTIFY_ALWAYS;
		else
			nIDRadio = IDC_RADIO_NOTIFY_AUTOCHECK;

		((CButton*)GetDlgItem(nIDRadio))->SetCheck(1);

		m_cboKeyboard.ResetContent();
		LPCTSTR szSelect = NULL;
		for(int n = 0; n < sizeof(m_KeyMap)/sizeof(m_KeyMap[0]); n++)
		{
			m_cboKeyboard.AddString(m_KeyMap[n].Caption);
			if(m_KeyMap[n].Value == m_KeyboardLight)
				szSelect = m_KeyMap[n].Caption;
		}
		m_cboKeyboard.SetCurSel(0);
		if(szSelect) m_cboKeyboard.SelectString(-1, szSelect);
		m_chkKeyboard.SetCheck(m_KeyboardLight == KEYBOARDLIGHT::None ? 0 : 1);
	}

}

BOOL CPropPage_Notification::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	if(m_bShowBallonOption == FALSE)
		m_chkBallon.ShowWindow(SW_HIDE);

	OnNotifySound_Click();
	OnNewMailTip();
	OnCheckNotifyKeyboard();

	InitGUIText();

	return TRUE;  
}

BEGIN_MESSAGE_MAP(CPropPage_Notification, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_Notification)
	ON_BN_CLICKED(IDC_BUTTON_SELECT_SOUND, OnSelectSoundFile)
	ON_BN_CLICKED(IDC_CHECK_NOTIFY_SOUND, OnNotifySound_Click)
	ON_BN_CLICKED(IDC_BUTTON_PLAY_SOUND, OnPlaySound)
	ON_BN_CLICKED(IDC_CHECK_NOTIFY_NEWMAILTIP, OnNewMailTip)
	ON_BN_CLICKED(IDC_CHECK_NOTIFY_KEYBOARD, OnCheckNotifyKeyboard)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()

void CPropPage_Notification::OnSelectSoundFile() 
{
	CString stFileName;
	m_editSoundFile.GetWindowText(stFileName);
		
	CFileDlgEx Dlg;
	Dlg.AddFilter(i18n("Sound (*.wav)"), _T("*.wav"));
		
	if(!Dlg.GetOpenFileName(stFileName, _T("wav")))
		return;

	m_editSoundFile.SetWindowText(stFileName);
}

void CPropPage_Notification::OnNotifySound_Click() 
{
	BOOL bUpdateEnabled = (1 == m_chkSound.GetCheck());
	((CWnd*)GetDlgItem(IDC_EDIT_SOUND_FILE))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_BUTTON_SELECT_SOUND))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_SOUND_FILE))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_BUTTON_PLAY_SOUND))->EnableWindow(bUpdateEnabled);
}


void CPropPage_Notification::OnPlaySound() 
{
	CString stFileName;
	m_editSoundFile.GetWindowText(stFileName);

	if(stFileName.GetLength() > 0)
		PlaySound(stFileName, 0, SND_FILENAME | SND_ASYNC);
	else
		MessageBeep(MB_ICONEXCLAMATION);
}


void CPropPage_Notification::OnCheckNotifyKeyboard() 
{
	BOOL bUpdateEnabled = (1 == m_chkKeyboard.GetCheck());
	((CWnd*)GetDlgItem(IDC_COMBO_KEYBOARD))->EnableWindow(bUpdateEnabled);
}

void CPropPage_Notification::OnNewMailTip() 
{
	BOOL bUpdateEnabled = (1 == m_chkNewMailTip.GetCheck());
	((CWnd*)GetDlgItem(IDC_CHECK_NOTIFY_DETAILED_NEWTIP))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_CHECK_NOTIFY_BALLONTIP))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_PROTECTEDONLYTIPS_CHECK))->EnableWindow(bUpdateEnabled);
}

void CPropPage_Notification::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_NOTIFY,					i18n("Notify when new mail arrives"));
	SetDlgItemText(IDC_RADIO_NOTIFY_ALWAYS,				i18n("Always"));
	SetDlgItemText(IDC_RADIO_NOTIFY_AUTOCHECK,			i18n("When AutoChecked or minimized"));
	SetDlgItemText(IDC_RADIO_NOTIFY_MINIMIZED,			i18n("Only when minimized"));
	SetDlgItemText(IDC_STATIC_ON_NEW,					i18n("On new mail"));
	SetDlgItemText(IDC_CHECK_NOTIFY_SOUND,				i18n("Acoustic notification"));
	SetDlgItemText(IDC_STATIC_SOUND_FILE,				i18n("Sound:"));
	SetDlgItemText(IDC_BUTTON_PLAY_SOUND,				i18n("&Test"));
	SetDlgItemText(IDC_CHECK_NOTIFY_SHOW_MAIN,			i18n("Show PopMan window"));
	SetDlgItemText(IDC_CHECK_NOTIFY_NEWMAILTIP,			i18n("Show new mail tip"));
	SetDlgItemText(IDC_PROTECTEDONLYTIPS_CHECK,			i18n("Limit tips to [PROTECTED] messages"));
	SetDlgItemText(IDC_CHECK_NOTIFY_DETAILED_NEWTIP,	i18n("Include sender and subject of new mails"));
	SetDlgItemText(IDC_CHECK_NOTIFY_BALLONTIP,			i18n("Use balloon tip"));
	SetDlgItemText(IDC_CHECK_NOTIFY_KEYBOARD,			i18n("Flash keyboard light:"));

	ArrangeControlPair(this, IDC_STATIC_NOTIFY, IDC_STATIC_NOTIFY_LINE);
	ArrangeControlPair(this, IDC_STATIC_ON_NEW,	IDC_STATIC_NEW_LINE);
	ArrangeControlPair(this, IDC_STATIC_SOUND_FILE,	IDC_EDIT_SOUND_FILE);
	ArrangeControlPair(this, IDC_CHECK_NOTIFY_KEYBOARD,	IDC_COMBO_KEYBOARD, TRUE);
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_Layout 
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CPropPage_Layout, CPropertyPage)

CPropPage_Layout::CPropPage_Layout(LPCTSTR szTitle): CPropertyPage(CPropPage_Layout::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_Layout)
	m_bPreview = FALSE;
	m_bMarkUnread = TRUE;
	//}}AFX_DATA_INIT
}

CPropPage_Layout::~CPropPage_Layout()
{
}


void CPropPage_Layout::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_Layout)
	DDX_Control(pDX, IDC_STATIC_ICON, m_staticIcon);
	DDX_Control(pDX, IDC_CHECK_LAYOUT_PREVIEW, m_chkPreview);
	DDX_Check(pDX, IDC_CHECK_LAYOUT_PREVIEW, m_bPreview);
	DDX_Check(pDX, IDC_CHECK_MARK_UNREAD, m_bMarkUnread);
	//}}AFX_DATA_MAP
	if(pDX->m_bSaveAndValidate)
	{
		BOOL bBelow = (((CButton*)GetDlgItem(IDC_RADIO_LAYOUT_BELOW))->GetCheck() == 1);
		m_PreviewLayout = bBelow ? PREVIEW_LAYOUT::layoutBelow : PREVIEW_LAYOUT::layoutBeside;
	}
	else
	{
		BOOL bBelow = (m_PreviewLayout == PREVIEW_LAYOUT::layoutBelow);
		((CButton*)GetDlgItem(IDC_RADIO_LAYOUT_BELOW))->SetCheck(bBelow ? 1 : 0);
		((CButton*)GetDlgItem(IDC_RADIO_LAYOUT_BESIDE))->SetCheck(bBelow ? 0 : 1);
	}
}

BEGIN_MESSAGE_MAP(CPropPage_Layout, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_Layout)
	ON_BN_CLICKED(IDC_CHECK_LAYOUT_PREVIEW, OnShowPreview_Click)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()



void CPropPage_Layout::OnShowPreview_Click() 
{
	BOOL bUpdateEnabled = (1 == m_chkPreview.GetCheck());
	((CWnd*)GetDlgItem(IDC_RADIO_LAYOUT_BELOW))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_RADIO_LAYOUT_BESIDE))->EnableWindow(bUpdateEnabled);
}

BOOL CPropPage_Layout::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	OnShowPreview_Click();

	m_staticIcon.SetIcon(CPopManApp::GetIcon(IDI_PREVIEW));

	InitGUIText();

	return TRUE;
}

void CPropPage_Layout::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_UNREAD,			i18n("Unread Messages"));
	SetDlgItemText(IDC_CHECK_MARK_UNREAD,		i18n("Display unread messages in bold font"));
	SetDlgItemText(IDC_STATIC_PREVIEW,			i18n("Preview Pane"));
	SetDlgItemText(IDC_STATIC_USE_PANE,			i18n("Use the preview pane to quickly view a message without opening a separate window."));
	SetDlgItemText(IDC_CHECK_LAYOUT_PREVIEW,	i18n("Show preview pane"));
	SetDlgItemText(IDC_RADIO_LAYOUT_BELOW,		i18n("Be&low messages"));
	SetDlgItemText(IDC_RADIO_LAYOUT_BESIDE,		i18n("Be&side messages"));

	ArrangeControlPair(this, IDC_STATIC_UNREAD, IDC_STATIC_UNREAD_LINE);
	ArrangeControlPair(this, IDC_STATIC_PREVIEW, IDC_STATIC_PREVIEW_LINE);
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_Logging 
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CPropPage_Logging, CPropertyPage)

CPropPage_Logging::CPropPage_Logging(LPCTSTR szTitle) : CPropertyPage(CPropPage_Logging::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_Logging)
	m_bLoggingEnabled = FALSE;
	m_bLoggingRules = FALSE;
	m_nMaxLogSizeRules = 100;
	m_nMaxLogSize = 100;
	m_bCompactLogging = FALSE;
	//}}AFX_DATA_INIT
}


CPropPage_Logging::~CPropPage_Logging()
{
}

BOOL CPropPage_Logging::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();

	m_spinLogSize.SetBase(10);
	m_spinLogSize.SetRange(10, 25*1024);
	m_spinLogSize.SetBuddy(GetDlgItem(IDC_EDIT_MAX_LOG_SIZE));

    m_spinLogSizeRules.SetBase(10);
	m_spinLogSizeRules.SetRange(10, 25*1024);
	m_spinLogSizeRules.SetBuddy(GetDlgItem(IDC_EDIT_MAX_LOG_SIZE_RULES));

	m_LogDir.SubclassDlgItem(IDC_STATIC_LOG_DIR_LINK, this);
	m_LogDir.m_stLink = CPopManApp::GetAppDataPath();
	SetDlgItemText(IDC_STATIC_LOG_DIR_LINK, CPopManApp::GetAppDataPath());

	m_staticIcon.SetIcon(CPopManApp::GetIcon(IDI_LOGGING));

	OnCheckEnableLogging();
	InitGUIText();
	return TRUE; 
}

void CPropPage_Logging::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_Logging)
	DDX_Control(pDX, IDC_SPIN_MAX_SIZE, m_spinLogSize);
	DDX_Control(pDX, IDC_SPIN_MAX_SIZE_RULES, m_spinLogSizeRules);
	DDX_Control(pDX, IDC_CHECK_ENABLE_RULELOGGING, m_chkLoggingRules);
	DDX_Control(pDX, IDC_STATIC_ICON, m_staticIcon);
	DDX_Control(pDX, IDC_CHECK_ENABLE_LOGGING, m_chkLogging);
	DDX_Check(pDX, IDC_CHECK_ENABLE_LOGGING, m_bLoggingEnabled);
	DDX_Check(pDX, IDC_CHECK_ENABLE_RULELOGGING, m_bLoggingRules);
	DDX_Text(pDX, IDC_EDIT_MAX_LOG_SIZE_RULES, m_nMaxLogSizeRules);
	DDV_MinMaxInt(pDX, m_nMaxLogSizeRules, 10, 25600);
	DDX_Text(pDX, IDC_EDIT_MAX_LOG_SIZE, m_nMaxLogSize);
	DDV_MinMaxInt(pDX, m_nMaxLogSize, 10, 25600);
	DDX_Check(pDX, IDC_CHECK_COMPACT_LOGGING, m_bCompactLogging);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CPropPage_Logging, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_Logging)
	ON_BN_CLICKED(IDC_CHECK_ENABLE_LOGGING, OnCheckEnableLogging)
    ON_BN_CLICKED(IDC_CHECK_ENABLE_RULELOGGING, OnCheckEnableLogging)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()


void CPropPage_Logging::OnCheckEnableLogging() 
{
	BOOL bUpdateEnabled = (1 == m_chkLogging.GetCheck());
	((CWnd*)GetDlgItem(IDC_CHECK_COMPACT_LOGGING))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_MAX_SIZE))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_EDIT_MAX_LOG_SIZE))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_KB))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_SPIN_MAX_SIZE))->EnableWindow(bUpdateEnabled);

    bUpdateEnabled = (1 == m_chkLoggingRules.GetCheck());
	((CWnd*)GetDlgItem(IDC_STATIC_MAX_SIZE_RULES))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_EDIT_MAX_LOG_SIZE_RULES))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_KB_RULES))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_SPIN_MAX_SIZE_RULES))->EnableWindow(bUpdateEnabled);
}

void CPropPage_Logging::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_LOGGING,			i18n("Use logging to record all data communication between PopMan and the email server."));
	SetDlgItemText(IDC_CHECK_ENABLE_LOGGING,	i18n("Enable logging"));
	SetDlgItemText(IDC_CHECK_COMPACT_LOGGING,	i18n("Only log compact mail info instead of entire data traffic"));
	SetDlgItemText(IDC_STATIC_MAX_SIZE,			i18n("Log file size:"));
	SetDlgItemText(IDC_STATIC_KB,				i18n("KB"));

    SetDlgItemText(IDC_CHECK_ENABLE_RULELOGGING,i18n("Enable logging of applied rules"));
    SetDlgItemText(IDC_STATIC_MAX_SIZE_RULES,	i18n("Log file size:"));
    SetDlgItemText(IDC_STATIC_KB_RULES,			i18n("KB"));

	SetDlgItemText(IDC_STATIC_LOG_DIR,			i18n("Location of the log file \"PopManLogFile.log\":"));

	ArrangeUpDown(this, IDC_STATIC_MAX_SIZE, IDC_EDIT_MAX_LOG_SIZE, IDC_SPIN_MAX_SIZE, IDC_STATIC_KB);
    ArrangeUpDown(this, IDC_STATIC_MAX_SIZE_RULES, IDC_EDIT_MAX_LOG_SIZE_RULES, IDC_SPIN_MAX_SIZE_RULES, IDC_STATIC_KB_RULES);
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_General 
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CPropPage_General, CPropertyPage)

CPropPage_General::CPropPage_General(LPCTSTR szTitle) : CPropertyPage(CPropPage_General::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_General)
	m_bAutoStart = FALSE;
	m_bCheckMailOnStartup = FALSE;
	m_bUseDialUp = FALSE;
	m_bMarkDelete = FALSE;
	//}}AFX_DATA_INIT
	m_nStartCheckDelay = 3;
	m_bXHide = FALSE;
	m_bMinHide = FALSE;
	m_bConfirmDeleteMain = TRUE;
	m_bConfirmDeleteMsg = TRUE;
	m_bPasswordProtect = FALSE;
}

CPropPage_General::~CPropPage_General()
{
}

void CPropPage_General::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_General)
	DDX_Control(pDX, IDC_STATIC_ICON, m_staticIcon);
	DDX_Check(pDX, IDC_CHECK_AUTO_START, m_bAutoStart);
	DDX_Check(pDX, IDC_CHECK_MAILS_ON_STARTUP, m_bCheckMailOnStartup);
	DDX_Check(pDX, IDC_CHECK_USE_DIAL_UP, m_bUseDialUp);
	DDX_Check(pDX, IDC_CHECK_MARK_DELETE, m_bMarkDelete);
	//}}AFX_DATA_MAP
	DDX_Check(pDX, IDC_CHECK_X_HIDE, m_bXHide);
	DDX_Check(pDX, IDC_CHECK_MIN_HIDE, m_bMinHide);
	DDX_Control(pDX, IDC_EDIT_STARTCHECK_DELAY, m_editStartCheckDelay);
	DDX_Control(pDX, IDC_SPIN_STARTCHECK_DELAY, m_spinStartCheckDelay);
	DDX_Control(pDX, IDC_CHECK_MAILS_ON_STARTUP, m_chkCheckMailOnStartup);
	DDX_Text(pDX, IDC_EDIT_STARTCHECK_DELAY, m_nStartCheckDelay);
	DDV_MinMaxUInt(pDX, m_nStartCheckDelay, 0, 1800);
	DDX_Check(pDX, IDC_CHECK_DELETE_CONFIRMATION_MAIN, m_bConfirmDeleteMain);
	DDX_Check(pDX, IDC_CHECK_DELETE_CONFIRMATION_MSG, m_bConfirmDeleteMsg);
	DDX_Check(pDX, IDC_CHECK_PASSWORD, m_bPasswordProtect);
	DDX_Text(pDX, IDC_EDIT_THE_PASSWORD, m_stPassword);

	
	if(pDX->m_bSaveAndValidate)
	{
		if(m_bPasswordProtect && m_stPassword.IsEmpty())
		{
			AfxMessageBox(i18n("Please enter the password."));
			pDX->PrepareEditCtrl(IDC_EDIT_THE_PASSWORD);
			pDX->Fail();
			return;
		}
		if(!m_bPasswordProtect && !m_stPassword.IsEmpty())
			m_stPassword.Empty();

	}
}

BEGIN_MESSAGE_MAP(CPropPage_General, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_General)
	ON_BN_CLICKED(IDC_CHECK_MAILS_ON_STARTUP, OnCheckMailsOnStartup)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
	ON_BN_CLICKED(IDC_CHECK_PASSWORD, OnCheckPassword)
END_MESSAGE_MAP()

BOOL CPropPage_General::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	m_spinStartCheckDelay.SetBase(10);
	m_spinStartCheckDelay.SetRange(0, 1800);
	m_spinStartCheckDelay.SetBuddy(GetDlgItem(IDC_EDIT_STARTCHECK_DELAY));
	
	m_staticIcon.SetIcon(CPopManApp::GetIcon(IDI_DIALUP));

	OnCheckPassword();
	OnCheckMailsOnStartup();

	InitGUIText();

	return TRUE;  
}


void CPropPage_General::OnCheckPassword() 
{
	BOOL bUpdateEnabled = ((CWnd*)GetDlgItem(IDC_CHECK_PASSWORD))->SendMessage(BM_GETCHECK);
	((CWnd*)GetDlgItem(IDC_EDIT_THE_PASSWORD))->EnableWindow(bUpdateEnabled);
	if(bUpdateEnabled)
		((CWnd*)GetDlgItem(IDC_EDIT_THE_PASSWORD))->SetFocus();

}

void CPropPage_General::OnCheckMailsOnStartup() 
{
	BOOL bUpdateEnabled = (1 == m_chkCheckMailOnStartup.GetCheck());
	((CWnd*)GetDlgItem(IDC_STATIC_WAIT))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_EDIT_STARTCHECK_DELAY))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_SECONDS))->EnableWindow(bUpdateEnabled);	
	((CWnd*)GetDlgItem(IDC_SPIN_STARTCHECK_DELAY))->EnableWindow(bUpdateEnabled);	
}

void CPropPage_General::InitGUIText()
{
	SetDlgItemText(IDC_CHECK_USE_DIAL_UP,				i18n("Use dial-up networking"));
	SetDlgItemText(IDC_CHECK_AUTO_START,				i18n("Launch PopMan on Windows startup"));
	SetDlgItemText(IDC_CHECK_MAILS_ON_STARTUP,			i18n("Check for new mail on startup"));
	SetDlgItemText(IDC_STATIC_WAIT,						i18n("First wait"));
	SetDlgItemText(IDC_STATIC_SECONDS,					i18n("seconds"));
	SetDlgItemText(IDC_CHECK_X_HIDE,					i18n("X button hides to tray"));
	SetDlgItemText(IDC_CHECK_MIN_HIDE,					i18n("Min button hides to tray"));
	SetDlgItemText(IDC_CHECK_DELETE_CONFIRMATION_MAIN,	i18n("Delete confirmation in main window"));
	SetDlgItemText(IDC_CHECK_DELETE_CONFIRMATION_MSG,	i18n("Delete confirmation in preview pane and message window"));
	SetDlgItemText(IDC_CHECK_MARK_DELETE,				i18n("Delete command marks message for deletion"));
	SetDlgItemText(IDC_CHECK_PASSWORD,					i18n("Enable password protection:"));
	ArrangeUpDown(this, IDC_STATIC_WAIT, IDC_EDIT_STARTCHECK_DELAY, IDC_SPIN_STARTCHECK_DELAY, IDC_STATIC_SECONDS);
	ArrangeControlPair(this, IDC_CHECK_PASSWORD, IDC_EDIT_THE_PASSWORD, TRUE);

/*

	CWnd* pCheck = GetDlgItem(IDC_CHECK_DELETE_CONFIRMATION_MSG);
	int nTextWidth = GetLabelWidth(this, pCheck) + 16;

	int nWidth = 0;
	int nHeight = 0;
	int nStdHeight = 0;

	GetWindowSize(GetDlgItem(IDC_CHECK_DELETE_CONFIRMATION_MAIN), nWidth, nStdHeight);

	GetWindowSize(pCheck, nWidth, nHeight);
	if(nWidth < nTextWidth)
		SetWindowSize(this, pCheck, nWidth, 2*nStdHeight);
	else
		SetWindowSize(this, pCheck, nWidth, nStdHeight);
	*/
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_AutoCheck
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CPropPage_AutoCheck, CPropertyPage)

CPropPage_AutoCheck::CPropPage_AutoCheck(LPCTSTR szTitle) : CPropertyPage(CPropPage_AutoCheck::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_AutoCheck)
	m_bAutoCheckMinimized = FALSE;
	m_bSuspendAutoCheck = FALSE;
	m_nUpdateInterval = 15;
	m_bEnableCustomAutoCheck = FALSE;
	//}}AFX_DATA_INIT
	m_pAccounts = NULL;
	m_nLabelEditIdx = -1;
	//m_nLastEditTime = COleDateTime::GetCurrentTime();
	m_bFirstFocus = true;
}

CPropPage_AutoCheck::~CPropPage_AutoCheck()
{
}

void CPropPage_AutoCheck::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_AutoCheck)
	DDX_Control(pDX, IDC_CHECK_ENABLE_CUSTOM, m_chkEnableCustomAutoCheck);
	DDX_Control(pDX, IDC_LIST_AUTOCHECK_ACCS, m_listAutoAccs);
	DDX_Control(pDX, IDC_TIME_SUSPEND_TO, m_dtTo);
	DDX_Control(pDX, IDC_TIME_SUSPEND_FROM, m_dtFrom);
	DDX_Control(pDX, IDC_CHECK_AUTOCHECK_SUSPEND, m_chkAutoCheckSuspend);
	DDX_Control(pDX, IDC_SPIN_UPDATEINTERVAL, m_spinUpdateInterval);
	DDX_Control(pDX, IDC_EDIT_UPDATEINTERVAL, m_editUpdateInterval);
	DDX_Check(pDX, IDC_CHECK_AUTOCHECK_MINIMIZED, m_bAutoCheckMinimized);
	DDX_Check(pDX, IDC_CHECK_AUTOCHECK_SUSPEND, m_bSuspendAutoCheck);
	DDX_Text(pDX, IDC_EDIT_UPDATEINTERVAL, m_nUpdateInterval);
	DDV_MinMaxUInt(pDX, m_nUpdateInterval, 1, 180);
	DDX_Check(pDX, IDC_CHECK_ENABLE_CUSTOM, m_bEnableCustomAutoCheck);
	//}}AFX_DATA_MAP

	if(pDX->m_bSaveAndValidate)
	{
		CTime Time;
		m_dtFrom.GetTime(Time);
		m_nSuspendFrom = Time.GetHour() * 3600 + Time.GetMinute() * 60 + Time.GetSecond();
		m_dtTo.GetTime(Time);
		m_nSuspendTo = Time.GetHour() * 3600 + Time.GetMinute() * 60 + Time.GetSecond();


		for(int i = 0; i < m_listAutoAccs.GetItemCount(); ++i) {
			AccInfo* info = (AccInfo*)m_listAutoAccs.GetItemData(i);
			info->pAcc->m_bCustomAutoCheckInterval = info->bCustom;
			info->pAcc->m_nCustomAutoCheckInterval = info->nInterval;
		}
		
	}
	else
	{
		UINT Hour, Minute, Second;

		Hour = (UINT)m_nSuspendFrom / 3600;
		Minute = (UINT)((m_nSuspendFrom - (Hour * 3600)) / 60);
		Second = m_nSuspendFrom - (Hour * 3600) - (Minute * 60);
		CTime TimeFrom(2003, 1, 1, Hour, Minute, Second);
		m_dtFrom.SetTime(&TimeFrom);

		Hour = (UINT)m_nSuspendTo / 3600;
		Minute = (UINT)((m_nSuspendTo - (Hour * 3600)) / 60);
		Second = m_nSuspendTo - (Hour * 3600) - (Minute * 60);
		CTime TimeTo(2003, 1, 1, Hour, Minute, Second);
		m_dtTo.SetTime(&TimeTo);
	}

}


BEGIN_MESSAGE_MAP(CPropPage_AutoCheck, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_AutoCheck)
	ON_BN_CLICKED(IDC_CHECK_AUTOCHECK_SUSPEND, OnAutocheckSuspend)
	ON_WM_DESTROY()
	ON_NOTIFY(LVN_BEGINLABELEDIT, IDC_LIST_AUTOCHECK_ACCS, OnBeginlabeleditListAutocheckAccs)
	ON_NOTIFY(LVN_ENDLABELEDIT, IDC_LIST_AUTOCHECK_ACCS, OnEndlabeleditListAutocheckAccs)
	ON_NOTIFY(NM_SETFOCUS, IDC_LIST_AUTOCHECK_ACCS, OnSetfocusListAutocheckAccs)
	ON_NOTIFY(NM_CLICK, IDC_LIST_AUTOCHECK_ACCS, OnClickListAutocheckAccs)
	ON_NOTIFY(NM_DBLCLK, IDC_LIST_AUTOCHECK_ACCS, OnDblclkListAutocheckAccs)
	ON_BN_CLICKED(IDC_CHECK_ENABLE_CUSTOM, OnCustomAutoCheck)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
	ON_NOTIFY(NM_CUSTOMDRAW, IDC_LIST_AUTOCHECK_ACCS, OnCustomDraw)
END_MESSAGE_MAP()


BOOL CPropPage_AutoCheck::OnInitDialog() 
{
	if(!m_pAccounts) return FALSE;

	CPropertyPage::OnInitDialog();
		
	m_spinUpdateInterval.SetBase(10);
	m_spinUpdateInterval.SetRange(1, 180);
	m_spinUpdateInterval.SetBuddy(GetDlgItem(IDC_EDIT_UPDATEINTERVAL));
		
	OnAutocheckSuspend();
	
	
	m_listAutoAccs.SetExtendedStyle(LVS_EX_GRIDLINES);

	m_listAutoAccs.InsertColumn(0, _T(""),	LVCFMT_RIGHT,	62,   0);
	m_listAutoAccs.InsertColumn(1, _T(""),	LVCFMT_CENTER,	66,   1); 
	m_listAutoAccs.InsertColumn(2, _T(""),	LVCFMT_LEFT,	170,  2); 
	 
	int IdxArray[] = {1,2,0};
	m_listAutoAccs.SetColumnOrderArray(3, IdxArray);

	int i = 0;
	POSITION pos = m_pAccounts->GetHeadPosition();
	while(pos != NULL)
	{
		CAccount* pAcc =  m_pAccounts->GetNext(pos);
		m_listAutoAccs.InsertItem(i, _T(""));

		LV_ITEM lvItem;
		lvItem.iItem = i;
		lvItem.mask = LVIF_TEXT;

		lvItem.pszText = (LPTSTR)(LPCTSTR)pAcc->m_stName;
		lvItem.iSubItem = 2;
		m_listAutoAccs.SetItem(&lvItem);

		
		CString stInterval;
		stInterval.Format(_T("%d"), pAcc->m_nCustomAutoCheckInterval);
		lvItem.pszText = (LPTSTR)(LPCTSTR)stInterval;
		lvItem.iSubItem = 0;
		m_listAutoAccs.SetItem(&lvItem);


		AccInfo* info = new AccInfo;
		info->pAcc = pAcc;
		info->bCustom = pAcc->m_bCustomAutoCheckInterval;
		info->nInterval = pAcc->m_nCustomAutoCheckInterval;

		m_listAutoAccs.SetItemData(i, (DWORD)info);
		++i;
	}

	InitGUIText();
	OnCustomAutoCheck();

	return TRUE;  
}

void CPropPage_AutoCheck::OnCustomAutoCheck() 
{
	BOOL bEnable = (1 == m_chkEnableCustomAutoCheck.GetCheck());
	m_listAutoAccs.EnableWindow(bEnable);
}

void CPropPage_AutoCheck::OnBeginlabeleditListAutocheckAccs(NMHDR* pNMHDR, LRESULT* pResult) 
{
	LV_DISPINFO* pDispInfo = (LV_DISPINFO*)pNMHDR;

	m_nLabelEditIdx = pDispInfo->item.iItem;
	
	*pResult = 0;
}

void CPropPage_AutoCheck::OnEndlabeleditListAutocheckAccs(NMHDR* pNMHDR, LRESULT* pResult) 
{
	BOOL bSuccess = FALSE;
	LV_DISPINFO* pDispInfo = (LV_DISPINFO*)pNMHDR;
	int nItem = pDispInfo->item.iItem;
	CString stText = pDispInfo->item.pszText;

//	TRACE("OnEndLabelEdit %s \n", static_cast<LPCTSTR>(stText));

	if(!stText.IsEmpty())
	{
		bool bNum = true;
		for(int i = 0; i < stText.GetLength(); ++i) {
			TCHAR c = stText[i];
			if(c < '0' || c > '9') {
				bNum = false;
				break;
			}
		}
		if(bNum)
		{
			AccInfo* info = (AccInfo*)m_listAutoAccs.GetItemData(nItem);
			info->nInterval = _tcstoul(stText, NULL, 10);
			bSuccess = TRUE;
		}
	}
	m_nLabelEditIdx = -1;
	*pResult = bSuccess;
}

void CPropPage_AutoCheck::OnDestroy() 
{
	for(int i = 0; i < m_listAutoAccs.GetItemCount(); ++i)
		delete (AccInfo*)m_listAutoAccs.GetItemData(i);
		
	CPropertyPage::OnDestroy();
}

void CPropPage_AutoCheck::OnCustomDraw(NMHDR* pNMHDR, LRESULT* pResult)
{
	LPNMLVCUSTOMDRAW  lplvcd = (LPNMLVCUSTOMDRAW)pNMHDR;

	switch(lplvcd->nmcd.dwDrawStage)
	{
	case CDDS_PREPAINT:
		*pResult = CDRF_NOTIFYITEMDRAW;          // ask for item notifications.
		break;

	case CDDS_ITEMPREPAINT:
	{
		// This is the beginning of an item's paint cycle.
		int  nItem = (int)lplvcd->nmcd.dwItemSpec;
		CDC* pDC   = CDC::FromHandle(lplvcd->nmcd.hdc);
		bool bListHasFocus = (m_listAutoAccs.GetSafeHwnd() == ::GetFocus());
		AccInfo* info = (AccInfo*)m_listAutoAccs.GetItemData(nItem);
	
		bool bDisabled = (1 != m_chkEnableCustomAutoCheck.GetCheck());

		COLORREF colorStdBkg = ::GetSysColor(COLOR_WINDOW);
		COLORREF colorStdText = ::GetSysColor(COLOR_BTNTEXT);

		if(bDisabled) {
			colorStdBkg = ::GetSysColor(COLOR_BTNFACE);
			colorStdText = ::GetSysColor(COLOR_GRAYTEXT);
		}

		LVITEM rItem;
		ZeroMemory(&rItem, sizeof(LVITEM));
		rItem.mask  = LVIF_STATE;
		rItem.iItem = nItem;
		rItem.stateMask = LVIS_SELECTED | LVIS_FOCUSED;
		m_listAutoAccs.GetItem(&rItem);

		// Get the rect that bounds the text label.
		CRect rcItem;
		GetSubItemBounds(nItem, 2, rcItem);
		rcItem.left += 1;
		rcItem.bottom -= 1;
		pDC->SetTextColor(colorStdText);
		pDC->FillSolidRect(rcItem, colorStdBkg);
		pDC->SetBkMode(TRANSPARENT);

		CRect rcText;
		rcText = rcItem;
		rcText.left += 3;
		rcText.top++;

		CString sText = m_listAutoAccs.GetItemText(nItem, 2);
		pDC->DrawText(sText, rcText, DT_NOPREFIX | DT_LEFT | DT_VCENTER | DT_SINGLELINE);

		
		// Draw the background of the list item.  Colors are selected 
		// according to the item's state.
		COLORREF crBkgnd, crText;
		if(info->bCustom)
			crText = colorStdText;
		else
			crText = ::GetSysColor(COLOR_GRAYTEXT);

		if (rItem.state & LVIS_SELECTED)
		{
			if (bListHasFocus)
			{
				crBkgnd = ::GetSysColor(COLOR_HIGHLIGHT);
				crText  = ::GetSysColor(COLOR_HIGHLIGHTTEXT);
			}
			else
				crBkgnd = ::GetSysColor(COLOR_BTNFACE);
		}
		else
			crBkgnd = colorStdBkg;
		

		if(m_nLabelEditIdx == nItem)
		{
			crBkgnd = colorStdBkg;
			crText = crBkgnd;	// white text on white background
		}

		pDC->SetTextColor(crText);

		// Draw the background & prep the DC for the text drawing.  Note
		// that the entire item RECT is filled in, so this emulates the full-
		// row selection style of normal lists.
		GetSubItemBounds(nItem, 0, rcItem);
		rcItem.left += 4;
		rcItem.bottom -= 1;
	
		// TRACE("FillSolidRect: %d : %x \n", nItem, crBkgnd);
		pDC->FillSolidRect(rcItem, crBkgnd);
		pDC->SetBkMode(TRANSPARENT);

		// Tweak the rect a bit for nicer-looking text alignment.
		rcText = rcItem;
		rcText.right -= 4;
		rcText.top++;
	
		// Draw the text.
		sText = m_listAutoAccs.GetItemText(nItem, 0);
		pDC->DrawText(sText, rcText, DT_RIGHT | DT_VCENTER | DT_SINGLELINE);

		// Draw a focus rect around the item if necessary.
		if(bListHasFocus && (rItem.state & LVIS_FOCUSED) && m_nLabelEditIdx != nItem)
			pDC->DrawFocusRect(rcItem);
	
	
		GetSubItemBounds(nItem, 1, rcItem);
		rcItem.left += 1;
		rcItem.bottom -= 1;
		pDC->FillSolidRect(rcItem, colorStdBkg);
		
		::DrawFrameControl(lplvcd->nmcd.hdc, &rcItem, DFC_BUTTON, DFCS_BUTTONCHECK | (info->bCustom ? DFCS_CHECKED : 0) | (bDisabled ? DFCS_INACTIVE : 0));
		
		*pResult = CDRF_SKIPDEFAULT;

		break;
	}
	default:
		*pResult = CDRF_DODEFAULT;
	}
}

void CPropPage_AutoCheck::GetSubItemBounds(int nItem, int nSubItem, RECT& rcItem)
{
	CRect rcHead;
	m_listAutoAccs.GetItemRect(nItem, &rcItem, LVIR_BOUNDS);
	m_listAutoAccs.GetHeaderCtrl()->GetItemRect(nSubItem, &rcHead);

	rcItem.left = rcHead.left;
	rcItem.right = rcHead.right;
}



void CPropPage_AutoCheck::OnClickListAutocheckAccs(NMHDR* pNMHDR, LRESULT* pResult) 
{
	POINT point;
	::GetCursorPos(&point);
	OnListClick(point);
	*pResult = 0;
}

void CPropPage_AutoCheck::OnDblclkListAutocheckAccs(NMHDR* pNMHDR, LRESULT* pResult) 
{
	POINT point;
	::GetCursorPos(&point);
	OnListClick(point);
	*pResult = 0;
}


BOOL CPropPage_AutoCheck::PreTranslateMessage(MSG* pMsg) 
{
	if(pMsg->message == WM_KEYDOWN && pMsg->wParam == VK_F2)
	{
		if(m_listAutoAccs.GetSafeHwnd() == ::GetFocus())
		{
			for(int i = 0; i < m_listAutoAccs.GetItemCount(); ++i) {
				if(LVIS_FOCUSED == m_listAutoAccs.GetItemState(i, LVIS_FOCUSED)) {
					m_listAutoAccs.SetItemState(i, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
					m_listAutoAccs.EditLabel(i);
					break;
				}
			}
		}
	}
	return CPropertyPage::PreTranslateMessage(pMsg);
}



void CPropPage_AutoCheck::OnListClick(POINT point)
{
	m_listAutoAccs.ScreenToClient(&point);

	LVHITTESTINFO testInfo;
	testInfo.pt = point;

	int idx = m_listAutoAccs.SubItemHitTest(&testInfo);

	if(idx > -1 && testInfo.iSubItem == 1)
	{	// toggle checkbox:
		AccInfo* info = (AccInfo*)m_listAutoAccs.GetItemData(idx);
		info->bCustom = !info->bCustom;
		CRect rcItem;
		GetSubItemBounds(idx, 1, rcItem);
		m_listAutoAccs.InvalidateRect(&rcItem);
		m_listAutoAccs.UpdateWindow();
	}

	idx = GetItemFromPoint(point);
	if(idx > -1)
	{
		if(LVIS_FOCUSED == m_listAutoAccs.GetItemState(idx, LVIS_FOCUSED))
		{
			CRect rect;
			GetSubItemBounds(idx, 0, rect);
			m_listAutoAccs.SetItemState(idx, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
			if(rect.PtInRect(point)) {
				m_listAutoAccs.EditLabel(idx);
//				TRACE("EditLabel idx:%d \n", idx);
			}
		}
		else
		{
//			TRACE("SetItemState idx:%d \n", idx);
			m_listAutoAccs.SetItemState(idx, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
			m_listAutoAccs.Invalidate();
			m_listAutoAccs.UpdateWindow();
		}
	}
}

void CPropPage_AutoCheck::OnSetfocusListAutocheckAccs(NMHDR* pNMHDR, LRESULT* pResult) 
{
	if(m_bFirstFocus)
	{
		m_bFirstFocus = false;
		POINT point;
		::GetCursorPos(&point);
		OnListClick(point);
	}
	*pResult = 0;
}

int CPropPage_AutoCheck::GetItemFromPoint(CPoint point)
{
	for(int i = m_listAutoAccs.GetTopIndex(); i < m_listAutoAccs.GetItemCount(); ++i)
	{
		CRect rect;
		m_listAutoAccs.GetItemRect(i, &rect, LVIR_BOUNDS);
		if(rect.PtInRect(point))
			return i;
	}
	return -1;
}

void CPropPage_AutoCheck::OnAutocheckSuspend() 
{
	BOOL bUpdateEnabled = (1 == m_chkAutoCheckSuspend.GetCheck());
	GetDlgItem(IDC_TIME_SUSPEND_FROM)->EnableWindow(bUpdateEnabled);
	GetDlgItem(IDC_TIME_SUSPEND_TO)->EnableWindow(bUpdateEnabled);
	GetDlgItem(IDC_STATIC_AND)->EnableWindow(bUpdateEnabled);
}

void CPropPage_AutoCheck::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_INTERVAL,				i18n("Timer interval:"));
	SetDlgItemText(IDC_STATIC_MINUTES,				i18n("minutes"));
	SetDlgItemText(IDC_CHECK_AUTOCHECK_MINIMIZED,	i18n("AutoCheck only while minimized"));
	SetDlgItemText(IDC_CHECK_AUTOCHECK_SUSPEND,		i18n("Don't check between:"));
	SetDlgItemText(IDC_STATIC_AND,					i18n("and"));
	SetDlgItemText(IDC_CHECK_ENABLE_CUSTOM,			i18n("Enable custom AutoCheck interval per account:"));

	LV_COLUMN column;
	column.mask = LVCF_TEXT;

	column.pszText = (LPTSTR)i18n("Interval");
	m_listAutoAccs.SetColumn(0, &column);

	column.pszText = (LPTSTR)i18n("Custom");
	m_listAutoAccs.SetColumn(1, &column);

	column.pszText = (LPTSTR)i18n("Account");
	m_listAutoAccs.SetColumn(2, &column);

	ArrangeUpDown(this, IDC_STATIC_INTERVAL, IDC_EDIT_UPDATEINTERVAL, IDC_SPIN_UPDATEINTERVAL, IDC_STATIC_MINUTES);
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_EmailClient 
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CPropPage_EmailClient, CPropertyPage)

CPropPage_EmailClient::CPropPage_EmailClient(LPCTSTR szTitle) : CPropertyPage(CPropPage_EmailClient::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_EmailClient)
	m_bHTMLReply = FALSE;
	m_bEnforceAddress = FALSE;
	m_stReplyParam = _T("");
	//}}AFX_DATA_INIT
	m_nReplyOption = REPLYOPTIONS::Eml;
	m_ClientLaunchAction = CLIENTLAUNCHACTION::NoAction;
	m_szNothing		= NULL;
	m_szHideToTray  = NULL;
	m_szQuit		= NULL;
}

CPropPage_EmailClient::~CPropPage_EmailClient()
{
}

void CPropPage_EmailClient::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_EmailClient)
	DDX_Control(pDX, IDC_COMBO_EMAIL_CLIENT, m_cbClient);
	DDX_Check(pDX, IDC_CHECK_HTML_REPLY, m_bHTMLReply);
	DDX_Check(pDX, IDC_CHECK_YAHOO_COMPATIBLE, m_bEnforceAddress);
	DDX_Text(pDX, IDC_EDIT_CMD_PARAM, m_stReplyParam);
	//}}AFX_DATA_MAP

	DDX_Control(pDX, IDC_COMBO_LAUNCH_ACTION, m_cboLaunchAction);


	if(pDX->m_bSaveAndValidate)
	{
		if(((CButton*)GetDlgItem(IDC_RADIO_REPLY_EML))->GetCheck() == 1)
			m_nReplyOption = REPLYOPTIONS::Eml;
		else if(((CButton*)GetDlgItem(IDC_RADIO_REPLY_MAPI))->GetCheck() == 1)
			m_nReplyOption = REPLYOPTIONS::MAPI;
		else if(((CButton*)GetDlgItem(IDC_RADIO_REPLY_MAILTO))->GetCheck() == 1)
			m_nReplyOption = REPLYOPTIONS::MailTo;
		else
			m_nReplyOption = REPLYOPTIONS::CmdParam;
	
		m_cbClient.GetWindowText(m_stEmailClient);
		int nPos = m_MailClients.FindName(m_stEmailClient);
		if(nPos > -1)
			m_stEmailClient = m_MailClients[nPos].m_stPath;

		CString stName;
		m_cboLaunchAction.GetWindowText(stName);
		if(stName == m_szHideToTray)
			m_ClientLaunchAction = CLIENTLAUNCHACTION::HideToTray;
		else if(stName == m_szQuit)
			m_ClientLaunchAction = CLIENTLAUNCHACTION::Quit;
		else
			m_ClientLaunchAction = CLIENTLAUNCHACTION::NoAction;

	}
	else
	{
		((CButton*)GetDlgItem(IDC_RADIO_REPLY_EML))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_REPLY_MAPI))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_REPLY_MAILTO))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_REPLY_CMD_PARAM))->SetCheck(0);

		UINT nIDRadio = 0;
	
		switch(m_nReplyOption) {
		case REPLYOPTIONS::Eml:		nIDRadio = IDC_RADIO_REPLY_EML;			break;
		case REPLYOPTIONS::MAPI:	nIDRadio = IDC_RADIO_REPLY_MAPI;		break;
		case REPLYOPTIONS::MailTo:	nIDRadio = IDC_RADIO_REPLY_MAILTO;		break;
		default:					nIDRadio = IDC_RADIO_REPLY_CMD_PARAM;	break;
		}

		((CButton*)GetDlgItem(nIDRadio))->SetCheck(1);
	}
}


BEGIN_MESSAGE_MAP(CPropPage_EmailClient, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_EmailClient)
	ON_BN_CLICKED(IDC_BUTTON_BROWSE, OnBrowseClient)
	ON_BN_CLICKED(IDC_BUTTON_CLIENT_LAUNCH, OnClientLaunch)
	ON_BN_CLICKED(IDC_RADIO_REPLY_MAPI, UpdateMAPIOptions)
	ON_BN_CLICKED(IDC_RADIO_REPLY_MAILTO, UpdateMAPIOptions)
	ON_BN_CLICKED(IDC_RADIO_REPLY_EML, UpdateMAPIOptions)
	ON_BN_CLICKED(IDC_RADIO_REPLY_CMD_PARAM, UpdateMAPIOptions)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()

void CPropPage_EmailClient::OnBrowseClient() 
{
	CFileDlgEx Dlg;
	Dlg.AddFilter(i18n("Executable (*.exe)"), _T("*.exe"));
	
	CString stFileName;
	if(!Dlg.GetOpenFileName(stFileName, _T("exe")))
		return;

	if(stFileName.Find(_T(' ')) > -1)
		stFileName = _T('\"') + stFileName + _T('\"');

	m_cbClient.SetWindowText(stFileName);
}

void CPropPage_EmailClient::OnClientLaunch() 
{
	CString stFileName;
	m_cbClient.GetWindowText(stFileName);

	int nPos = m_MailClients.FindName(stFileName);
	if(nPos > -1)
		stFileName = m_MailClients[nPos].m_stPath;

	CString stExe, stCmdParam;
	if(!SplitExeCmdParam(stFileName, stExe, stCmdParam))
		return;

	ShellExecute(NULL, _T("open"), stExe, stCmdParam, NULL, SW_SHOWNORMAL);
}

void CPropPage_EmailClient::UpdateMAPIOptions()
{
	BOOL bMapiEnabled = (1 == ((CButton*)GetDlgItem(IDC_RADIO_REPLY_MAPI))->GetCheck());
	GetDlgItem(IDC_CHECK_HTML_REPLY)->EnableWindow(bMapiEnabled);
	GetDlgItem(IDC_CHECK_YAHOO_COMPATIBLE)->EnableWindow(bMapiEnabled);

	BOOL bCmdParamEnabled = (1 == ((CButton*)GetDlgItem(IDC_RADIO_REPLY_CMD_PARAM))->GetCheck());
	GetDlgItem(IDC_EDIT_CMD_PARAM)->EnableWindow(bCmdParamEnabled);
}


BOOL CPropPage_EmailClient::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	m_MailClients.LoadFromRegistry();
	
	for(int n = 0; n < m_MailClients.GetSize(); n++)
		m_cbClient.AddString(m_MailClients[n].m_stName);
	
	
	int nPos = m_MailClients.FindPath(m_stEmailClient);
	if(nPos == -1)
	{
		m_cbClient.AddString(m_stEmailClient);
		m_cbClient.SelectString(-1, m_stEmailClient);
	}
	else
		m_cbClient.SelectString(-1, m_MailClients[nPos].m_stName);
	
	UpdateMAPIOptions();

	InitGUIText();
	
	return TRUE;  
}


int CPropPage_EmailClient::CMailClients::FindName(const CString& stName, int nPos)
{
	for(int n = nPos; n < GetSize(); n++)
	{
		if(GetAt(n).m_stName.CompareNoCase(stName) == 0)
			return n;
	}
	return -1;
}

int CPropPage_EmailClient::CMailClients::FindPath(const CString& stPath, int nPos)
{
	for(int n = nPos; n < GetSize(); n++)
	{
		if(GetAt(n).m_stPath.CompareNoCase(stPath) == 0)
			return n;
	}
	return -1;
}

void CPropPage_EmailClient::CMailClients::LoadFromRegistry()
{
	RemoveAll();

	const CString stClientsRootPath = _T("Software\\Clients\\Mail");

	CRegistryKey Reg;
	if(!Reg.Open(HKEY_LOCAL_MACHINE, stClientsRootPath, KEY_READ))
		return;

	CString stClientName;
	CString stClientPath;
	
	for(int n = 0; ; n++)
	{
		if(!Reg.EnumKey(n, stClientName))
			break;
		
		CString stOpenPath = stClientsRootPath;
		stOpenPath += _T('\\');
		stOpenPath += stClientName;
		stOpenPath += _T("\\shell\\open\\command");
		
		CRegistryKey RegClient;
		if(!RegClient.Open(HKEY_LOCAL_MACHINE, stOpenPath, KEY_READ))
			continue;

		if(!RegClient.QueryValue(_T(""), stClientPath))
			continue;

		TCHAR szExpandedPath[MAX_PATH] = _T("");
		if(ExpandEnvironmentStrings(stClientPath, szExpandedPath, sizeof(szExpandedPath)/sizeof(TCHAR)))
			stClientPath = szExpandedPath;


		MailClient Client(stClientName, stClientPath);
		Add(Client);
	}
} 

void CPropPage_EmailClient::InitGUIText()
{
	m_szNothing		= _T("  ");
	m_szHideToTray  = i18n("Hide PopMan to tray");
	m_szQuit		= i18n("Quit PopMan");

	m_cboLaunchAction.ResetContent();
	m_cboLaunchAction.AddString(m_szNothing);
	m_cboLaunchAction.AddString(m_szHideToTray);
	m_cboLaunchAction.AddString(m_szQuit);

	LPCTSTR szName = m_szNothing;
	if(m_ClientLaunchAction == CLIENTLAUNCHACTION::HideToTray)
		szName = m_szHideToTray;
	else if(m_ClientLaunchAction == CLIENTLAUNCHACTION::Quit)
		szName = m_szQuit;

	m_cboLaunchAction.SelectString(-1, szName);


	SetDlgItemText(IDC_STATIC_EMAIL_CLIENT,		i18n("Email client:"));
	SetDlgItemText(IDC_BUTTON_CLIENT_LAUNCH,	i18n("&Test"));
	SetDlgItemText(IDC_STATIC_LAUNCH_CLIENT,	i18n("When launching email client:"));
	SetDlgItemText(IDC_STATIC_WHEN_REPLY,		i18n("When replying"));
	SetDlgItemText(IDC_RADIO_REPLY_EML,			i18n("Use temporary eml file (Outlook Express)"));
	SetDlgItemText(IDC_RADIO_REPLY_MAILTO,		i18n("Use mailto (without quotted original message)"));
	SetDlgItemText(IDC_RADIO_REPLY_MAPI,		i18n("Use MAPI"));
	SetDlgItemText(IDC_CHECK_HTML_REPLY,		i18n("Use HTML linebreaks in reply body"));
	SetDlgItemText(IDC_CHECK_YAHOO_COMPATIBLE,	i18n("Enforce email address (Yahoo! Mail)"));
	SetDlgItemText(IDC_RADIO_REPLY_CMD_PARAM,	i18n("Launch email client with command line parameter:"));

	ArrangeControlPair(this, IDC_STATIC_LAUNCH_CLIENT, IDC_COMBO_LAUNCH_ACTION);
	ArrangeControlPair(this, IDC_STATIC_EMAIL_CLIENT,  IDC_COMBO_EMAIL_CLIENT);
}

/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_TrayIcon 
/////////////////////////////////////////////////////////////////////////////
IMPLEMENT_DYNCREATE(CPropPage_TrayIcon, CPropertyPage)

CPropPage_TrayIcon::CPropPage_TrayIcon(LPCTSTR szTitle) : CPropertyPage(CPropPage_TrayIcon::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_TrayIcon)
	m_bNumMailsInTray = FALSE;
	m_bHighColorTray = FALSE;
	m_nRotationInterval = 0;
	m_bRotateMailNum = FALSE;
	m_bAdvancedTrayInfo = FALSE;
	//}}AFX_DATA_INIT
	m_bHighColorTray = FALSE;
}

CPropPage_TrayIcon::~CPropPage_TrayIcon()
{
}

void CPropPage_TrayIcon::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_TrayIcon)
	DDX_Control(pDX, IDC_CHECK_NUM_ROTATE, m_chkRotate);
	DDX_Control(pDX, IDC_SPIN_ROTATION_INTERVAL, m_spinRotationInterval);
	DDX_Control(pDX, IDC_CHECK_NUM_MESSAGES, m_chkNumMails);
	DDX_Check(pDX, IDC_CHECK_NUM_MESSAGES, m_bNumMailsInTray);
	DDX_Check(pDX, IDC_CHECK_HIGH_COLOR, m_bHighColorTray);
	DDX_Text(pDX, IDC_EDIT_ROTATION_INTERVAL, m_nRotationInterval);
	DDV_MinMaxUInt(pDX, m_nRotationInterval, 1, 27000);
	DDX_Check(pDX, IDC_CHECK_NUM_ROTATE, m_bRotateMailNum);
	DDX_Check(pDX, IDC_CHECK_ADVANCED_INFO, m_bAdvancedTrayInfo);
	//}}AFX_DATA_MAP

	if(pDX->m_bSaveAndValidate)
	{
		if(((CButton*)GetDlgItem(IDC_RADIO_NUM_ALL))->GetCheck() == 1)
			m_numMails = NUMMAILSTRAY::All;
		else if(((CButton*)GetDlgItem(IDC_RADIO_NUM_UNREAD))->GetCheck() == 1)
			m_numMails = NUMMAILSTRAY::Unread;
		else 
			m_numMails = NUMMAILSTRAY::New;
	}
	else
	{
		((CButton*)GetDlgItem(IDC_RADIO_NUM_ALL))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_NUM_UNREAD))->SetCheck(0);
		((CButton*)GetDlgItem(IDC_RADIO_NUM_NEW))->SetCheck(0);

		UINT nIDRadio = 0;
		if(m_numMails == NUMMAILSTRAY::All)
			nIDRadio = IDC_RADIO_NUM_ALL;
		else if(m_numMails == NUMMAILSTRAY::Unread)
			nIDRadio = IDC_RADIO_NUM_UNREAD;
		else
			nIDRadio = IDC_RADIO_NUM_NEW;

		((CButton*)GetDlgItem(nIDRadio))->SetCheck(1);
	
	}
}


BEGIN_MESSAGE_MAP(CPropPage_TrayIcon, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_TrayIcon)
	ON_BN_CLICKED(IDC_CHECK_NUM_MESSAGES, OnCheckNumMessages)
	ON_BN_CLICKED(IDC_CHECK_NUM_ROTATE, OnRotateAccs)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()


BOOL CPropPage_TrayIcon::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	m_spinRotationInterval.SetBase(10);
	m_spinRotationInterval.SetRange(1, 27000);
	m_spinRotationInterval.SetBuddy(GetDlgItem(IDC_EDIT_ROTATION_INTERVAL));
	
	OnCheckNumMessages();
	if(m_bHighColorTrayEnabled)
		((CWnd*)GetDlgItem(IDC_CHECK_HIGH_COLOR))->ShowWindow(SW_HIDE);

	InitGUIText();

	return TRUE; 
}


void CPropPage_TrayIcon::OnCheckNumMessages() 
{
	BOOL bUpdateEnabled = (1 == m_chkNumMails.GetCheck());
	((CWnd*)GetDlgItem(IDC_CHECK_NUM_ROTATE))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_EDIT_ROTATION_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_SPIN_ROTATION_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_TS))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_RADIO_NUM_ALL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_RADIO_NUM_UNREAD))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_RADIO_NUM_NEW))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_CHECK_ADVANCED_INFO))->EnableWindow(bUpdateEnabled);
	if(bUpdateEnabled)
		OnRotateAccs();
}


void CPropPage_TrayIcon::OnRotateAccs() 
{
	BOOL bUpdateEnabled = (1 == m_chkRotate.GetCheck());
	((CWnd*)GetDlgItem(IDC_STATIC_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_EDIT_ROTATION_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_SPIN_ROTATION_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_TS))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_CHECK_ADVANCED_INFO))->EnableWindow(bUpdateEnabled);
}

void CPropPage_TrayIcon::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_NUMBER,		i18n("Number of messages"));
	SetDlgItemText(IDC_CHECK_NUM_MESSAGES,	i18n("Show number of messages in tray icon"));
	SetDlgItemText(IDC_RADIO_NUM_ALL,		i18n("Show number of all messages"));
	SetDlgItemText(IDC_RADIO_NUM_UNREAD,	i18n("Show number of unread messages"));
	SetDlgItemText(IDC_RADIO_NUM_NEW,		i18n("Show number of new messages"));
	SetDlgItemText(IDC_CHECK_NUM_ROTATE,	i18n("Rotate icon for each account"));
	SetDlgItemText(IDC_STATIC_INTERVAL,		i18n("Rotation interval:"));
	SetDlgItemText(IDC_STATIC_TS,			i18n("tenths of a second"));
	SetDlgItemText(IDC_CHECK_ADVANCED_INFO,	i18n("Show advanced account info"));
	SetDlgItemText(IDC_CHECK_HIGH_COLOR,	i18n("Enable HighColor icons in system tray"));

	ArrangeControlPair(this, IDC_STATIC_NUMBER, IDC_STATIC_NUMBER_LINE);
	ArrangeUpDown(this, IDC_STATIC_INTERVAL, IDC_EDIT_ROTATION_INTERVAL, IDC_SPIN_ROTATION_INTERVAL, IDC_STATIC_TS);
		
}


/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_MouseAction 
/////////////////////////////////////////////////////////////////////////////
IMPLEMENT_DYNCREATE(CPropPage_MouseAction, CPropertyPage)

CPropPage_MouseAction::CPropPage_MouseAction(LPCTSTR szTitle) : CPropertyPage(CPropPage_MouseAction::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_MouseAction)
	m_bDoubleClickDelay = FALSE;
	//}}AFX_DATA_INIT
}

CPropPage_MouseAction::~CPropPage_MouseAction()
{
}

void CPropPage_MouseAction::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_MouseAction)
	DDX_Control(pDX, IDC_COMBO_RIGHT, m_cboRight);
	DDX_Control(pDX, IDC_COMBO_MIDDLE, m_cboMiddle);
	DDX_Control(pDX, IDC_COMBO_LEFT, m_cboLeft);
	DDX_Control(pDX, IDC_COMBO_DOUBLE, m_cboDouble);
	DDX_Check(pDX, IDC_CHECK_CLICK_DELAY, m_bDoubleClickDelay);
	//}}AFX_DATA_MAP

	if(pDX->m_bSaveAndValidate)
	{
		m_actionLeftClick   = GetAction(m_cboLeft);
		m_actionRightClick  = GetAction(m_cboRight);  
		m_actionMiddleClick = GetAction(m_cboMiddle); 
		m_actionDoubleClick = GetAction(m_cboDouble); 
	}
}


void CPropPage_MouseAction::FillComboBox(CComboBox& Combo)
{
	Combo.ResetContent();
	for(int n = 0; n < m_ActionNameMap.GetSize(); n++)
		Combo.AddString(m_ActionNameMap[n].szName);
}

void CPropPage_MouseAction::SelectAction(CComboBox& Combo, TRAYACTIONS Action)
{
	LPCTSTR szName = NULL;

	for(int n = 0; n < m_ActionNameMap.GetSize(); n++)
	{
		if(m_ActionNameMap[n].Action == Action)
		{
			szName = m_ActionNameMap[n].szName;
			break;
		}
	}

	Combo.SelectString(-1, szName);
}

CPropPage_MouseAction::TRAYACTIONS CPropPage_MouseAction::GetAction(CComboBox& Combo)
{
	CString stName;
	Combo.GetWindowText(stName);

	for(int n = 0; n < m_ActionNameMap.GetSize(); n++)
	{
		if(m_ActionNameMap[n].szName == stName)
			return m_ActionNameMap[n].Action;
	}

	return TRAYACTIONS::Nothing;
}

void CPropPage_MouseAction::LoadActionMap()
{
	TrayAction trayAction;
	m_ActionNameMap.RemoveAll();

	trayAction.Action = TRAYACTIONS::Nothing;
	trayAction.szName = i18n("Nothing");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::ShowWindow;
	trayAction.szName = i18n("Show Window");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::CheckMail;
	trayAction.szName = i18n("Check Mail");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::PopUpMenu;
	trayAction.szName = i18n("PopUp Menu");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::RunEmailClient;
	trayAction.szName = i18n("Run Email Client");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::CheckAndShow;
	trayAction.szName = i18n("Check And Show");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::ToggleWindow;
	trayAction.szName = i18n("Toggle Window");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::ToggleAutoCheck;
	trayAction.szName = i18n("Toggle AutoCheck");
	m_ActionNameMap.Add(trayAction);

	trayAction.Action = TRAYACTIONS::ShowMailInfo;
	trayAction.szName = i18n("Show Mail Info");
	m_ActionNameMap.Add(trayAction);
}


BEGIN_MESSAGE_MAP(CPropPage_MouseAction, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_MouseAction)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()


BOOL CPropPage_MouseAction::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
		
	InitGUIText();

	return TRUE;  
}

void CPropPage_MouseAction::InitGUIText()
{
	LoadActionMap();

	FillComboBox(m_cboLeft);
	FillComboBox(m_cboRight);
	FillComboBox(m_cboMiddle);
	FillComboBox(m_cboDouble);
	
	SelectAction(m_cboLeft,   m_actionLeftClick);
	SelectAction(m_cboRight,  m_actionRightClick);
	SelectAction(m_cboMiddle, m_actionMiddleClick);
	SelectAction(m_cboDouble, m_actionDoubleClick);


	SetDlgItemText(IDC_STATIC_TRAY,			i18n("Tray Mouse Actions"));
	SetDlgItemText(IDC_STATIC_LEFT,			i18n("Left Click:"));
	SetDlgItemText(IDC_STATIC_RIGHT,		i18n("Right Click:"));
	SetDlgItemText(IDC_STATIC_MIDDLE,		i18n("Middle Click:"));
	SetDlgItemText(IDC_STATIC_DOUBLE,		i18n("Double Click:"));
	SetDlgItemText(IDC_CHECK_CLICK_DELAY,	i18n("Double click delay on click"));

	ArrangeControlPair(this, IDC_STATIC_TRAY, IDC_STATIC_TRAY_LINE);
}



/////////////////////////////////////////////////////////////////////////////
// Property page CPropPage_Language 
/////////////////////////////////////////////////////////////////////////////
IMPLEMENT_DYNCREATE(CPropPage_Language, CPropertyPage)

CPropPage_Language::CPropPage_Language(LPCTSTR szTitle) : CPropertyPage(CPropPage_Language::IDD)
{
	if(szTitle)
	{
		if(szTitle == GetLastOutputStr()) szTitle = GetLastInputStr();
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CPropPage_Language)
	//}}AFX_DATA_INIT
}

CPropPage_Language::~CPropPage_Language()
{
}

void CPropPage_Language::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPropPage_Language)
	//}}AFX_DATA_MAP
	DDX_Control(pDX, IDC_COMBO_LANGUAGES, m_cboLang);
}

BOOL CPropPage_Language::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	InitGUIText();

	if(CPopManApp::GetInstalledLanguages(m_Items))
	{
		SortLanguages();
		
		m_cboLang.ResetContent();
		for(int n = 0; n < m_Items.GetSize(); n++)
		{
			const LANGUAGE& Item = m_Items[n];
			m_cboLang.AddString(Item.stName);
		}

		CString stLangFile = GetCurrentLangFile();
		int Idx = stLangFile.ReverseFind(_T('\\'));
		stLangFile = stLangFile.Mid(Idx+1);
		stLangFile.MakeLower();

		for(int i = 0; i < m_Items.GetSize(); i++)
		{
			const LANGUAGE& Item = m_Items[i];
			CString stFile = Item.stFile;
			stFile.MakeLower();
			if(stFile == stLangFile)
			{
				m_cboLang.SelectString(-1, Item.stName);	
				ShowLangInfo(Item);
				break;
			}
		}
	}

	m_LinkEmail.SubclassDlgItem(IDC_STATIC_EMAIL_VALUE, this);
	m_LinkWeb.SubclassDlgItem(IDC_STATIC_WEBSITE, this);
	
	return TRUE;  
}

BEGIN_MESSAGE_MAP(CPropPage_Language, CPropertyPage)
	//{{AFX_MSG_MAP(CPropPage_Language)
	//}}AFX_MSG_MAP
	ON_CBN_SELCHANGE(IDC_COMBO_LANGUAGES, OnSelectLanguage)
	ON_COMMAND(ID_APP_UPDATE_GUI, InitGUIText)
END_MESSAGE_MAP()


void CPropPage_Language::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_LANG_INTRO,	i18n("Below you can choose between the translations currently on your system.\r\nIf you don't find your language, please visit the PopMan web site:"));
	SetDlgItemText(IDC_STATIC_LANG,			i18n("Language:"));
	SetDlgItemText(IDC_STATIC_AUTHOR,		i18n("Author:"));
	SetDlgItemText(IDC_STATIC_EMAIL,		i18n("Email:"));
	SetDlgItemText(IDC_STATIC_UPDATED,		i18n("Updated:"));
	SetDlgItemText(IDC_STATIC_COMMENT,		i18n("Comment:"));
}


void CPropPage_Language::SortLanguages()
{
	long k, i, j;
	LANGUAGE h;
	k = m_Items.GetSize() / 2;
	while(k > 0)
	{
		for(i = 0; i < m_Items.GetSize() - k; i++)
		{
			j = i;
			while(j >= 0 && m_Items[j].stName > m_Items[j+k].stName)
			{
				h = m_Items[j];
				m_Items[j] = m_Items[j+k];
				m_Items[j+k] = h;
				j -= k;
			}
		}
		k /= 2;
	}
}

void CPropPage_Language::ShowLangInfo(const LANGUAGE& Lang)
{
	SetDlgItemText(IDC_STATIC_AUTHOR_VALUE, Lang.stAuthor);
	SetDlgItemText(IDC_STATIC_EMAIL_VALUE, Lang.stEmail);
	SetDlgItemText(IDC_STATIC_UPDATED_VALUE, Lang.stUpdated);
	SetDlgItemText(IDC_STATIC_COMMENT_VALUE, Lang.stComment);
	m_LinkEmail.m_stLink = _T("mailto:") + Lang.stEmail;
}

void CPropPage_Language::OnSelectLanguage()
{
	CString stLang;
	m_cboLang.GetWindowText(stLang);
	
	for(int i = 0; i < m_Items.GetSize(); i++)
	{
		const LANGUAGE& Item = m_Items[i];
		if(Item.stName == stLang)
		{
			ShowLangInfo(Item);
			CString stLangFile;
			if(!Item.stFile.IsEmpty())
			{
				stLangFile = CPopManApp::GetAppLanguagePath();
				if(stLangFile.Right(1) != _T("\\")) stLangFile += _T('\\');
				stLangFile += Item.stFile;
			}
			LoadLanguageFile(stLangFile);
			GetParentOwner()->SendMessage(WM_COMMAND, ID_APP_UPDATE_GUI, 0);
			GetParentOwner()->SendMessageToDescendants(WM_COMMAND, ID_APP_UPDATE_GUI, 0);
			break;
		}
	}
}



