// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// DlgAccount.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "PopMan.h"
#include "DlgAccount.h"
#include <afxdlgs.h>
#include "Account.h"
#include "PopManDoc.h"
#include "PluginAccount.h"
#include "POP3Account.h"
#include "StrFunctions.h"
#include "DialogHelper.h"
#include <typeinfo.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//  i18nComment("Account Dialog")

/////////////////////////////////////////////////////////////////////////////
// Dialogfeld CDlgAccount 

IMPLEMENT_DYNAMIC(CDlgAccount, CPropertySheet)

CDlgAccount::CDlgAccount(CPopManDoc* pDoc, CWnd* pParentWnd, UINT iSelectPage)
	:CPropertySheet(i18n("Account"), pParentWnd, iSelectPage)
{
	m_psh.dwFlags |= PSH_NOAPPLYNOW;
	m_pDoc = pDoc;
}

CDlgAccount::~CDlgAccount()
{
}

int CDlgAccount::DoModal(CAccount*& pAcc, CProtocols& Protocols)
{
	if(pAcc == NULL)
		return IDCANCEL;

	while(GetPageCount()) RemovePage(0);

	CAcc_General AccGeneral(Protocols, i18n("General"));
	AccGeneral.m_stName = pAcc->m_stName;
	AccGeneral.m_stUser = pAcc->m_stUser;
	AccGeneral.m_stPass = pAcc->m_stPass;
	AccGeneral.m_stServer = pAcc->m_stServer;
	AccGeneral.m_nPort = pAcc->m_nPort;
	AccGeneral.m_bActive = pAcc->m_bActive;

	CString oldProtocolName = pAcc->GetProtocolName();
    CProtocol* pOldProtocol = NULL;

	if(typeid(*pAcc) == typeid(CPluginAccount))
    {
        CPluginAccount& pluginAccount = *dynamic_cast<CPluginAccount*>(pAcc);
		AccGeneral.m_stParameters = pluginAccount.GetParameters();
        pOldProtocol = &pluginAccount.GetProtocol();
    }

//	CProtocol* pOldProtocol = m_pDoc->GetProtocolFromName(pAcc->GetProtocolName());
	AccGeneral.m_stProtocolName = oldProtocolName;
	AccGeneral.m_pProtocol = pOldProtocol;
	AddPage(&AccGeneral);
	
	CAcc_Colors AccColors(i18n("Colors"));
	AccColors.m_bCustomColor = pAcc->m_bCustomColor;
	AccColors.m_CustomColor = pAcc->m_CustomColor;
	AccColors.m_TrayColor = pAcc->m_TrayColor;
	AddPage(&AccColors);

	CAcc_Ex AccEx(i18n("Advanced"));
	AccEx.m_nTimeOutVal = pAcc->GetTimeOut();
	//AccEx.m_bLimitLoadSize = pAcc->m_bLimitLoadSize;

    AccEx.m_RetrieveMode = (CAcc_Ex::ACC_RETRIEVE_MODE)pAcc->m_RetrieveMode;
    AccEx.m_nLines= pAcc->m_nLines;
    AccEx.m_nMaxSizeOfSmallMails = pAcc->m_nMaxSizeOfSmallMails;
    AccEx.m_nLinesBigMails = pAcc->m_nLinesBigMails;
    
	AccEx.m_bCustomAutoCheck = pAcc->m_bCustomAutoCheckInterval;
	AccEx.m_nAutoCheckInterval = pAcc->m_nCustomAutoCheckInterval;
	AccEx.m_bUseUIDLForDelete = pAcc->m_bUseUIDLForDelete;
	AddPage(&AccEx);
	

	int res = CPropertySheet::DoModal();

	if(res == IDOK)
	{
		if (pOldProtocol != AccGeneral.m_pProtocol || oldProtocolName != AccGeneral.m_stProtocolName)
		{
			CAccount* pOldAcc = pAcc;
			
			if (AccGeneral.m_pProtocol == NULL) {
				const bool tls = AccGeneral.m_stProtocolName == _T("POP3 TLS");
				pAcc = new CPOP3Account(m_pDoc, tls);
			}
			else
				pAcc = new CPluginAccount(m_pDoc, *AccGeneral.m_pProtocol);

			*pAcc = *pOldAcc;  // call standard assignment operator

			pAcc->UpdateMailAccReferenz(); // the emails have a new parent
			pOldAcc->m_Mails.RemoveAll();  // prevent emails' destruction

			delete pOldAcc;
		}

	
		pAcc->m_stName = AccGeneral.m_stName;
		pAcc->m_stUser = AccGeneral.m_stUser;
		pAcc->m_stPass = AccGeneral.m_stPass;
		pAcc->m_stServer = AccGeneral.m_stServer;
		pAcc->m_nPort = AccGeneral.m_nPort;
		pAcc->m_bActive = AccGeneral.m_bActive;
		
		if(typeid(*pAcc) == typeid(CPluginAccount))
			dynamic_cast<CPluginAccount*>(pAcc)->SetParameters(AccGeneral.m_stParameters);


		pAcc->m_bCustomColor = AccColors.m_bCustomColor;
		if(pAcc->m_bCustomColor) pAcc->m_CustomColor = AccColors.m_CustomColor;
		pAcc->m_TrayColor = AccColors.m_TrayColor;
		pAcc->SetTimeOut(AccEx.m_nTimeOutVal);

//		pAcc->m_bLimitLoadSize = AccEx.m_bLimitLoadSize;

        pAcc->m_RetrieveMode = (CAccount::RETRIEVE_MODE)AccEx.m_RetrieveMode;
        pAcc->m_nLines = AccEx.m_nLines;
        pAcc->m_nMaxSizeOfSmallMails = AccEx.m_nMaxSizeOfSmallMails;
        pAcc->m_nLinesBigMails = AccEx.m_nLinesBigMails;

		pAcc->m_bCustomAutoCheckInterval = AccEx.m_bCustomAutoCheck ;
		pAcc->m_nCustomAutoCheckInterval = AccEx.m_nAutoCheckInterval;
		pAcc->m_bUseUIDLForDelete = AccEx.m_bUseUIDLForDelete;
	}
	while(GetPageCount()) RemovePage(0);

	return res;
}


void CDlgAccount::DoDataExchange(CDataExchange* pDX) 
{
	CPropertySheet::DoDataExchange(pDX);

	if(m_pDoc->ConnectionsPending()) {
		AfxMessageBox(i18n("Can not change account settings while there are busy accounts."));
		pDX->Fail();
		return;
	}
}

BOOL CDlgAccount::OnInitDialog() 
{
	BOOL bResult = CPropertySheet::OnInitDialog();
	
	SetDlgItemText(IDOK, i18n("OK"));
	SetDlgItemText(IDCANCEL, i18n("Cancel"));
	
	return bResult;
}



BEGIN_MESSAGE_MAP(CDlgAccount, CPropertySheet)
	//{{AFX_MSG_MAP(CDlgAccount)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


///////////////////////////////////////////////////////////////////////////////////////
// CAcc_General


CAcc_General::CAcc_General(CProtocols& Protocols, LPCTSTR szTitle) : CPropertyPage(CAcc_General::IDD), m_Protocols(Protocols)
{
	if(szTitle)
	{
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}

	//{{AFX_DATA_INIT(CAcc_General)
	m_stName = _T("");
	m_stUser = _T("");
	m_stPass = _T("");
	m_stServer = _T("");
	m_nPort = 110;
	m_bActive = TRUE;
	//}}AFX_DATA_INIT
	m_pProtocol = NULL;
	m_currPlugin = NULL;
}

CAcc_General::~CAcc_General()
{
}


BOOL CAcc_General::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	InitGUIText();

	m_cboProtocols.AddString(_T("POP3"));
	m_cboProtocols.SetItemDataPtr(0, NULL);
	m_vectorProtocols.push_back(_T("POP3"));

	m_cboProtocols.AddString(_T("POP3 TLS"));
	m_cboProtocols.SetItemDataPtr(1, NULL);
	m_vectorProtocols.push_back(_T("POP3 TLS"));

	POSITION pos = m_Protocols.GetHeadPosition();
	while(pos != NULL)
	{
		CProtocol* Proto = m_Protocols.GetNext(pos);
		if(Proto == NULL) continue;
		m_cboProtocols.AddString(Proto->GetName());
		m_vectorProtocols.push_back(Proto->GetName());
		m_cboProtocols.SetItemDataPtr(m_cboProtocols.GetCount()-1, Proto);
	}

	CString proto = m_stProtocolName;
    
	if(CB_ERR == m_cboProtocols.SelectString(-1, proto) && m_pProtocol != NULL) 
    {
	    int idx = m_cboProtocols.AddString(m_pProtocol->GetName());
	    m_cboProtocols.SetItemDataPtr(idx, m_pProtocol);
        m_cboProtocols.SelectString(-1, m_pProtocol->GetName());
    }
	m_currPlugin = m_pProtocol ? m_pProtocol->GetPlugin() : NULL; 

	OnSelchangeComboProtocols();

	CString stPort;
	stPort.Format(_T("%d"), m_nPort);
	SetDlgItemText(IDC_ACCOUNT_PORT, stPort);

	OnCheckVisibilityGoogleSignIn();

	m_btnSignInGoogle.SetBitmap(LoadBitmap(AfxGetInstanceHandle(), MAKEINTRESOURCE(IDB_BITMAP_GOOGLE)));

	return TRUE; 
}

void CAcc_General::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAcc_General)
	DDX_Control(pDX, IDC_COMBO_PROTOCOLS, m_cboProtocols);
	DDX_Text(pDX, IDC_ACCOUNT_NAME, m_stName);
	DDX_Text(pDX, IDC_ACCOUNT_USER, m_stUser);
	DDX_Text(pDX, IDC_ACCOUNT_PASS, m_stPass);
	DDX_Text(pDX, IDC_ACCOUNT_SERVER, m_stServer);
	DDX_Check(pDX, IDC_CHECK_ACCOUNT_ACTIVE, m_bActive);
	//}}AFX_DATA_MAP

	if (!pDX->m_bSaveAndValidate)
		DDX_Text(pDX, IDC_ACCOUNT_PORT, m_nPort);
	else
	{
		m_pProtocol = GetCurrProtocol();

		if (m_pProtocol == NULL || m_pProtocol->NeedsServerAndPort()) {
			DDV_TextEmpty(pDX, IDC_ACCOUNT_SERVER, m_stServer, i18n("Please enter the address of the Server!"));
			DDX_Text(pDX, IDC_ACCOUNT_PORT, m_nPort);
			DDV_MinMaxUInt(pDX, m_nPort, 1, 65535);
		}

		DDV_TextEmpty(pDX, IDC_ACCOUNT_NAME, m_stName, i18n("Please enter the account name!"));
		DDV_TextEmpty(pDX, IDC_ACCOUNT_USER, m_stUser, i18n("Please enter user name!"));

		int Idx = m_cboProtocols.GetCurSel();
		if (Idx == CB_ERR)
			pDX->Fail();
		else {
			m_stProtocolName = m_vectorProtocols.at(Idx);
		}
	}
	DDX_Control(pDX, IDC_BUTTON_SIGN_IN, m_btnSignInGoogle);
}


void CAcc_General::DDV_TextEmpty(CDataExchange* pDX, int nIDC, CString& value, LPCTSTR szMessage)
{
	if(!pDX->m_bSaveAndValidate)
		return;
	
	if(value.IsEmpty())
	{
		AfxMessageBox(szMessage);
		pDX->PrepareEditCtrl(nIDC);
		pDX->Fail();	
	}
}


CProtocol* CAcc_General::GetCurrProtocol()
{
	int Idx = m_cboProtocols.GetCurSel();
	if(Idx == CB_ERR)
		return NULL;
	else
		return (CProtocol*)m_cboProtocols.GetItemDataPtr(Idx);
}


void CAcc_General::OnSelchangeComboProtocols() 
{
	CProtocol* pProtocol = GetCurrProtocol();
	
	int defaultPort = m_cboProtocols.GetCurSel() == 1 ? 995 : 110;
	if(pProtocol) defaultPort = pProtocol->GetDefaultPort();
	CString stPort;
	stPort.Format(_T("%d"), defaultPort);
	SetDlgItemText(IDC_ACCOUNT_PORT, stPort);

	bool bEnableServerPort = (pProtocol == NULL || pProtocol->NeedsServerAndPort());

	GetDlgItem(IDC_ACCOUNT_SERVER)->EnableWindow(bEnableServerPort);
	GetDlgItem(IDC_ACCOUNT_PORT)->EnableWindow(bEnableServerPort);
	GetDlgItem(IDC_STATIC_SERVER)->EnableWindow(bEnableServerPort);
	GetDlgItem(IDC_STATIC_PORT)->EnableWindow(bEnableServerPort);


	if(!bEnableServerPort)	{	
		GetDlgItemText(IDC_ACCOUNT_SERVER, m_stSavedServer);
		SetDlgItemText(IDC_ACCOUNT_SERVER, _T(""));
		SetDlgItemText(IDC_ACCOUNT_PORT,   _T(""));
	}
	else {
		if(!m_stSavedServer.IsEmpty())
			SetDlgItemText(IDC_ACCOUNT_SERVER, m_stSavedServer);
	}

	bool bParams = (pProtocol && pProtocol->SupportsParameters());

	if(bParams) {
		if(m_currPlugin != pProtocol->GetPlugin())
			m_stParameters = pProtocol->GetDefaultParameters();
	}
	else
		m_stParameters = "";

	m_currPlugin = pProtocol ? pProtocol->GetPlugin() : NULL; 

	DWORD showhide = bParams ? SW_SHOW : SW_HIDE;
	GetDlgItem(IDC_BUTTON_OPTIONS)->ShowWindow(showhide);
}


void CAcc_General::OnButtonOptions() 
{
	CDlgParameters dlgParams(*GetCurrProtocol(), this);
	dlgParams.m_stParameters = m_stParameters;
	if(dlgParams.DoModal() == IDOK) {
		m_stParameters = dlgParams.m_stParameters;
	}
}

void CAcc_General::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_NAME,				i18n("Account name:"));
	SetDlgItemText(IDC_STATIC_PROTOCOL,			i18n("Protocol:"));
	SetDlgItemText(IDC_STATIC_SERVER,			i18n("Server:"));
	SetDlgItemText(IDC_STATIC_PORT,				i18n("Port:"));
	SetDlgItemText(IDC_STATIC_USER,				i18n("User name:"));
	SetDlgItemText(IDC_STATIC_PASS,				i18n("Password:"));
	SetDlgItemText(IDC_CHECK_ACCOUNT_ACTIVE,	i18n("Include this account when receiving mail"));
}

#include "OAuth2.h"

#define WM_CHANGE_TO_UI_THREAD (WM_USER+36)

struct ThreadInfo {
	//HANDLE hChildStd_OUT_Rd;
	HWND hWnd;
	CString user;
	UINT port;
};

//UINT ReadFromPipe(LPVOID lParam);

UINT DoHttp(LPVOID lParam) {

	ThreadInfo* pInfo = (ThreadInfo*)lParam;
	ThreadInfo info = *pInfo;
	delete pInfo;
	pInfo = NULL;

	CString res = DoOAuth(info.user, info.port);

	SendMessage(info.hWnd, WM_CHANGE_TO_UI_THREAD, 0, (LPARAM)&res);
	
	return 0;
}

static UINT DoGetFreePort() {
	CAsyncSocket socket;
	socket.Create(0, SOCK_STREAM, FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE, _T("127.0.0.1"));
	socket.Listen();
	CString s;
	UINT freePort = 0;
	socket.GetSockName(s, freePort);
	socket.Close();
	return freePort;
}

void CAcc_General::OnBnClickedButtonSignIn() {

	CString user;
	GetDlgItemText(IDC_ACCOUNT_USER, user);

	const UINT freePort = DoGetFreePort();

	auto pInfo = new ThreadInfo();
	pInfo->hWnd = this->GetSafeHwnd();
	//pInfo->hChildStd_OUT_Rd = 0;
	pInfo->user = user;
	pInfo->port = freePort;

	AfxBeginThread(DoHttp, pInfo);
}

LRESULT CAcc_General::OnGotSignInResult(WPARAM wParam, LPARAM lParam) {

	CString& str = *(CString*)lParam;
	//CString& strUTF8 = *(CString*)lParam;
	//CString str = DecodeUTF8(strUTF8);

	str.Trim();

	const CString tokenLeft = _T("<token>");
	const CString tokenRight = _T("</token>");

	if (str.Left(tokenLeft.GetLength()) == tokenLeft && str.Right(tokenRight.GetLength()) == tokenRight) {
		CString token = str.Mid(tokenLeft.GetLength(), str.GetLength() - tokenLeft.GetLength() - tokenRight.GetLength());
		CString pass = OAuthPassPrefix + token;
		SetDlgItemText(IDC_ACCOUNT_PASS, pass);
		MessageBox(i18n("Sign-in completed successfully."), i18n("Result"), MB_OK | MB_ICONINFORMATION);
		return 0;
	}
	
	const CString errLeft = _T("<error>");
	const CString errRight = _T("</error>");
	
	if (str.Left(errLeft.GetLength()) == errLeft && str.Right(errRight.GetLength()) == errRight) {
		CString err = str.Mid(errLeft.GetLength(), str.GetLength() - errLeft.GetLength() - errRight.GetLength());
		MessageBox(err, i18n("Error"), MB_OK | MB_ICONERROR);
		return 0;
	}

	MessageBox(i18n("Sign-in failed."), i18n("Error"), MB_OK | MB_ICONERROR);

	return 0; 
}

void CAcc_General::OnCheckVisibilityGoogleSignIn()
{
	CString server;
	GetDlgItemText(IDC_ACCOUNT_SERVER, server);
	server.MakeLower();

	if (server.Find(_T(".gmail.")) >= 0) {
		GetDlgItem(IDC_BUTTON_SIGN_IN)->ShowWindow(SW_SHOW);
	}
	else {
		GetDlgItem(IDC_BUTTON_SIGN_IN)->ShowWindow(SW_HIDE);
	}

	CString user;
	GetDlgItemText(IDC_ACCOUNT_USER, user);
	if (user.Find(_T("@")) > 0) {
		GetDlgItem(IDC_BUTTON_SIGN_IN)->EnableWindow(1);
	}
	else {
		GetDlgItem(IDC_BUTTON_SIGN_IN)->EnableWindow(0);
	}
}

BEGIN_MESSAGE_MAP(CAcc_General, CPropertyPage)
	//{{AFX_MSG_MAP(CAcc_General)
	ON_CBN_SELCHANGE(IDC_COMBO_PROTOCOLS, OnSelchangeComboProtocols)
	ON_BN_CLICKED(IDC_BUTTON_OPTIONS, OnButtonOptions)
	//}}AFX_MSG_MAP
	ON_BN_CLICKED(IDC_BUTTON_SIGN_IN, &CAcc_General::OnBnClickedButtonSignIn)
	ON_EN_CHANGE(IDC_ACCOUNT_SERVER, &CAcc_General::OnCheckVisibilityGoogleSignIn)
	ON_EN_CHANGE(IDC_ACCOUNT_USER, &CAcc_General::OnCheckVisibilityGoogleSignIn)
	ON_MESSAGE(WM_CHANGE_TO_UI_THREAD, &CAcc_General::OnGotSignInResult)
END_MESSAGE_MAP()



///////////////////////////////////////////////////////////////////////////////////////
// CAcc_Ex

IMPLEMENT_DYNCREATE(CAcc_Ex, CPropertyPage)

CAcc_Ex::CAcc_Ex(LPCTSTR szTitle) : CPropertyPage(CAcc_Ex::IDD)
{
	if(szTitle)
	{
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}
	//{{AFX_DATA_INIT(CAcc_Ex)
	m_nLines = 200;
	m_bCustomAutoCheck = FALSE;
	m_nAutoCheckInterval = 0;
	m_bUseUIDLForDelete = FALSE;
	m_nLinesBigMails = 200;
	m_nMaxSizeOfSmallMails = 100;
	//}}AFX_DATA_INIT
	m_nTimeOutVal = 30;
}

BOOL CAcc_Ex::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	UpdateTimeOut();

	m_spinLines.SetBase(10);
	m_spinLines.SetRange(0, 20000);
	m_spinLines.SetBuddy(GetDlgItem(IDC_EDIT_LINES));

	m_spinInterval.SetBase(10);
	m_spinInterval.SetRange(0, 10000);
	m_spinInterval.SetBuddy(GetDlgItem(IDC_EDIT_INTERVAL));

    m_spinMaxSizeOfSmallMails.SetBase(10);
	m_spinMaxSizeOfSmallMails.SetRange(1, 0x7FFF);
	m_spinMaxSizeOfSmallMails.SetBuddy(GetDlgItem(IDC_EDIT_MAX_SIZE_RETR));

    m_spinLinesBigMails.SetBase(10);
	m_spinLinesBigMails.SetRange(1, 0x7FFF);
	m_spinLinesBigMails.SetBuddy(GetDlgItem(IDC_EDIT_LINES_BIG_MAILS));

	OnLimitLoadMails();
	InitGUIText();
	OnCheckCustomAutocheck();

	return TRUE;
}

CAcc_Ex::~CAcc_Ex()
{
}

void CAcc_Ex::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAcc_Ex)
	DDX_Control(pDX, IDC_EDIT_LINES_BIG_MAILS, m_editLinesBigMails);
	DDX_Control(pDX, IDC_EDIT_MAX_SIZE_RETR, m_editMaxSizeOfSmallMails);
	DDX_Control(pDX, IDC_SPIN_LINES_SMALL_SIZE, m_spinMaxSizeOfSmallMails);
	DDX_Control(pDX, IDC_SPIN_LINES_BIG_MAILs, m_spinLinesBigMails);
	DDX_Control(pDX, IDC_SPIN_INTERVAL, m_spinInterval);
	DDX_Control(pDX, IDC_SPIN_LINES, m_spinLines);
	DDX_Control(pDX, IDC_EDIT_LINES, m_editLines);
	DDX_Control(pDX, IDC_SLIDER_TIMEOUT, m_Slider);
	DDX_Text(pDX, IDC_EDIT_LINES, m_nLines);
	DDV_MinMaxUInt(pDX, m_nLines, 0, 10000);
	DDX_Check(pDX, IDC_CHECK_CUSTOM_AUTOCHECK, m_bCustomAutoCheck);
	DDX_Text(pDX, IDC_EDIT_INTERVAL, m_nAutoCheckInterval);
	DDX_Check(pDX, IDC_CHECK_UIDL_DELETE, m_bUseUIDLForDelete);
	DDX_Text(pDX, IDC_EDIT_LINES_BIG_MAILS, m_nLinesBigMails);
	DDX_Text(pDX, IDC_EDIT_MAX_SIZE_RETR, m_nMaxSizeOfSmallMails);
	//}}AFX_DATA_MAP

    RadioMapping<ACC_RETRIEVE_MODE> mapping[] = 
    {
        RadioMapping<ACC_RETRIEVE_MODE>(IDC_RADIO_LOAD_HEADER,   ACC_RETRIEVE_MODE::AllHeaderOnly),
        RadioMapping<ACC_RETRIEVE_MODE>(IDC_RADIO_LOAD_SMALL,    ACC_RETRIEVE_MODE::SmallComplete),
        RadioMapping<ACC_RETRIEVE_MODE>(IDC_RADIO_LOAD_COMPLETE, ACC_RETRIEVE_MODE::AllComplete)
    };
    DDX_MapRadio(pDX, m_RetrieveMode, mapping, sizeof(mapping)/sizeof(mapping[0]));


	if(pDX->m_bSaveAndValidate) 
    {
		m_nTimeOutVal = m_Slider.GetPos();
	}
	else
	{
		m_Slider.SetLineSize(10);
		m_Slider.SetPageSize(60);
		m_Slider.SetRange(10, 3600, TRUE);
		m_Slider.SetPos(m_nTimeOutVal);
		UpdateTimeOut();
	}
}

BEGIN_MESSAGE_MAP(CAcc_Ex, CPropertyPage)
	//{{AFX_MSG_MAP(CAcc_Ex)
	ON_BN_CLICKED(IDC_RADIO_LOAD_HEADER, OnLimitLoadMails)
	ON_EN_CHANGE(IDC_EDIT_LINES, OnChangeEditLines)
	ON_BN_CLICKED(IDC_CHECK_CUSTOM_AUTOCHECK, OnCheckCustomAutocheck)
	ON_BN_CLICKED(IDC_RADIO_LOAD_COMPLETE, OnLimitLoadMails)
	ON_BN_CLICKED(IDC_RADIO_LOAD_SMALL, OnLimitLoadMails)
	//}}AFX_MSG_MAP
	ON_WM_HSCROLL()
END_MESSAGE_MAP()



void CAcc_Ex::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar)
{
	switch(nSBCode)
	{
	case TB_BOTTOM:
	case TB_ENDTRACK:
	case TB_LINEDOWN:
	case TB_LINEUP:
	case TB_PAGEDOWN:
	case TB_PAGEUP:
	case TB_THUMBPOSITION:
	case TB_THUMBTRACK:
	case TB_TOP :
			
		UpdateTimeOut();
	}
}

void CAcc_Ex::UpdateTimeOut()
{
	int nPos = m_Slider.GetPos();

	int nMin = nPos / 60;
	int nSec = nPos - nMin * 60;

	CString stTime;
	if(nMin > 0)
	{
		if(nMin == 1)
			stTime = i18n("1 minute");
		else
			stTime = StrFormat(i18n("{1} minutes"), _T("d"), nMin);
	}

	if(nMin > 0 && nSec > 0)
		stTime += _T(", ");

	if(nSec > 0)
	{
		CString stFormat;
		if(nSec == 1)
			stFormat = i18n("1 second");
		else
			stFormat = StrFormat(i18n("{1} seconds"), _T("d"), nSec);

		stTime += stFormat;
	}

	GetDlgItem(IDC_STATIC_TIMEOUT)->SetWindowText(stTime);
}

void CAcc_Ex::OnLimitLoadMails() 
{
	BOOL bUpdateEnabled = (((CButton*)GetDlgItem(IDC_RADIO_LOAD_HEADER))->GetCheck() == 1);
	m_editLines.EnableWindow(bUpdateEnabled);
	m_spinLines.EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_MESSAGE_LINES))->EnableWindow(bUpdateEnabled);

    bUpdateEnabled = (((CButton*)GetDlgItem(IDC_RADIO_LOAD_SMALL))->GetCheck() == 1);
	m_editLinesBigMails.EnableWindow(bUpdateEnabled);
    m_editMaxSizeOfSmallMails.EnableWindow(bUpdateEnabled);
	m_spinLinesBigMails.EnableWindow(bUpdateEnabled);
    m_spinMaxSizeOfSmallMails.EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_MESSAGE_SIZE_SMALL))->EnableWindow(bUpdateEnabled);
    ((CWnd*)GetDlgItem(IDC_STATIC_MESSAGE_LINES_BIG))->EnableWindow(bUpdateEnabled);
}

void CAcc_Ex::OnCheckCustomAutocheck() 
{
	BOOL bUpdateEnabled = (((CButton*)GetDlgItem(IDC_CHECK_CUSTOM_AUTOCHECK))->GetCheck() == 1);
	m_spinInterval.EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_EDIT_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_INTERVAL))->EnableWindow(bUpdateEnabled);
	((CWnd*)GetDlgItem(IDC_STATIC_MINUTES))->EnableWindow(bUpdateEnabled);
}


void CAcc_Ex::InitGUIText()
{
	SetDlgItemText(IDC_CHECK_UIDL_DELETE,			i18n("Use UIDL when deleting messages"));

	SetDlgItemText(IDC_CHECK_CUSTOM_AUTOCHECK,		i18n("Use custom AutoCheck timer interval"));
	SetDlgItemText(IDC_STATIC_INTERVAL,				i18n("Timer interval:"));
	SetDlgItemText(IDC_STATIC_MINUTES,				i18n("minutes"));

	SetDlgItemText(IDC_STATIC_TIMEOUTS,				i18n("Server Timeouts"));
	SetDlgItemText(IDC_STATIC_SHORT,				i18n("Short"));
	SetDlgItemText(IDC_STATIC_LONG,					i18n("Long"));
	SetDlgItemText(IDC_RADIO_LOAD_COMPLETE,			i18n("Load all emails completely"));
	CString stLines;
	GetDlgItemText(IDC_EDIT_LINES, stLines);
	SetDlgItemText(IDC_RADIO_LOAD_HEADER,			StrFormat(i18n("Retrieve header and {1} lines of message body"), _T("s"), (LPCTSTR)stLines));
	SetDlgItemText(IDC_STATIC_MESSAGE_LINES,		i18n("Number of lines:"));

    SetDlgItemText(IDC_RADIO_LOAD_SMALL,			i18n("Load small emails completely"));
    SetDlgItemText(IDC_STATIC_MESSAGE_SIZE_SMALL,	i18n("Small emails are < "));
    SetDlgItemText(IDC_STATIC_MESSAGE_LINES_BIG,	i18n("Number of lines:"));

}


void CAcc_Ex::OnChangeEditLines() 
{
	InitGUIText();
}




///////////////////////////////////////////////////////////////////////////////////////
// CAcc_Colors


IMPLEMENT_DYNCREATE(CAcc_Colors, CPropertyPage)

CAcc_Colors::CAcc_Colors(LPCTSTR szTitle) : CPropertyPage(CAcc_Colors::IDD)
{
	if(szTitle)
	{
		m_psp.pszTitle = szTitle;
		m_psp.dwFlags |= PSP_USETITLE;
	}

	//{{AFX_DATA_INIT(CAcc_Colors)
	m_bCustomColor = FALSE;
	//}}AFX_DATA_INIT
	m_CustomColor = GetSysColor(COLOR_WINDOWTEXT);
	m_TrayColor = 0;
}

CAcc_Colors::~CAcc_Colors()
{
}


BOOL CAcc_Colors::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	OnCustomColor();
	InitGUIText();

	return TRUE;  
}

void CAcc_Colors::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAcc_Colors)
	DDX_Control(pDX, IDC_CHECK_ACCOUNT_CUSTOM_COLOR, m_chkCustomColor);
	DDX_Check(pDX, IDC_CHECK_ACCOUNT_CUSTOM_COLOR, m_bCustomColor);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CAcc_Colors, CPropertyPage)
	//{{AFX_MSG_MAP(CAcc_Colors)
	ON_BN_CLICKED(IDC_BUTTON_COLOR_TRAY, OnColorTray)
	ON_BN_CLICKED(IDC_CHECK_ACCOUNT_CUSTOM_COLOR, OnCustomColor)
	ON_WM_DRAWITEM()
	ON_BN_CLICKED(IDC_BUTTON_COLOR, OnChooseCustomColor)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


void CAcc_Colors::OnColorTray() 
{
	CColorDialog cDlg(m_TrayColor);
	if(IDOK != cDlg.DoModal())
		return;

	m_TrayColor = cDlg.GetColor();

	GetDlgItem(IDC_BUTTON_COLOR_TRAY)->Invalidate();
	
}


void CAcc_Colors::OnCustomColor() 
{
	GetDlgItem(IDC_BUTTON_COLOR)->Invalidate();
}


void CAcc_Colors::OnChooseCustomColor() 
{
	if(1 != m_chkCustomColor.GetCheck())
		return;

	CColorDialog cDlg(m_CustomColor);
	if(IDOK != cDlg.DoModal())
		return;

	m_CustomColor = cDlg.GetColor();

	GetDlgItem(IDC_BUTTON_COLOR)->Invalidate();
}


void CAcc_Colors::OnDrawItem(int nIDCtl, LPDRAWITEMSTRUCT lpDrawItemStruct) 
{
	if(nIDCtl == IDC_BUTTON_COLOR || nIDCtl == IDC_BUTTON_COLOR_TRAY)
	{
		CDC dc;
		dc.Attach(lpDrawItemStruct->hDC);

		if(nIDCtl == IDC_BUTTON_COLOR)
			dc.FillSolidRect(&lpDrawItemStruct->rcItem, (1 != m_chkCustomColor.GetCheck()) ? GetSysColor(COLOR_3DLIGHT) : m_CustomColor);
		else
			dc.FillSolidRect(&lpDrawItemStruct->rcItem, m_TrayColor);

		dc.Draw3dRect(&lpDrawItemStruct->rcItem, GetSysColor(COLOR_3DHILIGHT), GetSysColor(COLOR_3DSHADOW));
		dc.Detach();
	}

	CDialog::OnDrawItem(nIDCtl, lpDrawItemStruct);

}


void CAcc_Colors::InitGUIText()
{
	SetWindowText(i18n("Parameters"));
	SetDlgItemText(IDC_STATIC_COLORS,				i18n("Colors"));
	SetDlgItemText(IDC_CHECK_ACCOUNT_CUSTOM_COLOR,	i18n("Use individual font color:"));
	SetDlgItemText(IDC_STATIC_TRAY_COLOR,			i18n("Circle color in system tray:"));
}



/////////////////////////////////////////////////////////////////////////////
// CDlgParameters 

CDlgParameters::CDlgParameters(const CProtocol& Proto, CWnd* pParent /*=NULL*/)
	: CDialog(CDlgParameters::IDD, pParent), m_Protocol(Proto)
{
	//{{AFX_DATA_INIT(CDlgParameters)
	m_stParameters = _T("");
	//}}AFX_DATA_INIT
}

BOOL CDlgParameters::OnInitDialog() 
{
	CDialog::OnInitDialog();
	
	InitGUIText();	

	return TRUE;
}

void CDlgParameters::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CDlgParameters)
	DDX_Text(pDX, IDC_EDIT_PARAMS, m_stParameters);
	//}}AFX_DATA_MAP

	if(pDX->m_bSaveAndValidate && !m_Protocol.ValidateParameters(m_stParameters))
	{
		AfxMessageBox(i18n("Invalid parameter string!"));
		pDX->PrepareEditCtrl(IDC_EDIT_PARAMS);
		pDX->Fail();
	}
}

BEGIN_MESSAGE_MAP(CDlgParameters, CDialog)
	//{{AFX_MSG_MAP(CDlgParameters)
	ON_BN_CLICKED(ID_DEFAULT, OnDefault)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

void CDlgParameters::OnDefault() 
{
	SetDlgItemText(IDC_EDIT_PARAMS, m_Protocol.GetDefaultParameters());
}


void CDlgParameters::InitGUIText()
{
	SetDlgItemText(IDOK,		i18n("OK"));
	SetDlgItemText(IDCANCEL,	i18n("Cancel"));
	SetDlgItemText(ID_DEFAULT,	i18n("Default"));
}
