// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// DlgAccImport.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "Plugin.h"
#include "DlgAccImport.h"
#include "RegKey.h"
#include "Account.h"
#include "POP3Account.h"
#include "DlgAccount.h"
#include "PopManDoc.h"
#include "PluginAccount.h"
#include "StrFunctions.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//  i18nComment("Import Accounts Dialog")

CDlgAccImport::CDlgAccImport(CPopManDoc* pDoc, BOOL bPreSelect, CWnd* pParent /*=NULL*/)
	: CDialog(CDlgAccImport::IDD, pParent)
{
	m_pDoc = pDoc;
	m_bPreSelect = bPreSelect;
	//{{AFX_DATA_INIT(CDlgAccImport)
	//}}AFX_DATA_INIT
}


void CDlgAccImport::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CDlgAccImport)
	DDX_Control(pDX, IDC_STATIC_ICON, m_staticIcon);
	DDX_Control(pDX, IDC_LIST_IMPORT_ACCOUNTS, m_AccList);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CDlgAccImport, CDialog)
	//{{AFX_MSG_MAP(CDlgAccImport)
	ON_NOTIFY(NM_DBLCLK, IDC_LIST_IMPORT_ACCOUNTS, OnDblclkListAccounts)
	ON_BN_CLICKED(IDC_ACCOUNT_IMPORT_PROPERTIES, OnAccountProperties)
	//}}AFX_MSG_MAP
	ON_NOTIFY(LVN_ITEMCHANGED, IDC_LIST_IMPORT_ACCOUNTS, OnItemChanged)
END_MESSAGE_MAP()


BOOL CDlgAccImport::OnInitDialog() 
{
	CDialog::OnInitDialog();
	CString stText;

	m_AccList.SetExtendedStyle(LVS_EX_CHECKBOXES | LVS_EX_FULLROWSELECT );
	
	m_AccList.InsertColumn(0, i18n("Account"), LVCFMT_LEFT, 110, 0);
    m_AccList.InsertColumn(1, i18n("Type"), LVCFMT_LEFT, 110, 0);
	m_AccList.InsertColumn(2, i18n("User"), LVCFMT_LEFT, 160, 1);
	m_AccList.InsertColumn(3, i18n("Server"), LVCFMT_LEFT, 104, 2);
	m_AccList.InsertColumn(4, i18n("Port"), LVCFMT_RIGHT, 40, 3);

    m_pDoc->GetAvailableProtocols(m_Protocols);
	
	LoadAccounts();

	GetDlgItem(IDC_ACCOUNT_IMPORT_PROPERTIES)->EnableWindow(m_AccList.GetSelectedCount() == 1);

	m_staticIcon.SetIcon(CPopManApp::GetIcon(IDI_OPEN_FOLDER));

	InitGUIText();

	return TRUE;  
}


int CDlgAccImport::LoadAccounts(BOOL bCountOnly)
{
    return LoadAccountsOE(bCountOnly) + LoadAccountsWindowsMail(bCountOnly);
}

int CDlgAccImport::LoadAccountsOE(BOOL bCountOnly)
{
	CRegistryKey Reg;
	const CString stPath = _T("Software\\Microsoft\\Internet Account Manager\\Accounts");

	if(!Reg.Open(HKEY_CURRENT_USER, stPath, KEY_READ))
		return 0;

	CString stName;
	
	int nAccCount = 0;
	for(int n = 0; ; n++)
	{
		if(!Reg.EnumKey(n, stName))
			break;
		
		CRegistryKey RegAcc;
		if(!RegAcc.Open(HKEY_CURRENT_USER, stPath + _T("\\") + stName, KEY_READ))
			continue;

        BOOL bImap = FALSE;
		CString stAccName;
		CString stServer;
		CString stUserName;
		DWORD dwPort = 0;

		if(!RegAcc.QueryValue(_T("Account Name"), stAccName))
			continue;

		if(RegAcc.QueryValue(_T("POP3 Server"), stServer))
        {
		    if(!RegAcc.QueryValue(_T("POP3 User Name"), stUserName))
			    continue;

		    if(!RegAcc.QueryValue(_T("POP3 Port"), dwPort))
			    dwPort = 110;
        }
        else
        {
            if(!RegAcc.QueryValue(_T("IMAP Server"), stServer))
			    continue;

            if(!RegAcc.QueryValue(_T("IMAP User Name"), stUserName))
			    continue;

		    if(!RegAcc.QueryValue(_T("IMAP Port"), dwPort))
			    dwPort = 143;

            bImap = TRUE;
        }

		if(bCountOnly == FALSE)
			AddAccount(bImap, stAccName, stUserName, stServer, dwPort);

		nAccCount++;
	}

	return nAccCount;
}

CString ReadUtf16TextFile(const CString& file) 
{
    CString data;
    try 
    {
        CFile File;

        if(!File.Open(file, CFile::shareDenyNone | CFile::modeRead | CFile::typeBinary))
            return data;

        const unsigned int nFileSize = static_cast<unsigned int>(File.SeekToEnd());
        File.Seek(0, CFile::begin);

#ifndef _UNICODE     
        char* pBuffer = new char[nFileSize];

		File.Read(pBuffer, nFileSize);
        File.Close();

        const int bufferSize = nFileSize/2 + 1;
        LPSTR pDest = data.GetBuffer(bufferSize);
        
        WideCharToMultiByte(CP_ACP, 0, (LPCWSTR)pBuffer, nFileSize/2, pDest, bufferSize, NULL, NULL);
        data.ReleaseBuffer(bufferSize-1);
		delete[] pBuffer;
#else
		// TODO: Unicode
		const int bufferSize = nFileSize / 2 + 1;

		LPWSTR pDest = data.GetBuffer(bufferSize);

		UINT read = File.Read(pDest, nFileSize);
		File.Close();
		data.ReleaseBuffer(bufferSize - 1);

#endif
        
    }
	catch(CFileException* e)
	{
		e->Delete();
	}

    return data;
}

bool GetXMLTagContent(const CString& data, const CString& tag, CString& content) 
{
    CString leftPattern = _T("<");
    leftPattern += tag;
    leftPattern += _T(" ");

    int s = FindNoCase(data, leftPattern);
    if(s < 0)
        return false;

    s += leftPattern.GetLength();

    while(s < data.GetLength() && data[s] != _T('>')) 
    {
        s++;
    }

    s++;

    if(s >= data.GetLength())
        return false;

    int e = s;
    while(e < data.GetLength() && data[e] != _T('<')) 
    {
        e++;
    }

    if(e >= data.GetLength())
        return false;

    content = data.Mid(s, e-s);

    content.Replace(_T("&lt;"), _T("<"));
    content.Replace(_T("&gt;"), _T(">"));
    content.Replace(_T("&quot;"), _T("\""));
    content.Replace(_T("&amp;"), _T("&"));

    return true;
}

bool IsImapWinMailFile(const CString& file, CString& stAccName, CString& stServer, CString& stUserName, DWORD& dwPort) 
{
    CString text = ReadUtf16TextFile(file);
        
    if(!GetXMLTagContent(text, _T("Account_Name"), stAccName))
        return false;

    if(!GetXMLTagContent(text, _T("IMAP_Server"), stServer))
        return false;

    if(!GetXMLTagContent(text, _T("IMAP_User_Name"), stUserName))
        return false;

    CString port = _T("0000008f"); // 143
    GetXMLTagContent(text, _T("IMAP_Port"), port);

    dwPort = _tcstoul(port, NULL, 16);

    return true;
}

bool IsPOP3WinMailFile(const CString& file, CString& stAccName, CString& stServer, CString& stUserName, DWORD& dwPort) 
{
    CString text = ReadUtf16TextFile(file);
    
    if(!GetXMLTagContent(text, _T("Account_Name"), stAccName))
        return false;

    if(!GetXMLTagContent(text, _T("POP3_Server"), stServer))
        return false;

    if(!GetXMLTagContent(text, _T("POP3_User_Name"), stUserName))
        return false;

    CString port = _T("00000064"); // 110
    GetXMLTagContent(text, _T("POP3_Port"), port);

    dwPort = _tcstoul(port, NULL, 16);

    return true;
}

CString GetLocalAppDataPath()
{
	LPITEMIDLIST pidl;

    const int _CSIDL_LOCAL_APPDATA = 0x001C;
	if(NOERROR != ::SHGetSpecialFolderLocation(NULL, _CSIDL_LOCAL_APPDATA, &pidl))
		return _T("");

	TCHAR szPath[MAX_PATH+1];
	if(!::SHGetPathFromIDList(pidl, szPath))
		return _T("");

	LPMALLOC pMalloc;
	if(NOERROR == ::SHGetMalloc(&pMalloc))
	{
		pMalloc->Free(pidl);
		pMalloc->Release();
	}

	return CString(szPath);
}

void FindAllFilesRec(const CString& path, const CString& ending, CStringArray& files)
{
    CString stPattern = path + _T("*");
    
    WIN32_FIND_DATA FileData;
	
	const HANDLE hHandle = FindFirstFile(stPattern, &FileData);
	if(hHandle == INVALID_HANDLE_VALUE)
		return;

    do 
    {
        if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0) 
        {
            if(FileData.cFileName[0] != _T('.')) 
            {
                CString newPath(path + FileData.cFileName + _T("\\"));
                FindAllFilesRec(newPath, ending, files);
            }
        }
        else 
        {
            CString fileName(FileData.cFileName); 
            if(fileName.GetLength() > ending.GetLength() && fileName.Right(ending.GetLength()).CompareNoCase(ending) == 0)
		        files.Add(path + fileName);
        }
    }
    while(FindNextFile(hHandle, &FileData) != 0);

	FindClose(hHandle);
}

int CDlgAccImport::LoadAccountsWindowsMail(BOOL bCountOnly)
{
    CString stAppData = GetLocalAppDataPath();
    if(stAppData.GetLength() == 0)
        return 0;
	if(stAppData.Right(1) != _T("\\")) stAppData += _T('\\');
    stAppData += _T("Microsoft\\Windows Mail\\");

	CStringArray Files;	
	FindAllFilesRec(stAppData, _T(".oeaccount"), Files);
	
    int nAccCount = 0;
	for(int n = 0; n < Files.GetSize(); n++)
	{
		CString stFile = Files[n];
		CString stAccName;
		CString stServer;
		CString stUserName;
		DWORD dwPort = 0;

        if(IsImapWinMailFile(stFile, stAccName, stServer, stUserName, dwPort)) 
        {
            if(bCountOnly == FALSE)
			    AddAccount(true, stAccName, stUserName, stServer, dwPort);
            nAccCount++;
        }
        else if (IsPOP3WinMailFile(stFile, stAccName, stServer, stUserName, dwPort)) 
        {
            if(bCountOnly == FALSE)
			    AddAccount(false, stAccName, stUserName, stServer, dwPort);
            nAccCount++;
        }	
	}

    return nAccCount;
}

BOOL CDlgAccImport::AddAccount(BOOL bIMAP, CString stAccount, CString stUser, CString stServer, DWORD dwPort)
{
	if(m_AccList.m_hWnd == NULL)
		return FALSE;

    CAccount* pAcc = NULL;

    if(!bIMAP)
    {
        if(dwPort == 995)
        {
			pAcc = new CPOP3Account(m_pDoc, true);
        }
        else
        {
            pAcc = new CPOP3Account(m_pDoc, false);
        }
    }
    else
    {
        if(dwPort == 993)
        {
            CProtocol* prot = m_Protocols.GetProtocolByName(_T("IMAP4rev1 (SSL)"));
            if(prot != NULL)
                pAcc = new CPluginAccount(m_pDoc, *prot);      
        }
        else
        {
            CProtocol* prot = m_Protocols.GetProtocolByName(_T("IMAP4rev1"));
            if(prot != NULL)
                pAcc = new CPluginAccount(m_pDoc, *prot); 
        }
    }


    if(pAcc == NULL)
        return FALSE;

    LV_ITEM lvItem;
	lvItem.mask = LVIF_TEXT;
	
	// neue Zeile einfgen:
	lvItem.iItem = m_AccList.GetItemCount();
	lvItem.iSubItem = 0;
	lvItem.pszText = NULL;
	m_AccList.InsertItem(&lvItem);


	pAcc->m_stName = stAccount;
	pAcc->m_stUser = stUser;
	pAcc->m_stServer = stServer;
	pAcc->m_nPort = dwPort;
	m_AccList.SetItemData(m_AccList.GetItemCount() - 1, (DWORD)pAcc);
	
	if(m_bPreSelect)
		m_AccList.SetCheck(m_AccList.GetItemCount() - 1);

	UpdateLine(m_AccList.GetItemCount() - 1, pAcc);
	
	return TRUE;
}


void CDlgAccImport::OnDblclkListAccounts(NMHDR* pNMHDR, LRESULT* pResult) 
{
	if(pResult) *pResult = 0;
	
	POSITION pos = m_AccList.GetFirstSelectedItemPosition();
	if(pos == NULL)
		return;

	int nItem = m_AccList.GetNextSelectedItem(pos);
	
	CAccount* pAcc = GetValidAcc(m_AccList.GetItemData(nItem));
	if(pAcc == NULL)
		return;
	
	CDlgAccount DlgAcc(m_pDoc);
			
	if(DlgAcc.DoModal(pAcc, m_Protocols) == IDOK)
	{
		m_AccList.SetItemData(nItem, (DWORD)pAcc);
		UpdateLine(nItem, pAcc);
	}
}

void CDlgAccImport::UpdateLine(int nLine, CAccount* pAcc)
{
	if(nLine >= m_AccList.GetItemCount() || pAcc == NULL)
		return;

	LV_ITEM lvItem;
	lvItem.mask = LVIF_TEXT;
	lvItem.iItem = nLine;

	
	lvItem.pszText = (LPTSTR)(LPCTSTR)pAcc->m_stName;
	lvItem.iSubItem = 0;
	m_AccList.SetItem(&lvItem);

    CString accType = pAcc->GetProtocolName();
    lvItem.pszText = (LPTSTR)(LPCTSTR)accType;
	lvItem.iSubItem = 1;
	m_AccList.SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pAcc->m_stUser;
	lvItem.iSubItem = 2;
	m_AccList.SetItem(&lvItem);

	lvItem.pszText = (LPTSTR)(LPCTSTR)pAcc->m_stServer;
	lvItem.iSubItem = 3;
	m_AccList.SetItem(&lvItem);

	CString stPort;
	stPort.Format(_T("%u"), pAcc->m_nPort);
	lvItem.pszText = (LPTSTR)(LPCTSTR)stPort;
	lvItem.iSubItem = 4;
	m_AccList.SetItem(&lvItem);
}

void CDlgAccImport::OnAccountProperties() 
{
	OnDblclkListAccounts(NULL, NULL);
}

void CDlgAccImport::OnItemChanged(NMHDR* pNMHDR, LRESULT* pResult)
{		
	GetDlgItem(IDC_ACCOUNT_IMPORT_PROPERTIES)->EnableWindow(m_AccList.GetSelectedCount() == 1);

	*pResult = 0;
}

void CDlgAccImport::OnOK() 
{
	for(int n = 0; n < m_AccList.GetItemCount(); n++)
	{
		CAccount* pAcc = GetValidAcc(m_AccList.GetItemData(n));
		if(pAcc == NULL)
			continue;
	
        if(m_AccList.GetCheck(n)) 
        {
			m_pDoc->m_Accounts.AddTail(pAcc);

            // Hack: CAccount is not a CObject!
            m_pDoc->UpdateAllViews(NULL, CPopManDoc::NOTIFICATIONS::ntNewAccount, (CObject*)pAcc);
        }
		else
			delete pAcc;
	}
	
	CDialog::OnOK();
}


void CDlgAccImport::OnCancel() 
{
	for(int n = 0; n < m_AccList.GetItemCount(); n++)
		delete GetValidAcc(m_AccList.GetItemData(n));
	
	CDialog::OnCancel();
}

CAccount* CDlgAccImport::GetValidAcc(DWORD pPointer)
{
	return (CAccount*)pPointer;
	/*CAccount* pAcc = NULL;

	try 
	{
		pAcc = dynamic_cast<CAccount*>((CObject*)pPointer);
		if(pAcc == NULL)
			return NULL;

		return pAcc;
	} 
	catch (...)
	{
		return NULL;
	}*/
}

void CDlgAccImport::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_HINT, i18n("Please select the accounts that you want to import from Outlook Express."));
	SetDlgItemText(IDC_ACCOUNT_IMPORT_PROPERTIES, i18n("&Properties..."));
	SetDlgItemText(IDOK,		i18n("OK"));
	SetDlgItemText(IDCANCEL,	i18n("Cancel"));
	SetWindowText(i18n("Import email accounts"));
}