// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ColorStatusBar.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "ColorStatusBar.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CColorStatusBar

CColorStatusBar::CColorStatusBar()
{
	PaneArray = NULL;
	m_nPaneCount = 0;
	m_nTimerID = 0;
	m_nCurrPane = -1;
	m_fnOnPaneDblClick = NULL;
}

CColorStatusBar::~CColorStatusBar()
{
	if(PaneArray != NULL)
	{
		delete [] PaneArray;
		PaneArray = NULL;
	}
}


BEGIN_MESSAGE_MAP(CColorStatusBar, CStatusBar)
	//{{AFX_MSG_MAP(CColorStatusBar)
	ON_WM_MOUSEMOVE()
	ON_WM_TIMER()
	ON_WM_LBUTTONDBLCLK()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CColorStatusBar::SetIndicators(const UINT* lpIDArray, int nIDCount)
{
	if(PaneArray != NULL)
	{
		delete [] PaneArray;
		PaneArray = NULL;
	}

	if(nIDCount > 0)
		PaneArray = new PaneInfo[nIDCount];

	for(int n = 0; n < nIDCount; n++)
	{
		PaneArray[n].bOwnerDrawn = FALSE;
		PaneArray[n].TextColor = 0;
	}
	
	m_nPaneCount = nIDCount;

	return CStatusBar::SetIndicators(lpIDArray, nIDCount);
}


void CColorStatusBar::GetPaneText(int nIndex, CString& rString) const
{
	if(!IsOwnerDrawn(nIndex))
	{
		CStatusBar::GetPaneText(nIndex, rString);
		return;
	}

	ASSERT(PaneArray);

	rString = PaneArray[nIndex].Text;
}
	

CString CColorStatusBar::GetPaneText(int nIndex) const
{
	if(!IsOwnerDrawn(nIndex))
		return CStatusBar::GetPaneText(nIndex);

	ASSERT(PaneArray);

	return PaneArray[nIndex].Text;
}
	
BOOL CColorStatusBar::SetPaneText(int nIndex, LPCTSTR lpszNewText, BOOL bUpdate)
{
	if(nIndex == m_nCurrPane)
		m_ToolTip.SetTip(lpszNewText);
	
	if(!IsOwnerDrawn(nIndex))
		return CStatusBar::SetPaneText(nIndex, lpszNewText, bUpdate);

	ASSERT(PaneArray);

	PaneArray[nIndex].Text = lpszNewText;
	
	if(bUpdate) {
		Invalidate();
		RedrawWindow();
	}

	return TRUE;
}

void CColorStatusBar::SetPaneTextColor(int nIndex, COLORREF TextColor)
{
	ASSERT(PaneArray);

	PaneArray[nIndex].Text = GetPaneText(nIndex);
	PaneArray[nIndex].TextColor = TextColor;
	PaneArray[nIndex].bOwnerDrawn = TRUE;

	SendMessage(SB_SETTEXT, nIndex | SBT_OWNERDRAW, 0);

}


COLORREF CColorStatusBar::GetPaneTextColor(int nIndex)
{
	if(IsOwnerDrawn(nIndex))
		return PaneArray[nIndex].TextColor;
	else
		return ::GetSysColor(COLOR_WINDOWTEXT);
}

void CColorStatusBar::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct)
{
	ASSERT(PaneArray);

	if(PaneArray == NULL)
		return;

	HDC hDC = lpDrawItemStruct->hDC;
	int Idx = lpDrawItemStruct->itemID;

	COLORREF oldTextColor = ::SetTextColor(hDC, PaneArray[Idx].TextColor);
	int oldBkMode = ::SetBkMode(hDC, TRANSPARENT); 
	::DrawText(hDC, PaneArray[Idx].Text, -1, &lpDrawItemStruct->rcItem, DT_VCENTER | DT_LEFT | DT_NOPREFIX | DT_SINGLELINE);
	::SetTextColor(hDC, oldTextColor);
	::SetBkMode(hDC, oldBkMode); 
}

BOOL CColorStatusBar::IsOwnerDrawn(int PaneIdx) const
{
	ASSERT(PaneArray);
	return PaneArray[PaneIdx].bOwnerDrawn;
}


int CColorStatusBar::GetCurrentPane()
{
	POINT ptPos;
	GetCursorPos(&ptPos);
	
	return GetPaneFromPoint(ptPos, TRUE);
}


int CColorStatusBar::GetPaneFromPoint(POINT ptPoint, BOOL bScreen)
{
	if(bScreen)
		ScreenToClient(&ptPoint);

	RECT rect;
	for(int n = 0; n < m_nPaneCount; n++)
	{
		GetItemRect(n, &rect);
		if(PtInRect(&rect, ptPoint))
			return n;
	}
	return -1;
}

BOOL CColorStatusBar::IsToolTipNeeded(int nPaneIdx)
{
	if(nPaneIdx < 0)
		return FALSE;

	SIZE Size;
	CString stPaneText = GetPaneText(nPaneIdx);
	HDC hDC = ::GetDC(GetSafeHwnd());
	
	HFONT oldFont = (HFONT)::SelectObject(hDC, GetFont()->GetSafeHandle());
	::GetTextExtentPoint32(hDC, stPaneText, stPaneText.GetLength(), &Size);
	::SelectObject(hDC, oldFont);
	
	::ReleaseDC(GetSafeHwnd(), hDC);
	RECT rect;
	GetItemRect(nPaneIdx, &rect);
	return (rect.right-rect.left < Size.cx);
}

void CColorStatusBar::OnMouseMove(UINT nFlags, CPoint point) 
{
	CStatusBar::OnMouseMove(nFlags, point);

	int nCurrPane = GetCurrentPane();
	if(nCurrPane < 0 || IsToolTipNeeded(nCurrPane) == FALSE)
		return;

	StartTimer();
}

void CColorStatusBar::OnTimer(UINT nIDEvent) 
{

//	TRACE(_T("OnTimer\n"));

	CStatusBar::OnTimer(nIDEvent);
	
	int nCurrPane = GetCurrentPane();

	if(nCurrPane == -1 || IsToolTipNeeded(nCurrPane) == FALSE)  
	{
		StopTimer();
		HideToolTip();
//		TRACE(_T("-> 1\n"));
		return;
	}


	if(m_nCurrPane == -1)
	{
		ShowToolTip(nCurrPane);
	//	TRACE(_T("-> 2\n"));
		return;
	}


	if(m_nCurrPane != nCurrPane)
	{
		HideToolTip();
		ShowToolTip(nCurrPane);
//		TRACE(_T("-> 3\n"));
		return;
	}
}

void CColorStatusBar::StartTimer()
{
//	TRACE(_T("startTimer\n"));

	StopTimer();
	
	m_nTimerID = ::SetTimer(GetSafeHwnd(), 1, 400, NULL);
}

void CColorStatusBar::StopTimer()
{
//	TRACE(_T("StopTimer\n"));

	if(m_nTimerID != 0)
		::KillTimer(GetSafeHwnd(), 1);

	m_nTimerID = 0;
}


void CColorStatusBar::HideToolTip()
{
//	TRACE(_T("HideToolTip\n"));

	m_nCurrPane = -1;
	m_ToolTip.Hide();

}

void CColorStatusBar::ShowToolTip(int nPaneIdx)
{
	m_nCurrPane = nPaneIdx;

	RECT rect;
	GetItemRect(nPaneIdx, &rect);
	ClientToScreen(&rect);

	m_ToolTip.Show(GetPaneText(nPaneIdx), GetFont(), GetPaneTextColor(nPaneIdx), CPoint(rect.left, rect.top), this);
}


void CColorStatusBar::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	CStatusBar::OnLButtonDblClk(nFlags, point);

	int nPane = GetPaneFromPoint(point, FALSE);
	if(m_fnOnPaneDblClick != NULL && nPane != -1)
		m_fnOnPaneDblClick(nPane);
}

