// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// AutoUpdate.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "AutoUpdate.h"
#include "StrFunctions.h"
#include "PopMan.h"
#include <afxinet.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//  i18nComment("Update Wizard")


HINSTANCE				CAutoUpdate_Page_Two::m_hDecompressDLL	= 0;
BuffToBuffDecompressFn	CAutoUpdate_Page_Two::m_fnDecompress	= NULL;
CAutoUpdate_Page_Two::CFileSignatures			CAutoUpdate_Page_Two::m_LocalFiles;
CAutoUpdate_Page_Two::CFileSignatures			CAutoUpdate_Page_Two::m_NewFiles;

CVersion CDlgAutoUpdate::vNewVersion;


CString  g_stURL;
CString  g_stDownloadedFile;
BOOL	 g_bCanceled = FALSE;
HWND	 g_hwndReceiver = NULL;

const int WM_DOWNLOAD_COMPLETE	= WM_USER + 1;
const int WM_DOWNLOAD_FAILED	= WM_USER + 2;
const int WM_DOWNLOAD_PROGRESS	= WM_USER + 3;

UINT DownloadThread(LPVOID lParam);


enum DownloadError
{
	errURL  = 1,
	errSize = 2,
	errRead = 3,
	errUnknown = 4,
};


UINT DownloadThread(LPVOID lParam)
{
	g_stDownloadedFile.Empty();

	CString stClient;
	stClient.Format(_T("PopMan Update Wizard %s"), cstAppVersion.GetString());
	

	CInternetSession Session(stClient, 1, INTERNET_OPEN_TYPE_PRECONFIG);
	CHttpFile* pFile = NULL;

	try
	{
		pFile = (CHttpFile*)Session.OpenURL(g_stURL, 1, INTERNET_FLAG_PRAGMA_NOCACHE | INTERNET_FLAG_RELOAD | INTERNET_FLAG_DONT_CACHE | INTERNET_FLAG_TRANSFER_BINARY);
		if(pFile == NULL)
		{
			Session.Close();
			::PostMessage(g_hwndReceiver, WM_DOWNLOAD_FAILED, errURL, 0);
			return 1;
		}
		
		DWORD dwFileLen = 0;
		if(0 == pFile->QueryInfo(HTTP_QUERY_FLAG_NUMBER|HTTP_QUERY_CONTENT_LENGTH, dwFileLen, NULL))
		{
			pFile->Close();
			delete pFile;
			Session.Close();
			::PostMessage(g_hwndReceiver, WM_DOWNLOAD_FAILED, errSize, 0);
			return 1;
		}

		int iAllReceived = 0;
		int iBytesReceived = 0;
		
		do
		{
			if(g_bCanceled) 
			{
				pFile->Close();
				delete pFile;
				Session.Close();
				return 0;
			}

			// ToDo: UNICODE-Konvertierung!!!
            const int BUFSIZE = 4024;

			CString stBuf;
			TCHAR* pBuf = stBuf.GetBuffer(BUFSIZE);

#ifdef _UNICODE

            unsigned char BUFFER[BUFSIZE];
            iBytesReceived = pFile->Read(BUFFER, BUFSIZE);
        
            for(int i = 0; i < iBytesReceived; ++i) {
                pBuf[i] = BUFFER[i];
            }
#else
            
    	    iBytesReceived = pFile->Read(pBuf, BUFSIZE);
#endif
     
			iAllReceived += iBytesReceived;

			if(iBytesReceived <= 0)
				stBuf.ReleaseBuffer(0);	
			else
				stBuf.ReleaseBuffer(iBytesReceived);
		
			g_stDownloadedFile += stBuf;

			if(g_bCanceled)
			{
				pFile->Close();
				delete pFile;
				Session.Close();
				return 0;
			}
			::PostMessage(g_hwndReceiver, WM_DOWNLOAD_PROGRESS, iAllReceived, dwFileLen);

		} while (iBytesReceived > 0);

		pFile->Close();
		delete pFile;
		Session.Close();
	}
	catch(CException* e)
	{
		if(pFile)
		{
			pFile->Close();
			delete pFile;
		}
		Session.Close();
		::PostMessage(g_hwndReceiver, WM_DOWNLOAD_FAILED, errUnknown, (LPARAM)e);
		return 1;
	}

	::PostMessage(g_hwndReceiver, WM_DOWNLOAD_COMPLETE, 0, 0);
	return 0;
}
//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////


IMPLEMENT_DYNAMIC(CDlgAutoUpdate, CPropertySheet)

CDlgAutoUpdate::CDlgAutoUpdate(CWnd* pParentWnd, UINT iSelectPage)
	:CPropertySheet(_T(""), pParentWnd, iSelectPage)
{
	AddPage(&m_PageOne);
	AddPage(&m_PageTwo);
	SetWizardMode();
}

CDlgAutoUpdate::~CDlgAutoUpdate()
{
}

BOOL CDlgAutoUpdate::OnInitDialog() 
{
	CPropertySheet::OnInitDialog();
	
	SetTitle(i18n("Update Wizard"));

	return TRUE;  
}

int CDlgAutoUpdate::DoModal() 
{
	return CPropertySheet::DoModal();
}

BEGIN_MESSAGE_MAP(CDlgAutoUpdate, CPropertySheet)
	//{{AFX_MSG_MAP(CDlgAutoUpdate)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()



/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////



IMPLEMENT_DYNCREATE(CAutoUpdate_Page_One, CPropertyPage)

CAutoUpdate_Page_One::CAutoUpdate_Page_One() : CPropertyPage(CAutoUpdate_Page_One::IDD)
{
	//{{AFX_DATA_INIT(CAutoUpdate_Page_One)
	//}}AFX_DATA_INIT
	
	m_bError = FALSE;
}

CAutoUpdate_Page_One::~CAutoUpdate_Page_One()
{
}

void CAutoUpdate_Page_One::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAutoUpdate_Page_One)
	DDX_Control(pDX, IDC_STATIC_ICON, m_staticIcon);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CAutoUpdate_Page_One, CPropertyPage)
	//{{AFX_MSG_MAP(CAutoUpdate_Page_One)
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_DOWNLOAD_COMPLETE, OnDownloadComplete)
	ON_MESSAGE(WM_DOWNLOAD_FAILED, OnDownloadFailed)
	ON_WM_CTLCOLOR()
END_MESSAGE_MAP()


BOOL CAutoUpdate_Page_One::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	InitGUIText();

	m_staticIcon.SetIcon(CPopManApp::GetIcon(IDI_AUTO_UPDATE));

	CFont* pFont = GetDlgItem(IDC_STATIC_WELCOME)->GetFont();
	LOGFONT lf;
	pFont->GetLogFont(&lf);
	lf.lfWeight = FW_BOLD;
	m_FontStatic.CreateFontIndirect(&lf);

	GetDlgItem(IDC_STATIC_WELCOME)->SetFont(&m_FontStatic);


	g_stURL = _T("https://www.ch-software.de/popman/updateinfo.dat");
	g_bCanceled = FALSE;
	g_hwndReceiver = GetSafeHwnd();
	SetDlgItemText(IDC_STATIC_STATE, i18n("Connecting..."));

	AfxBeginThread(DownloadThread, NULL, 0, 0, 0, 0);
	
	return TRUE;  
}


BOOL CAutoUpdate_Page_One::OnSetActive() 
{	
	BOOL bRes = CPropertyPage::OnSetActive();

	CPropertySheet* parent = (CPropertySheet*)GetParent();
	parent->SetWizardButtons(0);  //PSWIZB_NEXT

	CWnd* pBtn = parent->GetDlgItem(ID_WIZBACK);
	if(pBtn) pBtn->ShowWindow(SW_HIDE);
	
	return bRes;
}


LRESULT CAutoUpdate_Page_One::OnDownloadComplete(WPARAM wParam, LPARAM lParam)
{
	CString  stInfoLink;
	CVersion Version;

	if(!NewVersionAvailable(g_stDownloadedFile, Version, stInfoLink)) {
		SetDlgItemText(IDC_STATIC_STATE, i18n("There are no updates available."));
		return 0;
	}
	
	CString stMsg;
	CString stVersion;
	stVersion.Format(_T("%d.%d"), Version.GetMajor(), Version.GetMinor());

	if(!Version.IsBeta())
	{
		if(Version.GetSub() > 0)
			stVersion = Version.GetString();
	
		stMsg = StrFormat(i18n("There is a new version {1} available!"), _T("s"), (LPCTSTR)stVersion);
	}
	else 		
		stMsg = StrFormat(i18n("There is a new {1} Beta version available!"), _T("s"), (LPCTSTR)stVersion);
		

	if(!stInfoLink.IsEmpty()) {
		m_Link.SubclassDlgItem(IDC_STATIC_FIRST_LINE, this);
		m_Link.m_stLink = stInfoLink;
	}

	SetDlgItemText(IDC_STATIC_FIRST_LINE, stMsg); 
	SetDlgItemText(IDC_STATIC_STATE, i18n("Click \"Next\" to update your copy of PopMan..."));
	

	CPropertySheet* parent = (CPropertySheet*)GetParent();
	parent->SetWizardButtons(PSWIZB_NEXT); 

	CDlgAutoUpdate::vNewVersion = Version;

    return 0;

}


LRESULT CAutoUpdate_Page_One::OnDownloadFailed(WPARAM wParam, LPARAM lParam)
{
	CString stErr = i18n("Unable to retrieve version infomation!");
	
	if(lParam == NULL)
	{
		stErr += _T("\r\n\r\n");
		stErr += StrFormat(i18n("Error number: {1}"), _T("d"), wParam);
	}
	else
	{
		CException* e = (CException*)lParam;
		TCHAR szBuf[1024] = _T("");
		e->GetErrorMessage(szBuf, sizeof(szBuf) / sizeof(TCHAR));
		e->Delete();
		stErr += _T("\r\n\r\n\"");
		CString stMsg = szBuf;
		stMsg.TrimLeft();
		stMsg.TrimRight();
		stErr += stMsg;
		stErr += _T('\"');
	}
	
	m_bError = TRUE;
	SetDlgItemText(IDC_STATIC_STATE, stErr);

    return 0;

}

BOOL CAutoUpdate_Page_One::NewVersionAvailable(const CString& stData, CVersion& Version, CString& stInfoLink)
{
	CVersion vRelease, vBeta;
	CString  stReleaseInfo, stBetaInfo;
	if(!CVersion::ParseVersionInfo(stData, vRelease, stReleaseInfo, vBeta, stBetaInfo))
		return FALSE;

	BOOL bNewVersion = FALSE;

	if(vRelease > cstAppVersion) {
		bNewVersion = TRUE;
		Version = vRelease;
		stInfoLink = stReleaseInfo;
	} 
	else if(vBeta > cstAppVersion) {
		bNewVersion = TRUE;
		Version = vBeta;
		stInfoLink = stBetaInfo;
	}

	if(!cstAppVersion.IsBeta() && Version.IsBeta())
		return FALSE;

	return bNewVersion;
}

HBRUSH CAutoUpdate_Page_One::OnCtlColor(CDC* pDC, CWnd* pWnd, UINT nCtlColor)
{
	HWND hwndState = GetDlgItem(IDC_STATIC_STATE)->GetSafeHwnd();
	HWND hwndLink = GetDlgItem(IDC_STATIC_FIRST_LINE)->GetSafeHwnd();

	if(pWnd->GetSafeHwnd() == hwndLink && m_Link.GetSafeHwnd() != 0)
		return CWnd::OnCtlColor(pDC, pWnd, nCtlColor);

	if(pWnd->GetSafeHwnd() == hwndState || pWnd->GetSafeHwnd() == hwndLink)
	{
		pDC->SetTextColor(m_bError ? RGB(220, 10, 10) : RGB(20, 20, 180));
		pDC->SetBkMode(TRANSPARENT);
		if(m_BkgBrush.m_hObject == NULL)
			m_BkgBrush.CreateSolidBrush(GetSysColor(COLOR_3DFACE));
	
		return m_BkgBrush;
	}

	return CWnd::OnCtlColor(pDC, pWnd, nCtlColor);
}


BOOL CAutoUpdate_Page_One::OnQueryCancel()
{
	g_bCanceled = TRUE;
	return TRUE;
}


void CAutoUpdate_Page_One::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_WELCOME, i18n("Welcome to the PopMan Update Wizard"));
	SetDlgItemText(IDC_STATIC_INFO, i18n("The wizard is now connecting to the PopMan webserver checking for updates. If there is a new version available, all necessary files can be downloaded and installed automatically."));
}
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

DWORD* CAutoUpdate_Page_Two::m_pdwCrc32Table = NULL;

IMPLEMENT_DYNCREATE(CAutoUpdate_Page_Two, CPropertyPage)

CAutoUpdate_Page_Two::CAutoUpdate_Page_Two() : CPropertyPage(CAutoUpdate_Page_Two::IDD)
{
	//{{AFX_DATA_INIT(CAutoUpdate_Page_Two)
	//}}AFX_DATA_INIT
	m_bProcessing = FALSE;
	m_bError = FALSE;
	m_nCurrentFile = 0;
	m_nTotalDownloadSize = 0;
	m_nBytesDownloaded = 0;

	// Init CRC table:

	DWORD dwPolynomial = 0xEDB88320;
	int i, j;

	m_pdwCrc32Table = new DWORD[256];

	DWORD dwCrc;
	for(i = 0; i < 256; i++)
	{
		dwCrc = i;
		for(j = 8; j > 0; j--)
		{
			if(dwCrc & 1)
				dwCrc = (dwCrc >> 1) ^ dwPolynomial;
			else
				dwCrc >>= 1;
		}
		m_pdwCrc32Table[i] = dwCrc;
	}
}

CAutoUpdate_Page_Two::~CAutoUpdate_Page_Two()
{
	delete m_pdwCrc32Table;
	m_pdwCrc32Table = NULL;
	
	if(m_hDecompressDLL != NULL)
		::FreeLibrary(m_hDecompressDLL);
}

void CAutoUpdate_Page_Two::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAutoUpdate_Page_Two)
	//}}AFX_DATA_MAP
	DDX_Control(pDX, IDC_PROGRESS_ALL, m_ProgressAll);
	DDX_Control(pDX, IDC_PROGRESS_CURRENT, m_ProgressCurrent);
}

BEGIN_MESSAGE_MAP(CAutoUpdate_Page_Two, CPropertyPage)
	//{{AFX_MSG_MAP(CAutoUpdate_Page_Two)
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_DOWNLOAD_COMPLETE, OnDownloadComplete)
	ON_MESSAGE(WM_DOWNLOAD_FAILED, OnDownloadFailed)
	ON_MESSAGE(WM_DOWNLOAD_PROGRESS, OnDownloadProgress)
	ON_WM_CTLCOLOR()
END_MESSAGE_MAP()


BOOL CAutoUpdate_Page_Two::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();

	SetDlgItemText(IDC_STATIC_ALL_SIZE, _T(""));
	SetDlgItemText(IDC_STATIC_CURRENT_SIZE, _T(""));
	
	InitGUIText();

	CFont* pFont = GetDlgItem(IDC_STATIC_TITLE)->GetFont();
	LOGFONT lf;
	pFont->GetLogFont(&lf);
	lf.lfWeight = FW_BOLD;
	m_FontStatic.CreateFontIndirect(&lf);
	GetDlgItem(IDC_STATIC_TITLE)->SetFont(&m_FontStatic);


	m_bProcessing = TRUE;

	
	m_hDecompressDLL = ::LoadLibrary(_T("Decompress.dll"));
	if(m_hDecompressDLL != NULL)
		m_fnDecompress = (BuffToBuffDecompressFn)::GetProcAddress(m_hDecompressDLL, "BZ2_bzBuffToBuffDecompress");

	if(m_hDecompressDLL == NULL || m_fnDecompress == NULL)
	{
		Error(i18n("Failed to load Decompress.dll!"));
		return TRUE;
	}

	if(CDlgAutoUpdate::vNewVersion.IsBeta())
		g_stURL = _T("https://www.ch-software.de/popman/currBeta.info");
	else
		g_stURL = _T("https://www.ch-software.de/popman/currRelease.info");

	g_bCanceled = FALSE;
	g_hwndReceiver = GetSafeHwnd();
	SetDlgItemText(IDC_STATIC_STATE, i18n("Initializing..."));
	m_UpdateState = updInit;
	
	m_NewFiles.RemoveAll();
	m_LocalFiles.RemoveAll();
	AfxBeginThread(InitThread, &m_LocalFiles, 0, 0, 0, 0);


/*
	for(int n = 0; n < m_LocalFiles.GetSize(); n++)
	{
		const FILESIGNATURE& File = m_LocalFiles[n];
		TRACE(" %X \t %d \t %s \n", File.dwCRC, File.dwSize, (LPCTSTR)File.stName);
	}
*/
	return TRUE;  
}


void CAutoUpdate_Page_Two::ListDir(CFileSignatures& Files, LPCTSTR szDir, LPCTSTR szBaseDir)
{	
	if(g_bCanceled) return;

	CString stPath = szDir;
	if(stPath.Right(1) != _T("\\")) stPath += _T('\\');
	stPath += _T('*');
	
	WIN32_FIND_DATA FileData;
	HANDLE hHandle = FindFirstFile(stPath, &FileData);
	if(hHandle == INVALID_HANDLE_VALUE)
		return;

	CStringArray SubDirs;
	do
	{
		if(g_bCanceled) return;

		if((FileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0)
		{
			FILESIGNATURE File;
			File.dwSize = FileData.nFileSizeLow;
			
			File.stName = szDir;
			File.stName += _T('\\');
			File.stName += FileData.cFileName;
			File.dwCRC  = GetFileCheckSum(File.stName);
			
			int nBaseDirLen = _tcslen(szBaseDir);
			int nDirLen = _tcslen(szDir);
			File.stName = File.stName.Mid(nBaseDirLen + (nBaseDirLen == nDirLen ? 1 : 0));
			
			if(g_bCanceled) return;
			Files.Add(File);
		}
		else
		{
			CString stDir = FileData.cFileName;
			if(stDir != _T(".") && stDir != _T(".."))
			{
				stDir = szDir;
				stDir += _T('\\');
				stDir += FileData.cFileName;
				SubDirs.Add(stDir);
			}
		}
			
	} while(FindNextFile(hHandle, &FileData) != 0);

	FindClose(hHandle);

	for(int n = 0; n < SubDirs.GetSize(); n++)
		ListDir(Files, SubDirs[n], szBaseDir);
}	


BOOL CAutoUpdate_Page_Two::OnSetActive() 
{	
	BOOL bRes = CPropertyPage::OnSetActive();
	
	CPropertySheet* parent = (CPropertySheet*)GetParent();
	parent->SetWizardButtons(0);

	CWnd* pBtn = parent->GetDlgItem(ID_WIZBACK);
	if(pBtn) pBtn->ShowWindow(SW_HIDE);
	
	return bRes;
}

BOOL CAutoUpdate_Page_Two::OnQueryCancel()
{
	BOOL bContinueCancel = TRUE;
	if(m_bProcessing)
	{
		UINT nRes = AfxMessageBox(i18n("Do you really want to cancel the update process?"), MB_YESNO | MB_DEFBUTTON2 | MB_ICONEXCLAMATION);
		if(nRes == IDYES)
			g_bCanceled = TRUE;
		else
			bContinueCancel = FALSE;
	}

	return (bContinueCancel);
}


LRESULT CAutoUpdate_Page_Two::OnDownloadFailed(WPARAM wParam, LPARAM lParam)
{
	CString stErr = i18n("Update process failed!");

	if(lParam == NULL)
	{
		stErr += _T("\r\n");
		stErr += StrFormat(i18n("Error number: {1}"), _T("d"), wParam);
	}
	else
	{
		CException* e = (CException*)lParam;
		TCHAR szBuf[1024] = _T("");
		e->GetErrorMessage(szBuf, sizeof(szBuf) / sizeof(TCHAR));
		e->Delete();
		stErr += _T("\r\n\"");
		CString stMsg = szBuf;
		stMsg.TrimLeft();
		stMsg.TrimRight();
		stErr += stMsg;
		stErr += _T('\"');
	}
	Error(stErr);

	return 0;

}



LRESULT CAutoUpdate_Page_Two::OnDownloadProgress(WPARAM wParam, LPARAM lParam)
{
//	TRACE("OnDownloadProgress: All: %d; Done: %d \n", lParam, wParam);
	
	int nAll = lParam / 1024;
	if(nAll < 1) nAll = 1;
	int nDone = wParam / 1024;
	CString stSize;
	stSize.Format(_T("%d KB / %d KB"), nDone, nAll);
	SetDlgItemText(IDC_STATIC_CURRENT_SIZE, stSize);
	m_ProgressCurrent.SetPos(nDone*100/nAll);

	nAll = m_nTotalDownloadSize / 1024;
	if(nAll < 1) nAll = 1;
	nDone = (m_nBytesDownloaded + wParam) / 1024;
	stSize.Format(_T("%d KB / %d KB"), nDone, nAll);
	SetDlgItemText(IDC_STATIC_ALL_SIZE, stSize);
	m_ProgressAll.SetPos(nDone*100/nAll);
//	TRACE("\t\t: m_nTotalDownloadSize: %d; m_nBytesDownloaded + wParam: %d \n", m_nTotalDownloadSize, m_nBytesDownloaded + wParam);

	return 0;

}


LRESULT CAutoUpdate_Page_Two::OnDownloadComplete(WPARAM wParam, LPARAM lParam)
{
	switch(m_UpdateState)
	{
	case updInit:
	{
		CString stVersion;
		CFileSignatures AvailableFiles;

		if(!ParseVersionFile(g_stDownloadedFile, m_stServer, m_stBaseDir, stVersion, m_stRestartCmd, AvailableFiles))
		{
			Error(i18n("Error while downloading initialization file!"), TRUE);
			return 0;
		}
		
		m_NewFiles.RemoveAll();
	
		for(int n = 0; n < AvailableFiles.GetSize(); n++)
		{
			BOOL bAddFile = TRUE;
			FILESIGNATURE& AF =  AvailableFiles[n];
			for(int m = 0; m < m_LocalFiles.GetSize(); m++)
			{
				const FILESIGNATURE& LF = m_LocalFiles[m];
				//if(AF.stName != LF.stName) continue;
				if(AF.stName.CompareNoCase(LF.stName) != 0) continue;

				if(AF.dwCRC == LF.dwCRC && AF.dwSize == LF.dwSize)
					bAddFile = FALSE;

				break;
			}
			if(bAddFile) {
				m_NewFiles.Add(AF);
				m_nTotalDownloadSize += AF.dwSizeCompressed;
			}
		}

		if(m_NewFiles.GetSize() == 0)
		{
			SetDlgItemText(IDC_STATIC_STATE, i18n("No updates available."));
			m_bProcessing = FALSE;
			return 0;		
		}


		TCHAR szPath[_MAX_PATH] = _T("");
		DWORD dwRet = ::GetTempPath(sizeof(szPath)/sizeof(TCHAR), szPath);
		m_stLocalBaseDir = szPath;
		if(dwRet == 0 || m_stLocalBaseDir.IsEmpty())
		{
			Error(i18n("Failed to retrieve temporary directory!"), TRUE);
			return 0;
		}
		if(m_stLocalBaseDir.Right(1) != _T("\\")) m_stLocalBaseDir += _T('\\');
		m_stLocalBaseDir += _T("PopMan Update\\");
		m_stLocalBaseDir += CDlgAutoUpdate::vNewVersion.GetString();

		StartDownloadingFile(0);
	}
		break;


	case updFileDownload:
		
		m_nBytesDownloaded += m_NewFiles[m_nCurrentFile].dwSizeCompressed;
		
		CString stFileName = ParseReverse(CString(m_NewFiles[m_nCurrentFile].stName), _T("\\"));
		SetDlgItemText(IDC_STATIC_STATE, StrFormat(i18n("Saving {1} ..."), _T("s"), (LPCTSTR)stFileName));
	

		try 
		{
			CString stFilePath = m_stLocalBaseDir;
			if(stFilePath.Right(1) != _T("\\")) stFilePath += _T('\\');

			CString stFileName = m_NewFiles[m_nCurrentFile].stName;
			if(stFileName[0] == _T('\\')) stFileName.Delete(0);
				
			stFilePath += stFileName;
			stFileName = stFilePath;
			
			ParseReverse(stFilePath, _T("\\"));

			if(!CreateDirectoryRec(stFilePath))
			{ 
				CString stErr = i18n("Failed to create directory:");
				stErr += _T(' ');
				stErr += stFilePath;
				Error(stErr);
				return 0;
			}


			TRACE(_T("Speichere: %s \n"), (LPCTSTR)stFileName);

			CFile File(stFileName, CFile::modeCreate | CFile::modeWrite);
			
			CString stData;

			if(m_NewFiles[m_nCurrentFile].bCompressed)
			{
				if(!Decompress(g_stDownloadedFile, stData))
				{
					CString stErr = i18n("Failed to decompress file:");
					stErr += _T(' ');
					stErr += ParseReverse(stFileName, _T("\\"));
					Error(stErr);
					return 0;
				}
			}
			else
			{
				stData = g_stDownloadedFile;
			}

			
			DWORD dwCheckSum = GetDataCheckSum(stData);
			if(m_NewFiles[m_nCurrentFile].dwCRC != dwCheckSum)
			{
				CString stErr = i18n("Failed to download file:");
				stErr += _T(' ');
				stErr += ParseReverse(stFileName, _T("\\"));
				stErr += _T("\r\n");
				stErr += i18n("(checksum error)");
				stErr += _T(' ');
				stErr += i18n("Please try again later.");
				Error(stErr);
				return 0;
			}

			#ifndef _UNICODE
		
				File.Write(stData, stData.GetLength());

			#else

				int nLen = stData.GetLength();
				char* szBuf = new char[nLen];
				WideCharToMultiByte(CP_ACP, 0, (LPCWSTR)stData, nLen, szBuf, nLen, NULL, NULL);
				
				File.Write(szBuf, nLen);
			
				delete[] szBuf;

			#endif

			File.Close();

		}
		catch(CFileException* e)
		{
			e->ReportError();
			e->Delete();
			
			Error(i18n("Error while saving downloaded file!"));
		
			return 0;
		}

		
		SetDlgItemText(IDC_STATIC_CURRENT_SIZE, _T(""));
		m_ProgressCurrent.SetPos(0);
	
		
		if(m_NewFiles.GetSize() > m_nCurrentFile + 1)
			StartDownloadingFile(m_nCurrentFile + 1);
		else
		{
			CString stMsg = i18n("All files downloaded successfully.");
			stMsg += _T("\r\n\r\n");
			stMsg += i18n("Press the \"Finish\" button to complete the update process...");

			SetDlgItemText(IDC_STATIC_STATE, stMsg);

			CPropertySheet* parent = (CPropertySheet*)GetParent();
			parent->SetWizardButtons(ID_WIZFINISH);
		}
		
		break;
	}

	return 0;

}

void CAutoUpdate_Page_Two::Error(LPCTSTR szError, BOOL bResetProgress)
{
	m_bError = TRUE;
	m_bProcessing = FALSE;
	if(szError)
		SetDlgItemText(IDC_STATIC_STATE, szError);
	if(bResetProgress)
	{
		SetDlgItemText(IDC_STATIC_CURRENT_SIZE, _T(""));
		m_ProgressCurrent.SetPos(0);
		SetDlgItemText(IDC_STATIC_ALL_SIZE, _T(""));
		m_ProgressAll.SetPos(0);
	}
}

BOOL CAutoUpdate_Page_Two::OnWizardFinish()
{
	BOOL bMoveUpdated = FALSE;
	for(int n = 0; n < m_NewFiles.GetSize(); n++)
	{
		if(m_NewFiles[n].stName.CompareNoCase(_T("mv.exe")) == 0) {
			bMoveUpdated = TRUE;
			break;
		}
	}

	CString stMovePath;
	if(bMoveUpdated)
		stMovePath = m_stLocalBaseDir;
	else
		stMovePath = CPopManApp::GetAppPath();

	if(stMovePath.Right(1) != _T("\\")) stMovePath += _T('\\');
	stMovePath += _T("mv.exe");

	CString stParam;
	if(bMoveUpdated) {
		stParam = _T("-dest \"");
		stParam += CPopManApp::GetAppPath();
		stParam += _T('\"');
	}
	else {
		stParam = _T("-src \"");
		stParam += m_stLocalBaseDir;
		stParam += _T('\"');
	}

	if(!m_stRestartCmd.IsEmpty()) 
	{
		stParam += _T(" \"");
		stParam += m_stRestartCmd;
		stParam += _T('\"');
	}

	int res = (int)::ShellExecute(0, _T("open"), stMovePath, stParam, NULL, SW_SHOWNORMAL);

	if(res <= 32) {
		CString stErr = i18n("Failed to initialize the final copy process!");
		stErr += _T("\r\n\r\n");
		stErr += StrFormat(i18n("Error number: {1}"), _T("d"), res);
		AfxMessageBox(stErr, MB_OK | MB_ICONERROR);
	}
	else
		AfxGetMainWnd()->PostMessage(WM_CLOSE);

	return TRUE;
}


BOOL CAutoUpdate_Page_Two::StartDownloadingFile(int nFileIdx)
{
	if(m_bError || !m_bProcessing) return FALSE;

	if(nFileIdx >= m_NewFiles.GetSize())
		return FALSE;

	m_nCurrentFile = nFileIdx;
	const FILESIGNATURE& File = m_NewFiles[m_nCurrentFile];
	CString stFileName = File.stName;
	stFileName.Replace(_T('\\'), _T('/'));

	g_stURL = _T("https://");
	g_stURL += m_stServer;
	g_stURL += m_stBaseDir;
	if(stFileName[0] != _T('/'))
		g_stURL += _T('/');
	g_stURL += stFileName;
	if(File.bCompressed) g_stURL += _T(".bz2");

	g_bCanceled = FALSE;
	g_hwndReceiver = GetSafeHwnd();
	
	stFileName = ParseReverse(stFileName, _T("/"));
	SetDlgItemText(IDC_STATIC_STATE, StrFormat(i18n("Downloading {1} ..."), _T("s"), (LPCTSTR)stFileName));
	m_UpdateState = updFileDownload;

	AfxBeginThread(DownloadThread, NULL, 0, 0, 0, 0);

	return TRUE;
}

HBRUSH CAutoUpdate_Page_Two::OnCtlColor(CDC* pDC, CWnd* pWnd, UINT nCtlColor)
{
	if(pWnd->GetSafeHwnd() == GetDlgItem(IDC_STATIC_STATE)->GetSafeHwnd())
	{
		pDC->SetTextColor(m_bError ? RGB(220, 10, 10) : RGB(20, 20, 200));
		pDC->SetBkMode(TRANSPARENT);
		if(m_BkgBrush.m_hObject == NULL)
			m_BkgBrush.CreateSolidBrush(GetSysColor(COLOR_3DFACE));
	
		return m_BkgBrush;
	}

	return CWnd::OnCtlColor(pDC, pWnd, nCtlColor);
}

DWORD CAutoUpdate_Page_Two::GetDataCheckSum(const CString& stData)
{
	if(m_pdwCrc32Table == NULL)
			return 0;

	DWORD dwCrc32 = 0xFFFFFFFF;
	for(int n = 0; n < stData.GetLength(); ++n)
		CalcCrc32(static_cast<unsigned char>(stData[n]), dwCrc32);

	return ~dwCrc32;
}


DWORD CAutoUpdate_Page_Two::GetFileCheckSum(LPCTSTR szFile)
{
	DWORD dwCrc32 = 0xFFFFFFFF;

	try
	{
        // Is the table initialized?
		if(m_pdwCrc32Table == NULL)
			return 0;

        CFile File(szFile, CFile::modeRead);

		char buffer[8192];
        int nCount = File.Read(buffer, sizeof(buffer));

		while(nCount)
		{
			for(int nLoop = 0; nLoop < nCount; nLoop++)
				CalcCrc32(buffer[nLoop], dwCrc32);

            nCount = File.Read(buffer, sizeof(buffer));
		}
        File.Close();
	}
	catch(CFileException* e)
	{
        e->Delete();
		return 0;
	}

	return ~dwCrc32;
}


inline void CAutoUpdate_Page_Two::CalcCrc32(const BYTE byte, DWORD& dwCrc32)
{
	dwCrc32 = ((dwCrc32) >> 8) ^ m_pdwCrc32Table[(byte) ^ ((dwCrc32) & 0x000000FF)];
}

UINT CAutoUpdate_Page_Two::InitThread(LPVOID lParam)
{
	CFileSignatures* pFiles = (CFileSignatures*)lParam;
	if(pFiles == NULL)
		return 1;

	pFiles->RemoveAll();
	CString stAppPath = CPopManApp::GetAppPath();
	ListDir(*pFiles, stAppPath, stAppPath);
	
	if(g_bCanceled) return 1;

	DownloadThread(0);
	
	return 0;
}

BOOL CAutoUpdate_Page_Two::ParseVersionFile(CString stData, CString& stServer, CString& stBaseDir, CString& stVersion, CString& stRestartCmd, CFileSignatures& Files)
{
	CString stLine;
	int nIdx = 0;

	CString stHead = Parse(stData, _T("\r\n\r\n"));
	
	while(nIdx > -1)
	{
		stLine = ParseConst(stHead, _T("\r\n"), nIdx);
		if(stLine.Find(_T("Server=")) == 0)
		{
			Parse(stLine, _T("Server="));
			stServer = stLine;
		}
		else if(stLine.Find(_T("BaseDir=")) == 0)
		{
			Parse(stLine, _T("BaseDir="));
			stBaseDir = stLine;
		}
		else if(stLine.Find(_T("Version=")) == 0)
		{
			Parse(stLine, _T("Version="));
			stVersion = stLine;
		}
		else if(stLine.Find(_T("RestartCmd=")) == 0)
		{
			Parse(stLine, _T("RestartCmd="));
			stRestartCmd = stLine;
		}
	}

	if(stServer.IsEmpty() || stBaseDir.IsEmpty())
		return FALSE;
	
	nIdx = 0;
	while(nIdx > -1)
	{
		stLine = ParseConst(stData, _T("\r\n"), nIdx);
		stLine.TrimLeft();
		if(stLine.GetLength() == 0) 
			continue;

		FILESIGNATURE File;
	
		File.stName = Parse(stLine, _T("\t"));
		File.stName.Replace(_T('/'), _T('\\'));
		stLine.TrimLeft();
		File.bCompressed = (_T("1") == Parse(stLine, _T("\t")));
		stLine.TrimLeft();
		File.dwCRC = _tcstoul(Parse(stLine, _T("\t")), NULL, 16);
		stLine.TrimLeft();
		File.dwSize = _ttoi(Parse(stLine, _T("\t")));
		stLine.TrimLeft();
		File.dwSizeCompressed = _ttoi(Parse(stLine, _T("\t")));
		
		Files.Add(File);
	}
	
	return Files.GetSize() > 0;
}



BOOL CAutoUpdate_Page_Two::CreateDirectoryRec(LPCTSTR szDir)
{
	DWORD dwAtt = GetFileAttributes(szDir);
	if(dwAtt != 0xFFFFFFFF && (dwAtt & FILE_ATTRIBUTE_DIRECTORY) != 0)
		return TRUE;

	CString stDir = szDir;
	int Idx = stDir.ReverseFind(_T('\\')); 
	if(Idx < 2) return FALSE;

	stDir.Delete(Idx, stDir.GetLength() - Idx);

	if(!CreateDirectoryRec(stDir))
		return FALSE;

	return (0 != ::CreateDirectory(szDir, 0));
}



BOOL CAutoUpdate_Page_Two::Decompress(const CString& stIn, CString& stOut)
{
	const int BZ_OUTBUFF_FULL	= -8;
	const int BZ_OK				=  0;

	if(m_fnDecompress == NULL)
		return FALSE;

	int res = 0;
	for(int n = 5; n < 600; ++n)
	{
		unsigned int nDestLen = n * stIn.GetLength();

        unsigned char* ReadBuffer = new unsigned char[stIn.GetLength()+1];
        for(int n = 0; n < stIn.GetLength(); ++n)
            ReadBuffer[n] = static_cast<unsigned char>(stIn[n]);
		
        LPTSTR pBuf = stOut.GetBuffer(nDestLen + 1);
        unsigned char* WriteBuffer = new unsigned char[nDestLen + 1];

		res = m_fnDecompress((char*)WriteBuffer, &nDestLen, (char*)ReadBuffer, stIn.GetLength(), 0, 0);

        for(unsigned int i = 0; i < nDestLen; ++i)
            pBuf[i] = WriteBuffer[i];

		stOut.ReleaseBuffer(nDestLen);

        delete [] ReadBuffer;
        delete [] WriteBuffer;

		if(res != BZ_OUTBUFF_FULL) 
			break;
	}

	return (res == BZ_OK);
}


void CAutoUpdate_Page_Two::InitGUIText()
{
	SetDlgItemText(IDC_STATIC_PROGRESS_ALL,		i18n("Progress all:"));
	SetDlgItemText(IDC_STATIC_PROGRESS_CURRENT, i18n("Progress current file:"));
	SetDlgItemText(IDC_STATIC_TITLE,			i18n("Download of updated files"));
}
