// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Account.h
//
////////////////////////////////////////////////////////////////////////////////


#if !defined(AFX_ACCOUNT_H_)
#define AFX_ACCOUNT_H_

#if _MSC_VER > 1000
#pragma once
#endif 

#include <afxtempl.h>
#include "Mail.h"
#include "Settings.h"


class CPopManDoc;



class CacheItem
{
public:
    CacheItem() : m_bRead(FALSE) { }
    CacheItem(const CString& stSignature, BOOL bRead) : 
              m_stSignature(stSignature), m_bRead(bRead) {  }

	CString m_stSignature;
	BOOL	m_bRead;
};


class CacheItems
{
public:

    CacheItems() : m_bChanged(false) {}

	CacheItems& operator=(const CacheItems& items) 
	{
		if(this == &items) return *this;
		m_Items.RemoveAll();
		m_Items.Copy(items.m_Items);
        m_bChanged = true;
		return *this;
	}

	int FindSignature(const CString& stSignature, int nPos = 0) const
	{
		for(int n = nPos; n < GetSize(); n++)
		{
			if(m_Items.GetAt(n).m_stSignature == stSignature)
				return n;
		}
		return -1;
	}

    void Clear() { 
        if(GetSize() > 0)
        {
            m_Items.RemoveAll(); 
            m_bChanged = true;
        }
    }
    
    void Add(const CacheItem& item) { 
        m_Items.Add(item);
        m_bChanged = true;
    }
    
    void RemoveAt(int i) { 
        m_Items.RemoveAt(i); 
        m_bChanged = true;
    }   
    
    int  GetSize() const { return m_Items.GetSize(); }

    const CacheItem& operator[](int nIndex) const 
    { 
        ASSERT(nIndex >= 0 && nIndex < GetSize());
        return m_Items.GetData()[nIndex]; 
    }

    void SetReadAt(int idx, BOOL bRead)
    {
        if(m_Items[idx].m_bRead != bRead)
        {
            m_Items[idx].m_bRead = bRead;
            m_bChanged = true;
        }
    }

    bool IsChanged() const { return m_bChanged; }
    void ResetChanged() { m_bChanged = false; }

private:
    CArray<CacheItem, const CacheItem&> m_Items;
    bool m_bChanged;
};


class CAccount
{

protected:	
	CAccount(CPopManDoc* pD);

public:
	virtual ~CAccount();

	enum ERRORS
	{
		erNoError = 0,		
		erCanceled,			
		erBusy,				
		erNotConnected,		
		erConnectFailed,	
		erResponseInvalid,	
		erTimedOut,			
		erAuthFailed,		// authentification failed (invalid user name, invalid password, mailbox locked)
		erMailMissing,		// unable to find the message (which is to be deleted or downloaded)
		erMailLocFailure	// unable to locate message because of server failure or compatibility problems
	};

	enum STATES
	{
		stIdle = 0,
		stConnecting,
		stConnected,
		stLogin,
		stLoggedIn,
		stLoadingMails,
		stDeletingMails,
		stMarkAsSeen,
		stLoadingMail,
		stError
	};

	CString  m_stName;

	CString  m_stServer;
	UINT     m_nPort;

	CString  m_stUser;
	CString  m_stPass;
	CString  m_stPassVolatile;
	BOOL	 m_bActive;

	BOOL	 m_bCustomColor;
	COLORREF m_CustomColor;
	COLORREF m_TrayColor;

	BOOL	 m_bCustomAutoCheckInterval;
	UINT	 m_nCustomAutoCheckInterval; // in minutes

	COleDateTime& GetLastCheckTime() { return m_LastCheckTime; }

	CMails		m_Mails;
	CPopManDoc* m_pDoc;	

	
	CacheItems m_MailCache;


	CString m_stServerErrMessage;
	UINT	m_nErrorLocation;


    enum RETRIEVE_MODE 
	{
		AllComplete   = 0,
		AllHeaderOnly = 1,
		SmallComplete = 2
	};
    RETRIEVE_MODE m_RetrieveMode;

    
    UINT m_nMaxSizeOfSmallMails; // used for RETRIEVE_MODE::SmallComplete

    UINT m_nLines;               // used for RETRIEVE_MODE::AllHeaderOnly
    UINT m_nLinesBigMails;       // used for RETRIEVE_MODE::SmallComplete
    UINT NumberOfLinesForIncompleteMail() { return (m_RetrieveMode==RETRIEVE_MODE::SmallComplete) ? m_nLinesBigMails : m_nLines; }
	
	BOOL	m_bUseUIDLForDelete;

	virtual void SetTimeOut(UINT nSecs) = 0;
	virtual UINT GetTimeOut()  const  = 0; 

	virtual BOOL  IsBusy()     const = 0;
	virtual BOOL  IsCanceled() const = 0;
	virtual BOOL  IsTimedOut() const = 0;

	virtual void Cancel() {}
	virtual void Close()  {}
	
	virtual BOOL ListMails(BOOL bHardReload) = 0;
	virtual BOOL DeleteMails(CMails* pMailList) = 0;
	virtual BOOL MarkMailsAsSeen(CMails* pMailList) = 0;
	virtual BOOL DownloadMail(CMail* pMail, BOOL bKeepConnectionAlive) = 0;


	virtual BOOL ReadSettings(const CSettingsKey& settings);
	virtual void SaveSettings(CSettingsKey& settings) const;

	virtual CString GetProtocolName() const = 0;

	ERRORS GetLastError()   const { return m_LastError; }
	STATES GetState()       const { return m_stState; }
	UINT   GetCurrentMail() const { return m_nCurrentMail; }
	UINT   MailsCount()     const { return m_nMailsCount; }

	int    CountUnreadMails()  const;
	int    GetSizeOfAllMails() const;
	void   ClearMails();
	CMail* GetMailFromSignature(const CString& stSignature) const;

	CString GetErrorDescription() const;
	CString GetStateDescription() const;

	void UpdateMailCache(CMail* pMail);  // the "Read" status of pMail has changed

	void UpdateMailAccReferenz();

protected:
	void UpdateMailCache();  // remove obsolete mails from the mail cache
	void SyncMailListWithUIDLs();

	void GetCurrentUIDLs(CStringArray& UIDLArray);
	void GetCurrentSizes(CStringArray& SizeArray);
	
	class MyStringArray : public CStringArray {
	public:
		MyStringArray& operator=(const MyStringArray& array) 
		{
			if(this == &array) return *this;
			RemoveAll();
			Copy(array);
			return *this;
		}
	};
	MyStringArray m_UIDLCache;
	BOOL GetMailUIDL(int nMailNo, CString& stUIDL);

	void LogMailNotFound(CMail* pMail);
	void StateChanged(STATES State);

	static BOOL VerifyIntegrity(CStringArray& arOld, CStringArray& arNew);

public:
	void OnReceivedData(const void* pData, int nDataLen); 
	void OnSentData(const void* pData, int nDataLen); 
	
protected:
	
	ERRORS		m_LastError;
	int			m_nCurrentMail;	// number of the mail which is currently loaded
	int			m_nMailsCount;	// number of mails on the server
	STATES		m_stState;

	COleDateTime m_LastCheckTime;

    BOOL m_bTopMessages;
    int  m_nTopMessagesCount;

};



class CAccounts : public CTypedPtrList<CPtrList, CAccount*>
{
public:

	CAccount* FindByName(LPCTSTR szName)
	{
		POSITION pos = GetHeadPosition();
		while(pos != NULL)
		{
			CAccount* pAcc = GetNext(pos);
			if(pAcc->m_stName.CompareNoCase(szName) == 0)
				return pAcc;
		}
		return NULL;
	}

	CAccounts& operator=(const CAccounts& list) 
	{
		if(this == &list) return *this;
		RemoveAll();
		
		POSITION pos = list.GetHeadPosition();
		while(pos)
			AddTail(list.GetNext(pos));

		return *this;
	}

    typedef BOOL (*fnMailPredicate)(const CMail& mail);

    int CountMails(fnMailPredicate predicate) const;
    bool ExistsMail(fnMailPredicate predicate) const;
};



#endif // AFX_ACCOUNT_H_
