// PopMan - a Windows POP3 manager
//
// Copyright (C) 2002-2010 Christian Hbner (chuebner@ch-software.de)
// All Rights Reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Account.cpp
//
////////////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "Account.h"
#include "PopManDoc.h"
#include "StrFunctions.h"
#include "Rule.h"

#include <afxcoll.h>
#include <mmsystem.h>


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


int CAccounts::CountMails(fnMailPredicate predicate) const
{
    int counter = 0;
	POSITION pos = GetHeadPosition();
	while(pos != NULL)
	{
		const CAccount* pAcc = GetNext(pos);	
		const CMails& Mails = pAcc->m_Mails;
		POSITION pos2 = Mails.GetHeadPosition();
		while(pos2 != NULL)
		{
			const CMail& Mail = *Mails.GetNext(pos2);
            if(predicate(Mail))
	            ++counter;
		}
	}
    return counter;
}

bool CAccounts::ExistsMail(fnMailPredicate predicate) const
{
	POSITION pos = GetHeadPosition();
	while(pos != NULL)
	{
		const CAccount* pAcc = GetNext(pos);	
		const CMails& Mails = pAcc->m_Mails;
		POSITION pos2 = Mails.GetHeadPosition();
		while(pos2 != NULL)
		{
			const CMail& Mail = *Mails.GetNext(pos2);
            if(predicate(Mail)) return true;
		}
	}
    return false;
}

CAccount::CAccount(CPopManDoc* pD)
{
	m_pDoc = pD;
	m_bActive = TRUE;
	m_LastError = ERRORS::erNoError;
	m_nCurrentMail = 0;
	m_nMailsCount = 0;	
	m_stState = STATES::stIdle;
	m_nErrorLocation = 0;
	m_bCustomColor = FALSE;
	m_CustomColor = GetSysColor(COLOR_WINDOWTEXT);
	m_TrayColor = RGB(0, 0, 180);
	
	//m_bLimitLoadSize = TRUE;
    m_RetrieveMode = RETRIEVE_MODE::SmallComplete;
    m_nLines = 900; 
    m_nLinesBigMails = 900;
    m_nMaxSizeOfSmallMails = 1024; // KB

	m_nPort = 110;
	m_bCustomAutoCheckInterval = FALSE;
	m_nCustomAutoCheckInterval = 10;
	m_LastCheckTime = COleDateTime::GetCurrentTime(); // - COleDateTimeSpan(1,0,0,0);
	m_bUseUIDLForDelete = TRUE;

    m_nTopMessagesCount = 10;
    m_bTopMessages = FALSE;
}

CAccount::~CAccount()
{
	ClearMails();
}

void CAccount::ClearMails()
{
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		delete pMail;
	}
	m_Mails.RemoveAll();
}

void CAccount::LogMailNotFound(CMail* pMail) 
{
	if(pMail == NULL) return;
	CString stLog;
	stLog.Format(_T("\r\n\r\n -> Mail not found: UIDL=%s Size=%d \r\n\t From:    %s\r\n\t Subject: %s\r\n\r\n"),
					pMail->GetUIDL(), pMail->GetSize(), pMail->GetFrom(), pMail->GetSubject());
	m_pDoc->Log(stLog);
}


void CAccount::StateChanged(STATES State)
{
	m_stState = State;
	m_pDoc->OnStatusChanged(this);
}

void CAccount::OnReceivedData(const void* pData, int nDataLen)
{
	m_pDoc->OnDataTraffic(TRUE, pData, nDataLen); 
}

void CAccount::OnSentData(const void* pData, int nDataLen)
{
	m_pDoc->OnDataTraffic(FALSE, pData, nDataLen); 
}


void CAccount::GetCurrentUIDLs(CStringArray& UIDLArray)
{
	UIDLArray.RemoveAll();
	UIDLArray.SetSize(m_Mails.GetCount());

	int nCount = 0;
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		UIDLArray.SetAt(nCount++, pMail->GetUIDL());
	}
}


void CAccount::GetCurrentSizes(CStringArray& SizeArray)
{
	SizeArray.RemoveAll();
	SizeArray.SetSize(m_Mails.GetCount());

	int nCount = 0;
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		CString stSize;
		stSize.Format(_T("%d"), pMail->GetSize());
		SizeArray.SetAt(nCount++, stSize);
	}
}



BOOL CAccount::VerifyIntegrity(CStringArray& arOld, CStringArray& arNew)
{
	if(arNew.GetSize() < 1 || arOld.GetSize() < 1)
		return FALSE;

#ifdef DEBUG
	// Debug -->
	for(int x = 0; x < arOld.GetSize(); x++)
	{
		TRACE(_T("arOld[%d] %s\n"), x, (LPCTSTR)arOld[x]);
	}
	
	for(int y = 0; y < arNew.GetSize(); y++)
	{
		TRACE(_T("arNew[%d] %s\n"), y, (LPCTSTR)arNew[y]);
	}
	// <-- Debug
#endif

	int nIdx = FindInStrArray(arNew, arOld[0]);
	if(nIdx == -1)
	 	 return FALSE;
		
	 for(int n = 1; n < arOld.GetSize(); n++)
	 {
		if(arNew.GetSize() <= nIdx+n || arNew[nIdx+n] != arOld[n])
			return FALSE;
	 }

	 if(nIdx == 0)
		 return TRUE;

	 if(nIdx == (arNew.GetSize() - arOld.GetSize()))
		return TRUE;

	return FALSE;
}

void CAccount::UpdateMailCache(CMail* pMail)
{
	if(pMail == NULL)
		return;

	int nPos = m_MailCache.FindSignature(pMail->GetSignature());

	if(nPos == -1)
	{
		CacheItem Item(pMail->GetSignature(), pMail->IsRead());
		m_MailCache.Add(Item);
	}
    else
    {
        m_MailCache.SetReadAt(nPos, pMail->IsRead());
    }

    if(m_pDoc)
		m_pDoc->OnReadStateChanged();
}

int CAccount::CountUnreadMails() const
{
	int nUnreadMails = 0;
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		if(!pMail->IsRead())
			nUnreadMails++;
	}
	return nUnreadMails;
}

int CAccount::GetSizeOfAllMails() const
{
	int nSize = 0;
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		nSize += pMail->GetSize();
	}
	return nSize;
}


CMail* CAccount::GetMailFromSignature(const CString& stSignature) const
{
 
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		if(pMail->GetSignature() == stSignature)
			return pMail;

	}		
	return NULL;
 }


CString CAccount::GetErrorDescription() const
{
	CString stError;

	switch(GetLastError())
	{
	case CAccount::ERRORS::erNotConnected:
		stError = i18n("Not connected");
		break;

	case CAccount::ERRORS::erConnectFailed:
		stError = i18n("Connection failure"); 
		break;

	case CAccount::ERRORS::erCanceled:
		stError = i18n("Canceled");
		break;

	case CAccount::ERRORS::erResponseInvalid:
		stError = i18n("Invalid server response");
		break;
	
	case CAccount::ERRORS::erTimedOut:
		stError = i18n("Timeout");
		break;

	case CAccount::ERRORS::erAuthFailed:
		stError = i18n("Login failed"); 
		break;

	case CAccount::ERRORS::erMailMissing:
		stError = i18n("Message not found"); 
		break;

	case CAccount::ERRORS::erMailLocFailure:
		stError = i18n("Server failure: Message not found!");
		break;

	case CAccount::ERRORS::erBusy:
		stError = i18n("Busy");
		break;

	}
	return stError;
}



CString CAccount::GetStateDescription() const
{
	CString stState;

	switch(GetState())
	{
	case CAccount::STATES::stIdle:
		stState = i18n("Ready");
		break;

	case CAccount::STATES::stConnecting:
		stState = i18n("Connecting...");
		break;

	case CAccount::STATES::stConnected:
		stState = i18n("Connected!");
		break;

	case CAccount::STATES::stLogin:
		stState = i18n("Login...");
		break;

	case CAccount::STATES::stLoggedIn:
		stState = i18n("Successfully logged in!");
		break;

	case CAccount::STATES::stLoadingMail:
		stState = i18n("Loading message ...");
		break;

	case CAccount::STATES::stLoadingMails:
		stState = StrFormat(i18n("Loading message {1} of {2} ..."), _T("d"), GetCurrentMail(), _T("d"), MailsCount());
		break;

	case CAccount::STATES::stDeletingMails:
		stState = StrFormat(i18n("Deleting message {1} of {2} ..."), _T("d"), GetCurrentMail(), _T("d"), MailsCount());
		break;

	case CAccount::STATES::stMarkAsSeen:
		stState = StrFormat(i18n("Marking message {1} of {2} as seen..."), _T("d"), GetCurrentMail(), _T("d"), MailsCount());
		break;
	}

	return stState;	
}

void CAccount::UpdateMailCache()
{
    for(int i = 0; i < m_MailCache.GetSize(); i++)
	{
		const CString& stSig = m_MailCache[i].m_stSignature;
		BOOL bFound = FALSE;

		POSITION pos = m_Mails.GetHeadPosition();
		while(pos != NULL)
		{
			const CMail* pMail = m_Mails.GetNext(pos);
			if(pMail->GetSignature() == stSig)
			{
				bFound = TRUE;
				break;
			}
		}
	
		if(!bFound)
			m_MailCache.RemoveAt(i);	
	}
}


BOOL CAccount::GetMailUIDL(int nMailNo, CString& stUIDL)
{
	if(nMailNo < 1 || nMailNo > m_UIDLCache.GetSize())
		return FALSE;

	stUIDL = m_UIDLCache[nMailNo-1];
	return !stUIDL.IsEmpty();
}


BOOL CAccount::ReadSettings(const CSettingsKey& settings)
{	
	if(!settings.QueryValue(szAccNameValue, m_stName))
		return FALSE;
	
	if(!settings.QueryValue(szAccUserValue, m_stUser))
		return FALSE;

	if(!settings.QueryValue(szAccServerValue, m_stServer))
		return FALSE;
	
	if(!settings.QueryValue(szAccPortValue, m_nPort))
		return FALSE;

	CString stPass;
	if(!settings.QueryValue(szAccPassValue, stPass))
		return FALSE;
	m_stPass = Crypt(stPass, FALSE);

	settings.QueryValue(szAccActiveValue, m_bActive);

	settings.QueryValue(szAccUseCustomColorValue, m_bCustomColor);
	settings.QueryValue(szAccCustomColorValue, m_CustomColor);

	UINT nTimeOutVal = 30;
	if(settings.QueryValue(szAccTimeOutValValue, nTimeOutVal))
		SetTimeOut(nTimeOutVal);
		
	//settings.QueryValue(szAccLimitLoadSizeValue, m_bLimitLoadSize);

    UINT retrieveMode;
    if(settings.QueryValue(szAccRetrieveModeValue, retrieveMode))
        m_RetrieveMode = (RETRIEVE_MODE)retrieveMode;
    settings.QueryValue(szAccLinesValue, m_nLines);
	settings.QueryValue(szAccMaxSizeOfSmallMailsValue, m_nMaxSizeOfSmallMails);
	settings.QueryValue(szAccLinesBigMailsValue, m_nLinesBigMails);

	if(!settings.QueryValue(szAccTrayColorValue, m_TrayColor))
		m_TrayColor = m_CustomColor;

	settings.QueryValue(szAccUseCustomIntervalValue, m_bCustomAutoCheckInterval);
	settings.QueryValue(szAccCustomIntervalValue, m_nCustomAutoCheckInterval);

	settings.QueryValue(szAccUseUIDLForDeleteValue, m_bUseUIDLForDelete);

    settings.QueryValue(szAccTopMessagesValue,      m_bTopMessages);
    settings.QueryValue(szAccTopMessagesCountValue, m_nTopMessagesCount); 

	return TRUE;
}


void CAccount::SaveSettings(CSettingsKey& settings) const
{
	settings.SetValue(szAccNameValue,   m_stName);
	settings.SetValue(szAccServerValue, m_stServer);
	settings.SetValue(szAccPortValue,   m_nPort);
	settings.SetValue(szAccUserValue,   m_stUser);
	settings.SetValue(szAccPassValue,   Crypt(m_stPass));

	settings.SetValue(szAccActiveValue, m_bActive);
	settings.SetValue(szAccUseCustomColorValue, m_bCustomColor);
	settings.SetValue(szAccCustomColorValue, m_CustomColor);
	settings.SetValue(szAccTimeOutValValue, GetTimeOut());
	
	//settings.SetValue(szAccLimitLoadSizeValue, m_bLimitLoadSize);

    settings.SetValue(szAccRetrieveModeValue, m_RetrieveMode);
    settings.SetValue(szAccLinesValue, m_nLines);
	settings.SetValue(szAccMaxSizeOfSmallMailsValue, m_nMaxSizeOfSmallMails);
	settings.SetValue(szAccLinesBigMailsValue, m_nLinesBigMails);  

	settings.SetValue(szAccUseUIDLForDeleteValue, m_bUseUIDLForDelete);
	
	settings.SetValue(szAccTrayColorValue, m_TrayColor);
	settings.SetValue(szAccUseCustomIntervalValue, m_bCustomAutoCheckInterval);
	settings.SetValue(szAccCustomIntervalValue, m_nCustomAutoCheckInterval);

    settings.SetValue(szAccTopMessagesValue,      m_bTopMessages);
    settings.SetValue(szAccTopMessagesCountValue, m_nTopMessagesCount);
}

void CAccount::SyncMailListWithUIDLs()
{
	// Synchronisize mail list with UIDL list:
	// Necessary for servers, that insert new mail in front of the list!
	CMails mList;

	for(int n = 0; n < m_UIDLCache.GetSize(); n++)
	{
		const CString& stUIDL = m_UIDLCache[n];

		POSITION pos = m_Mails.GetHeadPosition();
		while(pos != NULL)
		{
			CMail* pMail = m_Mails.GetNext(pos);
			
			if(stUIDL == pMail->GetUIDL()) 
			{
				if(mList.Find(pMail) == NULL)
					mList.AddTail(pMail);
				else
					continue;
				break;
			}
		}
	}

	m_Mails.RemoveAll();

	POSITION pos = mList.GetHeadPosition();
	while(pos != NULL)
		m_Mails.AddTail(mList.GetNext(pos));
}

void CAccount::UpdateMailAccReferenz()
{
	POSITION pos = m_Mails.GetHeadPosition();
	while(pos != NULL)
	{
		CMail* pMail = m_Mails.GetNext(pos);
		pMail->m_pAccount = this;
	}
}
